package ru.yandex.travel.hotels.common.partners.travelline.utils;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.base.Preconditions;

import ru.yandex.travel.hotels.common.partners.travelline.model.ContactInfo;
import ru.yandex.travel.hotels.common.partners.travelline.model.Customer;
import ru.yandex.travel.hotels.common.partners.travelline.model.Email;
import ru.yandex.travel.hotels.common.partners.travelline.model.Guest;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelRef;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelReservation;
import ru.yandex.travel.hotels.common.partners.travelline.model.Phone;
import ru.yandex.travel.hotels.common.partners.travelline.model.PlacementRef;
import ru.yandex.travel.hotels.common.partners.travelline.model.ReservationRoomStay;
import ru.yandex.travel.hotels.common.partners.travelline.model.ReservationRoomType;
import ru.yandex.travel.hotels.common.partners.travelline.model.ReservationServiceReference;
import ru.yandex.travel.hotels.common.partners.travelline.model.StayDates;
import ru.yandex.travel.hotels.common.partners.travelline.model.dto.OfferDto;
import ru.yandex.travel.hotels.common.partners.travelline.model.dto.ServiceDto;
import ru.yandex.travel.hotels.common.partners.travelline.placements.PlacementSet;

public class RequestHelper {

    private static List<PlacementRef> getPlacementRefs(PlacementSet placementSet) {
        return placementSet.getPlacements().stream().map(
                placement -> PlacementRef.builder()
                        .code(placement.getCode())
                        .kind(placement.getKind())
                        .index(placement.getIndex())
                        .build())
                .collect(Collectors.toList());
    }

    private static List<ReservationServiceReference> getServiceRefs(OfferDto dto, List<String> selectedServiceCodes) {
        return selectedServiceCodes.stream().map(code -> {
            ServiceDto serviceDto = dto.getServices().get(code);
            Preconditions.checkArgument(serviceDto != null, "Unknown ServiceCode " + code);
            return ReservationServiceReference.builder()
                    .code(code)
                    // .quantity(1) TODO(tivelkov): add appropriate quantity here
                    .build();
        }).collect(Collectors.toList());
    }


    public static HotelReservation createHotelReservation(
            @Nonnull OfferDto offerDto,
            int placementIndex,
            @Nonnull List<String> selectedServiceCodes,
            @Nullable String yandexId,
            @Nullable List<ru.yandex.travel.hotels.common.orders.Guest> guests,
            @Nullable String customerPhone,
            @Nullable String customerEmail) {
        Preconditions.checkArgument(placementIndex < offerDto.getPossiblePlacements().size(),
                "placementIndex is out of range");
        PlacementSet placementSet = offerDto.getPossiblePlacements().get(placementIndex);


        ReservationRoomStay.ReservationRoomStayBuilder roomStayBuilder = ReservationRoomStay.builder()
                .stayDates(StayDates.builder()
                        .startDate(offerDto.getStayDates().getStartDate())
                        .endDate(offerDto.getStayDates().getEndDate())
                        .build())
                .roomType(ReservationRoomType.builder()
                        .code(offerDto.getRoomType().getCode())
                        .placements(getPlacementRefs(placementSet))
                        .build())
                .ratePlans(List.of(ReservationRoomStay.RatePlan.builder()
                        .code(offerDto.getRatePlan().getCode())
                        .build()))
                .guestCountInfo(placementSet.getGuestCountInfo())
                .services(getServiceRefs(offerDto, selectedServiceCodes));
        if (guests != null) {
            Preconditions.checkArgument(guests.size() == placementSet.getGuestPlacementIndexes().size(),
                    "Unexpected size of placement indexes list");
            roomStayBuilder.guests(
                    IntStream.range(0, guests.size()).mapToObj(i ->
                            Guest.builder()
                                    .firstName(guests.get(i).getFirstName())
                                    .lastName(guests.get(i).getLastName())
                                    .placement(
                                            Guest.Placement.builder()
                                                    .index(placementSet.getGuestPlacementIndexes().get(i))
                                                    .build())
                                    .build()
                    ).collect(Collectors.toList())
            );
        }
        HotelReservation.HotelReservationBuilder hotelReservationBuilder = HotelReservation.builder()
                .hotelRef(HotelRef.builder().code(offerDto.getHotel().getCode()).build())
                .roomStay(roomStayBuilder.build());
        if (yandexId != null) {
            hotelReservationBuilder.yandexNumber(yandexId);
        }
        if (customerPhone != null || customerEmail != null) {
            hotelReservationBuilder.customer(
                    Customer.builder()
                            .contactInfo(ContactInfo.builder()
                                    .phone(Phone.builder().phoneNumber(customerPhone).build())
                                    .email(Email.builder().emailAddress(customerEmail).build())
                                    .build())
                            .build()
            );
        }
        return hotelReservationBuilder.build();
    }

}
