package ru.yandex.travel.hotels.common.partners.travelline.utils;

import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.travel.hotels.common.partners.travelline.model.HotelOfferAvailability;
import ru.yandex.travel.hotels.common.partners.travelline.model.Placement;
import ru.yandex.travel.hotels.common.partners.travelline.model.ReservationServiceReference;
import ru.yandex.travel.hotels.common.partners.travelline.model.RoomStay;
import ru.yandex.travel.hotels.common.partners.travelline.model.Service;
import ru.yandex.travel.hotels.common.token.Occupancy;

public class ServicesHelper {

    //TODO consider children
    public static List<ReservationServiceReference> generateServiceRefs(List<HotelOfferAvailability.ServiceConditions> roomServices,
                                                                        String checkInDate,
                                                                        String checkOutDate,
                                                                        Map<String, Service> hotelServiceMap,
                                                                        Occupancy occupancy) {
        final LocalDate startDate = LocalDate.parse(checkInDate);
        final LocalDate endDate = LocalDate.parse(checkOutDate);
        final int numberOfDays = (int) ChronoUnit.DAYS.between(startDate, endDate);


        return roomServices.stream()
                .filter(service -> !service.isInclusive())  //HOTELS-4570 - we don't add included services
                .map(roomService -> {
                    var builder = ReservationServiceReference.builder();
                    builder.code(roomService.getCode());

                    switch (hotelServiceMap.get(roomService.getCode()).getChargeType()) {
                        case PER_PERSON:
                            builder.quantity(occupancy.getAdults());
                            break;
                        case PER_ROOM_PER_NIGHT:
                            builder.quantity(numberOfDays);
                            break;
                        case PER_OCCUPANCY_PER_NIGHT:
                        case PER_PERSON_PER_NIGHT:
                            builder.quantity(occupancy.getAdults() * numberOfDays);
                            break;
                        case PER_UNIT:
                        case PER_RESERVATION:
                        case PER_STAY:
                        default:
                            throw new RuntimeException("Unexpected charge type of meal service:");
                    }
                    return builder.build();
                }).collect(Collectors.toList());
    }

    public static double calculateOfferPrice(RoomStay.RoomTypeRef roomType,
                                             List<String> guestPlacementCodes,
                                             List<ReservationServiceReference> serviceRefs,
                                             List<HotelOfferAvailability.ServiceConditions> roomServices) {
        Set<String> placementCodes = new HashSet<>(guestPlacementCodes);
        double placementsAmount = roomType.getPlacements()
                .stream()
                .filter(placement -> placementCodes.contains(placement.getCode()))
                .map(Placement::getPriceBeforeTax)
                .reduce(0.0, Double::sum);
        double servicesAmount = serviceRefs.stream()
                .flatMap(serviceRef -> roomServices.stream() //there can be problems here with performance
                        .filter(roomService -> roomService.getCode().equals(serviceRef.getCode()) && !roomService.isInclusive())
                        .findFirst()
                        .stream()
                        .map(roomService -> roomService.getPrice().getPriceAfterTax() * serviceRef.getQuantity()))
                .reduce(0.0, Double::sum);
        return placementsAmount + servicesAmount;
    }

}
