package ru.yandex.travel.hotels.common.partners.tvil;

import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.google.common.base.Preconditions;
import lombok.Getter;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.hotels.common.partners.base.BaseClient;
import ru.yandex.travel.hotels.common.partners.base.ClientMethods;
import ru.yandex.travel.hotels.common.partners.tvil.model.TvilSearchRequest;
import ru.yandex.travel.hotels.common.partners.tvil.model.TvilSearchResponse;

public class DefaultTvilClient extends BaseClient<TvilClientProperties> implements TvilClient {
    private static final String AVAILABILITY = "Availability";
    @Getter
    private static final ClientMethods methods;

    static {
        methods = new ClientMethods()
                .register(AVAILABILITY, "/availability", HttpMethod.POST, TvilSearchResponse.class);
    }

    public DefaultTvilClient(TvilClientProperties properties, AsyncHttpClientWrapper ahcWrapper, Retry retry) {
        super(properties, ahcWrapper, createObjectMapper(), retry);
        Preconditions.checkArgument(!properties.getBaseUrl().endsWith("/"),
                "The base url shouldn't end with a '/' symbol; url %s", properties.getBaseUrl());
    }

    public static ObjectMapper createObjectMapper() {
        return new ObjectMapper()
                .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE)
                .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    }

    @Override
    protected ClientMethods getClientMethods() {
        return methods;
    }

    @Override
    public CompletableFuture<TvilSearchResponse> searchOffers(TvilSearchRequest request) {
        String commaSeparatedHotelIds = request.getHotelIds().stream()
                .map(String::valueOf).sorted().collect(Collectors.joining(","));
        String guests = request.getGuests().stream()
                .map(TvilClientHelper::convertGuestsParam)
                .collect(Collectors.joining(";"));
        return call(AVAILABILITY, new CallArguments()
                .withQueryParam("hotels", commaSeparatedHotelIds)
                .withQueryParam("check_in", request.getCheckIn().toString())
                .withQueryParam("check_out", request.getCheckOut().toString())
                .withQueryParam("guests", guests)
                .withQueryParam("language", request.getLanguage())
                .withQueryParam("currency", request.getCurrency())
                .withQueryParam("user_country", request.getUserCountry())
                .withRequestId(request.getRequestId())
        );
    }

    @Override
    protected <R> R bodyAs(String body, Class<R> type) {
        if ("[]".equals(body)) {
            Preconditions.checkArgument(type.equals(TvilSearchResponse.class),
                    "Only %s response type is expected but got %s", TvilSearchResponse.class, type);
            return type.cast(new TvilSearchResponse());
        }
        return super.bodyAs(body, type);
    }
}
