package ru.yandex.travel.hotels.common.partners.vipservice

import com.fasterxml.jackson.databind.DeserializationFeature
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.databind.PropertyNamingStrategy
import com.fasterxml.jackson.databind.module.SimpleModule
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer
import com.fasterxml.jackson.module.kotlin.KotlinModule
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper
import ru.yandex.travel.commons.proto.ECurrency
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit
import ru.yandex.travel.commons.retry.Retry
import ru.yandex.travel.hotels.common.partners.base.BaseClient
import ru.yandex.travel.hotels.common.partners.base.ClientMethods
import ru.yandex.travel.hotels.common.partners.vipservice.model.CurrencyDeserializer
import ru.yandex.travel.hotels.common.partners.vipservice.model.Offer
import ru.yandex.travel.hotels.common.partners.vipservice.model.SearchQuery
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.format.DateTimeFormatter
import java.util.concurrent.CompletableFuture

class DefaultVipserviceClient(
    properties: VipserviceClientProperties,
    ahcWrapper: AsyncHttpClientWrapper,
    retry: Retry,
) :
    BaseClient<VipserviceClientProperties>(properties, ahcWrapper, mapper, retry), VipserviceClient {

    override fun getCommonHeaders(destinationMethod: String) =
        mutableMapOf(API_KEY_HEADER to properties.apiKey)

    override fun getClientMethods() =
        methods

    override fun findOffers(
        hotelIds: List<Int>,
        checkIn: LocalDate,
        checkOut: LocalDate,
        adults: Int,
        children: List<Int>,
        currency: ECurrency,
    ): CompletableFuture<Array<Offer>> =
        call<Array<Offer>, Any>(
            SEARCH, CallArguments().withBody(
                SearchQuery(
                    hotelIds = hotelIds,
                    checkIn = checkIn,
                    checkOut = checkOut,
                    adults = adults,
                    children = children,
                    currency = ProtoCurrencyUnit.fromProtoCurrencyUnit(currency),
                )
            )
        )

    companion object {
        private const val API_KEY_HEADER = "X-HS-Token"
        private const val SEARCH = "Search"
        private const val DATE_TIME_PATTERN = "yyyy-MM-dd HH:mm:ss"

        val mapper: ObjectMapper = ObjectMapper()
            .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE)
            .configure(DeserializationFeature.ACCEPT_EMPTY_STRING_AS_NULL_OBJECT, true)
            .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
            .registerModules(
                KotlinModule(),
                JavaTimeModule()
                    .addDeserializer(
                        LocalDate::class.java,
                        LocalDateDeserializer(DateTimeFormatter.ISO_LOCAL_DATE)
                    )
                    .addSerializer(
                        LocalDate::class.java,
                        LocalDateSerializer(DateTimeFormatter.ISO_LOCAL_DATE)
                    )
                    .addDeserializer(
                        LocalDateTime::class.java,
                        LocalDateTimeDeserializer(DateTimeFormatter.ofPattern(DATE_TIME_PATTERN))
                    )
                    .addSerializer(
                        LocalDateTime::class.java,
                        LocalDateTimeSerializer(DateTimeFormatter.ofPattern(DATE_TIME_PATTERN))
                    ),
                SimpleModule()
                    .addDeserializer(
                        ProtoCurrencyUnit::class.java,
                        CurrencyDeserializer()
                    )
            )

        @JvmStatic
        val methods: ClientMethods = ClientMethods()
            .register(SEARCH, "/meta/search", HttpMethod.POST, Array<Offer>::class.java)
    }
}
