# coding=utf-8
import argparse
import datetime
import logging
import os
import pwd
import re
import socket
import sys
from collections import OrderedDict


import six
from travel.hotels.lib.python import ytlib
import travel.library.python.tools as tools

LOG = logging.getLogger(__name__)


class Session(object):
    def __init__(self):
        self._annotations = OrderedDict()
        self._timestamp = datetime.datetime.utcnow()

    def annotate(self, key, value):
        self._annotations[str(key)] = str(value)

    def to_dict(self):
        result = dict(timelabel=self.timelabel)
        result.update(self._annotations)
        return result

    @property
    def timestamp(self):
        return self._timestamp

    @property
    def timelabel(self):
        label = self._timestamp.isoformat()
        label = label.split(".")[0] + "Z"
        return label


class ProcessEnvironment(object):
    def __init__(self, inheritance_chain, user, host):
        self.inheritance_chain = inheritance_chain
        self.user = user
        self.host = host
        self.args = None

    @property
    def caller_cls(self):
        return self.inheritance_chain[0]


class VersionedProcess(object):
    name = 'versioned_process'
    category_name = 'general'

    def __init__(self, session, args):
        """
        Derived classes may override this method, but must call super().
        This is enforced with the appropriate check.
        """
        self.session = session
        assert ytlib.yt.exists(args.base_dir.rstrip('/')), "Unable to launch versioned process - base_dir YT path \"{base_dir}\" doesn't exist".format(base_dir=args.base_dir)
        self.base_dir = args.base_dir
        self.keep_last_dirs_count = args.keep_last_dirs_count
        self.keep_weekly_milestones = not args.do_not_keep_weekly_milestones
        self.debug = args.debug
        self.__inited = True
        self._run_dir = None

        LOG.info("The run directory is %s", self.get_run_dir())
        LOG.info("See at %s", ytlib.get_url(self.get_run_dir()))
        ytlib.ensure_dir_exists(self.get_run_dir())
        ytlib.yt.set(self.get_run_dir() + "/@annotations", self.session.to_dict())
        ytlib.yt.set(self.get_run_dir() + "/@nightly_compression_settings", {
            'enabled': True,
            'compression_codec': 'zstd_9',
            'erasure_codec': 'lrc_12_2_2',
            'min_table_age': 2 * 24 * 60 * 60,
            'pool': ytlib.get_user_name()
        })

    def get_process_dir(self):
        return ytlib.join(self.base_dir, self.name)

    def get_run_dir(self):  # Directory of current run
        if self._run_dir is None:
            short_path = ytlib.join(self.get_process_dir(), self.session.timelabel[:10])
            if not ytlib.yt.exists(short_path):
                self._run_dir = short_path
            else:
                self._run_dir = ytlib.join(self.get_process_dir(), self.session.timelabel)
        return self._run_dir

    def get_latest_path(self):
        return ytlib.join(self.get_process_dir(), "latest")

    def get_table_path(self, name):
        return ytlib.join(self.get_run_dir(), name)

    def update_latest(self):
        ytlib.link(self.get_run_dir(), self.get_latest_path())
        LOG.info("Results were published at %s", ytlib.get_url(self.get_latest_path()))

    def cleanup(self):
        ytlib.cleanup(
            self.get_process_dir(),
            keep_last_nodes_count=self.keep_last_dirs_count,
            keep_weekly_milestones=self.keep_weekly_milestones
        )
        LOG.info("Directory %s was cleaned up", self.get_process_dir())

    def run(self):
        pass

    @staticmethod
    def configure_arg_parser(parser, proc_env):
        yt_group = parser.add_argument_group(VersionedProcess.name + " - YT")
        yt_group.add_argument("--base-dir", default="//home/travel/{}/{}"
                              .format(proc_env.user, proc_env.caller_cls.category_name))
        yt_group.add_argument("--yt-proxy", default="hahn")
        yt_group.add_argument("--yt-token")
        yt_group.add_argument("--keep-last-dirs-count", default=7, type=int)
        yt_group.add_argument("--do-not-keep-weekly-milestones", action="store_true", default=False)

        others_group = parser.add_argument_group(VersionedProcess.name + " - others")
        others_group.add_argument("--verbose", action="store_true", default=False,
                                  help="enable debug logging")
        others_group.add_argument("--annotate", action="append", metavar="K=V", type=str, default=[],
                                  help="extra annotations for the session")
        others_group.add_argument("--debug", action="store_true", default=False,
                                  help="enable debug mode")

    @staticmethod
    def configure_session(session, proc_env):
        for annotation in proc_env.args.annotate:
            key, value = annotation.split("=", 1)
            session.annotate(key, value)

    @classmethod
    def get_inheritance_chain(cls):
        inheritance_chain = [cls]
        while cls != VersionedProcess:
            assert len(cls.__bases__) == 1, "Multiple inheritance is not allowed"
            cls = cls.__bases__[0]
            inheritance_chain.append(cls)
        return inheritance_chain

    @classmethod
    def main(cls, args=None):
        inheritance_chain = cls.get_inheritance_chain()
        for c in inheritance_chain:
            assert c.name and re.match(r"^[a-z0-9_]+$", c.name), "Versioned process name must be a simple lowercase identifier"
        user = pwd.getpwuid(os.getuid()).pw_name
        host = socket.getfqdn()
        proc_env = ProcessEnvironment(inheritance_chain, user, host)

        parser = argparse.ArgumentParser(description="Versioned process '{}' ({})"
                                         .format(cls.name, '->'.join(c.name for c in inheritance_chain)),
                                         formatter_class=argparse.ArgumentDefaultsHelpFormatter)
        for c, prev_c in zip(inheritance_chain, inheritance_chain[1:] + [None]):
            if prev_c is None or prev_c.configure_arg_parser != c.configure_arg_parser:
                c.configure_arg_parser(parser, proc_env)
        args = proc_env.args = parser.parse_args(args=tools.replace_args_from_env(args))

        logging.basicConfig(level=(logging.DEBUG if args.verbose else logging.INFO),
                            format="%(asctime)-15s | %(levelname)s | %(message)s",
                            stream=sys.stdout)

        session = Session()
        session.annotate("sys_user", user)
        session.annotate("sys_host", host)
        for c, prev_c in zip(inheritance_chain, inheritance_chain[1:] + [None]):
            if prev_c is None or prev_c.configure_session != c.configure_session:
                c.configure_session(session, proc_env)

        LOG.info("Starting versioned process '%s'", cls.name)
        for key, value in six.iteritems(session.to_dict()):
            LOG.info("    %s=%s", key, value)

        ytlib.configure(args.yt_proxy, args.yt_token)

        instance = cls(session, args)
        assert instance.__inited, "Constructor of VersionedProcess was not called (did you forget a super() call?)"
        instance.run()
        if not instance.debug:
            instance.update_latest()
            instance.cleanup()
        else:
            LOG.info("Results were published at %s", ytlib.get_url(instance.get_run_dir()))
