import logging
import platform
import sys
from collections import Counter, namedtuple

import dateutil.parser
import six.moves
import yt.wrapper as yt
from yt.wrapper.http_helpers import get_user_name

LOG = logging.getLogger(__name__)

URL_FMT = 'https://yt.yandex-team.ru/{cluster}/?page=navigation&path={path}'
ARGS_TO_HIDE = {"password", "token", "secret", "auth"}


def configure(proxy, token):
    if proxy is not None:
        if not proxy.endswith(".yt.yandex.net"):
            proxy = proxy + ".yt.yandex.net"
        yt.config["proxy"]["url"] = proxy
    if token is not None:
        yt.config["token"] = token

    # OS X workaround
    if platform.system() == "Darwin":
        yt.config["pickling"]["module_filter"] = lambda module: hasattr(module, "__file__") and \
            not module.__file__.endswith(".so") and module.__name__ != "lxml"
        yt.config["pickling"]["force_using_py_instead_of_pyc"] = True
        yt.config["pickling"]["enable_modules_compatibility_filter"] = True


def ensure_dir_exists(dir):
    if not yt.exists(dir):
        yt.create("map_node", dir, recursive=True)


def join(*args):
    return yt.ypath_join(*args)


def get_cluster():
    return yt.config["proxy"]["url"].split(".")[0]


def get_url(path):
    return URL_FMT.format(cluster=get_cluster(), path=path)


# workaround to stop import optimizer from removing
get_user_name = get_user_name


def add_hidden_arg(arg):
    ARGS_TO_HIDE.add(arg)


def hide_sys_args():
    class ArgsHider(object):
        def __init__(self):
            self.old = None

        def __enter__(self):
            self.old = sys.argv
            new = list(self.old)

            for i in six.moves.xrange(1, len(new) - 1):
                arg = new[i]
                if arg == "*****":
                    continue
                for v in ARGS_TO_HIDE:
                    if v in arg.lower():
                        new[i + 1] = "*****"
                        break
            sys.argv = new

        def __exit__(self, exc_type, exc_val, exc_tb):
            sys.argv = self.old
            return False

    return ArgsHider()


def link(target_path, link_path):
    yt.link(target_path, link_path, force=True)


def diff_tables(left_table, right_table):
    ignore_fields = set(["table", "actualizationTime"])

    def compare(left, right, diff_counter, fields_counter):
        left_fields = set(left.keys())
        right_fields = set(right.keys())
        fields = (left_fields | right_fields) - ignore_fields
        fields_changed = False
        for field in fields:
            if field.startswith("_"):
                continue
            left_value = left.get(field, None)
            right_value = right.get(field, None)
            changes = 0
            if left_value != right_value:
                fields_changed = True
                changes = 1
            fields_counter[field] += changes
        if fields_changed:
            diff_counter["changed"] += 1

    @yt.with_context
    def mark_rows(row, context):
        row["table"] = context.table_index
        yield row

    @yt.reduce_aggregator
    def calculate_diff(row_groups):
        diff_counter = Counter(total=0, added=0, removed=0, changed=0)
        fields_counter = Counter()
        for _, rows in row_groups:
            diff_counter["total"] += 1
            rows = list(rows)
            count = len(rows)
            if count == 2:
                left, right = rows
                compare(left, right, diff_counter, fields_counter)
            else:
                if rows[0]["table"] == 0:
                    diff_counter["removed"] += 1
                else:
                    diff_counter["added"] += 1
        diff_counter["total"] -= diff_counter["removed"]
        yield {"counts": dict(diff_counter), "fields": dict(fields_counter)}

    with yt.TempTable() as diff_table:
        LOG.info("Computing difference: [%s, %s] -> [%s]", left_table, right_table, diff_table)
        with hide_sys_args():
            yt.run_map_reduce(mark_rows, calculate_diff, [left_table, right_table], diff_table, reduce_by="originalId")
        diff_counter = Counter(total=0, added=0, removed=0, changed=0)
        fields_counter = Counter()
        [(diff_counter.update(itm["counts"]), fields_counter.update(itm["fields"])) for itm in yt.read_table(diff_table)]
        LOG.info("Diff: %r", diff_counter)
        LOG.info("Fields: %r", fields_counter)
        return dict(diff_counter), dict(fields_counter)


def count_events(log_table):
    @yt.aggregator
    def calculate_counts(rows):
        c = Counter(error=0, warning=0)
        for row in rows:
            c[row["level"]] += 1
        yield dict(c)

    with yt.TempTable() as counts_table:
        LOG.info("Counting events: [%s] -> [%s]", log_table, counts_table)
        with hide_sys_args():
            yt.run_map(calculate_counts, log_table, counts_table)
        c = Counter(error=0, warning=0)
        [c.update(itm) for itm in yt.read_table(counts_table)]
        LOG.info("Result: %r", c)
        return dict(c)


Node = namedtuple('Node', ['path', 'type', 'ts'])


def cleanup(path, keep_last_nodes_count, keep_weekly_milestones, node_types=('map_node',)):
    '''
    Cleans up subdirectories below `path`, according to rules (see st/HOTELS-2934):
        1. Cleanup only directories (not links)
        2. Keep `keep_last_dirs_count` directories
        3. If `keep_weekly_milestones`, keep the first directory of each iso week
    '''
    dirs = yt.list(path, attributes=['type', 'creation_time'], absolute=True)
    dirs_with_ts = []
    node_types = set(node_types)
    for d in dirs:
        node_type = d.attributes['type']
        if node_type in node_types:
            dirs_with_ts.append(Node(str(d), node_type, dateutil.parser.parse(d.attributes['creation_time'])))
    dirs_with_ts = sorted(dirs_with_ts, key=lambda x: x.ts)
    dirs_with_ts = dirs_with_ts[:-keep_last_nodes_count]
    prev_week_nr = None
    with yt.Transaction():
        for d in dirs_with_ts:
            c = d.ts.isocalendar()
            week_nr = (c[0], c[1])
            if not keep_weekly_milestones or (prev_week_nr is not None and prev_week_nr == week_nr):
                LOG.info("Remove %s %s", d.type, d.path)
                yt.remove(d[0], recursive=True)
            prev_week_nr = week_nr
