from typing import Any, Dict

from jinja2 import Template
from library.python import resource

from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder
from yql.client.results import CantGetResultsException
from yt.wrapper import YPath


def create_client(*args, **kwargs):  # Only to avoid PEERDIR and import to YQL. Just PEERDIR and import this lib
    return YqlClient(*args, **kwargs)


def build_parameter_value(value):
    """
    :param value:
    :type value: str, dict, list, float, int
    :return: yql-client compatible parameter value.
    """
    if value is None:
        return ValueBuilder.make_null()
    if hasattr(value, 'to_json'):  # value already built
        return value
    if isinstance(value, str):
        return ValueBuilder.make_string(value)
    if isinstance(value, YPath):
        return ValueBuilder.make_string(str(value))
    if isinstance(value, bool):
        return ValueBuilder.make_bool(value)
    if isinstance(value, float):
        return ValueBuilder.make_float(value)
    if isinstance(value, int):
        return ValueBuilder.make_int64(value)
    if isinstance(value, list):
        return ValueBuilder.make_list([build_parameter_value(v) for v in value])
    if isinstance(value, dict):
        return ValueBuilder.make_dict([(build_parameter_value(k), build_parameter_value(v)) for k, v in value.items()])
    if isinstance(value, tuple) and hasattr(value, '_asdict'):  # namedtuple special processing
        params = {k: build_parameter_value(v) for k, v in value._asdict().items()}
        return ValueBuilder.make_struct(params)
    raise ValueError("Value type {} needs manual build. value type. Look at /arc/trunk/arcadia/yql/library/python/yql/api/v1/examples/parameters.py for examples".format(type(value)))


def build_yql_parameters(parameters, build_values=False):
    """
    :param parameters: parameters for yql query. key should include '$' symbol.
    :type parameters: dict
    :param build_values: Whether to try to use corresponding ValueBuilder.make_* method for unbuilt parameter values or not.
    :type build_values: bool
    :return: prepares parameters for yql client query
    """
    try:
        return ValueBuilder.build_json_map(parameters)
    except AttributeError:
        # need to build individual values
        if build_values:
            return ValueBuilder.build_json_map({k: build_parameter_value(v) for k, v in parameters.items()})
        else:
            raise AttributeError("""Need to build individual values first.\n Use build_values=True for basic cast that works for"""
                                 """string, list, dict and double. \nOr look at /arc/trunk/arcadia/yql/library/python/yql/api/v1/examples/parameters.py"""
                                 """ for examples""")


def display_query_with_parameters(query, parameters=None, cluster=None, run_dir=None):
    # todo: metod that validates request and shows compilation errors.
    if parameters is None:
        parameters = dict()
    print("-" * 60)
    if cluster is not None:
        print("USE {};".format(cluster))
    if run_dir is not None:
        print("PRAGMA TablePathPrefix = \"{}\";\n".format(run_dir))
    for k, v in parameters.items():
        try:
            val = v.to_json()
        except AttributeError:
            val = v
        print('{} = "{}";'.format(k, val))
    for line in query.split('\n'):
        if not line.lower().startswith('declare'):
            print(line)
    print("-" * 60)


def run_query(client, query, parameters=None, run_dir=None, debug=False, syntax_version=1, title=None, attaches=None,
              transaction_id=None):
    """
    Runs query with specified client and parameters.
    In debug mode - only prints query out with display_query_with_parameters method.
    :param query: query to run
    :param parameters: query parameters
    :param client: yql client object, default YqlClient if None.
    :type client: YqlClient
    :param
    :param debug: debug mode. Print out queries instead of launching them.
    :type debug: bool
    :return: request object
    """
    if parameters is None:
        parameters = dict()
    if run_dir is not None:
        query = "PRAGMA TablePathPrefix = \"{}\";\n".format(run_dir) + query
    request = client.query(query, syntax_version=syntax_version, title=title)
    if attaches:
        for name, data in attaches.items():
            request.attach_file_data(data, name)
    if debug:
        display_query_with_parameters(query, parameters, client.config.db, run_dir)
    else:
        request.run(parameters=build_yql_parameters(parameters, build_values=True), transaction_id=transaction_id)
    return request


def wait_results(*requests, print_result=False):
    """
    Wait for requests to finish. Check if requests were completed successfully
    :param requests:
    :param print_result:
    :return: link to operation if
    """
    failed_requests = []
    for request in requests:
        try:
            if print_result:
                raise NotImplementedError()
            else:
                request.get_results()
            if not request.is_success:
                failed_requests.append(request.share_url)
        except CantGetResultsException:
            print("WARNING: query was not launched. Supposedly you're in debug mode")
    if len(failed_requests) > 0:
        raise RuntimeError('Following YQL operations have failed:\n{}'.format('\n'.join(failed_requests)))


def run_yql_file(
    client: YqlClient,
    resource_name: str,
    project_name: str,
    title: str = None,
    parameters: Dict = None,
    encoding: str = 'utf-8',
    sync: bool = True,
    format_args: Dict = None,
    attaches: Dict = None,
    transaction_id: str = None,
    template_parameters: Dict[str, Any] = None,
):
    if title is None:
        title = resource_name

    query = resource.find(resource_name)
    if query is None:
        raise Exception(f"Resource not found by name {resource_name}")
    query = query.decode(encoding)

    if template_parameters is not None:
        query_template = Template(query)
        query = query_template.render(**template_parameters)

    if format_args is not None:
        query = query.format(**format_args)
    r = run_query(
        client,
        query,
        title=f'YQL:{project_name}:{title}',
        parameters=parameters,
        attaches=attaches,
        transaction_id=transaction_id,
    )
    if sync:
        wait_results(r)
    return r
