from typing import Dict
from dataclasses import dataclass
from asyncio import sleep
import logging

from yql.api.v1.client import YqlClient

from travel.hotels.lib.python3.yql.yqllib import run_query, run_yql_file


@dataclass
class YqlConnectionData:
    db: str
    token: str


class AsyncYqlClient:
    def __init__(self, cluster: str, token: str, project_name: str, ping_time: float = 1.0):
        self.connection_data = YqlConnectionData(db=cluster, token=token)
        self.project_name = project_name
        self.ping_time = ping_time

        self.logger = logging.getLogger(__file__)

    async def run_query_async(self, query, parameters=None, run_dir=None, debug=False, syntax_version=1, title=None):
        client = YqlClient(
            db=self.connection_data.db,
            token=self.connection_data.token,
        )
        request = run_query(client, query, parameters, run_dir, debug, syntax_version, title)
        request.get_results(wait=False)
        while request.status in request.IN_PROGRESS_STATUSES:
            await sleep(self.ping_time)
            request.get_results(wait=False)
        return request

    async def run_yql_file_async(
        self,
        resource_name: str,
        title: str = None,
        parameters: Dict = None,
        encoding: str = 'utf-8',
        format_args: Dict = None
    ):
        client = YqlClient(
            db=self.connection_data.db,
            token=self.connection_data.token,
        )
        request = run_yql_file(client, resource_name, self.project_name, title, parameters, encoding, False, format_args)
        request.get_results(wait=False)

        while request.status in request.IN_PROGRESS_STATUSES:
            await sleep(self.ping_time)
            request.get_results(wait=False)
        for warning in request.warnings:
            logging.warning(str(warning))
        for error in request.errors:
            logging.error(str(error))
        if not request.is_success:
            raise Exception(f'YQL failed: {request.share_url}')
        return request
