from typing import Union, List, Dict
from yt.wrapper import YPath, ypath_join, YtClient, create, exists, get, run_merge, ypath_split
from yt.transfer_manager.client import TransferManager
import getpass
import logging

LOG = logging.getLogger(__name__)


def create_client(*args, **kwargs):  # Only to avoid PEERDIR and import to YT. Just PEERDIR and import this lib
    return YtClient(*args, **kwargs)


def join(*parts: Union[YPath, str]) -> YPath:
    return YPath(ypath_join(*map(str, parts)))


def create_table(table_path: Union[YPath, str], yt_client: YtClient, schema: List[Dict] = None) -> None:
    if schema:
        create('table', path=table_path, ignore_existing=True, recursive=True, attributes={"schema": schema},
               client=yt_client)
    else:
        create('table', path=table_path, ignore_existing=True, recursive=True, client=yt_client)


def ensure_table_exists(table_path: Union[YPath, str], yt_client: YtClient, schema: List[Dict] = None) -> None:
    if not exists(table_path, client=yt_client):
        create_table(table_path, yt_client, schema)
    elif schema is not None:
        actual_schema = get(str(table_path) + '/@schema', client=yt_client)
        actual_schema_dict = _schema_to_dict(actual_schema)
        schema_dict = _schema_to_dict(schema)
        if actual_schema_dict != schema_dict:
            raise Exception(f'Table "{table_path}" already exists and its schema is different. Expected schema {schema_dict} but schema was {actual_schema_dict}')


def recreate_table(table_path: Union[YPath, str], yt_client: YtClient, schema: List[Dict] = None) -> None:
    if exists(table_path, client=yt_client):
        yt_client.remove(table_path)
    create_table(table_path, yt_client, schema)


def _schema_to_dict(schema: List[Dict]):
    result = {}
    for column in schema:
        result[column['name']] = column['type']
    return result


def schema_from_dict(fields: Dict):
    return [{'name': k, 'type': v} for k, v in fields.items()]


def merge_chunks(table: YPath, yt_client: YtClient) -> None:
    run_merge(table, table, spec={"combine_chunks": True}, client=yt_client)


def link(node: Union[YPath, str], cluster: str = 'hahn') -> str:
    return f'https://yt.yandex-team.ru/{cluster}/navigation?path={str(node)}&'


def get_default_user_path(additional_path: str = '') -> YPath:
    username = getpass.getuser()
    return join(f'//home/travel/{username}', additional_path)


def ensure_dir(yt_client, yt_path):
    yt_client.create('map_node', yt_path, recursive=True, ignore_existing=True)


def transfer_results(path: str, source_cluster: str, destination_cluster: str, yt_token: str, link_to_path: str = None, is_dir=True):
    transfer_manager = TransferManager(token=yt_token)
    transfer_client = YtClient(proxy=destination_cluster, token=yt_token)

    if is_dir:
        transfer_client.create('map_node', path, recursive=True)
    else:
        transfer_client.create('map_node', ypath_split(path)[0], recursive=True, ignore_existing=True)
    LOG.info("Transferring to {}".format(destination_cluster))
    transfer_tasks = transfer_manager.add_tasks(
        source_cluster=source_cluster,
        source_pattern=str(path),
        destination_cluster=destination_cluster,
        destination_pattern=str(path),
        enable_failed_tasks_restarting=True,
        include_files=True,
        sync=True
    )
    LOG.info("Transferring results to {}, created tasks: {}".format(destination_cluster, transfer_tasks))
    if link_to_path is not None:
        transfer_client.link(path, link_to_path, force=True)


def abspath(path: Union[str, YPath], yt_client: YtClient) -> str:
    target_path = str(path)
    if target_path.endswith('/'):
        target_path = target_path[:-1]
    if yt_client.has_attribute(target_path + '&', 'target_path'):
        target_path = yt_client.get(target_path + '&/@target_path')
        LOG.info(f'{path} resolved into {target_path}')
        return target_path
    else:
        return path
