#include "cache.h"

#include <travel/hotels/lib/cpp/mon/tools.h>
#include <travel/hotels/lib/cpp/scheduler/scheduler.h>
#include <travel/hotels/lib/cpp/util/sizes.h>

#include <library/cpp/logger/global/global.h>

#include <util/string/printf.h>
#include <util/generic/ymath.h>

namespace {
    /*
     * Outdated-записями считаем записи, у которых уже истёк рекомендованный сёрчером срок годности, но которые мы по прежнему храним в кеше
     * Обычная запись в момент превращения в outdated проходит через фиктивное удаление и новое добавление в кеш
     *
     * IsExpired отвечает на вопрос "на момент now нужно ли считать эту запись отсутствующей в кеше из-за устаревания"
     * Outdated-записи в зависимости от флага allowOutdated или всегда считаются устаревшими, или устаревают после своего ExpireTimestamp
     *
     * */
    bool IsExpired(const NTravel::NOfferCache::TCacheRecordRef& rec, TInstant now, bool allowOutdated) {
        return (rec->IsOutdated && !allowOutdated) || rec->ExpireTimestamp < now || rec->IsInvalidated;
    }
}

namespace NTravel {
namespace NOfferCache {

const std::initializer_list<int> RECORD_LIFETIME_BUCKETS = {5, 10, 20, 30, 40, 60, 90, 120, 180, 240, 300, 360, 540, 720, 900, 1080, 1260, 1440};

TCache::TCounters::TCounters()
    : NRecordLifetimes(RECORD_LIFETIME_BUCKETS, "Inf")
{
}

void TCache::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NPreKeys));
    ct->insert(MAKE_COUNTER_PAIR(NPreKeysEmpty));
    ct->insert(MAKE_COUNTER_PAIR(NPreKeysPartial));
    ct->insert(MAKE_COUNTER_PAIR(NPreKeysFull));

    ct->insert(MAKE_COUNTER_PAIR(NRecords));
    ct->insert(MAKE_COUNTER_PAIR(NEmptyRecords));
    ct->insert(MAKE_COUNTER_PAIR(NFullRecords));
    ct->insert(MAKE_COUNTER_PAIR(NOutdatedRecords));

    ct->insert(MAKE_COUNTER_PAIR(NBytes));
    ct->insert(MAKE_COUNTER_PAIR(NBytesEmptyRecords));
    ct->insert(MAKE_COUNTER_PAIR(NBytesFullRecords));
    ct->insert(MAKE_COUNTER_PAIR(NBytesOutdatedRecords));

    ct->insert(MAKE_COUNTER_PAIR(NSkippedRecords));
    ct->insert(MAKE_COUNTER_PAIR(NRecordsAdded));
    ct->insert(MAKE_COUNTER_PAIR(NRecordsRemoved));

    NRecordLifetimes.QueryCounters("NRecordLifetimeIsUpTo", "", ct);

    ct->insert(MAKE_COUNTER_PAIR(AddBulkWaitTimeNs));
    ct->insert(MAKE_COUNTER_PAIR(AddBulkRunTimeNs));
    ct->insert(MAKE_COUNTER_PAIR(RemoveBulkWaitTimeNs));
    ct->insert(MAKE_COUNTER_PAIR(RemoveBulkRunTimeNs));
    ct->insert(MAKE_COUNTER_PAIR(ComplexSearchWaitTimeNs));
    ct->insert(MAKE_COUNTER_PAIR(ComplexSearchRunTimeNs));
}

void TCache::TPerOpCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NOffers));
}

void TCache::TPerPartnerCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NBytesEmptyRecords));
    ct->insert(MAKE_COUNTER_PAIR(NBytesFullRecords));
    ct->insert(MAKE_COUNTER_PAIR(NBytesOutdatedRecords));
    ct->insert(MAKE_COUNTER_PAIR(NEmptyRecords));
    ct->insert(MAKE_COUNTER_PAIR(NFullRecords));
    ct->insert(MAKE_COUNTER_PAIR(NOutdatedRecords));
}

size_t CacheRecordSize(const TCacheRecordRef& val) {
    return sizeof *val + val->GetAllocSize();
}

TCache::TCache(const NTravelProto::NOfferCache::TConfig::TCache& config, const TCacheInvalidationService& cacheInvalidationService, TOutdatedRecordBuilder& outdatedRecordBuilder)
    : MaximalTotalSizeInBytes_(config.GetMaximalTotalSizeMBytes() * 1024ULL * 1024ULL)
    , CleanupPeriod_(TDuration::MilliSeconds(config.GetMemoryCleanupPeriodMSec()))
    , InvalidationPeriod_(TDuration::MilliSeconds(config.GetInvalidationPeriodMSec()))
    , PerOpCounters_({"operator"})
    , PerPartnerCounters_({"partner"})
    , TotalSizeInBytes_(0)
    , CacheInvalidationService(cacheInvalidationService)
    , CacheUsageLogger_("CacheUsageLogger", config.GetCacheUsageLogger())
    , CleanupBucketIdx_(0)
    , OutdatedRecordBuilder_(outdatedRecordBuilder)
{
}

TCache::~TCache() {
}

void TCache::RegisterCounters(NMonitor::TCounterSource& source) {
    source.RegisterSource(&Counters_, "Cache");
    source.RegisterSource(&PerOpCounters_, "CachePerOp");
    source.RegisterSource(&PerPartnerCounters_, "CachePerPartner");
    CacheUsageLogger_.RegisterCounters(source);
}

void TCache::AddBulk(const TVector<TCacheRecordRef>& records, TInstant now, const TString& searcherReqId) {
    // Нельзя просто не обрабатывать expired - т.к. они могут вытеснять более старые записи из кэша
    TVector<TCacheRecordRef> evictedRecords;
    TProfileTimer started;
    TProfileTimer waited;
    TDuration waitDuration;
    for (const auto& rec : records) {
        auto expired = IsExpired(rec, now, true);
        if (TotalSizeInBytes_ >= MaximalTotalSizeInBytes_) {
            Counters_.NSkippedRecords.Inc();
            continue;
        }
        auto& bucket = ByPreKey_.GetBucketForKey(rec->Key.PreKey);
        waited.Reset();
        TWriteGuard g(bucket.GetMutex());
        waitDuration += waited.Get();
        auto pIt = bucket.GetMap().find(rec->Key.PreKey);
        if (pIt == bucket.GetMap().end()) {
            if (expired) {
                continue;
            }
            pIt = bucket.GetMap().insert(std::make_pair(rec->Key.PreKey, TPreKeyRecord())).first;
            UpdatePreKeyRecordCounters(+1, pIt->second);
        }
        TPreKeyRecord& prec = pIt->second;
        auto dIt = prec.ByDate.find(rec->Key.SubKey.Date);
        if (dIt == prec.ByDate.end()) {
            if (expired) {
                continue;
            }
            dIt = prec.ByDate.insert(std::make_pair(rec->Key.SubKey.Date, TDateRecord())).first;
        }
        TDateRecord& drec = dIt->second;
        auto nIt = drec.ByNights.find(rec->Key.SubKey.Nights);
        if (nIt == drec.ByNights.end()) {
            if (expired) {
                continue;
            }
            nIt = drec.ByNights.insert(std::make_pair(rec->Key.SubKey.Nights, TNightRecord())).first;
        }
        TNightRecord& nrec = nIt->second;
        auto cIt = nrec.ByCapacity.find(rec->Key.SubKey.Capacity);
        if (cIt == nrec.ByCapacity.end()) {
            if (expired) {
                continue;
            }
            cIt = nrec.ByCapacity.insert(std::make_pair(rec->Key.SubKey.Capacity, rec)).first;
            OnRecordAdd(rec, prec);
        } else {
            if (cIt->second->Timestamp > rec->Timestamp) {
                // Более старыми данными кеш никогда не обновляем
                continue;
            }
            if (cIt->second->Timestamp == rec->Timestamp) {
                if (expired) {
                    // Обновлять кеш expired-записью при совпадении timestamp-ов нельзя
                    // Таким образом мы в частности не затрём outdated-запись устаревшей не-outdated записью при чтении шины на старте
                    continue;
                }
                if ((cIt->second->IsOutdated && rec->IsOutdated) || (!cIt->second->IsOutdated && !rec->IsOutdated)) {
                    // менять outdated на outdated и не-outdated на не-outdated нет смысла, это вообще странная ситуация
                    continue;
                }
                if (!cIt->second->IsOutdated && rec->IsOutdated) {
                    // не-outdated можно заменить на outdated, если не-outdated уже просрочена. Иначе оставляем не-outdated
                    if (!IsExpired(cIt->second, now, true)) {
                        continue;
                    }
                }
                // поменять outdated на не-outdated - это ок, так может происходить при первоначальном чтении шины (если outdated-запись попала в кеш раньше основной)
            }
            OnRecordDel(cIt->second, prec, &evictedRecords);
            if (expired) {
                // Новая запись убила старую, но при этом она сразу устаревшая -> удалим старую запись
                nrec.ByCapacity.erase(cIt);
                if (nrec.ByCapacity.empty()) {
                    drec.ByNights.erase(nIt);
                    if (drec.ByNights.empty()) {
                        prec.ByDate.erase(dIt);
                        if (prec.ByDate.empty()) {
                            UpdatePreKeyRecordCounters(-1, pIt->second);
                            bucket.GetMap().erase(pIt);
                        }
                    }
                }
            } else {
                // Новая запись заместила старую
                cIt->second = rec;
                OnRecordAdd(cIt->second, prec);
            }
        }
    }
    LogEvictedRecords(evictedRecords, false, searcherReqId);
    Counters_.AddBulkWaitTimeNs += waitDuration.NanoSeconds();
    Counters_.AddBulkRunTimeNs += (started.Get() - waitDuration).NanoSeconds();
}

void TCache::OnRecordAdd(const TCacheRecordRef& newRecord, TPreKeyRecord& pkRecord) {
    Counters_.NRecordsAdded.Inc();
    UpdateRecordCounters(+1, newRecord, pkRecord);

    auto perDayKey = TCacheDayStatKey(newRecord->Key.PreKey.Currency, newRecord->Key.SubKey.Date);
    auto& bucket = PerDayStat_.GetBucketForKey(perDayKey);
    TWriteGuard g(bucket.GetMutex());
    auto& stat = bucket.GetMap()[perDayKey].ByNights[newRecord->Key.SubKey.Nights];
    ++stat.RecordCount;
    if (newRecord->Offers.empty()) {
        ++stat.EmptyRecordCount;
    }
    if (newRecord->IsOutdated) {
        ++stat.OutdatedRecordCount;
    }
}

void TCache::OnRecordDel(const TCacheRecordRef& oldRecord, TPreKeyRecord& pkRecord, TVector<TCacheRecordRef>* evictedRecords) {
    Counters_.NRecordsRemoved.Inc();
    UpdateRecordCounters(-1, oldRecord, pkRecord);

    evictedRecords->push_back(oldRecord);
    {
        auto perDayKey = TCacheDayStatKey(oldRecord->Key.PreKey.Currency, oldRecord->Key.SubKey.Date);
        auto& bucket = PerDayStat_.GetBucketForKey(perDayKey);
        TWriteGuard g(bucket.GetMutex());
        auto& stat = bucket.GetMap()[perDayKey].ByNights[oldRecord->Key.SubKey.Nights];
        --stat.RecordCount;
        if (oldRecord->Offers.empty()) {
            --stat.EmptyRecordCount;
        }
        if (oldRecord->IsOutdated) {
            --stat.OutdatedRecordCount;
        }
    }
}

void TCache::UpdatePreKeyRecordCounters(i64 sign, const TPreKeyRecord& pkRecord) {
    Counters_.NPreKeys += sign;
    if (pkRecord.RecordsEmpty == pkRecord.RecordsTotal) {
        Counters_.NPreKeysEmpty += sign;
    } else if (pkRecord.RecordsEmpty == 0) {
        Counters_.NPreKeysFull += sign;
    } else {
        Counters_.NPreKeysPartial += sign;
    }
}

void TCache::UpdateRecordCounters(i64 sign, const TCacheRecordRef& record, TPreKeyRecord& pkRecord) {
    UpdatePreKeyRecordCounters(-1, pkRecord);

    i64 byteDiff = CacheRecordSize(record) * sign;
    TotalSizeInBytes_ += byteDiff;
    Counters_.NBytes += byteDiff;

    Counters_.NRecords += sign;
    Counters_.NRecordLifetimes.Update((record->ExpireTimestamp - record->Timestamp).Minutes(), sign);

    record->SourceCounters->NCacheRecords += sign;
    pkRecord.RecordsTotal += sign;

    auto perPartnerCounters = PerPartnerCounters_.GetOrCreate({ToString(record->Key.PreKey.HotelId.PartnerId)});
    if (record->IsOutdated) {
        Counters_.NOutdatedRecords += sign;
        Counters_.NBytesOutdatedRecords += byteDiff;
        record->SourceCounters->NCacheRecordsOutdated += sign;
        perPartnerCounters->NOutdatedRecords += sign;
        perPartnerCounters->NBytesOutdatedRecords += byteDiff;
    } else if (record->Offers.empty()) {
        Counters_.NEmptyRecords += sign;
        Counters_.NBytesEmptyRecords += byteDiff;
        pkRecord.RecordsEmpty += sign;
        record->SourceCounters->NCacheRecordsEmpty += sign;
        perPartnerCounters->NEmptyRecords += sign;
        perPartnerCounters->NBytesEmptyRecords += byteDiff;
    } else {
        Counters_.NFullRecords += sign;
        Counters_.NBytesFullRecords += byteDiff;
        record->SourceCounters->NCacheRecordsFull += sign;
        for (const auto& offer: record->Offers) {
            PerOpCounters_.GetOrCreate({ToString(offer.OperatorId)})->NOffers += sign;
        }
        perPartnerCounters->NFullRecords += sign;
        perPartnerCounters->NBytesFullRecords += byteDiff;
    }

    UpdatePreKeyRecordCounters(+1, pkRecord);
}


void TCache::ComplexSearch(const THashSet<EOperatorId>& enabledOpIds,
                           const THashMap<TPreKey, THashSet<TPermalink>>& preKeysForSearch,
                           TInstant startedTs,
                           const TSearchSubKeyRange& subKeyRange,
                           const TSearchSubKey& defaultSubKey,
                           const TSearchSubKey& userSubKey,
                           TSearchSubKey* subKey,
                           THashMap<TPermalink, THashMap<THotelId, TVector<TCacheRecordRef>>>* records,
                           TCacheSearchStat* stat,
                           bool allowOutdated,
                           bool allowOutdatedForKeyDetectionByUserDates,
                           bool allowOutdatedForKeyDetectionByAnyDate) const {
    TProfileTimer started;
    TProfileTimer waited;
    if (!subKey->IsComplete()) {
        DetermineBestSubKey(preKeysForSearch, enabledOpIds, startedTs, subKeyRange, defaultSubKey, userSubKey, subKey,
            stat, allowOutdatedForKeyDetectionByUserDates, allowOutdatedForKeyDetectionByAnyDate);
        stat->PreDuration += started.Get();
    }
    // Предполагается, что сюда приходит полный subKey, в котором задано всё
    // Из-за нечеткого сравнения Ages может быть более одной записи на каждый HotelId
    Scan(preKeysForSearch, startedTs, {TSearchSubKeyRange::FromSearchSubKey(*subKey)}, [records, stat]
            (const THashSet<TPermalink>& permalinks, const TCacheRecordRef& rec) {
        ++stat->RecordsGood;
        for (TPermalink p: permalinks) {
            (*records)[p][rec->Key.PreKey.HotelId].push_back(rec);
        }
        AtomicIncrement(rec->UsageCount);
    }, stat, allowOutdated);
    TDuration prevWaitLockDuration = stat->WaitLockDuration;
    TDuration dur = started.Get();
    TDuration thisWaitLockDuration = stat->WaitLockDuration - prevWaitLockDuration;
    stat->Duration += dur;
    Counters_.ComplexSearchWaitTimeNs += thisWaitLockDuration.NanoSeconds();
    Counters_.ComplexSearchRunTimeNs += (dur - thisWaitLockDuration).NanoSeconds();
}

void TCache::Start() {
    CacheUsageLogger_.Start();
}

void TCache::StartPeriodicalJobs() {
    TScheduler::Instance().EnqueuePeriodical(CleanupPeriod_, [this] {
        Cleanup();
    });
    TScheduler::Instance().EnqueuePeriodical(InvalidationPeriod_, [this] {
        MarkInvalidatedRecords();
    });
}

void TCache::Stop() {
    CacheUsageLogger_.Stop();
}

void TCache::ReopenCacheUsageLog() {
    CacheUsageLogger_.Reopen();
}

TCacheDayStat TCache::GetDayStat(NTravelProto::ECurrency currency, NOrdinalDate::TOrdinalDate date) const {
    auto perDayKey = TCacheDayStatKey(currency, date);
    auto& bucket = PerDayStat_.GetBucketForKey(perDayKey);
    TReadGuard g(bucket.GetMutex());
    auto it = bucket.GetMap().find(perDayKey);
    if (it == bucket.GetMap().end()) {
        return TCacheDayStat();
    }
    return it->second;
}

void TCache::LogEvictedRecords(const TVector<TCacheRecordRef>& evictedRecords, bool byTTL, const TString& otherSearcherReqId) {
    NTravelProto::NOfferCache::TCacheUsageLogRecord log;
    log.set_unixtime(TInstant::Now().Seconds());
    log.SetByTTL(byTTL);
    if (otherSearcherReqId) {
        log.SetOtherSearcherReqId(otherSearcherReqId);
    }
    for (const TCacheRecordRef& record: evictedRecords) {
        log.SetSearcherReqId(record->SearcherReqId);
        log.SetUsageCount(AtomicGet(record->UsageCount));
        const TCacheKey& cacheKey = record->Key;
        log.SetCapacity(cacheKey.SubKey.Capacity.ToCapacityString());
        log.SetDate(NOrdinalDate::ToString(cacheKey.SubKey.Date));
        log.SetNights(cacheKey.SubKey.Nights);
        cacheKey.PreKey.HotelId.ToProto(log.MutableHotelId());
        log.SetOfferCacheClientId(record->OfferCacheClientId);
        log.SetTimeStamp(record->Timestamp.Seconds());
        log.SetExpireTimestamp(record->ExpireTimestamp.Seconds());
        CacheUsageLogger_.AddRecord(log);
    }
}

bool TCache::SubKeyStatLess(const TSubKeyStat& lhs, const TSubKeyStat& rhs) const {
    if (lhs.PermalinksWithOffers.size() != rhs.PermalinksWithOffers.size()) {
        return lhs.PermalinksWithOffers.size() < rhs.PermalinksWithOffers.size();
    }
    if (lhs.OperatorsWithOffers.size() != rhs.OperatorsWithOffers.size()) {
        return lhs.OperatorsWithOffers.size() < rhs.OperatorsWithOffers.size();
    }
    if (lhs.PermalinksWithoutOffers.size() != rhs.PermalinksWithoutOffers.size()) {
        return lhs.PermalinksWithoutOffers.size() < rhs.PermalinksWithoutOffers.size();
    }
    if (lhs.DefaultKeyDistance != rhs.DefaultKeyDistance) {
        return lhs.DefaultKeyDistance > rhs.DefaultKeyDistance;
    }
    return false;
}

static int Distance(const TSearchSubKey& lhs, const TSearchSubKey& rhs) {
    int distance = 0;
    distance += Abs(lhs.Date - rhs.Date);
    distance += (lhs.Nights > rhs.Nights) ? (lhs.Nights - rhs.Nights) : (rhs.Nights - lhs.Nights);
    if (lhs.Ages != rhs.Ages) {
        distance *= 2;
    }
    return distance;
}

void TCache::DetermineBestSubKey(const THashMap<TPreKey, THashSet<TPermalink>>& preKeys, const THashSet<EOperatorId>& enabledOpIds,
                                 TInstant startedTs,
                                 const TSearchSubKeyRange& subKeyRange,
                                 const TSearchSubKey& defaultSubKey,
                                 const TSearchSubKey& userSubKey,
                                 TSearchSubKey* subKey,
                                 TCacheSearchStat* stat,
                                 bool allowOutdatedForKeyDetectionByUserDates,
                                 bool allowOutdatedForKeyDetectionByAnyDate) const {
    // Если что-то не задано, то надо найти лучший вариант
    // Тут сложная логика, см HOTELS-2532 + HOTELS-2717
    // Цель: выбрать лучшие Date, Ages, Nights.
    // Способ выбора лучших (в порядке приоритетности)
    // 1. Макс кол-во отелей с этим ключом
    // 2. максимальное общее количество предложений по отелям.
    // 3. При прочих равных сравниваем по близости к дефолтному ключу
    TSubKeyStatMap subKeyStatMap;
    TVector<TSearchSubKeyRange> subKeyRanges;
    subKeyRanges.push_back(subKeyRange);
    const bool useUserSubKey = userSubKey.IsComplete();
    if (useUserSubKey) {
        subKeyRanges.push_back(TSearchSubKeyRange::FromSearchSubKey(userSubKey));
    }
    Scan(preKeys, startedTs, subKeyRanges, [&enabledOpIds, &subKeyStatMap, allowOutdatedForKeyDetectionByAnyDate, stat](const THashSet<TPermalink>& permalinks, const TCacheRecordRef& rec) {
        TSearchSubKey searchKey;
        searchKey.Date = rec->Key.SubKey.Date;
        searchKey.Nights = rec->Key.SubKey.Nights;
        searchKey.Ages = rec->Key.SubKey.Capacity.DowncastToAges();
        TSubKeyStat& subKeyStat = subKeyStatMap[searchKey];
        bool haveOffers = false;
        for (const auto& offer: rec->Offers) {
            if (enabledOpIds.contains(offer.OperatorId)) {
                subKeyStat.HasOutdatedOffers |= rec->IsOutdated;
                if (!rec->IsOutdated || allowOutdatedForKeyDetectionByAnyDate) {
                    haveOffers = true;
                    subKeyStat.OperatorsWithOffers.insert(offer.OperatorId);
                }
            }
        }
        for (TPermalink permalink: permalinks) {
            if (haveOffers) {
                subKeyStat.PermalinksWithOffers.insert(permalink);
            } else {
                subKeyStat.PermalinksWithoutOffers.insert(permalink);
            }
        }
        ++stat->RecordsPreGood;
    }, stat, allowOutdatedForKeyDetectionByUserDates || allowOutdatedForKeyDetectionByAnyDate);
    auto endIt = subKeyStatMap.end();
    auto bestIt = endIt;
    for (auto it = subKeyStatMap.begin(); it != endIt; ++it) {
        const TSearchSubKey& key = it->first;
        TSubKeyStat& subKeyStat = it->second;
        if (useUserSubKey && key == userSubKey) {
            if (!subKeyStat.PermalinksWithOffers.empty() ||
                (subKeyStat.HasOutdatedOffers && allowOutdatedForKeyDetectionByUserDates)) {
                // По пользовательскому ключу есть хоть что-то - и ура
                subKey->Merge(key);
                return;
            }
        }

        if (!subKeyStat.PermalinksWithOffers.empty()) {
            // Не учитываем при поиске ключи, по которым офферов нет, уж лучше взять дефолт
            subKeyStat.DefaultKeyDistance += Distance(key, defaultSubKey);
            if (bestIt == endIt || SubKeyStatLess(bestIt->second, subKeyStat)) {
                bestIt = it;
            }
        }
    }
    TSearchSubKey bestKey;
    if (bestIt == endIt) {
        // Если ничего не найдено - ключ заполняем дефолтами
        if (useUserSubKey) {
            bestKey = userSubKey;
        } else {
            bestKey = defaultSubKey;
        }
    } else {
        bestKey = bestIt->first;
    }
    subKey->Merge(bestKey);
}

void TCache::Scan(const THashMap<TPreKey, THashSet<TPermalink>>& preKeys, TInstant startedTs,
                  const TVector<TSearchSubKeyRange>& subKeyRanges, TScanFunc scanFunc, TCacheSearchStat* stat,
                  bool allowOutdated) const {
    TProfileTimer waited;
    for (auto inputIt = preKeys.begin(); inputIt != preKeys.end(); ++inputIt) {
        // By Hotel
        auto& bucket = ByPreKey_.GetBucketForKey(inputIt->first);
        waited.Reset();
        TReadGuard g(bucket.GetMutex());
        stat->WaitLockDuration += waited.Get();
        auto pIt = bucket.GetMap().find(inputIt->first);
        if (pIt == bucket.GetMap().end()) {
            continue;
        }
        const TPreKeyRecord& prec = pIt->second;
        TPreKeyRecord::TByDateMap::const_iterator dItEnd = prec.ByDate.end();
        TPreKeyRecord::TByDateMap::const_iterator dItFrom, dItTo;
        for (const auto& subKeyRange: subKeyRanges) {
            if (subKeyRange.DateFrom != NOrdinalDate::g_DateZero) {
                dItFrom = prec.ByDate.lower_bound(subKeyRange.DateFrom);
                if (dItFrom == dItEnd) {
                    continue;
                }
            } else {
                dItFrom = prec.ByDate.begin();
            }
            if (subKeyRange.DateTo != NOrdinalDate::g_DateZero) {
                dItTo = prec.ByDate.upper_bound(subKeyRange.DateTo);
            } else {
                dItTo = dItEnd;
            }
            for (auto dIt = dItFrom; dIt != dItTo; ++dIt) {
                // By Date
                const TDateRecord& drec = dIt->second;
                TDateRecord::TByNightsMap::const_iterator nItEnd = drec.ByNights.end();
                TDateRecord::TByNightsMap::const_iterator nItFrom, nItTo;
                if (subKeyRange.NightsFrom != g_NightsZero) {
                    nItFrom = drec.ByNights.lower_bound(subKeyRange.NightsFrom);
                    if (nItFrom == nItEnd) {
                        continue;
                    }
                } else {
                    nItFrom = drec.ByNights.begin();
                }
                if (subKeyRange.NightsTo != g_NightsZero) {
                    nItTo = drec.ByNights.upper_bound(subKeyRange.NightsTo);
                } else {
                    nItTo = nItEnd;
                }
                for (auto nIt = nItFrom; nIt != nItTo; ++nIt) {
                    // By Nights
                    const TNightRecord& nrec = nIt->second;
                    for (auto cIt = nrec.ByCapacity.begin(); cIt != nrec.ByCapacity.end(); ++cIt) {
                        // By Ages
                        ++stat->RecordsScanned;
                        const TCacheRecordRef& rec = cIt->second;
                        if (!IsExpired(rec, startedTs, allowOutdated)) {
                            if (subKeyRange.AgeMatches(rec->Key.SubKey.Capacity)) {
                                scanFunc(inputIt->second, rec);
                            }
                        }
                    }
                }
            }
        }
    }
}

void TCache::Cleanup() {
    if (++CleanupBucketIdx_ >= ByPreKey_.Buckets.size()) {
        CleanupBucketIdx_ = 0;
    }
    auto& bucket = ByPreKey_.Buckets[CleanupBucketIdx_];

    TInstant now = Now();
    TVector<TCacheRecordRef> recordsToClean;
    TVector<TCacheRecordRef> recordsToMarkOutdated;
    TProfileTimer started;
    {
        TReadGuard g(bucket.GetMutex());
        for (auto pIt = bucket.GetMap().begin(); pIt != bucket.GetMap().end(); ++pIt) {
            TPreKeyRecord& prec = pIt->second;
            for (auto dIt = prec.ByDate.begin(); dIt != prec.ByDate.end(); ++dIt) {
                TDateRecord& drec = dIt->second;
                for (auto nIt = drec.ByNights.begin(); nIt != drec.ByNights.end(); ++nIt) {
                    TNightRecord& nrec = nIt->second;
                    for (auto cIt = nrec.ByCapacity.begin(); cIt != nrec.ByCapacity.end(); ++cIt) {
                        TCacheRecordRef& rec = cIt->second;
                        if (IsExpired(rec, now, true)) {
                            if (!rec->IsOutdated && OutdatedRecordBuilder_.AreOutdatedPricesEnabled()) {
                                // Свежая запись expired - превращаем её в outdated
                                TMaybe<TCacheRecordRef> newRec = OutdatedRecordBuilder_.ConvertToOutdated(rec);
                                if (newRec.Defined()) {
                                    recordsToMarkOutdated.push_back(newRec.GetRef());
                                } else {
                                    // Если перевоплощение не удалось - просто удаляем запись
                                    recordsToClean.push_back(rec);
                                }
                            } else {
                                // Старая запись expired - остаётся только удалять
                                recordsToClean.push_back(rec);
                            }
                        }
                    }
                }
            }
        }
    }
    TDuration maxLockDuration;
    size_t evictedCount = 0;
    if (recordsToClean) {
        RemoveBulk(now, recordsToClean, bucket, &maxLockDuration, &evictedCount);
    }
    if (recordsToMarkOutdated) {
        AddBulk(recordsToMarkOutdated, now, {});
    }
    DEBUG_LOG << "Cleaned up bucket " << CleanupBucketIdx_ << " in " << started.Get()
              << ", maxLockDuration " << maxLockDuration
              << ", evicted " << evictedCount << " records"
              << ", recordsToClean.size(): " << recordsToClean.size()
              << ", recordsToMarkOutdated.size(): " << recordsToMarkOutdated.size()
              << Endl;
}


void TCache::MarkInvalidatedRecords() {
    TProfileTimer started;
    int count = 0;
    for (auto& bucket: ByPreKey_.Buckets) {
        TReadGuard g(bucket.GetMutex());
        for (auto& [preKey, prec]: bucket.GetMap()) {
            for (auto& [date, drec]: prec.ByDate) {
                for (auto& [nights, nrec]: drec.ByNights) {
                    auto invalidationTs = CacheInvalidationService.GetInvalidationTimestamp(preKey, date, date + nights);
                    for (auto& [_, rec]: nrec.ByCapacity) {
                        if (invalidationTs >= rec->Timestamp) {
                            if (!rec->IsOutdated && rec->IsInvalidated.TrySet()) {
                                count++;
                            }
                        }
                    }
                }
            }
        }
    }
    DEBUG_LOG << "Marked invalidated records in " << started.Get() << ". Marked " << count << " records" << Endl;
}

void TCache::RemoveBulk(TInstant now, const TVector<TCacheRecordRef>& records,
                        TByPrekeyMap::TBucket& bucket,
                        TDuration* maxLockDuration, size_t* evictedCount) {
    TVector<TCacheRecordRef> evictedRecords;
    TProfileTimer started;
    TProfileTimer waited;
    TDuration waitDuration;
    for (const auto& delRec : records) {
        waited.Reset();
        TWriteGuard g(bucket.GetMutex());// Per record lock intentionally!
        waitDuration += waited.Get();
        TProfileTimer lockTimer;
        auto pIt = bucket.GetMap().find(delRec->Key.PreKey);
        if (pIt != bucket.GetMap().end()) {
            TPreKeyRecord& prec = pIt->second;
            auto dIt = prec.ByDate.find(delRec->Key.SubKey.Date);
            if (dIt != prec.ByDate.end()) {
                TDateRecord& drec = dIt->second;
                auto nIt = drec.ByNights.find(delRec->Key.SubKey.Nights);
                if (nIt != drec.ByNights.end()) {
                    TNightRecord& nrec = nIt->second;
                    auto cIt = nrec.ByCapacity.find(delRec->Key.SubKey.Capacity);
                    if (cIt != nrec.ByCapacity.end()) {
                        TCacheRecordRef& rec = cIt->second;
                        if (IsExpired(rec, now, true)) {// Запись могла быть обновлена в кэше
                            // Удаляем запись о subkey
                            OnRecordDel(rec, prec, &evictedRecords);
                            nrec.ByCapacity.erase(cIt);
                            if (nrec.ByCapacity.empty()) {
                                drec.ByNights.erase(nIt);
                                if (drec.ByNights.empty()) {
                                    prec.ByDate.erase(dIt);
                                    if (prec.ByDate.empty()) {
                                        UpdatePreKeyRecordCounters(-1, pIt->second);
                                        bucket.GetMap().erase(pIt);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        TDuration lockDuration = lockTimer.Get();
        if (lockDuration > *maxLockDuration) {
            *maxLockDuration = lockDuration;
        }
    }
    *evictedCount = evictedRecords.size();
    LogEvictedRecords(evictedRecords, true, {});
    Counters_.RemoveBulkWaitTimeNs += waitDuration.NanoSeconds();
    Counters_.RemoveBulkRunTimeNs += (started.Get() - waitDuration).NanoSeconds();
}

}// namespace NOfferCache
}// namespace NTravel
