#pragma once

#include "data.h"
#include "outdated_record_builder.h"

#include <travel/hotels/offercache/proto/config.pb.h>
#include <travel/hotels/offercache/proto/cacheusage_logrecord.pb.h>

#include <travel/hotels/proto2/hotels.pb.h>
#include <travel/hotels/lib/cpp/offer_cache_invalidation/cache_invalidation_service.h>
#include <travel/hotels/lib/cpp/mon/counter.h>
#include <travel/hotels/lib/cpp/mon/counter_hypercube.h>
#include <travel/hotels/lib/cpp/mon/tools.h>
#include <travel/hotels/lib/cpp/logging/common_logger.h>
#include <travel/hotels/lib/cpp/util/flag.h>
#include <travel/hotels/lib/cpp/util/profiletimer.h>

#include <library/cpp/containers/concurrent_hash/concurrent_hash.h>
#include <util/generic/vector.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/map.h>
#include <util/system/rwlock.h>
#include <functional>

namespace NTravel {
namespace NOfferCache {

struct TCacheSearchStat {
    TDuration Duration;
    TDuration WaitLockDuration;
    TDuration PreDuration;
    size_t RecordsScanned = 0;
    size_t RecordsPreGood = 0;
    size_t RecordsGood = 0;
};

struct TCacheDayNightsStat {
    size_t RecordCount = 0;
    size_t EmptyRecordCount = 0;
    size_t OutdatedRecordCount = 0;
};

struct TCacheDayStat {
    THashMap<TNights, TCacheDayNightsStat> ByNights;
};

class TCache {
public:
    TCache(const NTravelProto::NOfferCache::TConfig::TCache& config, const TCacheInvalidationService& cacheInvalidationService, TOutdatedRecordBuilder& outdatedRecordBuilder);
    ~TCache();

    void RegisterCounters(NMonitor::TCounterSource& source);

    void AddBulk(const TVector<TCacheRecordRef>& records, TInstant now, const TString& searcherReqId);

    // Для каждого отеля в result будут записи (м.б. несколько из-за capacity), происходит выбор по HotelId
    // Заодно выбирается лучший subKey, если надо
    void ComplexSearch(const THashSet<EOperatorId>& enabledOpIds,
                       const THashMap<TPreKey, THashSet<TPermalink>>& preKeysForSearch,
                       TInstant startedTs,
                       const TSearchSubKeyRange& subKeyRange,
                       const TSearchSubKey& defaultSubKey,
                       const TSearchSubKey& userSubKey,
                       TSearchSubKey* subKey,
                       THashMap<TPermalink, THashMap<THotelId, TVector<TCacheRecordRef>>>* records,
                       TCacheSearchStat* stat,
                       bool allowOutdated,
                       bool allowOutdatedForKeyDetectionByUserDates,
                       bool allowOutdatedForKeyDetectionByAnyDate) const;

    void Start();
    void StartPeriodicalJobs();
    void Stop();
    void ReopenCacheUsageLog();
    TCacheDayStat GetDayStat(NTravelProto::ECurrency currency, NOrdinalDate::TOrdinalDate date) const;
private:
    struct TCounters : public NMonitor::TCounterSource {
        // Размеры
        NMonitor::TCounter NPreKeys;         // Total
        NMonitor::TCounter NPreKeysEmpty;    // у которых все записи - пустые
        NMonitor::TCounter NPreKeysPartial;  // у которых часть записей - пустые
        NMonitor::TCounter NPreKeysFull;     // у которых все записи - непустые

        NMonitor::TCounter NRecords;         // Total
        NMonitor::TCounter NEmptyRecords;    // Пустые записи
        NMonitor::TCounter NFullRecords;     // Непустые
        NMonitor::TCounter NOutdatedRecords; // Устаревшие

        NMonitor::TCounter NBytes;                // Total
        NMonitor::TCounter NBytesEmptyRecords;    // Пустые записи
        NMonitor::TCounter NBytesFullRecords;     // Непустые
        NMonitor::TCounter NBytesOutdatedRecords; // Устаревшие

        NMonitor::TDerivCounter NSkippedRecords; // Пропущено из-за переполнения

        NMonitor::TDerivCounter NRecordsAdded;
        NMonitor::TDerivCounter NRecordsRemoved;

        NMonitor::THistogramCounter NRecordLifetimes;

        // Benchmark counters
        NMonitor::TDerivCounter AddBulkWaitTimeNs;
        NMonitor::TDerivCounter AddBulkRunTimeNs;
        NMonitor::TDerivCounter RemoveBulkWaitTimeNs;
        NMonitor::TDerivCounter RemoveBulkRunTimeNs;
        NMonitor::TDerivCounter ComplexSearchWaitTimeNs;
        NMonitor::TDerivCounter ComplexSearchRunTimeNs;

        TCounters();
        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    struct TPerOpCounters : public NMonitor::TCounterSource {
        NMonitor::TCounter NOffers;

        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    struct TPerPartnerCounters : public NMonitor::TCounterSource {
        NMonitor::TCounter NBytesEmptyRecords;
        NMonitor::TCounter NBytesFullRecords;
        NMonitor::TCounter NBytesOutdatedRecords;
        NMonitor::TCounter NEmptyRecords;
        NMonitor::TCounter NFullRecords;
        NMonitor::TCounter NOutdatedRecords;

        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    struct TSubKeyStat {
        // Нам важно максимизировать кол-во пермалинков с офферами. Среди одинаковых - с макс кол-вом операторов с офферами.
        // Среди одинаковых - с макс кол-вом пермалинков без офферов (пустых)
        // Далее по близости к дефолтному ключу
        THashSet<TPermalink>   PermalinksWithOffers;
        THashSet<EOperatorId>  OperatorsWithOffers;
        THashSet<TPermalink>   PermalinksWithoutOffers;
        bool                   HasOutdatedOffers;
        int                    DefaultKeyDistance = 0; // близость к DefaultSubKey
    };

    using TSubKeyStatMap = THashMap<TSearchSubKey, TSubKeyStat>;

    struct TNightRecord {
        using TByCapacityMap = THashMap<TCapacity, TCacheRecordRef>;
        TByCapacityMap ByCapacity;
    };

    struct TDateRecord {
        using TByNightsMap = TMap<TNights, TNightRecord>;
        TByNightsMap ByNights;
    };

    struct TPreKeyRecord {
        using TByDateMap = TMap<NOrdinalDate::TOrdinalDate, TDateRecord>;
        TByDateMap                           ByDate;
        size_t RecordsEmpty = 0;
        size_t RecordsTotal = 0;
    };

    static const int NUM_BUCKETS = 64;

    using TCacheDayStatKey = std::pair<NTravelProto::ECurrency, NOrdinalDate::TOrdinalDate>;

    using TScanFunc = std::function<void (const THashSet<TPermalink> permalinks, const TCacheRecordRef& rec)>;

    using TByPrekeyMap = TConcurrentHashMap<TPreKey, TPreKeyRecord, NUM_BUCKETS, TRWMutex> ;


    const ui64 MaximalTotalSizeInBytes_;
    const TDuration CleanupPeriod_;
    const TDuration InvalidationPeriod_;

    mutable TCounters Counters_;
    mutable NMonitor::TCounterHypercube<TPerOpCounters> PerOpCounters_;
    mutable NMonitor::TCounterHypercube<TPerPartnerCounters> PerPartnerCounters_;

    std::atomic<ui64> TotalSizeInBytes_;
    TByPrekeyMap ByPreKey_;
    TConcurrentHashMap<TCacheDayStatKey, TCacheDayStat, NUM_BUCKETS, TRWMutex> PerDayStat_;

    const TCacheInvalidationService& CacheInvalidationService;

    TCommonLogger<NTravelProto::NOfferCache::TCacheUsageLogRecord> CacheUsageLogger_;

    size_t CleanupBucketIdx_;

    TOutdatedRecordBuilder& OutdatedRecordBuilder_;

    void LogEvictedRecords(const TVector<TCacheRecordRef>& evictedRecords, bool byTTL, const TString& otherSearcherReqId);

    void OnRecordAdd(const TCacheRecordRef& newRecord, TPreKeyRecord& pkRecord);
    void OnRecordDel(const TCacheRecordRef& oldRecord, TPreKeyRecord& pkRecord, TVector<TCacheRecordRef>* evictedRecords);

    void UpdatePreKeyRecordCounters(i64 sign, const TPreKeyRecord& pkRecord);
    void UpdateRecordCounters(i64 sign, const TCacheRecordRef& record, TPreKeyRecord& pkRecord);

    bool SubKeyStatLess(const TSubKeyStat& lhs, const TSubKeyStat& rhs) const;
    void DetermineBestSubKey(const THashMap<TPreKey, THashSet<TPermalink>>& preKeys, const THashSet<EOperatorId>& enabledOpIds,
                             TInstant startedTs,
                             const TSearchSubKeyRange& subKeyRange,
                             const TSearchSubKey& defaultSubKey,
                             const TSearchSubKey& userSubKey,
                             TSearchSubKey* subKey,
                             TCacheSearchStat* stat,
                             bool allowOutdatedForKeyDetectionByUserDates,
                             bool allowOutdatedForKeyDetectionByAnyDate) const;
    void Scan(const THashMap<TPreKey, THashSet<TPermalink>>& preKeys, TInstant startedTs,
              const TVector<TSearchSubKeyRange>& subKeyRanges, TScanFunc scanFunc, TCacheSearchStat* stat,
              bool allowOutdated) const;

    void Cleanup();
    void MarkInvalidatedRecords();

    void RemoveBulk(TInstant now, const TVector<TCacheRecordRef>& records,
                    TByPrekeyMap::TBucket& bucket,
                    TDuration* maxLockDuration, size_t* evictedCount);
};

}// namespace NOfferCache
}// namespace NTravel
