#include "cache_heatmap.h"

#include "data.h"
#include "cache.h"

#include <travel/hotels/lib/cpp/protobuf/tools.h>

#include <library/cpp/logger/global/global.h>

#include <util/string/builder.h>

namespace NTravel {
namespace NOfferCache {

ui32 GetOffersFromBucket(const NTravelProto::NOfferCache::TConfig::TCacheHeatmap::TBucket& buck) {
    return buck.GetOffers();
}

size_t GetValueByShowMode(const TCacheDayNightsStat& stat, NTravelProto::NOfferCache::NApi::THeatmapReq_EShowMode showMode) {
    switch (showMode) {
        case NTravelProto::NOfferCache::NApi::THeatmapReq_EShowMode_SHOW_RECORDS:
            return stat.RecordCount;
        case NTravelProto::NOfferCache::NApi::THeatmapReq_EShowMode_SHOW_EMPTY_RECORDS:
            return stat.EmptyRecordCount;
        case NTravelProto::NOfferCache::NApi::THeatmapReq_EShowMode_SHOW_WITHOUT_OUTDATED_RECORDS:
            return stat.RecordCount - stat.OutdatedRecordCount;
        case NTravelProto::NOfferCache::NApi::THeatmapReq_EShowMode_SHOW_ONLY_OUTDATED_RECORDS:
            return stat.OutdatedRecordCount;
    }
}

TString ConvertColorIntoShadesOfRed(TString color) {
    if (color.size() != 7 || color[0] != '#') {
        ERROR_LOG << "Cannot convert a non-hex color into shades of red" << Endl;
        return color;
    }
    auto red = color.substr(1, 2);
    auto green = color.substr(3, 2);
    auto blue = color.substr(5, 2);
    return "#" + green + red + blue;
}

TCacheHeatmap::TCacheHeatmap(const TCache& cache, const NTravelProto::NOfferCache::TConfig::TCacheHeatmap& pbCfg)
    : Cache_(cache)
{
    for (const auto& bucket: pbCfg.GetBucket()) {
        Buckets_.push_back(bucket);
    }
    SortBy(Buckets_, GetOffersFromBucket);
    if (Buckets_.empty()) {
        throw yexception() << "Cache heatmap buckets is empty";
    }
}

TCacheHeatmap::~TCacheHeatmap() {
}

TString TCacheHeatmap::BuildHTMLCacheHeatmap(const NTravelProto::NOfferCache::NApi::THeatmapReq& req) const {
    auto cur = CurrencyFromOCFormat(req.GetCurrency());
    TStringBuilder out;
    out << "<html>";
    if (req.GetRefreshSec()) {
        out << "<head><meta http-equiv=\"refresh\" content=\"" << req.GetRefreshSec() << "\"></head>";
    }
    out << "<body bgcolor=\"" << Buckets_[0].GetColor() << "\"><table>";
    out << "<tr><td width=\"100px\" align=\"right\">Nights</td>";
    for (TNights n = req.GetNightOffset(); n < (req.GetNightOffset() + req.GetNightCount()); ++n) {
        out << "<td width=\"40px\" align=\"right\">" << n << "</td>";
    }
    out << "</tr><tr>";
    NOrdinalDate::TOrdinalDate dtBegin = NOrdinalDate::FromInstant(::Now()) + req.GetDateOffset();
    for (size_t dtAdd = 0; dtAdd < req.GetDateCount(); ++dtAdd) {
        NOrdinalDate::TOrdinalDate dt = dtBegin + dtAdd;
        out << "<tr><td>" << NOrdinalDate::ToString(dt) << "</td>";
        auto stats = Cache_.GetDayStat(cur, dt);
        for (TNights n = req.GetNightOffset(); n < (req.GetNightOffset() + req.GetNightCount()); ++n) {
            size_t value = GetValueByShowMode(stats.ByNights[n], req.GetShowMode());
            auto it = LowerBoundBy(Buckets_.begin(), Buckets_.end(), value, GetOffersFromBucket);
            TString color;
            if (it == Buckets_.end()) {
                color = Buckets_.rbegin()->GetColor();
            } else {
                color = it->GetColor();
            }
            if (req.GetShadesOfRed()) {
                color = ConvertColorIntoShadesOfRed(color);
            }
            out << "<td align=\"right\" bgcolor=\" " << color << "\">&nbsp;" << value;
            if (req.GetWithPercentage()) {
                double percentage = 0.0;
                if (stats.ByNights[n].RecordCount > 0) {
                    percentage = 100.0 * value / stats.ByNights[n].RecordCount;
                }
                out << "&nbsp;(" << FloatToString(percentage, PREC_POINT_DIGITS, 1) << "%)";
            }
            out << "</td>";
        }
        out << "</tr>";
    }
    out << "</table></body></html>";
    return out;
}

void TCacheHeatmap::BuildProtoCacheHeatmap(const NTravelProto::NOfferCache::NApi::THeatmapReq& req, NTravelProto::NOfferCache::NApi::THeatmapResp* resp) const {
    auto cur = CurrencyFromOCFormat(req.GetCurrency());
    NOrdinalDate::TOrdinalDate dtBegin = NOrdinalDate::FromInstant(::Now()) + req.GetDateOffset();
    for (size_t dtAdd = 0; dtAdd < req.GetDateCount(); ++dtAdd) {
        NOrdinalDate::TOrdinalDate dt = dtBegin + dtAdd;
        auto pbDayStat = resp->AddByDate();
        pbDayStat->SetDate(NOrdinalDate::ToString(dt));
        auto stats = Cache_.GetDayStat(cur, dt);
        for (TNights n = req.GetNightOffset(); n < (req.GetNightOffset() + req.GetNightCount()); ++n) {
            pbDayStat->AddOffersByNights(GetValueByShowMode(stats.ByNights[n], req.GetShowMode()));
        }
    }
}

void TCacheHeatmap::BuildCacheHeatmap(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) const {
    NTravelProto::NOfferCache::NApi::THeatmapReq req;
    try {
        NTravel::NProtobuf::ParseCgiRequest(httpReq.Query(), &req);
    } catch (...) {
        ERROR_LOG << "Bad Heatmap request: " << CurrentExceptionMessage() << Endl;
        responseCb(NHttp::TResponse::CreateText(CurrentExceptionMessage() + "\n", HTTP_BAD_REQUEST));
        return;
    }
    if (req.GetAsJSON()) {
        NTravelProto::NOfferCache::NApi::THeatmapResp resp;
        BuildProtoCacheHeatmap(req, &resp);
        responseCb(NHttp::TResponse::CreateJson(resp));
    } else {
        responseCb(NHttp::TResponse::CreateHTML(BuildHTMLCacheHeatmap(req)));
    }
}

}// namespace NOfferCache
}// namespace NTravel
