#include "counters.h"

#include "service.h"

namespace NTravel {
namespace NOfferCache {

static std::initializer_list<int> g_TimeBuckets = {100, 500, 1000, 2500, 5000, 10000, 15000, 30000, 40000, 50000, 60000, 70000, 80000, 90000, 100000};
static std::initializer_list<int> g_RespSizeBuckets = {1, 5, 10, 25, 50, 100, 250, 500, 750, 1000, 1001};
static std::initializer_list<int> g_LabelSizeBuckets = {200, 400, 600, 800, 1000, 1500, 2000};
static std::initializer_list<int> g_PctBuckets = {0, 1, 5, 10, 25, 50, 75, 90, 100};

TServiceCounters::TServiceCounters(TService& service)
    : NCacheTimeXus(g_TimeBuckets)
    , NCacheTimeXusMid(g_TimeBuckets)
    , NCacheTimeXusBig(g_TimeBuckets)
    , NCacheTimeXusLarge(g_TimeBuckets)
    , NCacheTimeXusExtraLarge(g_TimeBuckets)
    , RespSizeXKb(g_RespSizeBuckets)
    , RespSizeXKbMid(g_RespSizeBuckets)
    , RespSizeXKbBig(g_RespSizeBuckets)
    , RespSizeXKbLarge(g_RespSizeBuckets)
    , RespSizeXKbExtraLarge(g_RespSizeBuckets)
    , LabelSize(g_LabelSizeBuckets)
    , NCatRoomOtherPctLess(g_PctBuckets)
    , BaseFiltersWasteHotelsPct(g_PctBuckets)
    , Service_(service)
{
}

void TServiceCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    // HTTP Only
    ct->insert(MAKE_COUNTER_PAIR(NHttpJobs));
    ct->insert(MAKE_COUNTER_PAIR(NRequestParseErrors));
    NCacheTimeXus.QueryCounters("CacheTimeLess", "", ct);
    NCacheTimeXusMid.QueryCounters("CacheTimeMidLess", "", ct);
    NCacheTimeXusBig.QueryCounters("CacheTimeBigLess", "", ct);
    NCacheTimeXusLarge.QueryCounters("CacheTimeLargeLess", "", ct);
    NCacheTimeXusExtraLarge.QueryCounters("CacheTimeExtraLargeLess", "", ct);
    RespSizeXKb.QueryCounters("RespSizeLess", "Kb", ct);
    RespSizeXKbMid.QueryCounters("RespSizeMidLess", "Kb", ct);
    RespSizeXKbBig.QueryCounters("RespSizeBigLess", "Kb", ct);
    RespSizeXKbLarge.QueryCounters("RespSizeLargeLess", "Kb", ct);
    RespSizeXKbExtraLarge.QueryCounters("RespSizeExtraLargeLess", "Kb", ct);

    LabelSize.QueryCounters("LabelSize", "", ct);

    // gRPC only
    ct->insert(MAKE_COUNTER_PAIR(NGrpcJobs));
    ct->insert(MAKE_COUNTER_PAIR(NGrpcRequests));

    // gRPC Searcher client
    ct->insert(MAKE_COUNTER_PAIR(NSearcherRequests));
    ct->insert(MAKE_COUNTER_PAIR(NSearcherInFly));

    // Общесервисная всячина
    IsReady = Service_.IsReady() ? 1: 0;
    PingIsMissing = Service_.IsPinged() ? 0 : 1;
    ct->insert(MAKE_COUNTER_PAIR(IsReady));
    ct->insert(MAKE_COUNTER_PAIR(PingIsMissing));
    ct->insert(MAKE_COUNTER_PAIR(MsgParseNs));
    ct->insert(MAKE_COUNTER_PAIR(MsgProcessCacheNs));
    ct->insert(MAKE_COUNTER_PAIR(MsgProcessReqCacheNs));
    ct->insert(MAKE_COUNTER_PAIR(MsgProcessOtherNs));
    ct->insert(MAKE_COUNTER_PAIR(MsgProcessEncodingNs));

    // Catroom-related
    // Статусы показов категорий номеров
    ct->insert(MAKE_COUNTER_PAIR(NCatRoomOK));
    ct->insert(MAKE_COUNTER_PAIR(NCatRoomOnlyOther));
    ct->insert(MAKE_COUNTER_PAIR(NCatRoomPermalinkNotPublished));
    ct->insert(MAKE_COUNTER_PAIR(NCatRoomWrongPermaroomId));
    ct->insert(MAKE_COUNTER_PAIR(NCatRoomEmptyPermaroomId));

    NCatRoomOtherPctLess.QueryCounters("NCatRoomOtherPctLess", "", ct);
    ct->insert(MAKE_COUNTER_PAIR(NCatRoomHiddenOffers));

    ct->insert(MAKE_COUNTER_PAIR(DummyPayloadSize));

    ct->insert(MAKE_COUNTER_PAIR(BaseFiltersHasOffers));
    ct->insert(MAKE_COUNTER_PAIR(BaseFiltersSkippedByUserFilter));
    ct->insert(MAKE_COUNTER_PAIR(BaseFiltersSkippedOther));
    ct->insert(MAKE_COUNTER_PAIR(BaseFiltersNoOffers));
    BaseFiltersWasteHotelsPct.QueryCounters("BaseFiltersWasteHotelsPct", "", ct);

    ct->insert(MAKE_COUNTER_PAIR(NoTravellinePartnerSpecificData));
    ct->insert(MAKE_COUNTER_PAIR(NoDolphinPartnerSpecificData));
    ct->insert(MAKE_COUNTER_PAIR(NoBNovoPartnerSpecificData));
    ct->insert(MAKE_COUNTER_PAIR(ExtraPartnerSpecificData));
    ct->insert(MAKE_COUNTER_PAIR(PartnerSpecificDataInOutdatedRecord));

    ct->insert(MAKE_COUNTER_PAIR(NoPartnerSpecificDataInStoredOffer));

    ct->insert(MAKE_COUNTER_PAIR(UnknownTravellineRatePlan));
    ct->insert(MAKE_COUNTER_PAIR(UnknownDolphinTour));
    ct->insert(MAKE_COUNTER_PAIR(UnknownDolphinPansion));
    ct->insert(MAKE_COUNTER_PAIR(UnknownDolphinRoom));
    ct->insert(MAKE_COUNTER_PAIR(UnknownDolphinRoomCat));
    ct->insert(MAKE_COUNTER_PAIR(UnknownBNovoRatePlan));

    ct->insert(MAKE_COUNTER_PAIR(NMultipleBoYDirect));

    ct->insert(MAKE_COUNTER_PAIR(UnknownOperatorForCatRoomMapping));

    ct->insert(MAKE_COUNTER_PAIR(InvalidRefundRulesError));
    ct->insert(MAKE_COUNTER_PAIR(InvalidRefundRulesWarn));
    ct->insert(MAKE_COUNTER_PAIR(FractionalRefundRulePenalty));

    ct->insert(MAKE_COUNTER_PAIR(InvalidMirPromo));
}

TServiceCountersPerOperator::TServiceCountersPerOperator()
    : NCatRoomOtherPctLess(g_PctBuckets)
{
}

void TServiceCountersPerOperator::QueryCounters(NMonitor::TCounterTable* ct) const {
    NCatRoomOtherPctLess.QueryCounters("NCatRoomOtherPctLess", "", ct);
}

void TServiceCountersPerPartner::IncCacheHitCounter(NTravelProto::NOfferCache::NApi::EHotelCacheStatus currentPartnerStatus, NTravelProto::NOfferCache::NApi::EHotelCacheStatus otherPartnersStatus) {
    NCacheStatus[currentPartnerStatus][otherPartnersStatus].Inc();
}

TString TServiceCountersPerPartner::GetCacheHitStatusName(NTravelProto::NOfferCache::NApi::EHotelCacheStatus st) {
    static const TString result[]  = {
        "Miss", "Hit", "Empty", "Error", "Restr"
    };
    return result[st];

}

void TServiceCountersPerPartner::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NSearchSubKeyRestricted));
    for (size_t cur = NTravelProto::NOfferCache::NApi::EHotelCacheStatus_MIN; cur <= NTravelProto::NOfferCache::NApi::EHotelCacheStatus_MAX; ++cur) {
        for (size_t others = NTravelProto::NOfferCache::NApi::EHotelCacheStatus_MIN; others <= NTravelProto::NOfferCache::NApi::EHotelCacheStatus_MAX; ++others) {
            TString name = "NCacheStatusCur" + GetCacheHitStatusName((NTravelProto::NOfferCache::NApi::EHotelCacheStatus)cur)
                + "Others" + GetCacheHitStatusName((NTravelProto::NOfferCache::NApi::EHotelCacheStatus)others);
            ct->insert(MAKE_NAMED_COUNTER_PAIR(name, NCacheStatus[cur][others]));
        }
    }
}

void TCatRoomCountersPerDataSource::QueryCounters(NMonitor::TCounterTable* ct) const {
    // Catroom-related
    ct->insert(MAKE_COUNTER_PAIR(NStatusOK));
    ct->insert(MAKE_COUNTER_PAIR(NStatusNoMarkup));
    ct->insert(MAKE_COUNTER_PAIR(NStatusTooMuchOther));
    ct->insert(MAKE_COUNTER_PAIR(NStatusWrongPermaroomId));
    ct->insert(MAKE_COUNTER_PAIR(NStatusEmptyPermaroomId));
}


TSourceCounters::TSourceCounters()
    : RequestHotelCount({1, 3, 7, 12, 20, 30, 50, 100, 250, 500, 1000})
    , RequestSubHotelCount({1, 10, 25, 50, 100, 250, 500, 1000, 2500, 5000, 10000})
{
}

void TSourceCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    // HTTP-only запросы
    ct->insert(MAKE_COUNTER_PAIR(NRequests));
    ct->insert(MAKE_COUNTER_PAIR(NRequestsCache));
    ct->insert(MAKE_COUNTER_PAIR(NRequestsCacheFull));
    ct->insert(MAKE_COUNTER_PAIR(NRequestsCacheSearcher));
    ct->insert(MAKE_COUNTER_PAIR(NRequestsCacheSearcherFull));
    ct->insert(MAKE_COUNTER_PAIR(NRequestsSearcher));
    ct->insert(MAKE_COUNTER_PAIR(NRequestsErrors));
    ct->insert(MAKE_COUNTER_PAIR(NRequestsWrongNights));
    ct->insert(MAKE_COUNTER_PAIR(NRequestsCheckInInPast));
    ct->insert(MAKE_COUNTER_PAIR(NRequestsCheckInAndOutInPast));
    ct->insert(MAKE_COUNTER_PAIR(NRequestsWrongAges));

    ct->insert(MAKE_COUNTER_PAIR(NPermalinksEmpty));
    ct->insert(MAKE_COUNTER_PAIR(NPermalinksGreylisted));
    ct->insert(MAKE_COUNTER_PAIR(NPermalinksBlacklisted));
    ct->insert(MAKE_COUNTER_PAIR(NPermalinksPartiallyBlacklisted));
    ct->insert(MAKE_COUNTER_PAIR(NPermalinksWizardBanned));
    ct->insert(MAKE_COUNTER_PAIR(NPermalinksOK));

    ct->insert(MAKE_COUNTER_PAIR(NMainPermalinksSearchFormHidden));
    ct->insert(MAKE_COUNTER_PAIR(NMainPermalinksOK));

    // HTTP-only отели в запросах
    ct->insert(MAKE_COUNTER_PAIR(NHotelNoTravelId));
    ct->insert(MAKE_COUNTER_PAIR(NHotelNoPartnerIdsNoTravelIdRecovered));
    ct->insert(MAKE_COUNTER_PAIR(NHotelNoPartnerIdsNoTravelId));
    RequestHotelCount.QueryCounters("RequestHotelCount", "", ct);
    RequestSubHotelCount.QueryCounters("RequestSubHotelCount", "", ct);

    // HTTP & gRPC service - независимые счетчики на каждый HotelId в запросе
    ct->insert(MAKE_COUNTER_PAIR(NSubReq));
    ct->insert(MAKE_COUNTER_PAIR(NSubReqError));
    ct->insert(MAKE_COUNTER_PAIR(NSubReqWrongField));
    ct->insert(MAKE_COUNTER_PAIR(NSubReqUnknownPartnerCode));
    ct->insert(MAKE_COUNTER_PAIR(NSubReqBlacklisted));
    ct->insert(MAKE_COUNTER_PAIR(NSubReqDroppedByWhitelist));

    // generic, Searcher - запросы/ответы по подзапросам
    ct->insert(MAKE_COUNTER_PAIR(NSearcherSubReq));
    ct->insert(MAKE_COUNTER_PAIR(NSearcherSubReqInfly));
    ct->insert(MAKE_COUNTER_PAIR(NSearcherSubRespError));
    ct->insert(MAKE_COUNTER_PAIR(NSearcherSubRespOK));

    // Cache records
    ct->insert(MAKE_COUNTER_PAIR(NCacheRecords));
    ct->insert(MAKE_COUNTER_PAIR(NCacheRecordsFull));
    ct->insert(MAKE_COUNTER_PAIR(NCacheRecordsEmpty));
    ct->insert(MAKE_COUNTER_PAIR(NCacheRecordsOutdated));

    // HTTP-only ответы
    ct->insert(MAKE_COUNTER_PAIR(NRespInProgress));
    ct->insert(MAKE_COUNTER_PAIR(NRespCacheHit));
    ct->insert(MAKE_COUNTER_PAIR(NRespCacheHitEmpty));
    ct->insert(MAKE_COUNTER_PAIR(NRespCacheMiss));

    ct->insert(MAKE_COUNTER_PAIR(NInitialCacheHitContrib));
}

void TCacheHitCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    // HTTP & gRPC service - независимые счетчики на каждый партнёрский HotelId в запросе
    ct->insert(MAKE_COUNTER_PAIR(NCacheHit));
    ct->insert(MAKE_COUNTER_PAIR(NCacheHitEmpty));
    ct->insert(MAKE_COUNTER_PAIR(NCacheMiss));

    // HTTP - per permalink cache status
    ct->insert(MAKE_COUNTER_PAIR(NPermalinkCacheMiss));
    ct->insert(MAKE_COUNTER_PAIR(NPermalinkCacheHitPartialEmpty));
    ct->insert(MAKE_COUNTER_PAIR(NPermalinkCacheHitPartial));
    ct->insert(MAKE_COUNTER_PAIR(NPermalinkCacheHitFullEmpty));
    ct->insert(MAKE_COUNTER_PAIR(NPermalinkCacheHitFull));
}

void TOfferShowCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NAnyOffer));
    ct->insert(MAKE_COUNTER_PAIR(NFirstOffer));
    ct->insert(MAKE_COUNTER_PAIR(NBoYHotelFirstOffer));
    ct->insert(MAKE_COUNTER_PAIR(NDirectBoYHotelFirstOffer));
}

void TServiceCountersPerStage::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(TimeMicroseconds));
}

void TServiceCountersPerBlendingStage::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(TimeMicroseconds));
}

}// namespace NOfferCache
}// namespace NTravel
