#pragma once

#include "travel/hotels/proto/offercache_api/enums.pb.h"

#include <travel/hotels/lib/cpp/mon/tools.h>
#include <travel/hotels/lib/cpp/mon/counter.h>
#include <travel/hotels/lib/cpp/mon/counter_hypercube.h>

namespace NTravel {
namespace NOfferCache {

class TService;

//---------------------------------------------------------------------------------------
struct TServiceCounters : public NMonitor::TCounterSource {
    // Dimensions:
    //   None

    // HTTP Only
    NMonitor::TCounter          NHttpJobs;
    NMonitor::TDerivCounter     NRequestParseErrors;   // совсем ошибка в параметрах HTTP запроса
    // Скорости выполнения HTTP запроса, Cache в названии по историческим причинам
    NMonitor::TDerivHistogramCounter NCacheTimeXus;      // 1 <= Hotels < Mid
    NMonitor::TDerivHistogramCounter NCacheTimeXusMid;   // Mid <= Hotels < Big
    NMonitor::TDerivHistogramCounter NCacheTimeXusBig;   // Big <= Hotels < Large
    NMonitor::TDerivHistogramCounter NCacheTimeXusLarge; // Large <= Hotels < ExtraLarge
    NMonitor::TDerivHistogramCounter NCacheTimeXusExtraLarge; // ExtraLarge <= Hotels

    NMonitor::TDerivHistogramCounter RespSizeXKb;      // 1 <= Hotels < Mid
    NMonitor::TDerivHistogramCounter RespSizeXKbMid;   // Mid <= Hotels < Big
    NMonitor::TDerivHistogramCounter RespSizeXKbBig;   // Big <= Hotels < Large
    NMonitor::TDerivHistogramCounter RespSizeXKbLarge; // Large <= Hotels < ExtraLarge
    NMonitor::TDerivHistogramCounter RespSizeXKbExtraLarge; // ExtraLarge <= Hotels

    NMonitor::THistogramCounter LabelSize;

    // gRPC only
    NMonitor::TCounter          NGrpcJobs;
    NMonitor::TDerivCounter     NGrpcRequests;

    // gRPC Searcher client
    NMonitor::TDerivCounter     NSearcherRequests;
    NMonitor::TCounter          NSearcherInFly;

    // Общесервисная всячина
    mutable NMonitor::TCounter  IsReady;
    mutable NMonitor::TCounter  PingIsMissing; // Давно не пинга от балансера (негативная логика лучше для алертов соломона)

    NMonitor::TDerivCounter     MsgParseNs;
    NMonitor::TDerivCounter     MsgProcessCacheNs;
    NMonitor::TDerivCounter     MsgProcessReqCacheNs;
    NMonitor::TDerivCounter     MsgProcessEncodingNs;
    NMonitor::TDerivCounter     MsgProcessOtherNs;

    // Catroom-related
    // Статусы показов категорий номеров
    NMonitor::TDerivCounter     NCatRoomOK; // Успешных показов пермарумов
    NMonitor::TDerivCounter     NCatRoomOnlyOther; // Были только "прочие" пермарумы
    NMonitor::TDerivCounter     NCatRoomPermalinkNotPublished; // Не показан из-за неопубликованности
    NMonitor::TDerivCounter     NCatRoomWrongPermaroomId; // Проставлен неизвестный PermaroomId (внутренняя ошибка, так быть не должно)
    NMonitor::TDerivCounter     NCatRoomEmptyPermaroomId; // Проставлен пустой PermaroomId (внутренняя ошибка, так быть не должно)

    NMonitor::TDerivHistogramCounter NCatRoomOtherPctLess; // Процент прочих пермарумов. 0 - все сматчилось, 100 - ничего не сматчилось
    NMonitor::TDerivCounter          NCatRoomHiddenOffers; // Кол-во скрытых офферов (из-за ShowOther=false)

    NMonitor::TCounter          DummyPayloadSize;

    // Метрики для измерения качества фильтрации на базовом: сколько отелей запроса остались без офферов (а значит зря попали в выдачу)
    NMonitor::TDerivCounter BaseFiltersHasOffers;
    NMonitor::TDerivCounter BaseFiltersSkippedByUserFilter;
    NMonitor::TDerivCounter BaseFiltersSkippedOther;
    NMonitor::TDerivCounter BaseFiltersNoOffers;
    NMonitor::TDerivHistogramCounter BaseFiltersWasteHotelsPct;

    NMonitor::TDerivCounter NoTravellinePartnerSpecificData;
    NMonitor::TDerivCounter NoDolphinPartnerSpecificData;
    NMonitor::TDerivCounter NoBNovoPartnerSpecificData;
    NMonitor::TDerivCounter ExtraPartnerSpecificData;
    NMonitor::TDerivCounter PartnerSpecificDataInOutdatedRecord;

    NMonitor::TDerivCounter NoPartnerSpecificDataInStoredOffer;

    NMonitor::TDerivCounter UnknownTravellineRatePlan;
    NMonitor::TDerivCounter UnknownDolphinTour;
    NMonitor::TDerivCounter UnknownDolphinPansion;
    NMonitor::TDerivCounter UnknownDolphinRoom;
    NMonitor::TDerivCounter UnknownDolphinRoomCat;
    NMonitor::TDerivCounter UnknownBNovoRatePlan;

    NMonitor::TDerivCounter NMultipleBoYDirect;

    NMonitor::TDerivCounter UnknownOperatorForCatRoomMapping;

    NMonitor::TDerivCounter InvalidRefundRulesError;
    NMonitor::TDerivCounter InvalidRefundRulesWarn;
    NMonitor::TDerivCounter FractionalRefundRulePenalty;

    NMonitor::TDerivCounter InvalidMirPromo;

    TService& Service_;

    TServiceCounters(TService& service);
    void QueryCounters(NMonitor::TCounterTable* ct) const override;
};
using TServiceCountersRef = TAtomicSharedPtr<TServiceCounters>;

//---------------------------------------------------------------------------------------

struct TServiceCountersPerOperator : public NMonitor::TCounterSource {
    // Dimensions:
    //   1. Operator

    NMonitor::TDerivHistogramCounter NCatRoomOtherPctLess; // Процент прочих пермарумов. 0 - все сматчилось, 100 - ничего не сматчилось

    TServiceCountersPerOperator();
    void QueryCounters(NMonitor::TCounterTable* ct) const override;
};
using TServiceCountersPerOperatorRef = TAtomicSharedPtr<TServiceCountersPerOperator>;

//---------------------------------------------------------------------------------------
struct TServiceCountersPerPartner : public NMonitor::TCounterSource {
    // Dimensions:
    //   1. Partner

    NMonitor::TDerivCounter NSearchSubKeyRestricted;

    NMonitor::TDerivCounter NCacheStatus[NTravelProto::NOfferCache::NApi::EHotelCacheStatus_ARRAYSIZE][NTravelProto::NOfferCache::NApi::EHotelCacheStatus_ARRAYSIZE];

    void IncCacheHitCounter(NTravelProto::NOfferCache::NApi::EHotelCacheStatus currentPartnerStatus, NTravelProto::NOfferCache::NApi::EHotelCacheStatus otherPartnersStatus);
    static TString GetCacheHitStatusName(NTravelProto::NOfferCache::NApi::EHotelCacheStatus st);
    void QueryCounters(NMonitor::TCounterTable* ct) const override;
};
using TServiceCountersPerPartnerRef = TAtomicSharedPtr<TServiceCountersPerPartner>;

//---------------------------------------------------------------------------------------

struct TCatRoomCountersPerDataSource: public NMonitor::TCounterSource {
    // Dimensions:
    //   1. CatroomDataSource

    NMonitor::TDerivCounter     NStatusOK; // Успешных показов пермарумов
    NMonitor::TDerivCounter     NStatusNoMarkup;
    NMonitor::TDerivCounter     NStatusTooMuchOther;
    NMonitor::TDerivCounter     NStatusWrongPermaroomId;
    NMonitor::TDerivCounter     NStatusEmptyPermaroomId;

    void QueryCounters(NMonitor::TCounterTable* ct) const override;
};
using TCatRoomCountersPerDataSourceRef = TAtomicSharedPtr<TCatRoomCountersPerDataSource>;

//---------------------------------------------------------------------------------------

struct TSourceCounters : public NMonitor::TCounterSource {
    // Dimensions:
    //   1. OfferCacheClientId

    // HTTP-only запросы
    NMonitor::TDerivCounter     NRequests;              // Все запросы
    NMonitor::TDerivCounter     NRequestsCache;         // Запросы чисто к кэшу Full=0
    NMonitor::TDerivCounter     NRequestsCacheFull;     // Запросы чисто к кэшу Full=1
    NMonitor::TDerivCounter     NRequestsCacheSearcher; // запросы кэшу и к сёрчеру, Full=0
    NMonitor::TDerivCounter     NRequestsCacheSearcherFull; // запросы кэшу и к сёрчеру, Full=1
    NMonitor::TDerivCounter     NRequestsSearcher;      // запросы ТОЛЬКО к сёрчеру
    NMonitor::TDerivCounter     NRequestsErrors;        // ошибка в параметрах
    NMonitor::TDerivCounter     NRequestsWrongNights;   //
    NMonitor::TDerivCounter     NRequestsCheckInInPast;
    NMonitor::TDerivCounter     NRequestsCheckInAndOutInPast;
    NMonitor::TDerivCounter     NRequestsWrongAges;

    NMonitor::TDerivCounter     NPermalinksEmpty;                // количество пермалинков, для которых не известно ни одного original id-а
    NMonitor::TDerivCounter     NPermalinksGreylisted;           // original id-ы Permalink-а находятся в HotelsGreylist
    NMonitor::TDerivCounter     NPermalinksBlacklisted;          // в точности все original id-ы Permalink-а находятся в HotelsBlacklist
    NMonitor::TDerivCounter     NPermalinksPartiallyBlacklisted; // только часть (но не все) original id-ов Permalink-а забанена HotelsBlacklist
    NMonitor::TDerivCounter     NPermalinksWizardBanned;         // Колдун забанен для пермалинка => поиск не выполняется
    NMonitor::TDerivCounter     NPermalinksOK;                   // количество пермалинков, не попадающих ни под одну из категорий выше

    NMonitor::TDerivCounter     NMainPermalinksSearchFormHidden; // Форма выбора дата скрыта
    NMonitor::TDerivCounter     NMainPermalinksOK;               //

    // HTTP-only отели в запросах
    NMonitor::TDerivCounter          NHotelNoTravelId;                      // TravelId-, PartnerIds+
    NMonitor::TDerivCounter          NHotelNoPartnerIdsNoTravelIdRecovered; // TravelId-, PartnerIds-, recovered (by permalink)
    NMonitor::TDerivCounter          NHotelNoPartnerIdsNoTravelId;          // TravelId-, PartnerIds-, not recovered (by permalink)
    NMonitor::TDerivHistogramCounter RequestHotelCount;                     // Количество отелей в запросе
    NMonitor::TDerivHistogramCounter RequestSubHotelCount;                  // Количество подотелей отелей в запросе

    // HTTP & gRPC service - независимые счетчики на каждый партнёрский HotelId в запросе
    NMonitor::TDerivCounter     NSubReq;
    NMonitor::TDerivCounter     NSubReqError;
    NMonitor::TDerivCounter     NSubReqWrongField;
    NMonitor::TDerivCounter     NSubReqUnknownPartnerCode;
    NMonitor::TDerivCounter     NSubReqBlacklisted;
    NMonitor::TDerivCounter     NSubReqDroppedByWhitelist;

    // generic, Searcher - запросы/ответы по подзапросам
    NMonitor::TDerivCounter     NSearcherSubReq;
    NMonitor::TCounter          NSearcherSubReqInfly;
    NMonitor::TDerivCounter     NSearcherSubRespError;
    NMonitor::TDerivCounter     NSearcherSubRespOK;

    // Cache records
    NMonitor::TCounter          NCacheRecords;
    NMonitor::TCounter          NCacheRecordsFull;
    NMonitor::TCounter          NCacheRecordsEmpty;
    NMonitor::TCounter          NCacheRecordsOutdated;

    // HTTP-only ответы
    NMonitor::TDerivCounter     NRespInProgress;
    NMonitor::TDerivCounter     NRespCacheHit;
    NMonitor::TDerivCounter     NRespCacheHitEmpty;  // Кэш хит, но предложений нет
    NMonitor::TDerivCounter     NRespCacheMiss;

    NMonitor::TDerivCounter NInitialCacheHitContrib; // Кэшхит по офферам, инициатором создания которых является данный источник

    TSourceCounters();
    void QueryCounters(NMonitor::TCounterTable* ct) const override;
};
using TSourceCountersRef = TAtomicSharedPtr<TSourceCounters>;

//---------------------------------------------------------------------------------------

enum class ERequestType {
    Test,
    Initial,
    Full,
    Searcher,
    SearcherFull,
    OnlySearcher,
};
// Cube dimension: Permalink type
enum class EPermalinkType {
    // Members of this enum should have default values (from 0 to cnt - 1) to allow enumeration in http_job
    Main,
    Usual,
    Similar,
};

//---------------------------------------------------------------------------------------

struct TCacheHitCounters : public NMonitor::TCounterSource {
    // Dimensions:
    //   1. ERequestType
    //   2. EPermalinkType
    //   3. Source (OfferCacheClientId)

    // HTTP & gRPC service - независимые счетчики на каждый партнёрский HotelId в запросе
    NMonitor::TDerivCounter     NCacheHit;
    NMonitor::TDerivCounter     NCacheHitEmpty;
    NMonitor::TDerivCounter     NCacheMiss;

    // HTTP - per permalink cache status
    NMonitor::TDerivCounter     NPermalinkCacheMiss;            // Записи в кэше не найдены
    NMonitor::TDerivCounter     NPermalinkCacheHitPartialEmpty; // Найдены, но не все, и они - пустые
    NMonitor::TDerivCounter     NPermalinkCacheHitPartial;      // Найдены, но не все, хотя бы одна - не пустая
    NMonitor::TDerivCounter     NPermalinkCacheHitFullEmpty;    // Все записи найдены и они все пустые
    NMonitor::TDerivCounter     NPermalinkCacheHitFull;         // Все записи найдены, хотя бы одна - не пустая

    void QueryCounters(NMonitor::TCounterTable* ct) const override;
};
using TCacheHitCountersRef = TAtomicSharedPtr<TCacheHitCounters>;

//---------------------------------------------------------------------------------------

struct TOfferShowCounters : public NMonitor::TCounterSource {
    // Dimensions:
    //   1. ERequestType
    //   2. EPermalinkType
    //   3. EOperatorId
    NMonitor::TDerivCounter     NAnyOffer;// Любой показ оффера
    NMonitor::TDerivCounter     NFirstOffer;// показ первого оффера для любого отеля
    NMonitor::TDerivCounter     NBoYHotelFirstOffer;// Показ первого оффера для отеля с BoY провязками
    NMonitor::TDerivCounter     NDirectBoYHotelFirstOffer;// Показ первого оффера для отеля с Direct-BoY провязками

    void QueryCounters(NMonitor::TCounterTable* ct) const override;
};
using TOfferShowCountersRef = TAtomicSharedPtr<TOfferShowCounters>;

//---------------------------------------------------------------------------------------

enum class ERequestSize {
    Small,
    Medium,
    Big,
    Large,
    ExtraLarge,
};

enum class ERequestStage {
    Init,
    Start,
    Parse,
    SearchInCache,
    DoSearcherRequest,
    ReplyFieldsBeforePutRecordsToResp,
    ReplyFieldsPutRecordsToResp,
    ReplyFieldsAfterPutRecordsToResp,
    ReplyBuildJson,
    ReplyResponseCb,
};

struct TServiceCountersPerStage: public NMonitor::TCounterSource {
    // Dimensions:
    //   1. ERequestSize
    //   2. ERequestStage
    //   3. UseSearcher

    NMonitor::TDerivCounter TimeMicroseconds;

    void QueryCounters(NMonitor::TCounterTable* ct) const override;
};
using TServiceCountersPerStageRef = TAtomicSharedPtr<TServiceCountersPerStage>;

//---------------------------------------------------------------------------------------

enum class EBlendingStage {
    Init,
    Start,
    GetAllOffers,
    RemoveDisabledOperators,
    RemoveBannedOfferTypes,
    RemoveRestrictedOffers,
    RemoveExperiments,
    RemoveDuplicatedOffersBeforeScanPermarooms,
    ScanPermarooms,
    ProcessCatRoomStatus,
    RemoveDuplicatedOffersAfterScanPermarooms,
    SelectBestBoY,
    ApplyAuxFilters,
    FillUserFiltersInfo,
    ApplyUserFilters,
    ApplyAuxPostUserFilters,
    FillFullStats,
    ReduceOffers,
    SortOffers,
    FillDefaultOffer,
    BumpOffers,
    CheckMedianMinPrice,
    CheckMinPriceBadge,
    PutOffersToResp,
    FillMirCashbackAvailability,
    LogSkippedOffers,
};

struct TServiceCountersPerBlendingStage: public NMonitor::TCounterSource {
    // Dimensions:
    //   1. NTravelProto::NOfferCache::NApi::ERespMode
    //   2. bool full
    //   3. EBlendingStage

    NMonitor::TDerivCounter TimeMicroseconds;

    void QueryCounters(NMonitor::TCounterTable* ct) const override;
};
using TServiceCountersPerBlendingStageRef = TAtomicSharedPtr<TServiceCountersPerBlendingStage>;

}// namespace NOfferCache
}// namespace NTravel
