#include "data.h"

#include <util/digest/multi.h>
#include <util/string/cast.h>
#include <util/stream/str.h>
#include <travel/hotels/lib/cpp/util/sizes.h>
#include <travel/hotels/lib/cpp/protobuf/tools.h>

#include <travel/hotels/proto/data_config/promo/mir_white_list.pb.h>

#include <travel/proto/user_order_counters/user_order_counters.pb.h>

namespace NTravel {
namespace NOfferCache {

//----------------

EOCPansion PansionToOCFormat(NTravelProto::EPansionType pt) {
    switch (pt) {
#define PT(_V_) case NTravelProto::PT_##_V_: return NTravelProto::NOfferCache::NApi::TOCPansion::_V_
        PT(UNKNOWN);
        PT(AI);
        PT(BB);
        PT(FB);
        PT(HB);
        PT(RO);
        PT(UAI);
        PT(LAI);
        PT(BD);
#undef PT
        default:
            throw yexception() << "Unknown PansionType: " << (int)pt;
    }
}

NTravelProto::ECurrency CurrencyFromOCFormat(NTravelProto::NOfferCache::NApi::TOCCurrency::ECurrency c) {
    switch (c) {
#define C(_V_) case NTravelProto::NOfferCache::NApi::TOCCurrency::_V_: return NTravelProto::C_##_V_
        C(UNKNOWN);
        C(RUB);
        C(USD);
#undef C
    }
}

bool TCacheSubKey::operator == (const TCacheSubKey& rhs) const {
    return Capacity == rhs.Capacity && Date == rhs.Date && Nights == rhs.Nights;
}

size_t TCacheSubKey::Hash() const {
    return MultiHash(Capacity, Date, Nights);
}

size_t TCacheSubKey::GetAllocSize() const {
    return Capacity.GetAllocSize();
}

//----------------

bool TCacheKey::operator == (const TCacheKey& rhs) const {
    return PreKey == rhs.PreKey && SubKey == rhs.SubKey;
}

size_t TCacheKey::Hash() const {
    return MultiHash(PreKey, SubKey);
}

size_t TCacheKey::GetAllocSize() const {
    return PreKey.GetAllocSize() + SubKey.GetAllocSize();
}

size_t TCacheRecord::GetAllocSize() const {
    // OfferCacheClientId is shared, thus not counted here
    size_t allocSize = Key.GetAllocSize() + TTotalByteSize<TString>()(SearcherReqId) - sizeof(SearcherReqId);
    for (const auto& offer: Offers) {
        allocSize += sizeof offer + offer.GetAllocSize();
    }
    if (SearchWarnings) {
        allocSize += sizeof(TSearchWarnings) + SearchWarnings->WarningCounts.capacity() * sizeof(decltype(SearchWarnings->WarningCounts)::value_type);
    }
    return allocSize;
}

bool TRefundRule::operator==(const TRefundRule& rhs) const {
    return Type == rhs.Type &&
        Penalty == rhs.Penalty &&
        StartsAtTimestampSec == rhs.StartsAtTimestampSec &&
        EndsAtTimestampSec == rhs.EndsAtTimestampSec;
}

size_t TRefundRule::Hash() const {
    return MultiHash(Type, Penalty, StartsAtTimestampSec, EndsAtTimestampSec);
}

size_t TRefundRule::CalcTotalByteSize() const {
    return sizeof(TRefundRule);
}

bool TOfferId::operator == (const TOfferId& rhs) const {
    return Value == rhs.Value;
}

size_t TOfferId::Hash() const {
    return THash<TGUID>()(Value);
}

TString TOfferId::AsString() const {
    return Value.AsUuidString();
}

size_t TOfferId::GetAllocSize() const {
    return 0;
}

TOfferId TOfferId::TryFromString(const TString& s) {
    return DoFromString(s, false);
}

TOfferId TOfferId::FromString(const TString& s) {
    return DoFromString(s, true);
}

TOfferId TOfferId::DoFromString(const TString& s, bool throwOnError) {
    TOfferId result{};
    auto valueIsParsed = GetUuid(s, result.Value);
    if (!valueIsParsed) {
        if (throwOnError) {
            throw yexception() << "Can't parse offerId as uuid (" << s << ")";
        } else {
            return {};
        }
    }
    auto resultIsValid = (ToString(result) == s);
    if (!resultIsValid) {
        if (throwOnError) {
            throw yexception() << "ToString(OfferId) != initial offer id (" << result << " != " << s << ")";
        } else {
            return {};
        }
    }
    return result;
}

bool TOfferTitleAndOriginalRoomId::operator==(const TOfferTitleAndOriginalRoomId& rhs) const {
    return Title == rhs.Title && OriginalRoomId == rhs.OriginalRoomId;
}

size_t TOfferTitleAndOriginalRoomId::Hash() const {
    return MultiHash(Title, OriginalRoomId);
}

size_t TOfferTitleAndOriginalRoomId::CalcTotalByteSize() const {
    return GetTotalByteSizeMulti(Title, OriginalRoomId);
}

size_t TOffer::GetAllocSize() const {
    return OfferId.GetAllocSize();
}

TSearchSubKeyRange TSearchSubKeyRange::FromSearchSubKey(const TSearchSubKey& subKey) {
    TSearchSubKeyRange res;
    res.DateFrom = subKey.Date;
    res.DateTo = subKey.Date;
    res.NightsFrom = subKey.Nights;
    res.NightsTo = subKey.Nights;
    res.ExactAges = subKey.Ages;
    return res;
}

bool TSearchSubKeyRange::AgeMatches(const TCapacity& cap) const {
    if (ExactAges.IsEmpty()) {
        return cap.Matches(AdultsFrom, AdultsTo, AllowChildren);
    } else {
        return cap.Matches(ExactAges);
    }
}

size_t TDefaultPartnerData::GetAllocSize() const {
    return 0;
}

size_t TDolphinData::GetAllocSize() const {
    return 0;
}

size_t TTravellineData::GetAllocSize() const {
    return TTotalByteSize<TString>()(HotelCode) - sizeof(HotelCode)
        + TTotalByteSize<TString>()(RatePlanCode) - sizeof(RatePlanCode);
}

size_t TBNovoData::GetAllocSize() const {
    return 0;
}

size_t TFakeOutdatedOfferPartnerData::GetAllocSize() const {
    return 0;
}

size_t TPartnerSpecificOfferData::GetAllocSize() const {
    struct TGetAllocSizeVisitor {
        size_t operator()(const TDefaultPartnerData& data) const {
            return data.GetAllocSize();
        }
        size_t operator()(const TDolphinData& data) const {
            return data.GetAllocSize();
        }
        size_t operator()(const TTravellineData& data) const {
            return data.GetAllocSize();
        }
        size_t operator()(const TBNovoData& data) const {
            return data.GetAllocSize();
        }
        size_t operator()(const TFakeOutdatedOfferPartnerData& data) const {
            return data.GetAllocSize();
        }
    };
    return std::visit(TGetAllocSizeVisitor(), Data);
}

size_t TPartnerSpecificOfferData::Hash() const {
    struct THashVisitor {
        size_t operator()(const TDefaultPartnerData&) const {
            return 0;
        }
        size_t operator()(const TDolphinData& data) const {
            return MultiHash(data.Tour, data.Pansion, data.Room, data.RoomCat);
        }
        size_t operator()(const TTravellineData& data) const {
            return MultiHash(data.HotelCode, data.RatePlanCode);
        }
        size_t operator()(const TBNovoData& data) const {
            return MultiHash(data.AccountId, data.RatePlanId);
        }
        size_t operator()(const TFakeOutdatedOfferPartnerData&) const {
            return 1;
        }
    };
    return std::visit(THashVisitor(), Data);
}

size_t TPartnerSpecificOfferData::CalcTotalByteSize() const {
    return sizeof(TPartnerSpecificOfferData) + GetAllocSize();
}

bool TPartnerSpecificOfferData::operator ==(const TPartnerSpecificOfferData& rhs) const {
    struct TEqualVisitor {
        TEqualVisitor(const TPartnerSpecificOfferData& rhs)
            : Rhs(rhs)
        {
        }

        bool operator()(const TDefaultPartnerData&) const {
            return std::holds_alternative<TDefaultPartnerData>(Rhs.Data);
        }
        bool operator()(const TDolphinData& data) const {
            if (!std::holds_alternative<TDolphinData>(Rhs.Data)) {
                return false;
            }
            const auto& rhsData = std::get<TDolphinData>(Rhs.Data);
            return data.Tour == rhsData.Tour &&
                data.Pansion == rhsData.Pansion &&
                data.Room == rhsData.Room &&
                data.RoomCat == rhsData.RoomCat;
        }
        bool operator()(const TTravellineData& data) const {
            if (!std::holds_alternative<TTravellineData>(Rhs.Data)) {
                return false;
            }
            const auto& rhsData = std::get<TTravellineData>(Rhs.Data);
            return data.HotelCode == rhsData.HotelCode &&
                data.RatePlanCode == rhsData.RatePlanCode;
        }
        bool operator()(const TBNovoData& data) const {
            if (!std::holds_alternative<TBNovoData>(Rhs.Data)) {
                return false;
            }
            const auto& rhsData = std::get<TBNovoData>(Rhs.Data);
            return data.AccountId == rhsData.AccountId &&
                data.RatePlanId == rhsData.RatePlanId;
        }
        bool operator()(const TFakeOutdatedOfferPartnerData&) const {
            return std::holds_alternative<TFakeOutdatedOfferPartnerData>(Rhs.Data);
        }

        const TPartnerSpecificOfferData& Rhs;
    };
    return std::visit(TEqualVisitor(rhs), Data);
}

bool TOfferRestrictions::IsAnyRestrictionSet() const {
    return RequiresMobile || RequiresRestrictedUser;
}

void TBlendingStageTimes::PutStep(const TString& name) {
    Times[name] = Timer.Step();
}

void TBlendingStageTimes::operator += (const TBlendingStageTimes& other) {
    for (const auto& [name, dur]: other.Times) {
        Times[name] += dur;
    }
}

void TOfferShowStats::TOfferShowStatsInner::operator += (const TOfferShowStatsInner& other) {
    NAnyOffer += other.NAnyOffer;
    NFirstOffer += other.NFirstOffer;
    NBoYHotelFirstOffer += other.NBoYHotelFirstOffer;
    NDirectBoYHotelFirstOffer += other.NDirectBoYHotelFirstOffer;
}

void TOfferShowStats::operator += (const TOfferShowStats& other) {
    for (size_t i = 0; i < GetEnumItemsCount<EPermalinkType>(); i++) {
        for (size_t j = 0; j < NTravelProto::EOperatorId_ARRAYSIZE; j++) {
            Stats[i][j] += other.Stats[i][j];
        }
    }
}

//----------------

NTravelProto::NOfferCache::NApi::EHotelCacheStatus CombineCacheStatuses(NTravelProto::NOfferCache::NApi::EHotelCacheStatus s1, NTravelProto::NOfferCache::NApi::EHotelCacheStatus s2) {
    using namespace NTravelProto::NOfferCache::NApi;
    const auto HCS_Restr = HCS_RestrictedBySearchSubKey;
    static constexpr EHotelCacheStatus matrix[EHotelCacheStatus_ARRAYSIZE][EHotelCacheStatus_ARRAYSIZE] = {
         // Miss    Found      Empty      Error      Restr
        {HCS_Miss,  HCS_Found, HCS_Empty, HCS_Error,  HCS_Restr}, // Miss
        {HCS_Found, HCS_Found, HCS_Found, HCS_Found,  HCS_Found}, // Found
        {HCS_Empty, HCS_Found, HCS_Empty, HCS_Empty,  HCS_Empty}, // Empty
        {HCS_Error, HCS_Found, HCS_Empty, HCS_Error,  HCS_Error}, // Error
        {HCS_Restr, HCS_Found, HCS_Empty, HCS_Error,  HCS_Restr}, // Restr
    };
    return matrix[s1][s2];
}

TMaybe<TUserIdentifier> TUserIdentifier::Create(TMaybe<TYandexUid> yandexUid, TMaybe<TPassportUid> passportUid) {
    if (passportUid) {
        return TUserIdentifier{true, passportUid.GetRef()};
    }
    if (yandexUid) {
        return TUserIdentifier{false, yandexUid.GetRef()};
    }
    return Nothing();
}

bool TUserIdentifier::operator == (const TUserIdentifier& rhs) const {
    return IsPassportUid == rhs.IsPassportUid && Uid == rhs.Uid;
}

size_t TUserIdentifier::Hash() const {
    return MultiHash(IsPassportUid, Uid);
}

size_t TUserIdentifier::GetAllocSize() const {
    return 0;
}

const TString TCommonDeduplicatorKeys::TitleAndOriginalRoomId = "TitleAndOriginalRoomId";
const TString TCommonDeduplicatorKeys::RefundRules = "RefundRules";
const TString TCommonDeduplicatorKeys::PartnerSpecificOfferData = "PartnerSpecificOfferData";

}// namespace NOfferCache
}// namespace NTravel

//---------------------------------------------------------------------------------------

template <>
void Out<NTravel::NOfferCache::TCacheSubKey>(IOutputStream& out, const NTravel::NOfferCache::TCacheSubKey& key) {
    out << "Date: " << NTravel::NOrdinalDate::ToString(key.Date)
        << ", Nights: " << (int)key.Nights
        << ", Capacity: " << key.Capacity.ToCapacityString();
}

template <>
void Out<NTravel::NOfferCache::TCacheKey>(IOutputStream& out, const NTravel::NOfferCache::TCacheKey& key) {
    out << "HotelId: " << key.PreKey << ", " << key.SubKey;
}

template <>
void Out<NTravel::NOfferCache::EOCPansion>(IOutputStream& out, NTravel::NOfferCache::EOCPansion pt) {
    out << NTravelProto::NOfferCache::NApi::TOCPansion::EPansion_Name(pt);
}

template <>
void Out<NTravelProto::NOfferCache::NApi::ERespMode>(IOutputStream& out, NTravelProto::NOfferCache::NApi::ERespMode respMode) {
    out << NTravelProto::NOfferCache::NApi::ERespMode_Name(respMode);
}

template <>
void Out<NTravel::NOfferCache::TOfferId>(IOutputStream& out, const NTravel::NOfferCache::TOfferId& offerId) {
    out << offerId.AsString();
}

namespace NTravel::NProtobuf {

template <>
THotelId GetIdentifier<THotelId, NTravelProto::NConfig::TMirWhitelistRecord>(const NTravelProto::NConfig::TMirWhitelistRecord& data) {
    return THotelId{data.GetPartnerId(), data.GetOriginalId()};
}

template <>
THotelId GetIdentifier<THotelId, NTravelProto::THotelId>(const NTravelProto::THotelId& data) {
    return THotelId::FromProto(data);
}

template <>
NOfferCache::TPassportUid GetIdentifier<NOfferCache::TPassportUid, ru::yandex::travel::user_order_counters::TUserOrderCounter>(const ru::yandex::travel::user_order_counters::TUserOrderCounter& data) {
    return data.GetPassportId();
}

}
