#pragma once

#include "counters.h"

#include "travel/hotels/lib/cpp/data/data.h"

#include "travel/hotels/proto/offercache_api/enums.pb.h"
#include "travel/hotels/proto/offercache_api/main.pb.h"
#include <travel/hotels/proto/offercache_grpc/offercache_service.grpc.pb.h>
#include <travel/hotels/proto/promo_service/promo_service.grpc.pb.h>
#include <travel/hotels/offercache/proto/reqans_logrecord.pb.h>
#include "travel/hotels/offercache/service/counters.h_serialized.h"
#include "travel/hotels/proto2/hotels.pb.h"

#include <travel/hotels/lib/cpp/ordinal_date/ordinal_date.h>
#include <travel/hotels/lib/cpp/ages/ages.h>
#include <travel/hotels/lib/cpp/grpc/grpc_async_server.h>
#include <travel/hotels/lib/cpp/util/profiletimer.h>
#include <travel/hotels/lib/cpp/util/string_compressor.h>
#include <travel/hotels/lib/cpp/util/object_deduplicator.h>

#include <util/datetime/base.h>
#include <util/generic/vector.h>
#include <util/generic/hash.h>
#include <util/generic/ptr.h>

namespace NTravel {
namespace NOfferCache {

using TPollingId = ui64;

using TYandexUid = ui64;
using TPassportUid = ui64;

using EOCPansion = NTravelProto::NOfferCache::NApi::TOCPansion::EPansion;
EOCPansion PansionToOCFormat(NTravelProto::EPansionType pt);

NTravelProto::ECurrency CurrencyFromOCFormat(NTravelProto::NOfferCache::NApi::TOCCurrency::ECurrency c);

using EErrorCode = NTravelProto::EErrorCode;

using EOfferSkipReason = NTravelProto::NOfferCache::NApi::ESkipReason;

using TOfferCacheGrpcServer = NGrpc::TAsyncServer<NTravelProto::NOfferCacheGrpc::OfferCacheServiceV1::AsyncService>;

using TPromoServiceGrpcServer = NGrpc::TAsyncServer<NTravelProto::NPromoService::PromoServiceV1::AsyncService>;

struct TCacheSubKey {
    TCapacity                  Capacity;
    NOrdinalDate::TOrdinalDate Date   = NOrdinalDate::g_DateZero;
    TNights                    Nights = g_NightsZero;
    bool operator == (const TCacheSubKey& rhs) const;
    size_t Hash() const;
    size_t GetAllocSize() const;
};

struct TCacheKey  {
    TPreKey PreKey;
    TCacheSubKey SubKey;
    bool operator == (const TCacheKey& rhs) const;
    size_t Hash() const;
    size_t GetAllocSize() const;
};

using TPermaroomIdStr = TString;
using TPermaroomVersion = TString;
static const TString g_OtherPermaroomId = "1";
static const TString g_OtherPermaroomName = "Другие";

using TCatRoomDataSourceIdStr = TString;

struct TDefaultPartnerData {
    size_t GetAllocSize() const;
};

struct TDolphinData {
    i64 Tour;
    i64 Pansion;
    i64 Room;
    i64 RoomCat;
    size_t GetAllocSize() const;
};

struct TTravellineData {
    TString HotelCode;
    TString RatePlanCode;
    size_t GetAllocSize() const;
};

struct TBNovoData {
    i64 AccountId;
    i64 RatePlanId;
    size_t GetAllocSize() const;
};

struct TFakeOutdatedOfferPartnerData {
    size_t GetAllocSize() const;
};

struct TPartnerSpecificOfferData {
    std::variant<TDefaultPartnerData, TDolphinData, TTravellineData, TBNovoData, TFakeOutdatedOfferPartnerData> Data;
    size_t GetAllocSize() const;

    bool operator==(const TPartnerSpecificOfferData& rhs) const;
    size_t Hash() const;
    size_t CalcTotalByteSize() const;
};

struct TOfferRestrictions {
    bool RequiresMobile : 1 = false;
    bool RequiresRestrictedUser : 1 = false;

    bool IsAnyRestrictionSet() const;
};

struct TRefundRule {
    NTravelProto::ERefundType Type;
    TPriceVal Penalty;
    ui64 StartsAtTimestampSec;
    ui64 EndsAtTimestampSec;

    bool operator==(const TRefundRule& rhs) const;
    size_t Hash() const;
    size_t CalcTotalByteSize() const;
};

class TOfferId {
public:
    constexpr explicit operator bool() const noexcept {
        return !Value.IsEmpty();
    }
    bool operator == (const TOfferId& rhs) const;
    size_t Hash() const;
    size_t GetAllocSize() const;
    TString AsString() const;
    static TOfferId FromString(const TString& s);
    static TOfferId TryFromString(const TString& s);
private:
    TGUID Value;

    static TOfferId DoFromString(const TString& s, bool throwOnError);
};

struct TOfferTitleAndOriginalRoomId {
    TString Title;
    TString OriginalRoomId;

    bool operator==(const TOfferTitleAndOriginalRoomId& rhs) const;
    size_t Hash() const;
    size_t CalcTotalByteSize() const;
};

struct TOffer {
    /*
     * We have many TOffers, so we are trying not to waste memory here.
     * So, members here are properly ordered for the best aligning.
     * Also, some of them are trimmed to minimum possible byte size. Static or dynamic asserts are used for such fields when needed.
     */

    static constexpr ui32 ENUM_BIT_SIZE = 8;
    static_assert(NTravelProto::EOperatorId_MIN >= 0);
    static_assert(NTravelProto::EOperatorId_MAX < (1u << ENUM_BIT_SIZE));
    static_assert(NTravelProto::EPansionType_MIN >= 0);
    static_assert(NTravelProto::EPansionType_MAX < (1u << ENUM_BIT_SIZE));
    static_assert(sizeof(EFreeCancellationType) * 8 <= ENUM_BIT_SIZE);

    TOfferId     OfferId;
    TOfferIdHash OfferIdHash;
    TPriceVal    PriceVal;
    EOperatorId  OperatorId : ENUM_BIT_SIZE;
    TOfferRestrictions OfferRestrictions;
    EFreeCancellationType FreeCancellation: ENUM_BIT_SIZE;
    NTravelProto::EPansionType PansionType: ENUM_BIT_SIZE;
    i16       RoomCount;
    i16       SingleRoomAdultCount;
    TObjectDeduplicator::TUnique<TOfferTitleAndOriginalRoomId> TitleAndOriginalRoomId;
    TObjectDeduplicator::TUnique<TPartnerSpecificOfferData> PartnerSpecificOfferData;
    TObjectDeduplicator::TUnique<TVector<TRefundRule>> RefundRules;
    size_t GetAllocSize() const;
};

struct TSearchWarnings : public TThrRefBase {
    TVector<std::pair<NTravelProto::ESearchWarningCode, ui32>> WarningCounts;
};

using TSearchWarningsRef = TIntrusivePtr<TSearchWarnings>;

struct TCacheRecord : public TThrRefBase {
    TCacheKey  Key;
    TInstant   Timestamp;
    TInstant   ExpireTimestamp;
    TString    SearcherReqId;
    TSourceCountersRef SourceCounters;
    TString    OfferCacheClientId;
    TAtomic    UsageCount;
    TVector<TOffer> Offers;
    EErrorCode ErrorCode;
    TSearchWarningsRef SearchWarnings;
    TAtomicFlag IsInvalidated;
    bool        IsOutdated;
    size_t GetAllocSize() const;
};

using TCacheRecordRef = TIntrusivePtr<TCacheRecord>;

struct TSearchSubKeyRange {
    NOrdinalDate::TOrdinalDate DateFrom = NOrdinalDate::g_DateZero;
    NOrdinalDate::TOrdinalDate DateTo = NOrdinalDate::g_DateZero;
    TNights                    NightsFrom = 0;
    TNights                    NightsTo = 0;

    TAges                      ExactAges;// Если пусто, то используются следующие поля
    size_t                     AdultsFrom = 0;
    size_t                     AdultsTo = 0;
    bool                       AllowChildren = true;

    static TSearchSubKeyRange FromSearchSubKey(const TSearchSubKey& subKey);
    bool AgeMatches(const TCapacity& cap) const;
};

struct TReadJobStats {
    bool MainPermalinkHasPrices = false;
    ui32 TotalPermalinksWithPrices = 0;
    ui32 Top5PermalinksWithPrices = 0;
    ui32 Top10PermalinksWithPrices = 0;
    bool Exp2216Effect = false;
    TDuration FullDuration;
    TDuration FixPriceDuration;

    NTravelProto::NOfferCache::NReqAnsLog::TReqAnsLogRecord::TInfo Info;
};

struct TBlendingStageTimes {
    TProfileTimer Timer;
    THashMap<TString, TDuration> Times;

    void PutStep(const TString& name);
    void operator += (const TBlendingStageTimes& other);
};

struct TOfferShowStats {
    struct TOfferShowStatsInner {
        ui32 NAnyOffer;
        ui32 NFirstOffer;
        ui32 NBoYHotelFirstOffer;
        ui32 NDirectBoYHotelFirstOffer;

        void operator += (const TOfferShowStatsInner& other);
    };

    TOfferShowStatsInner Stats[GetEnumItemsCount<EPermalinkType>()][NTravelProto::EOperatorId_ARRAYSIZE] = {};

    void operator += (const TOfferShowStats& other);
};

NTravelProto::NOfferCache::NApi::EHotelCacheStatus CombineCacheStatuses(NTravelProto::NOfferCache::NApi::EHotelCacheStatus s1, NTravelProto::NOfferCache::NApi::EHotelCacheStatus s2);

static const TString g_WelcomePromocodeUtmTerm = "welcomehotelpromo";

struct TUserIdentifier {
    bool IsPassportUid;
    ui64 Uid;

    static TMaybe<TUserIdentifier> Create(TMaybe<TYandexUid> yandexUid, TMaybe<TPassportUid> passportUid);

    bool operator == (const TUserIdentifier& rhs) const;
    size_t Hash() const;
    size_t GetAllocSize() const;
};

struct TUserInfo {
    TMaybe<TYandexUid> YandexUid;
    TMaybe<TPassportUid> PassportUid;
    TMaybe<TUserIdentifier> UserIdentifier;
    size_t ConfirmedHotelOrderCount = 0;
};

struct TCommonDeduplicatorKeys {
    static const TString TitleAndOriginalRoomId;
    static const TString RefundRules;
    static const TString PartnerSpecificOfferData;
};

}// namespace NOfferCache
}// namespace NTravel
//----------------------------------


template <>
struct NTravel::TTotalByteSize<TVector<NTravel::NOfferCache::TRefundRule>, void> {
    inline size_t operator()(const TVector<NTravel::NOfferCache::TRefundRule>& item) const {
        auto res = sizeof(item);
        for (const auto& x : item) {
            res += TTotalByteSize<NTravel::NOfferCache::TRefundRule>()(x);
        }
        res += (item.capacity() - item.size()) * sizeof(NTravel::NOfferCache::TRefundRule);
        return res;
    }
};

template <>
struct THash<NTravel::NOfferCache::TCacheSubKey> {
    inline size_t operator() (const NTravel::NOfferCache::TCacheSubKey& v) const { return v.Hash(); }
};

template <>
struct THash<NTravel::NOfferCache::TCacheKey> {
    inline size_t operator() (const NTravel::NOfferCache::TCacheKey& v) const { return v.Hash(); }
};

template <>
struct THash<NTravel::NOfferCache::TUserIdentifier> {
    inline size_t operator() (const NTravel::NOfferCache::TUserIdentifier& v) const { return v.Hash(); }
};

template <>
struct THash<NTravel::NOfferCache::TOfferId> {
    inline size_t operator() (const NTravel::NOfferCache::TOfferId& v) const { return v.Hash(); }
};

template <>
struct THash<NTravel::NOfferCache::TRefundRule> {
    inline size_t operator() (const NTravel::NOfferCache::TRefundRule& v) const { return v.Hash(); }
};

template <>
struct THash<NTravel::NOfferCache::TPartnerSpecificOfferData> {
    inline size_t operator() (const NTravel::NOfferCache::TPartnerSpecificOfferData& v) const { return v.Hash(); }
};

template <>
struct THash<NTravel::NOfferCache::TOfferTitleAndOriginalRoomId> {
    inline size_t operator() (const NTravel::NOfferCache::TOfferTitleAndOriginalRoomId& v) const { return v.Hash(); }
};

template <>
struct THash<TVector<NTravel::NOfferCache::TRefundRule>>: public TSimpleRangeHash {};
