#include "grpc_job.h"

#include "service.h"

#define JOB_LOG LogPrefix

namespace NTravel {
namespace NOfferCache {

TGrpcJob::TGrpcJob(TService& svc, ui64 id, const TOfferCacheGrpcServer::TResponseCb<NTravelProto::TSearchOffersRpcRsp>& responseCb, TInstant started)
    : Service(svc)
    , LogPrefix("IdG_" + ToString(id) + ": ")
    , ResponseCb(responseCb)
    , Started(started)
{
    Service.GetCounters().NGrpcJobs.Inc();
}

TGrpcJob::~TGrpcJob() {
    WriteGrpcReqAnsLog();
    Service.GetCounters().NGrpcJobs.Dec();
}

void TGrpcJob::Start(const NTravelProto::TSearchOffersRpcReq& origReq, const NGrpc::TServerReqMetadata& srvMeta) {
    ReqMeta.CallId = srvMeta.CallId;
    ReqMeta.RemoteFQDN = srvMeta.RemoteFQDN;
    Req.CopyFrom(origReq);
    INFO_LOG << JOB_LOG << "Got gRPC request callId " << srvMeta.CallId << " from '" << srvMeta.RemoteFQDN << "', "
             << "have " << origReq.SubrequestSize() << " subreqs, ids: " << GetSearchOffersRpcReqIds(origReq) << Endl;
    Service.GetCounters().NGrpcRequests.Inc();
    NTravelProto::TSearchOffersRpcReq searcherReq;
    searcherReq.SetSync(origReq.GetSync());
    searcherReq.SetIncludeDebug(origReq.GetIncludeDebug());// и нафига?...
    TVector<size_t> origPos; // searcher position -> original position
    for (size_t pos = 0; pos < origReq.SubrequestSize(); ++pos) {
        const auto& origSubReq = origReq.GetSubrequest(pos);
        TSourceCountersRef sourceCounters = Service.GetSourceCounters(origSubReq);
        sourceCounters->NSubReq.Inc();

        try {
            ValidateGrpcSubrequest(origReq, origSubReq);
        } catch (...) {
            sourceCounters->NSubReqError.Inc();
            ERROR_LOG << JOB_LOG << "Subrequest " << pos << " has error: " << CurrentExceptionMessage() << ", request Id " << origSubReq.GetId() << Endl;
            if (auto err = Resp.AddSubresponse()->MutableError()) {
                err->SetCode(NTravelProto::EC_INVALID_ARGUMENT);
                err->SetMessage(CurrentExceptionMessage());
            }
            continue;
        }

        const auto clusterPermalink = Service.PermalinkToClusterMapper().GetClusterPermalink(TPermalink(origSubReq.GetPermalink()));
        if (!origSubReq.GetOfferCacheIgnoreBlacklist()) {
            if (auto blacklistedHotel = Service.HotelsBlacklist().GetMapping(clusterPermalink)) {
                if (blacklistedHotel->Contains(THotelId::FromProto(origSubReq.GetHotelId()))) {
                    sourceCounters->NSubReqBlacklisted.Inc();
                    if (auto err = Resp.AddSubresponse()->MutableError()) {
                        err->SetCode(NTravelProto::EC_UNAVAILABLE);
                        err->SetMessage("HotelId is blacklisted");
                    }
                    continue;
                }
            }
        }

        if (!origSubReq.GetOfferCacheIgnoreWhitelist()) {
            if (!Service.HotelsWhitelist().IsAllowedByWhitelist(clusterPermalink, THotelId::FromProto(origSubReq.GetHotelId()))) {
                sourceCounters->NSubReqDroppedByWhitelist.Inc();
                if (auto err = Resp.AddSubresponse()->MutableError()) {
                    err->SetCode(NTravelProto::EC_UNAVAILABLE);
                    err->SetMessage("HotelId is not in whitelist");
                }
                continue;
            }
        }

        Resp.AddSubresponse()->MutablePlaceholder();// it is empty!

        // Далее "Бррр"
        bool useSearcher = origSubReq.GetOfferCacheUseSearcher();
        if (origSubReq.GetOfferCacheUseCache()) {
            // Надо смотреть в кэш
            if (Service.ReqCache().GetRequestState(origSubReq) == TReqCache::EReqState::None) {
                // Не в кэше
                if (useSearcher) {// Надо послать в сёрчер
                    origPos.push_back(pos);
                    searcherReq.AddSubrequest()->CopyFrom(origSubReq);
                }
            }
        } else {
            // Кэш не интересен
            origPos.push_back(pos);
            searcherReq.AddSubrequest()->CopyFrom(origSubReq);
        }
    }
    if (searcherReq.SubrequestSize()) {
        // Используется source последнего subrequest-а
        NGrpc::TClientMetadata clientMeta;
        clientMeta.CallId = srvMeta.CallId;
        clientMeta.ForwardedFor = srvMeta.RemoteFQDN;
        INFO_LOG << JOB_LOG << "Doing request to searcher with " << searcherReq.SubrequestSize()
                 << " subreqs, ids: " << GetSearchOffersRpcReqIds(searcherReq) << Endl;
        TGrpcJobRef job = this;// Rescue me
        TProfileTimer started;
        Service.DoSearcherRequest(LogPrefix, searcherReq, clientMeta, [started, job, origPos, searcherReq]
                          (const NTravelProto::TSearchOffersRpcRsp& searcherResp) {
            job->OnSearcherRequestFinished(started.Get(), origPos, searcherReq, searcherResp);
        });
    } else {
        ResponseCb(Resp, NGrpc::TServerRespMetadata::BuildOk(LogPrefix));
    }
}

void TGrpcJob::OnSearcherRequestFinished(TDuration dur, const TVector<size_t> origPos,
                                         const NTravelProto::TSearchOffersRpcReq& searcherRpcReq, const NTravelProto::TSearchOffersRpcRsp& searcherRpcResp) {
    INFO_LOG << JOB_LOG << "Searcher request finished successfully in " << dur << Endl;
    for (size_t pos = 0; pos < searcherRpcResp.SubresponseSize(); ++pos) {
        const auto& searcherSubResp = searcherRpcResp.GetSubresponse(pos);
        if (searcherSubResp.HasError()) {
            if (pos < origPos.size()) {
                Resp.MutableSubresponse(origPos[pos])->MutableError()->CopyFrom(searcherSubResp.GetError());
            }
            TString requestId;
            if (pos < searcherRpcReq.SubrequestSize()) {
                requestId = searcherRpcReq.GetSubrequest(pos).GetId();
            }
            ERROR_LOG << JOB_LOG << "Searcher response has error in subresponse #" << pos << ", RequestId: " << requestId << ": " << searcherSubResp.GetError() << Endl;
        }
    }
    ResponseCb(Resp, NGrpc::TServerRespMetadata::BuildOk(LogPrefix));
}

void TGrpcJob::ValidateGrpcSubrequest(const NTravelProto::TSearchOffersRpcReq& req, const NTravelProto::TSearchOffersReq& subReq) const {
    if (req.GetSync()) {
        throw yexception() << "Sync requests are not supported";
    }
    if (!Service.IsGrpcClientIdKnown(subReq.GetAttribution().GetOfferCacheClientId())) {
        throw yexception() << "Unknown OfferCacheClientId: " << Service.IsGrpcClientIdKnown(subReq.GetAttribution().GetOfferCacheClientId());
    }
    if (!subReq.GetOfferCacheUseCache() && !subReq.GetOfferCacheUseSearcher()) {
        throw yexception() << "At least one should be true: OfferCacheUseCache or OfferCacheUseSearcher (or maybe both)";
    }
    if (!Service.GetPartnerCode(subReq.GetHotelId().GetPartnerId())) {
        throw yexception() << "Invalid partner id " << (int)subReq.GetHotelId().GetPartnerId();
    }
    TSearchSubKey searchSubKey;
    searchSubKey.Date = NOrdinalDate::FromString(subReq.GetCheckInDate());
    NOrdinalDate::TOrdinalDate checkOut = NOrdinalDate::FromString(subReq.GetCheckOutDate());
    if (checkOut <= searchSubKey.Date) {
        throw yexception() << "Checkout " << subReq.GetCheckOutDate() << " should be after checkin " << subReq.GetCheckInDate();
    }
    searchSubKey.Nights = checkOut - searchSubKey.Date;
    searchSubKey.Ages = TAges::FromOccupancyString(subReq.GetOccupancy());
    THotelId hotelId = THotelId::FromProto(subReq.GetHotelId());
    TString restrictReaon;
    if (!Service.IsSearchSubKeyAllowed(NOrdinalDate::FromInstant(Started), hotelId, searchSubKey, &restrictReaon)) {
        throw yexception() << restrictReaon;
    }
}

void TGrpcJob::WriteGrpcReqAnsLog() const {
    TInstant now = ::Now();

    Y_ASSERT(Resp.SubresponseSize() == Req.SubrequestSize());

    for (size_t i = 0; i < Req.SubrequestSize(); i++) {
        NTravelProto::NOfferCache::NGrpcReqAnsLog::TGrpcReqAnsLogRecord log;
        log.set_unixtime(now.Seconds());
        log.set_local_time(now.ToIsoStringLocal());
        log.MutableResp()->CopyFrom(Resp.GetSubresponse(i));
        log.MutableReq()->CopyFrom(Req.GetSubrequest(i));
        log.MutableInfo()->SetLogPrefix(LogPrefix);
        log.MutableInfo()->SetCallId(ReqMeta.CallId);
        log.MutableInfo()->SetRemoteFQDN(ReqMeta.RemoteFQDN);

        Service.GrpcReqAnsLogger().AddRecord(log);
    }
}

}// namespace NOfferCache
}// namespace NTravel
