#include "grpc_read_job.h"
#include "service.h"
#include "read_request_processor.h"

#include <util/digest/city.h>
#include <util/generic/set.h>
#include <util/generic/maybe.h>
#include <util/generic/algorithm.h>
#include <util/string/vector.h>

#include <type_traits>
#include <travel/hotels/lib/cpp/grpc/grpc_async_server.h>

#define JOB_LOG LogPrefix

namespace NTravel::NOfferCache {
    TGrpcReadJob::TGrpcReadJob(TService& svc, const TOfferCacheGrpcServer::TResponseCb<NTravelProto::NOfferCache::NApi::TReadResp>& onResp, TInstant started, ui64 id)
      : Service(svc)
      , ResponseCb(onResp)
      , Started(started)
      , LogPrefix("IdH_" + ToString(id) + ": ")
      , StatsPtr(MakeAtomicShared<TReadJobStats>())
    {
        StatsPtr->Info.MutableProcessingStagesStats()->SetInitMicros(StagesTimer.Step().MicroSeconds());
        Service.GetCounters().NHttpJobs.Inc();
    }

    TGrpcReadJob::~TGrpcReadJob() {
        Service.GetCounters().NHttpJobs.Dec();
    }

    void TGrpcReadJob::Start(const NTravelProto::NOfferCache::NApi::TReadReq& req, const NGrpc::TServerReqMetadata&) {
        INFO_LOG << JOB_LOG << "Start job via grpc" << Endl;
        StatsPtr->Info.MutableProcessingStagesStats()->SetStartMicros(StagesTimer.Step().MicroSeconds());

        TIntrusivePtr<TReadRequestProcessor> processor = new TReadRequestProcessor(StatsPtr, Service, Started, LogPrefix);
        try {
            processor->Parse(req);
            StatsPtr->Info.MutableProcessingStagesStats()->SetParseMicros(StagesTimer.Step().MicroSeconds());
        } catch (...) {
            ERROR_LOG << JOB_LOG << "Bad request: " << CurrentExceptionMessage() << Endl;
            if (processor->SourceCounters) {
                processor->SourceCounters->NRequestsErrors.Inc();
            }
            ResponseCb(
                NTravelProto::NOfferCache::NApi::TReadResp(),
                NGrpc::TServerRespMetadata::BuildFailed(LogPrefix, grpc::Status(grpc::StatusCode::INVALID_ARGUMENT, CurrentExceptionMessage()))
            );
            return;
        }
        try {
            processor->Process([this](const NTravelProto::NOfferCache::NApi::TReadResp& resp) {
                StatsPtr->FullDuration = LifespanTimer.Get();
                ResponseCb(resp, NGrpc::TServerRespMetadata::BuildOk(LogPrefix));
                return resp.ByteSizeLong();
            });
        } catch (...) {
            ResponseCb(
                NTravelProto::NOfferCache::NApi::TReadResp(),
                NGrpc::TServerRespMetadata::BuildFailed(LogPrefix, grpc::Status(grpc::StatusCode::INTERNAL, CurrentExceptionMessage()))
            );
        }
    }
}

