#include "hotels_whitelist.h"

#include <library/cpp/logger/global/global.h>

#include <util/generic/algorithm.h>

namespace NTravel {
namespace NOfferCache {

void THotelsWhitelist::TPartnerCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NPermalinkOnlyMatched));
    ct->insert(MAKE_COUNTER_PAIR(NOriginalIdOnlyMatched));
    ct->insert(MAKE_COUNTER_PAIR(NFullMatch));
    ct->insert(MAKE_COUNTER_PAIR(NNothingMatched));
}

THotelsWhitelist::THotelsWhitelist(const TConfig& config)
    : TPermalinkToOriginalIdsMapper("HotelsWhitelist", config.GetWhitelist())
    , PartnerCounters_({"partner"})
{
    for (const auto& partner: config.GetPartner()) {
        auto partnerId = partner.GetId();
        if (partner.GetIsWhitelistEnabled()) {
            HotelsWhitelistPartners_.insert(partnerId);
        } else {
            HotelsWhitelistPartners_.erase(partnerId); // Из-за возможных дублей
        }
    }

    auto createInverseMapping = [this]() {
        TPartnerHotels partnerHotels;
        for (const auto& [permalink, hotelIds]: *TPermalinkToOriginalIdsMapper::GetMapping()) {
            Y_UNUSED(permalink);
            for (const auto& [partnerId, originalId]: hotelIds.PartnerIds) {
                if (HotelsWhitelistPartners_.contains(partnerId)) {
                    partnerHotels[partnerId].push_back(originalId);
                }
            }
        }
        size_t allocSize = sizeof partnerHotels;
        for (auto& [partnerId, originalIds]: partnerHotels) {
            Y_UNUSED(partnerId);
            SortUnique(originalIds);
            originalIds.shrink_to_fit();
            allocSize += sizeof partnerId + sizeof originalIds + sizeof originalIds.back() * originalIds.size();
            if (originalIds.empty()) {
                INFO_LOG << Name_ << ": originalIds list for whitlisted partner " << partnerId << " is empty" << Endl;
            }
        }
        Counters_.NBytes += allocSize;
        {
            TWriteGuard g(Lock_);
            PartnerHotels_ = std::move(partnerHotels);
        }
    };
    SetOnFinishHandler(createInverseMapping);
}

void THotelsWhitelist::RegisterCounters(NMonitor::TCounterSource& source) {
    TPermalinkToOriginalIdsMapper::RegisterCounters(source);
    source.RegisterSource(&PartnerCounters_, "HotelsWhitelistQuality");
}

bool THotelsWhitelist::IsInWhitelist(TPermalink permalink, const THotelId& hotelId) const {
    if (!HotelsWhitelistPartners_.contains(hotelId.PartnerId)) {
        return false;
    }
    auto whitelistedHotel = GetMapping(permalink);
    return whitelistedHotel && whitelistedHotel->Contains(hotelId);
}

bool THotelsWhitelist::IsWhitelistedPartner(EPartnerId partnerId) {
    return HotelsWhitelistPartners_.contains(partnerId);
}

bool THotelsWhitelist::IsAllowedByWhitelist(TPermalink permalink, const THotelId& hotelId) const {
    if (!HotelsWhitelistPartners_.contains(hotelId.PartnerId)) {
        return true;
    }
    auto whitelistedHotel = GetMapping(permalink);
    return whitelistedHotel && whitelistedHotel->Contains(hotelId);
}

void THotelsWhitelist::FilterPartnerHotels(TPermalink permalink, THashSet<THotelId>* hotelIds) const {
    auto whitelistedHotel = GetMapping(permalink);
    // whitelistedHotel and PartnerHotels_ can diverge
    TReadGuard g(Lock_);
    auto it = hotelIds->cbegin();
    const auto end = hotelIds->cend();
    while (it != end) {
        if (HotelsWhitelistPartners_.contains(it->PartnerId) && !IsWhitelistedHotel(whitelistedHotel.get(), *it)) {
            hotelIds->erase(it++);
        } else {
            ++it;
        }
    }
}

bool THotelsWhitelist::IsWhitelistedHotel(const TValue* whitelistedHotel, const THotelId& hotelId) const {
    auto counters = PartnerCounters_.GetOrCreate({NTravelProto::EPartnerId_Name(hotelId.PartnerId)});
    if (whitelistedHotel) {
        if (whitelistedHotel->Contains(hotelId)) {
            counters->NFullMatch.Inc();
        } else {
            counters->NPermalinkOnlyMatched.Inc();
            return false;
        }
    } else {
        if (auto partner = PartnerHotels_.find(hotelId.PartnerId); partner != PartnerHotels_.end()) {
            const auto& originalIds = partner->second;
            if (BinarySearch(originalIds.cbegin(), originalIds.cend(), hotelId.OriginalId)) {
                counters->NOriginalIdOnlyMatched.Inc();
            } else {
                counters->NNothingMatched.Inc();
            }
        } else {
            counters->NNothingMatched.Inc();
        }
        return false;
    }
    return true;
}

}// namespace NOfferCache
}// namespace NTravel
