#include "http_read_job.h"

#include "service.h"
#include "read_request_processor.h"

#include <travel/hotels/lib/cpp/protobuf/tools.h>

#include <util/digest/city.h>
#include <util/generic/set.h>
#include <util/generic/maybe.h>
#include <util/generic/algorithm.h>
#include <util/string/vector.h>

#include <type_traits>

#define JOB_LOG LogPrefix

namespace NTravel {
namespace NOfferCache {

THttpReadJob::THttpReadJob(TService& svc, NHttp::TOnResponse onResp, TInstant started, ui64 id)
  : Service(svc)
  , ResponseCb(std::move(onResp))
  , Started(started)
  , LogPrefix("IdH_" + ToString(id) + ": ")
  , StatsPtr(MakeAtomicShared<TReadJobStats>())
{
    StatsPtr->Info.MutableProcessingStagesStats()->SetInitMicros(StagesTimer.Step().MicroSeconds());
    Service.GetCounters().NHttpJobs.Inc();
}

THttpReadJob::~THttpReadJob() {
    Service.GetCounters().NHttpJobs.Dec();
}

void THttpReadJob::Start(const NHttp::TRequest& httpReq) {
    INFO_LOG << JOB_LOG << "Start job, query: " << httpReq.Cgi() << Endl;
    StatsPtr->Info.MutableProcessingStagesStats()->SetStartMicros(StagesTimer.Step().MicroSeconds());

    NTravelProto::NOfferCache::NApi::TReadReq req;
    TIntrusivePtr<TReadRequestProcessor> processor = new TReadRequestProcessor(StatsPtr, Service, Started, LogPrefix, httpReq.Cgi());
    try {
        NTravel::NProtobuf::ParseCgiRequest(httpReq.Query(), &req);
        processor->Parse(req);
        StatsPtr->Info.MutableProcessingStagesStats()->SetParseMicros(StagesTimer.Step().MicroSeconds());
    } catch (...) {
        ERROR_LOG << JOB_LOG << "Bad request: " << CurrentExceptionMessage() << Endl;
        if (processor->SourceCounters) {
            processor->SourceCounters->NRequestsErrors.Inc();
        }
        Service.GetCounters().NRequestParseErrors.Inc();
        NHttp::TResponse resp = NHttp::TResponse::CreateText(CurrentExceptionMessage() + "\n", HTTP_BAD_REQUEST);
        resp.LogPrefix = LogPrefix;
        ResponseCb(resp);
        return;
    }
    processor->Process([this, &req](const NTravelProto::NOfferCache::NApi::TReadResp& resp) {
        TProfileTimer jsonStarted;
        auto json = NProtobufJson::Proto2Json(resp, NProtobufJson::TProto2JsonConfig()
            .SetFormatOutput(req.GetDebug())
            .SetMissingRepeatedKeyMode(NProtobufJson::TProto2JsonConfig::MissingKeyDefault)
            .SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName)
            .SetMapAsObject(true));
        StatsPtr->Info.MutableProcessingStagesStats()->SetReplyBuildJsonMicros(StagesTimer.Step().MicroSeconds());
        StatsPtr->Info.SetJsonBytes(json.size());
        auto httpResp = NHttp::TResponse::CreateJson(json, HTTP_OK);
        httpResp.EnableKeepAlive = Service.IsPingEnabled();
        httpResp.LogPrefix = LogPrefix;
        StatsPtr->FullDuration = LifespanTimer.Get();
        if (req.GetDebug()) {
            httpResp.AddHeader("X-Debug-Jsonize-us", ToString(jsonStarted.Get().MicroSeconds()));
            httpResp.AddHeader("X-Debug-Full-us", ToString(StatsPtr->FullDuration.MicroSeconds()));
        }
        ResponseCb(httpResp);
        return json.size();
    });
}

}// namespace NOfferCache
}// namespace NTravel

