#include "offer_blender.h"

#include "service.h"
#include "offer_utils.h"

#include <travel/hotels/proto/promo_service/promo_service.pb.h>
#include <travel/hotels/lib/cpp/data/data.h>

#include <library/cpp/string_utils/quote/quote.h>
#include <util/charset/utf8.h>

#include <tuple>
#include <cmath>

#define JOB_LOG ReadRequestProcessor.LogPrefix << " (" << Permalink << ")"

namespace {
    static TString ROOM_AREA_FEATURE_ID = "room-size-square-meters";
}

namespace NTravel {
namespace NOfferCache {

TString TOfferBlender::TSelectedOffer::GetDecompressedTitle() const {
    if (DecompressedTitle.Empty()) {
        DecompressedTitle = Offer->TitleAndOriginalRoomId.GetValue().Title;
    }
    return *DecompressedTitle.Get();
}


TOfferBlender::TOfferBlender(TReadRequestProcessor& readRequestProcessor,
                             TPermalink permalink,
                             const THashMap<THotelId, TVector<TCacheRecordRef>>& records,
                             const TUrl& redirUrl,
                             NTravelProto::TLabel& label,
                             NTravelProto::NOfferCache::NApi::TReadResp::THotel* hotel
)
    : ReadRequestProcessor(readRequestProcessor)
    , Service(readRequestProcessor.Service)
    , OfferTextPrefix(readRequestProcessor.Service.Config().GetOther().GetOfferTextPrefix())
    , Req(readRequestProcessor.Req)
    , Permalink(permalink)
    , IsMainPermalink(permalink == readRequestProcessor.MainPermalink)
    , IsSingleOrgPermalink(readRequestProcessor.SingleOrgPermalinks.contains(Permalink))
    , IsSimilarPermalink(readRequestProcessor.SimilarPermalinks.contains(Permalink))
    , Records(records)
    , RedirUrlBase(redirUrl)
    , BoostedOfferId(TOfferId::TryFromString(Req.GetBoostedOfferId()))
    , Label(label)
    , RespHotel(hotel)
    , ShowPermarooms(false)
    , NeedMinPriceBadge(false)
    , OfferToRoomKeyMapper(Service.GetCounters())
    , EnabledAfterFilteringOperators(Service.GetEnabledOperatorsDefault())
{
    switch (Req.GetRespMode()) {
        case NTravelProto::NOfferCache::NApi::RM_Auto:
        case NTravelProto::NOfferCache::NApi::RM_ByOperator:
            IsBrief = !IsSingleOrgPermalink;
            if (Req.GetUseNewCatRoom()) {
                AllowPermarooms = IsSingleOrgPermalink && Req.GetFull() && (Req.GetEnableCatRoom() || Req.GetEnableCatRoomProps());
            } else {
                AllowPermarooms = !IsBrief && !Req.GetCompactResponseForCalendar() && Req.GetFull();
            }
            if (Req.GetRespMode() == NTravelProto::NOfferCache::NApi::RM_ByOperator) {
                MinPriceCount = 0; // сколько операторов, столько цен
            } else {
                if (IsSingleOrgPermalink) {
                    MinPriceCount = Service.Config().GetOther().GetNotFullOfferCount(); // минимум  столько цен надо отдать , даже если операторы будут повторяться
                } else {
                    MinPriceCount = 0; // для карусели достаточно по одному офферу на оператора
                }
            }
            break;
        case NTravelProto::NOfferCache::NApi::RM_MultiHotel:
            IsBrief = IsSimilarPermalink;
            AllowPermarooms = false;
            MinPriceCount = 0; // сколько операторов, столько цен
            break;
    }

    BlendingStageTimes.PutStep("Init");
}

void TOfferBlender::RearrangeOffers() {
    SortOffers();
    BlendingStageTimes.PutStep("SortOffers");

    BumpOffers();
    BlendingStageTimes.PutStep("BumpOffers");
}

void TOfferBlender::ReduceOffers() {
    if (!Req.GetFull()) {
        ReduceOffersByOperator();
    }
    BlendingStageTimes.PutStep("ReduceOffersByOperator");

    if (IsSimilarPermalink) {
        if (SelectedOffers.size() > 1) {
            SelectedOffers.resize(1);
        }
    } else if (!IsBrief && !Req.GetCompactResponseForCalendar()) {
        FillDefaultOffer();
        BlendingStageTimes.PutStep("FillDefaultOffer");
    }

    if (Req.GetCompactResponseForSearch() || Req.GetCompactResponseForCalendar()) {
        FilterNotFirstOffers();
        BlendingStageTimes.PutStep("FilterNotFirstOffers");
    }
}

void TOfferBlender::FillRespAfterBlending() {
    PutOffersToResp();
    BlendingStageTimes.PutStep("PutOffersToResp");

    LogSkippedOffers();
    BlendingStageTimes.PutStep("LogSkippedOffers");

    // HOTELS-3350
    // wasFound д.б. true только если есть цены, или точно известно что данные от всех партнёров в кэше есть, а цен всё равно нет
    bool wasFound = !SelectedOffers.empty() || ReadRequestProcessor.FullyFoundPermalinks.contains(Permalink);
    RespHotel->SetWasFound(wasFound);

    PutPartnerRefsToResp();
    BlendingStageTimes.PutStep("PutPartnerRefsToResp");
}

void TOfferBlender::Blend() {
    BlendingStageTimes.PutStep("Start");

    GetAllOffers();
    BlendingStageTimes.PutStep("GetAllOffers");

    RemoveDisabledOperators();
    BlendingStageTimes.PutStep("RemoveDisabledOperators");

    RemoveBannedOfferTypes();
    BlendingStageTimes.PutStep("RemoveBannedOfferTypes");

    FilterOffersByMinPrice();
    BlendingStageTimes.PutStep("FilterOffersByMinPrice");

    RemoveRestrictedOffers();
    BlendingStageTimes.PutStep("RemoveRestrictedOffers");

    RemoveExperiments();
    BlendingStageTimes.PutStep("RemoveExperiments");

    RemoveMetaIfBoyAvailable();
    BlendingStageTimes.PutStep("RemoveMetaIfBoyAvailable");

    CalculatePromos();
    BlendingStageTimes.PutStep("CalculatePromos");

    SelectBestBoY();
    BlendingStageTimes.PutStep("SelectBestBoY");

    ApplyFilterPartnerIdAfterBoYSelection();
    BlendingStageTimes.PutStep("ApplyFilterPartnerIdAfterBoYSelection");

    RemoveDuplicatedOffersBeforeScanPermarooms();
    BlendingStageTimes.PutStep("RemoveDuplicatedOffersBeforeScanPermarooms");

    CatRoomStatus = ScanPermarooms();
    BlendingStageTimes.PutStep("ScanPermarooms");

    ProcessCatRoomStatus();
    BlendingStageTimes.PutStep("ProcessCatRoomStatus");

    RemoveDuplicatedOffersAfterScanPermarooms();
    BlendingStageTimes.PutStep("RemoveDuplicatedOffersAfterScanPermarooms");

    UpdateTotalPriceRange();
    BlendingStageTimes.PutStep("UpdateTotalPriceRange");

    ApplyAuxFilters();
    BlendingStageTimes.PutStep("ApplyAuxFilters");

    FillUserFiltersInfo();
    BlendingStageTimes.PutStep("FillUserFiltersInfo");

    ApplyUserFilters();
    BlendingStageTimes.PutStep("ApplyUserFilters");

    ApplyAuxPostUserFilters();
    BlendingStageTimes.PutStep("ApplyAuxPostUserFilters");

    FillFullStats();
    BlendingStageTimes.PutStep("FillFullStats");

    CheckMinPriceBadge();
    BlendingStageTimes.PutStep("CheckMinPriceBadge");

    FillMirCashbackAvailability();
    BlendingStageTimes.PutStep("FillMirCashbackAvailability");

    FillPerHotelPlusInfo();
    BlendingStageTimes.PutStep("FillPerHotelPlusInfo");

    FillWelcomePromocode();
    BlendingStageTimes.PutStep("FillWelcomePromocode");

    CalculateBadges();
    BlendingStageTimes.PutStep("CalculateBadges");

    PutAggregatedOfferInfoToResp();
    BlendingStageTimes.PutStep("PutAggregatedOfferInfoToResp");

    FillHasBoyPartner();
    BlendingStageTimes.PutStep("FillHasBoyPartner");

    FillIsPlusAvailable();
    BlendingStageTimes.PutStep("FillIsPlusAvailable");

    RearrangeOffers();

    PutHotelBadgesToResp();
    BlendingStageTimes.PutStep("PutHotelBadgesToResp");

    RemoveMetaStrikethroughPricesForBoyHotels();
    BlendingStageTimes.PutStep("RemoveMetaStrikethroughPricesForBoyHotels");


    ReduceOffers();
    FillRespAfterBlending();
    LogTime();
}

const TBlendingStageTimes& TOfferBlender::GetBlendingStageTimes() const {
    return BlendingStageTimes;
}

const TOfferShowStats& TOfferBlender::GetOfferShowStats() const {
    return OfferShowStats;
}

void TOfferBlender::GetAllOffers() {
    for (const auto& [hotelId, hotelRecords]: Records) {
        for (const auto& rec: hotelRecords) {
            for (const auto& offer: rec->Offers) {
                TSelectedOffer selectedOffer;
                selectedOffer.Offer = &offer;
                selectedOffer.Rec = rec;
                selectedOffer.FinalPrice = selectedOffer.Offer->PriceVal;
                SelectedOffers.push_back(selectedOffer);
            }
        }
    }
}

void TOfferBlender::RemoveDisabledOperators() {
    auto isDisabledOp = [this](EOperatorId operatorId) {
        if (ReadRequestProcessor.BoYMode && !IsSimilarPermalink) {
            if (!ReadRequestProcessor.EnabledOperatorsBoY.contains(operatorId)) {
                return true;
            }
        } else {
            if (!ReadRequestProcessor.EnabledOperators.contains(operatorId)) {
                return true;
            }
        }
        return false;
    };
    BanOperators([&isDisabledOp](EOperatorId operatorId) {
        return isDisabledOp(operatorId);
    });
    SkipOffers([&isDisabledOp](const TSelectedOffer& selectedOffer){
        if (isDisabledOp(selectedOffer.Offer->OperatorId)) {
            return NTravelProto::NOfferCache::NApi::SR_OperatorDisabled;
        }
        return NTravelProto::NOfferCache::NApi::SR_None;
    });
}

void TOfferBlender::RemoveExperiments() {
    const bool isExp4460 = ReadRequestProcessor.Exp.IsExp(4460);
    const bool isExp4471 = ReadRequestProcessor.Exp.IsExp(4471);
    const bool isExp5782 = ReadRequestProcessor.Exp.IsExp(5782);

    THashSet<EOperatorId> availableOperators;
    for (const TSelectedOffer& selectedOffer: SelectedOffers) {
        availableOperators.insert(selectedOffer.Offer->OperatorId);
    }

    SkipOffers([isExp4460, isExp4471, isExp5782, &availableOperators](const TSelectedOffer& selectedOffer){
        if (!isExp4460 && selectedOffer.Offer->RoomCount > 1 && selectedOffer.Offer->SingleRoomAdultCount == 1) {
            return NTravelProto::NOfferCache::NApi::SR_Multiroom4460;
        } else if (!isExp4471 && selectedOffer.Offer->RoomCount > 1 && selectedOffer.Offer->SingleRoomAdultCount > 1) {
            return NTravelProto::NOfferCache::NApi::SR_Multiroom4471;
        } else if (isExp5782 && selectedOffer.Offer->OperatorId == NTravelProto::OI_OSTROVOK && availableOperators.size() > 1) {
            return NTravelProto::NOfferCache::NApi::SR_Ostrovok5782;
        }
        return NTravelProto::NOfferCache::NApi::SR_None;
    });
}

void TOfferBlender::RemoveMetaIfBoyAvailable() {
    if (!Req.GetOnlyBoYWhenBoYAvailable()) {
        return;
    }

    bool hasBoy = false;
    for (const TSelectedOffer& selectedOffer: SelectedOffers) {
        if (Service.IsBoYOperator(selectedOffer.Offer->OperatorId)) {
            hasBoy = true;
        }
    }

    if (!hasBoy) {
        return;
    }

    SkipOffers([this](const TSelectedOffer& selectedOffer){
        if (!Service.IsBoYOperator(selectedOffer.Offer->OperatorId)) {
            return NTravelProto::NOfferCache::NApi::SR_DisableMetaWhenBoYAvailable;
        }
        return NTravelProto::NOfferCache::NApi::SR_None;
    });
}

void TOfferBlender::RemoveStrikethroughPrices(std::function<bool (TSelectedOffer& selectedOffer)> shouldRemoveDiscountInfo) {
    for (auto &selectedOffer: SelectedOffers ) {
        if (shouldRemoveDiscountInfo(selectedOffer)) {
            selectedOffer.StrikethroughPrice = 0;
            selectedOffer.StrikethroughPriceReason = NTravelProto::NOfferCache::NApi::SPR_Unknown;
        }
    }
}

void TOfferBlender::RemoveMetaStrikethroughPricesForBoyHotels() {
    std::function<bool (TSelectedOffer& selectedOffer)> shouldRemoveDiscountIfo = [this](TSelectedOffer& selectedOffer) {
        return !Service.IsBoYOperator(selectedOffer.Offer->OperatorId) && RespHotel->GetHasBoyOffers() && Req.GetHideMetaStrikethroughForBoyHotels();
    };
    RemoveStrikethroughPrices(shouldRemoveDiscountIfo);
}


void TOfferBlender::RemoveBannedOfferTypes() {
    if (Service.Config().GetPartnerDataFilter().GetEnablePartnerDataOfferFilter()) {
        TStringBuilder logPrefix;
        logPrefix << JOB_LOG;
        SkipOffers([this, &logPrefix](const TSelectedOffer& selectedOffer) {
            return Service.PartnerDataOfferFilter().GetSkipReasonOrNone(logPrefix, *selectedOffer.Offer);
        });
    }
}

void TOfferBlender::FilterOffersByMinPrice() {
    ui32 minPrice = Service.Config().GetOther().GetMinOfferPrice();
    SkipOffers([minPrice](const TSelectedOffer& selectedOffer) {
        if (selectedOffer.FinalPrice < minPrice) {
            return NTravelProto::NOfferCache::NApi::SR_MinPrice;
        } else {
            return NTravelProto::NOfferCache::NApi::SR_None;
        }
    });
}

void TOfferBlender::RemoveRestrictedOffers() {
    auto allowRestrictedByMobile = Service.Config().GetOther().GetAllowRestrictedByMobileOffers();
    auto allowRestrictedByUser = Service.Config().GetOther().GetAllowRestrictedByUserOffers();
    SkipOffers([this, allowRestrictedByMobile, allowRestrictedByUser](const TSelectedOffer& selectedOffer) {
        if (selectedOffer.Offer->OfferRestrictions.RequiresMobile && (!allowRestrictedByMobile || !Req.GetAllowMobileRates())) {
            return NTravelProto::NOfferCache::NApi::SR_RequiresMobile;
        }
        if (selectedOffer.Offer->OfferRestrictions.RequiresRestrictedUser && (!allowRestrictedByUser || !Req.GetAllowRestrictedUserRates())) {
            return NTravelProto::NOfferCache::NApi::SR_RequiresRestrictedUser;
        }
        return NTravelProto::NOfferCache::NApi::SR_None;
    });
}

template <class TDedupKey>
void TOfferBlender::RemoveDuplicatedOffers(std::function<TDedupKey(const TSelectedOffer& selectedOffer)> keyGetter, const TStPricesSettings& stPricesSettings) {
    struct TGroupInfo {
        TSelectedOffer* BestOffer = nullptr;
        TSelectedOffer* SecondOffer = nullptr;
        TSelectedOffer* BestNonRestrictedOffer = nullptr;
    };

    // Найдем лучший оффер в каждой группе, любой и non-restricted
    THashMap<TDedupKey, TGroupInfo> groups;
    for (TSelectedOffer& selectedOffer: SelectedOffers) {
        TGroupInfo& groupInfo = groups[keyGetter(selectedOffer)];
        if (!groupInfo.BestOffer) {
            // Самый первый попавшийся
            groupInfo.BestOffer = &selectedOffer;
        } else if (SelectedOfferLess(selectedOffer, *groupInfo.BestOffer)) {
            // Лучше лучшего
            groupInfo.SecondOffer = groupInfo.BestOffer;
            groupInfo.BestOffer->SkippedDuringDeduplication = true;
            groupInfo.BestOffer = &selectedOffer;
        } else {
            // Хуже лучшего
            selectedOffer.SkippedDuringDeduplication = true;
            if (!groupInfo.SecondOffer || SelectedOfferLess(selectedOffer, *groupInfo.SecondOffer)) {
                // Но лучше второго
                groupInfo.SecondOffer = &selectedOffer;
            }
        }
        if (!selectedOffer.Offer->OfferRestrictions.IsAnyRestrictionSet()) {
            if (!groupInfo.BestNonRestrictedOffer || SelectedOfferLess(selectedOffer, *groupInfo.BestNonRestrictedOffer)) {
                groupInfo.BestNonRestrictedOffer = &selectedOffer;
            }
        }
    }

    // Подсчет st-цен на основе restricted предложений
    if (stPricesSettings.DetermineForRestrictedOffers) {
        for (const auto& [key, group]: groups) {
            // Если лучший оффер - restricted, а у лучшего non-restricted цена - больше, то это и будет st-цена
            if (group.BestOffer->Offer->OfferRestrictions.IsAnyRestrictionSet() && group.BestNonRestrictedOffer &&
                group.BestOffer->FinalPrice < group.BestNonRestrictedOffer->FinalPrice) {
                group.BestOffer->StrikethroughPrice = group.BestNonRestrictedOffer->FinalPrice;
                group.BestOffer->StrikethroughPriceReason = NTravelProto::NOfferCache::NApi::SPR_RestrictedOffer;
            }
        }
    }

    // Подсчет st-цены на основе второй цены для чёрной пятницы, она может перебить restricted цену
    if (stPricesSettings.DetermineForBlackFriday2021) {
        for (const auto& [key, group]: groups) {
            if (group.BestOffer->PromoRsp.GetBlackFriday2021Status().GetHotelHasSpecialTariff() && group.SecondOffer != nullptr
                && ((group.BestOffer->StrikethroughPrice == 0) ||
                    (group.BestOffer->StrikethroughPrice < group.SecondOffer->FinalPrice))
                ) {
                group.BestOffer->StrikethroughPrice = group.SecondOffer->FinalPrice;
                group.BestOffer->StrikethroughPriceReason = NTravelProto::NOfferCache::NApi::SPR_BlackFriday2021;
            }
        }
    }

    // Поскипаем офферы, у которых не минимальная цена
    SkipOffers([](const TSelectedOffer& selectedOffer) {
        if (selectedOffer.SkippedDuringDeduplication) {
            return NTravelProto::NOfferCache::NApi::SR_NonDistinguishable;
        } else {
            return NTravelProto::NOfferCache::NApi::SR_None;
        }
    });
}

void TOfferBlender::RemoveDuplicatedOffersBeforeScanPermarooms() {
    if (Req.GetOfferDeduplicationMode() == NTravelProto::NOfferCache::NApi::ODM_None) {
        return;
    }
    TStPricesSettings stPricesSettings;
    stPricesSettings.DetermineForRestrictedOffers = true;
    stPricesSettings.DetermineForBlackFriday2021 = true;
    using TDedupKey = std::tuple<EOperatorId, TString/*Title*/, EFreeCancellationType, NTravelProto::EPansionType>;
    RemoveDuplicatedOffers<TDedupKey>([](const TSelectedOffer& selectedOffer){
        const TOffer& offer = *selectedOffer.Offer;
        return std::make_tuple(offer.OperatorId, selectedOffer.GetDecompressedTitle(), offer.FreeCancellation, offer.PansionType);
    }, stPricesSettings);
}

void TOfferBlender::RemoveDuplicatedOffersAfterScanPermarooms() {
    TStPricesSettings stPricesSettings{};
    if (ShowPermarooms && Req.GetOfferDeduplicationMode() == NTravelProto::NOfferCache::NApi::ODM_ByPermaroom) {
        using TDedupKey = std::tuple<EOperatorId, TPermaroomIdStr, EFreeCancellationType, NTravelProto::EPansionType>;
        RemoveDuplicatedOffers<TDedupKey>([](const TSelectedOffer& selectedOffer){
            const TOffer& offer = *selectedOffer.Offer;
            return std::make_tuple(offer.OperatorId, selectedOffer.PermaroomId, offer.FreeCancellation, offer.PansionType);
        }, stPricesSettings);
    } else if (ShowPermarooms && Req.GetOfferDeduplicationMode() == NTravelProto::NOfferCache::NApi::ODM_ByPermaroomOnly) {
        using TDedupKey = std::tuple<EOperatorId, TPermaroomIdStr>;
        RemoveDuplicatedOffers<TDedupKey>([](const TSelectedOffer& selectedOffer){
            const TOffer& offer = *selectedOffer.Offer;
            return std::make_tuple(offer.OperatorId, selectedOffer.PermaroomId);
        }, stPricesSettings);
    } else if (Req.GetOfferDeduplicationMode() == NTravelProto::NOfferCache::NApi::ODM_ByPermaroomOnlyWithByPartnerFallback) {
        if (ShowPermarooms) {
            using TDedupKey = std::tuple<EOperatorId, TPermaroomIdStr>;
            RemoveDuplicatedOffers<TDedupKey>([](const TSelectedOffer& selectedOffer){
                const TOffer& offer = *selectedOffer.Offer;
                return std::make_tuple(offer.OperatorId, selectedOffer.PermaroomId);
            }, stPricesSettings);
        } else {
            using TDedupKey = std::tuple<EOperatorId>;
            RemoveDuplicatedOffers<TDedupKey>([](const TSelectedOffer& selectedOffer){
                const TOffer& offer = *selectedOffer.Offer;
                return std::make_tuple(offer.OperatorId);
            }, stPricesSettings);
        }
    }
}

TOfferBlender::ECatRoomStatus TOfferBlender::ScanPermarooms() {
    struct TPermaroomInfo {
        TPermaroom Permaroom;
        const TSelectedOffer* MinSelectedOffer = nullptr;
        bool IsOther = false;
    };

    if (!AllowPermarooms) {
        return NTravelProto::NOfferCache::NApi::CRS_NotNeeded;
    }
    auto enabledPartners = Service.GetPartnersForOperators(EnabledAfterFilteringOperators);
    TCatRoomDataSourceIdStr dsId = Req.GetCatRoomDataSourceId();
    auto matchOnlyBoYOffers = false;
    auto foundDataSource = false;
    if (dsId.empty() || dsId == "0") {
        if (Req.GetAutoCatRoomOnlyForBoY() || Req.GetAutoCatRoomForAll()) {
            matchOnlyBoYOffers = Req.GetAutoCatRoomOnlyForBoY();
            auto handlePartner = [this, &dsId, &foundDataSource](EPartnerId partner) {
                if (partner == EPartnerId::PI_EXPEDIA || partner == EPartnerId::PI_TRAVELLINE || partner == EPartnerId::PI_BNOVO || partner == EPartnerId::PI_BRONEVIK) {
                    foundDataSource = true;
                    dsId = "BoY_" + NTravelProto::EPartnerId_Name(partner);
                    return true;
                } else if (partner == EPartnerId::PI_DOLPHIN && !Req.GetAutoCatRoomForAll()) {
                    foundDataSource = true;
                    dsId = ToString(static_cast<int>(partner));
                    return true;
                }
                return false;
            };
            for (const TSelectedOffer& selectedOffer: SelectedOffers) {
                auto partner = selectedOffer.Rec->Key.PreKey.HotelId.PartnerId;
                if (handlePartner(partner)) {
                    break;
                }
            }
            if (!foundDataSource) { // not found suitable partner among SelectedOffers
                for (const auto& hotelId : ReadRequestProcessor.ComplexHotelIds[Permalink].HotelIds) {
                    if (!enabledPartners.contains(hotelId.PartnerId)) {
                        continue;
                    }
                    if (handlePartner(hotelId.PartnerId)) {
                        break;
                    }
                }
            }
            if (!foundDataSource) { // not found suitable partner among SelectedOffers and HotelIds
                return NTravelProto::NOfferCache::NApi::CRS_NoMarkup; // stopping here to avoid using default catroom (dsId=0)
            }
        }
    }

    SelectedCatRoomDataSourceId = dsId;

    TPermaroom otherPermaroom;
    otherPermaroom.PermaroomId = g_OtherPermaroomId;
    otherPermaroom.Name = g_OtherPermaroomName;

    // Получаем пермарумы
    THashMap<THotelId, TRoomService::TGetHotelPermaroomsRequest> reqsMapping;
    for (const auto& hotelId : ReadRequestProcessor.ComplexHotelIds[Permalink].HotelIds) {
        if (!enabledPartners.contains(hotelId.PartnerId)) {
            continue;
        }
        if (matchOnlyBoYOffers && !Service.IsBoYPartner(hotelId.PartnerId)) {
            continue;
        }
        if (!reqsMapping.contains(hotelId)) {
            auto req = TRoomService::TGetHotelPermaroomsRequest();
            req.HotelId = hotelId;
            reqsMapping[hotelId] = req;
        }
    }
    for (TSelectedOffer& selectedOffer: SelectedOffers) {
        const TOffer& offer = *(selectedOffer.Offer);
        selectedOffer.CatRoomMappingKey = OfferToRoomKeyMapper.GetCatRoomMappingKey(offer, selectedOffer.GetDecompressedTitle());
        if (matchOnlyBoYOffers && !Service.IsBoYPartner(selectedOffer.Rec->Key.PreKey.HotelId.PartnerId)) {
            continue;
        }
        auto reqIt = reqsMapping.find(selectedOffer.Rec->Key.PreKey.HotelId);
        if (reqIt == reqsMapping.end()) {
            ERROR_LOG << "Selected offer has HotelId not from ComplexHotelIds (" << selectedOffer.Rec->Key.PreKey.HotelId << ")" << Endl;
            return NTravelProto::NOfferCache::NApi::CRS_NoMarkup;
        }
        reqIt->second.MappingKeys.emplace_back(offer.OperatorId, selectedOffer.Rec->Key.PreKey.HotelId.OriginalId, selectedOffer.CatRoomMappingKey);
    }
    TVector<TRoomService::TGetHotelPermaroomsRequest> reqs;
    reqs.reserve(reqsMapping.size());
    for (const auto& [_, req] : reqsMapping) {
        reqs.push_back(req);
    }
    auto permaroomResult = Service.RoomService().GetPermarooms(dsId, reqs);

    // Вычислим минимальную цену номера для каждого permaroom-а
    THashMap<TPermaroomIdStr, TPermaroomInfo> permarooms;
    THashMap<EOperatorId, TCatRoomStat> op2stat;
    TCatRoomStat totalStat{};
    for (TSelectedOffer& selectedOffer: SelectedOffers) {
        if (matchOnlyBoYOffers && !Service.IsBoYPartner(selectedOffer.Rec->Key.PreKey.HotelId.PartnerId)) {
            continue;
        }
        const TOffer& offer = *(selectedOffer.Offer);
        auto permaroomMappingKey = TPermaroomMappingKey(offer.OperatorId, selectedOffer.Rec->Key.PreKey.HotelId.OriginalId, selectedOffer.CatRoomMappingKey);

        auto& perOpStat = op2stat[offer.OperatorId];
        ++totalStat.OfferCountAll;
        ++perOpStat.OfferCountAll;

        auto permaroomId = permaroomResult.Mapping.at(permaroomMappingKey).GetOrElse(otherPermaroom.PermaroomId);
        auto isOther = permaroomId == otherPermaroom.PermaroomId;

        TPermaroomInfo& pi = permarooms[permaroomId];
        if (!pi.MinSelectedOffer) {
            pi.Permaroom = isOther ? otherPermaroom : permaroomResult.Permarooms.at(permaroomId);
            pi.MinSelectedOffer = &selectedOffer;
            pi.IsOther = isOther;
        }

        selectedOffer.PermaroomId = pi.Permaroom.PermaroomId;
        selectedOffer.PermaroomVersion = pi.Permaroom.PermaroomVersion;
        selectedOffer.RawPermaroomId = pi.Permaroom.PermaroomId;

        if (pi.IsOther) {
            ++totalStat.OfferCountOther;
            ++perOpStat.OfferCountOther;
        }

        if (!pi.IsOther || Req.GetCatRoomShowOther()) {
            perOpStat.PermaroomIds.insert(selectedOffer.PermaroomId);
            if (SelectedOfferLess(selectedOffer, *pi.MinSelectedOffer)) {
                pi.MinSelectedOffer = &selectedOffer;
            }
        }
    }
    auto status = ValidateCatRoomOtherPortionAndReportStats(dsId, totalStat, op2stat);
    if (status.Defined()) {
        return *status.Get();
    }
    // Отсортируем пермарумы по цене (и порядку)
    TVector<const TPermaroomInfo*> permaroomInfoVector;
    TVector<TPermaroomInfo> soldOutPermarooms;
    if (Req.GetShowPermaroomsWithNoOffers()) {
        permaroomInfoVector.reserve(permaroomResult.Permarooms.size() + 1); // all permarooms + other
        soldOutPermarooms.reserve(permaroomResult.Permarooms.size());
        for (const auto& [_, permaroomInfo] : permarooms) {
            permaroomInfoVector.push_back(&permaroomInfo);
        }
        for (const auto& [permaroomId, permaroom] : permaroomResult.Permarooms) {
            if (!permarooms.contains(permaroomId)) {
                auto& permaroomInfo = soldOutPermarooms.emplace_back();
                permaroomInfo.Permaroom = permaroom;
                permaroomInfoVector.push_back(&permaroomInfo);
            }
        }
    } else {
        permaroomInfoVector.reserve(permarooms.size());
        for (const auto& [_, permaroomInfo] : permarooms) {
            Y_VERIFY(permaroomInfo.MinSelectedOffer);
            permaroomInfoVector.push_back(&permaroomInfo);
        }
    }
    Sort(permaroomInfoVector, [this](const TPermaroomInfo* p1, const TPermaroomInfo* p2) {
        if (p1->IsOther != p2->IsOther) {
            return p2->IsOther;
        }
        if (!p1->MinSelectedOffer && !p2->MinSelectedOffer) {
            return p1->Permaroom.Name < p2->Permaroom.Name;
        }
        if (!p1->MinSelectedOffer || !p2->MinSelectedOffer) {
            return !p2->MinSelectedOffer;
        }
        auto byPrice = SelectedOfferLessByComplexPrice(*p1->MinSelectedOffer, *p2->MinSelectedOffer);
        if (byPrice.Defined()) {
            return byPrice.GetRef();
        }
        if (p1->Permaroom.Name != p2->Permaroom.Name) {
            return p1->Permaroom.Name < p2->Permaroom.Name;
        }
        return p1->Permaroom.PermaroomId < p2->Permaroom.PermaroomId;
    });

    if (permaroomInfoVector.empty()) {
        return NTravelProto::NOfferCache::NApi::CRS_NotNeeded;
    }

    if (IsMainPermalink) {
        ReadRequestProcessor.Stats.Info.SetMainPermalink_CatRoomAvailable(true);
    }

    if (Req.GetEnableCatRoomProps() && !Req.GetEnableCatRoom()) {
        return NTravelProto::NOfferCache::NApi::CRS_NotNeeded;
    }

    // Удалим невидимые офферы
    auto szPrev = SelectedOffers.size();

    if (!Req.GetCatRoomShowOther()) {
        SkipOffers([&otherPermaroom](const TSelectedOffer& selectedOffer){
            if (selectedOffer.PermaroomId == otherPermaroom.PermaroomId) {
                return NTravelProto::NOfferCache::NApi::SR_PermaroomHidden;
            }
            return NTravelProto::NOfferCache::NApi::SR_None;
        });
    }
    Service.GetCounters().NCatRoomHiddenOffers += szPrev - SelectedOffers.size();

    auto showBlackFriday2021Badge = Service.GetPromoService().ShowBlackFriday2021BadgeForHotel(
        ReadRequestProcessor.Started,
        ReadRequestProcessor.ComplexHotelIds[Permalink]
    );

    // А теперь положим пермарумы в ответ
    for (const TPermaroomInfo* permaroomInfo: permaroomInfoVector) {
        if (!permaroomInfo->IsOther || Req.GetCatRoomShowOther()) {
            AddPermaroomToHotel(permaroomInfo->Permaroom, showBlackFriday2021Badge);
        }
    }
    return NTravelProto::NOfferCache::NApi::CRS_OK;
}

TMaybe<TOfferBlender::ECatRoomStatus> TOfferBlender::ValidateCatRoomOtherPortionAndReportStats(
    const TCatRoomDataSourceIdStr& dsId,
    const TCatRoomStat& totalStat,
    const THashMap<EOperatorId, TCatRoomStat>& op2stat) {
    if (SelectedOffers.size() > 0) {
        Service.GetCounters().NCatRoomOtherPctLess.Update((100 * totalStat.OfferCountOther) / SelectedOffers.size());
    }
    for (const auto& [operatorId, perOpStat]: op2stat) {
        if (perOpStat.OfferCountAll > 0) {
            Service.GetCountersPerOperator(operatorId)->NCatRoomOtherPctLess.Update((100 * perOpStat.OfferCountOther) / perOpStat.OfferCountAll);
        }
    }
    int catRoomMaxOtherPct = Req.HasCatRoomMaxOtherPct() ? Req.GetCatRoomMaxOtherPct() : Service.Config().GetOther().GetCatRoomMaxOtherPct();
    bool tooMuchOther = (totalStat.OfferCountOther * 100) > (SelectedOffers.size() * catRoomMaxOtherPct);
    if (tooMuchOther) {
        DEBUG_LOG << JOB_LOG << "Not show permarooms because have too much other permarooms: " << totalStat.OfferCountOther << " of "
                  << SelectedOffers.size() << " visible offers, with total " << SelectedOffers.size() << " offers" << Endl;
        return NTravelProto::NOfferCache::NApi::CRS_TooMuchOther;
    }
    if (IsMainPermalink) {
        ReadRequestProcessor.Stats.Info.SetMainPermalink_CatRoomDSId(dsId);
        ReadRequestProcessor.Stats.Info.SetMainPermalink_CatRoomCategoryCount(totalStat.PermaroomIds.size());
        ReadRequestProcessor.Stats.Info.SetMainPermalink_CatRoomOtherOfferCount(totalStat.OfferCountOther);
        for (auto it = op2stat.begin(); it != op2stat.end(); ++it) {
            auto& byOp = (*ReadRequestProcessor.Stats.Info.MutableByOperators())[it->first];
            byOp.SetMainPermalink_CatRoomOtherOfferCount(it->second.OfferCountOther);
            byOp.SetMainPermalink_CatRoomCategoryCount(it->second.PermaroomIds.size());
        }
    }
    return {};
}


void TOfferBlender::ProcessCatRoomStatus() {
    auto catRoomCounters = Service.GetCatRoomCounters(SelectedCatRoomDataSourceId, Req.GetAutoCatRoomForAll());
    switch (CatRoomStatus) {
        case NTravelProto::NOfferCache::NApi::CRS_NotNeeded:
            return;
        case NTravelProto::NOfferCache::NApi::CRS_OK:
            ShowPermarooms = true;
            Service.GetCounters().NCatRoomOK.Inc();
            catRoomCounters->NStatusOK.Inc();
            break;
        case NTravelProto::NOfferCache::NApi::CRS_NoMarkup:
            Service.GetCounters().NCatRoomPermalinkNotPublished.Inc();
            catRoomCounters->NStatusNoMarkup.Inc();
            break;
        case NTravelProto::NOfferCache::NApi::CRS_TooMuchOther:
            Service.GetCounters().NCatRoomOnlyOther.Inc();
            catRoomCounters->NStatusTooMuchOther.Inc();
            break;
        case NTravelProto::NOfferCache::NApi::CRS_WrongPermaroomId:
            Service.GetCounters().NCatRoomWrongPermaroomId.Inc();
            catRoomCounters->NStatusWrongPermaroomId.Inc();
            break;
        case NTravelProto::NOfferCache::NApi::CRS_EmptyPermaroomId:
            Service.GetCounters().NCatRoomEmptyPermaroomId.Inc();
            catRoomCounters->NStatusEmptyPermaroomId.Inc();
            break;
    }
    if (IsMainPermalink) {
        ReadRequestProcessor.Stats.Info.SetMainPermalink_CatRoomShown(ShowPermarooms);
        ReadRequestProcessor.Stats.Info.SetMainPermalink_CatRoomTooManyOther(CatRoomStatus == NTravelProto::NOfferCache::NApi::CRS_TooMuchOther);
    }
    RespHotel->SetShowPermarooms(ShowPermarooms);
    RespHotel->SetCatRoomStatus(CatRoomStatus);
}

void TOfferBlender::AddPermaroomToHotel(const TPermaroom& permaroom, bool showBlackFriday2021Badge) {
    auto pbPermaroom = RespHotel->AddPermarooms();
    pbPermaroom->SetId(permaroom.PermaroomId);
    pbPermaroom->SetName(permaroom.Name);
    pbPermaroom->SetDescription(permaroom.Description);

    for (const auto& photo: permaroom.Photos) {
        auto pbPhoto = pbPermaroom->AddPhotos();
        for (const auto& size: photo.Sizes) {
            auto pbSize = pbPhoto->AddSizes();
            pbSize->SetHeight(size.Height);
            pbSize->SetWidth(size.Width);
            pbSize->SetSize(size.Size);
        }
        pbPhoto->SetUrlTemplate(photo.UrlTemplate);
    }
    for (const auto& feature: permaroom.BinaryFeatures) {
        auto pbFeature = pbPermaroom->AddBinaryFeatures();
        pbFeature->SetId(feature.Commons.Id);
        pbFeature->SetName(feature.Commons.Name);
        pbFeature->SetValue(feature.Value);
        if (feature.Commons.TopFeatureImportance.Defined()) {
            pbFeature->SetTopFeatureImportance(feature.Commons.TopFeatureImportance.GetRef());
        }
        pbFeature->SetCategory(feature.Commons.Category);
        pbFeature->SetCategoryName(feature.Commons.CategoryName);
        pbFeature->SetDisplayValue(feature.Commons.DisplayValue);
        pbFeature->SetIconId(feature.Commons.IconId);
    }
    for (const auto& feature: permaroom.EnumFeatures) {
        auto pbFeature = pbPermaroom->AddEnumFeatures();
        pbFeature->SetId(feature.Commons.Id);
        pbFeature->SetName(feature.Commons.Name);
        pbFeature->SetValueId(feature.ValueId);
        pbFeature->SetValueName(feature.ValueName);
        if (feature.Commons.TopFeatureImportance.Defined()) {
            pbFeature->SetTopFeatureImportance(feature.Commons.TopFeatureImportance.GetRef());
        }
        pbFeature->SetCategory(feature.Commons.Category);
        pbFeature->SetCategoryName(feature.Commons.CategoryName);
        pbFeature->SetDisplayValue(feature.Commons.DisplayValue);
        pbFeature->SetIconId(feature.Commons.IconId);
    }
    for (const auto& feature: permaroom.IntegerFeatures) {
        auto pbFeature = pbPermaroom->AddIntegerFeatures();
        pbFeature->SetId(feature.Commons.Id);
        pbFeature->SetName(feature.Commons.Name);
        pbFeature->SetValue(feature.Value);
        if (feature.Commons.TopFeatureImportance.Defined()) {
            pbFeature->SetTopFeatureImportance(feature.Commons.TopFeatureImportance.GetRef());
        }
        pbFeature->SetCategory(feature.Commons.Category);
        pbFeature->SetCategoryName(feature.Commons.CategoryName);
        pbFeature->SetDisplayValue(feature.Commons.DisplayValue);
        pbFeature->SetIconId(feature.Commons.IconId);
    }
    for (const auto& feature: permaroom.FloatFeatures) {
        auto pbFeature = pbPermaroom->AddFloatFeatures();
        pbFeature->SetId(feature.Commons.Id);
        pbFeature->SetName(feature.Commons.Name);
        pbFeature->SetValue(feature.Value);
        if (feature.Commons.TopFeatureImportance.Defined()) {
            pbFeature->SetTopFeatureImportance(feature.Commons.TopFeatureImportance.GetRef());
        }
        pbFeature->SetCategory(feature.Commons.Category);
        pbFeature->SetCategoryName(feature.Commons.CategoryName);
        pbFeature->SetDisplayValue(feature.Commons.DisplayValue);
        pbFeature->SetIconId(feature.Commons.IconId);
    }
    for (const auto& feature: permaroom.StringFeatures) {
        auto pbFeature = pbPermaroom->AddStringFeatures();
        pbFeature->SetId(feature.Commons.Id);
        pbFeature->SetName(feature.Commons.Name);
        pbFeature->SetValue(feature.Value);
        if (feature.Commons.TopFeatureImportance.Defined()) {
            pbFeature->SetTopFeatureImportance(feature.Commons.TopFeatureImportance.GetRef());
        }
        pbFeature->SetCategory(feature.Commons.Category);
        pbFeature->SetCategoryName(feature.Commons.CategoryName);
        pbFeature->SetDisplayValue(feature.Commons.DisplayValue);
        pbFeature->SetIconId(feature.Commons.IconId);
    }

    auto fillFeature = [](const TFeatureCommons& commons, const TString& name, NTravelProto::NOfferCache::NApi::TReadResp::TPermaroom::TFeature* pbFeature) {
        pbFeature->SetId(commons.Id);
        if (commons.DisplayValue.empty()) {
            pbFeature->SetName(name);
        } else {
            pbFeature->SetName(commons.DisplayValue);
        }
        if (commons.TopFeatureImportance.Defined()) {
            pbFeature->SetTopFeatureImportance(commons.TopFeatureImportance.GetRef());
        }
        pbFeature->SetCategory(commons.Category);
        pbFeature->SetCategoryName(commons.CategoryName);
        pbFeature->SetIconId(commons.IconId);
    };

    for (const auto& feature: permaroom.BinaryFeatures) {
        auto name = feature.Value ? feature.Commons.Name : feature.Commons.Name + ": нет";
        fillFeature(feature.Commons, name, pbPermaroom->AddFeatures());
    }
    for (const auto& feature: permaroom.EnumFeatures) {
        auto name = feature.ValueId == "yes" ? feature.Commons.Name : feature.Commons.Name + ": " + feature.ValueName;
        fillFeature(feature.Commons, name, pbPermaroom->AddFeatures());
    }
    for (const auto& feature: permaroom.IntegerFeatures) {
        if (feature.Commons.Id == ROOM_AREA_FEATURE_ID) {
            pbPermaroom->MutableRoomArea()->SetValue(feature.Value);
            pbPermaroom->MutableRoomArea()->SetUnit(NTravelProto::NOfferCache::NApi::TReadResp_TPermaroom_TRoomArea_EAreaUnit_AU_SquareMeters);
        } else {
            auto name = feature.Commons.Name + ": " + Sprintf("%d", feature.Value);
            fillFeature(feature.Commons, name, pbPermaroom->AddFeatures());
        }
    }
    for (const auto& feature: permaroom.FloatFeatures) {
        auto name = feature.Commons.Name + ": " + Sprintf("%.2f", feature.Value);
        fillFeature(feature.Commons, name, pbPermaroom->AddFeatures());
    }
    for (const auto& feature: permaroom.StringFeatures) {
        auto name = feature.Commons.Name + ": " + feature.Value;
        fillFeature(feature.Commons, name, pbPermaroom->AddFeatures());
    }

    auto getWordForm = [](int quantity, TString nominativeCaseSingular, TString genitiveCaseSingular, TString genitiveCasePlural) {
        quantity %= 100;
        if (quantity > 10 && quantity < 20) {
            return genitiveCasePlural;
        }
        if (quantity % 10 == 1) {
            return nominativeCaseSingular;
        }
        if (quantity % 10 == 2 || quantity % 10 == 3 || quantity % 10 == 4) {
            return genitiveCaseSingular;
        }
        return genitiveCasePlural;
    };

    for (const auto& bedGroup: permaroom.BedGroups) {
        auto pbBedGroup = pbPermaroom->AddBedGroups();
        for (const auto& configurationItem: bedGroup.Configuration) {
            auto pbConfigurationItem = pbBedGroup->AddConfiguration();
            pbConfigurationItem->SetType(configurationItem.Type);
            pbConfigurationItem->SetNameInitialForm(configurationItem.NominativeCaseSingular);
            pbConfigurationItem->SetNameInflectedForm(getWordForm(configurationItem.Quantity,
                                                                  configurationItem.NominativeCaseSingular,
                                                                  configurationItem.GenitiveCaseSingular,
                                                                  configurationItem.GenitiveCasePlural));
            pbConfigurationItem->SetQuantity(configurationItem.Quantity);
        }
        pbBedGroup->SetId(bedGroup.Id);
    }

    if (showBlackFriday2021Badge) {
        auto pbBage = pbPermaroom->AddBadges();
        pbBage->CopyFrom(Service.Config().GetOther().GetBlackFriday2021Badge());
    }
}

void TOfferBlender::SelectBestBoY() {
    if (Req.GetShowAllBoY()) {
        return;
    }

    const TSelectedOffer* bestBoYSelectedOffer = nullptr; // кроме boydirect, т.е. bnovo и tl
    // Первая фаза, сканируем
    THashSet<EOperatorId> boyDirectOpIds;
    for (const TSelectedOffer& selectedOffer: SelectedOffers) {
        EOperatorId opId = selectedOffer.Offer->OperatorId;
        if (Service.IsBoYDirectOperator(opId)) {
            boyDirectOpIds.insert(opId);
        } else if (Service.IsBoYOperator(opId)) {
            if (!bestBoYSelectedOffer || SelectedOfferLess(selectedOffer, *bestBoYSelectedOffer)) {
                bestBoYSelectedOffer = &selectedOffer;
            }
        }
    }
    bool removeBoYDirect = false;
    EOperatorId selectedBoYOpId = NTravelProto::OI_UNUSED;
    if (boyDirectOpIds.size() > 1) {
        WARNING_LOG << JOB_LOG << "Multiple boy-direct operators for permalink " << Permalink << Endl;
        Service.GetCounters().NMultipleBoYDirect.Inc();
        removeBoYDirect = true;
        boyDirectOpIds.clear();
    }
    if (boyDirectOpIds.size() == 1) {
        selectedBoYOpId = *boyDirectOpIds.begin();
    } else if (bestBoYSelectedOffer) {
        selectedBoYOpId = bestBoYSelectedOffer->Offer->OperatorId;
    }
    if (!removeBoYDirect && selectedBoYOpId == NTravelProto::OI_UNUSED) {
        // ничего удалять не надо
        return;
    }
    // Вторая фаза, фильтрация BoY, HOTELS-4408
    SkipOffers([this, removeBoYDirect, selectedBoYOpId](const TSelectedOffer& selectedOffer){
        EOperatorId opId = selectedOffer.Offer->OperatorId;
        if (removeBoYDirect && Service.IsBoYDirectOperator(opId)) {
            return NTravelProto::NOfferCache::NApi::SR_MultipleBoYDirect;
        }
        if (Service.IsBoYOperator(opId) && opId != selectedBoYOpId) {
            return NTravelProto::NOfferCache::NApi::SR_OtherBoYOperatorWon;
        }
        return NTravelProto::NOfferCache::NApi::SR_None;
    });
    BanOperators([this, removeBoYDirect, selectedBoYOpId](EOperatorId operatorId) {
        if (removeBoYDirect && Service.IsBoYDirectOperator(operatorId)) {
            return true;
        }
        if (Service.IsBoYOperator(operatorId) && operatorId != selectedBoYOpId) {
            return true;
        }
        return false;
    });
}

void TOfferBlender::SkipOffers(std::function<EOfferSkipReason (const TSelectedOffer& selectedOffer)> skipper) {
    for (auto it = SelectedOffers.begin(); it != SelectedOffers.end(); ) {
        EOfferSkipReason sr = skipper(*it);
        if (sr == NTravelProto::NOfferCache::NApi::SR_None) {
           ++it;
        } else {
            it->SkipReason = sr;
            SkippedOffers.push_back(*it);
            it = SelectedOffers.erase(it);
        }
    }
}

void TOfferBlender::BanOperators(const std::function<bool (EOperatorId operatorId)>& isBanned) {
    for (auto it = EnabledAfterFilteringOperators.begin(); it != EnabledAfterFilteringOperators.end(); ) {
        if (isBanned(*it)) {
            EnabledAfterFilteringOperators.erase(it++);
        } else {
            ++it;
        }
    }
}

void TOfferBlender::UpdateTotalPriceRange() {
    for (const TSelectedOffer& offer: SelectedOffers) {
        ReadRequestProcessor.TotalPriceValRange.Update(offer.FinalPrice);
    }
}

void TOfferBlender::ApplyAuxFilters() {
    SkipOffers([this](const TSelectedOffer& selectedOffer){
        if (IsOfferGoodForAuxFilter(selectedOffer)) {
            return NTravelProto::NOfferCache::NApi::SR_None;
        }
        return NTravelProto::NOfferCache::NApi::SR_AuxFilter;
    });
}

bool TOfferBlender::IsOfferGoodForAuxFilter(const TSelectedOffer& selectedOffer) const {
    const TOffer& offer = *selectedOffer.Offer;
    if (Req.FilterPansionAliasSize() > 0) {
        bool passesFilter = false;
        for (const auto& alias : Req.GetFilterPansionAlias()) {
            switch (alias) {
                case NTravelProto::NOfferCache::NApi::NO_PANSION:
                    if (HasNoPansion(offer.PansionType)) {
                        passesFilter = true;
                    }
                    break;
                case NTravelProto::NOfferCache::NApi::BREAKFAST:
                    if (HasBreakfast(offer.PansionType)) {
                        passesFilter = true;
                    }
                    break;
                case NTravelProto::NOfferCache::NApi::NO_BREAKFAST:
                    if (!HasBreakfast(offer.PansionType)) {
                        passesFilter = true;
                    }
                    break;
                case NTravelProto::NOfferCache::NApi::BREAKFAST_DINNER:
                    if (HasBreakfastDinner(offer.PansionType)) {
                        passesFilter = true;
                    }
                    break;
                case NTravelProto::NOfferCache::NApi::BREAKFAST_LUNCH_DINNER:
                    if (HasBreakfastLunchDinner(offer.PansionType)) {
                        passesFilter = true;
                    }
                    break;
                case NTravelProto::NOfferCache::NApi::ALL_INCLUSIVE:
                    if (HasAllInclusive(offer.PansionType)) {
                        passesFilter = true;
                    }
                    break;
                default:
                    break;
            }
        }
        if (!passesFilter) {
            return false;
        }
    }
    if (Req.HasFilterPriceFrom() && selectedOffer.FinalPrice < Req.GetFilterPriceFrom()) {
        return false;
    }
    if (Req.HasFilterPriceTo() && selectedOffer.FinalPrice > Req.GetFilterPriceTo()) {
        return false;
    }
    if (Req.GetOnlyRestrictedOffers() && !selectedOffer.Offer->OfferRestrictions.IsAnyRestrictionSet()) {
        return false;
    }
    return true;
}


void TOfferBlender::FillUserFiltersInfo() {
    if (IsBrief || !ReadRequestProcessor.Exp.IsExp(3750) || Req.GetCompactResponseForCalendar()) {
        return;
    }
    struct TFilterStat {
        size_t OfferCount = 0;
        TPriceVal MinPrice = 0;

        void Update(const TSelectedOffer& selectedOffer) {
            if (OfferCount == 0) {
                MinPrice = selectedOffer.FinalPrice;
            } else {
                MinPrice = Min(MinPrice, selectedOffer.FinalPrice);
            }
            ++OfferCount;
        }

        void PutToProto(NTravelProto::NOfferCache::NApi::TReadResp::THotel::TFilter::TValue& pb) const {
            pb.SetOfferCount(OfferCount);
            pb.SetMinPrice(MinPrice);
        }
    };

    THashMap<EOCPansion, TFilterStat> pansionTypes;
    THashMap<EFreeCancellationType, TFilterStat> freeCancellations;
    THashMap<EOperatorId, TFilterStat> operatorIds;

    for (const TSelectedOffer& selectedOffer: SelectedOffers) {
        const TOffer& offer = *selectedOffer.Offer;
        bool okByPansion = IsOfferGoodForUserFilterByPansion(offer);
        bool okByFreeCancellation = IsOfferGoodForUserFilterByFreeCancellation(offer);
        bool okByOperator = IsOfferGoodForUserFilterByOperator(offer);
        // Текущие фильтры влияют на статистику в других фильтрах, но не в себе, TRAVELBACK-585
        if (okByFreeCancellation && okByOperator) {
            pansionTypes[PansionToOCFormat(offer.PansionType)].Update(selectedOffer);
        }
        if (okByPansion && okByOperator) {
            freeCancellations[offer.FreeCancellation].Update(selectedOffer);
        }
        if (okByPansion && okByFreeCancellation) {
            operatorIds[offer.OperatorId].Update(selectedOffer);
        }
    }

    if (pansionTypes.size() > 1) {
        // Multiselect mode
        auto& filter = *RespHotel->AddFilters();
        filter = Service.Config().GetFilterPansion();
        Y_ASSERT(filter.ValuesSize() == 0);
        for (const auto& [pansion, filterStat]: pansionTypes) {
            if (pansion == NTravelProto::NOfferCache::NApi::TOCPansion::RO ||
                pansion == NTravelProto::NOfferCache::NApi::TOCPansion::UNKNOWN) {// See TRAVELBACK-635
                continue;
            }
            auto& value = *filter.AddValues();
            value.SetId(ToString(pansion));
            value.SetDisplayName(Service.GetPansionDisplayName(pansion));
            filterStat.PutToProto(value);
        }
        if (filter.ValuesSize() == 2) {
            filter.SetType(NTravelProto::NOfferCache::NApi::Oneof);
        }
        if (filter.ValuesSize() == 1) {
            filter.SetType(NTravelProto::NOfferCache::NApi::Checkbox);
        }
        if (filter.ValuesSize() == 0) {
            // Ну мало ли...
            RespHotel->MutableFilters()->RemoveLast();
        }
    }
    if ((freeCancellations.size() > 1) && freeCancellations.contains(EFreeCancellationType::Yes)) {
        auto& filter = *RespHotel->AddFilters();
        filter.CopyFrom(Service.Config().GetFilterFreeCancellation());
        if (filter.ValuesSize() > 0) {
            freeCancellations[EFreeCancellationType::Yes].PutToProto(*filter.MutableValues(0));
        }
    }
    if (operatorIds.size() > 1) {
        auto& filter = *RespHotel->AddFilters();
        filter = Service.Config().GetFilterOperator();
        Y_ASSERT(filter.ValuesSize() == 0);
        for (const auto& [operatorId, filterStat]: operatorIds) {
            auto& value = *filter.AddValues();
            value.SetId(ToString(std::underlying_type_t<EOperatorId>(operatorId)));
            value.SetDisplayName(Service.GetOperator(operatorId)->GetName());
            filterStat.PutToProto(value);
        }
    }
}

void TOfferBlender::ApplyUserFilters() {
    SkipOffers([this](const TSelectedOffer& selectedOffer){
        const TOffer& offer = *selectedOffer.Offer;
        if (IsOfferGoodForUserFilterByPansion(offer) && IsOfferGoodForUserFilterByFreeCancellation(offer) && IsOfferGoodForUserFilterByOperator(offer)) {
            return NTravelProto::NOfferCache::NApi::SR_None;
        }
        return NTravelProto::NOfferCache::NApi::SR_UserFilter;
    });
}

bool TOfferBlender::IsOfferGoodForUserFilterByPansion(const TOffer& offer) const {
    if ((Req.FilterPansionSize() != 0) && !FindPtr(Req.GetFilterPansion(), offer.PansionType)) {
        return false;
    }
    return true;
}

bool TOfferBlender::IsOfferGoodForUserFilterByFreeCancellation(const TOffer& offer) const {
    if (Req.GetFilterFreeCancellation() && (offer.FreeCancellation != EFreeCancellationType::Yes)) {
        return false;
    }
    return true;
}

bool TOfferBlender::IsOfferGoodForUserFilterByOperator(const TOffer& offer) const {
    if ((Req.FilterOperatorSize() != 0) && !FindPtr(Req.GetFilterOperator(), offer.OperatorId)) {
        return false;
    }
    return true;
}

void TOfferBlender::ApplyAuxPostUserFilters() {
    // Фильтр TRAVELBACK-939
    // Должен применяться после пользовательских фильтров
    bool skipAllOffers = false;
    if (Req.GetFilterRequireBoYOffer()) {
        skipAllOffers = true;
        for (const TSelectedOffer& selectedOffer: SelectedOffers) {
            if (Service.IsBoYPartner(selectedOffer.Rec->Key.PreKey.HotelId.PartnerId)) {
                skipAllOffers = false;
                break;
            }
        }
    }
    if (Req.GetFilterRequireMirOffer()) {
        skipAllOffers = true;
        for (const TSelectedOffer& selectedOffer: SelectedOffers) {
            if (selectedOffer.PromoRsp.GetMir().GetEligibility() == NTravelProto::NPromoService::ME_ELIGIBLE) {
                skipAllOffers = false;
                break;
            }
        }
    }
    if (skipAllOffers) {
        SkipOffers([](const TSelectedOffer&){
            return NTravelProto::NOfferCache::NApi::SR_AuxFilter;
        });
    }
}

void TOfferBlender::ApplyFilterPartnerIdAfterBoYSelection() {
    // TRAVELBACK-1950
    if (Req.FilterPartnerIdAfterBoYSelectionSize() > 0) {
        auto enabledPartners = THashSet<EPartnerId>();
        for (auto pId: Req.GetFilterPartnerIdAfterBoYSelection()) {
            enabledPartners.insert(static_cast<EPartnerId>(pId));
        }
        auto enabledOperators = Service.GetOperatorsForPartners(enabledPartners);
        SkipOffers([enabledOperators](const TSelectedOffer& selectedOffer){
            if (enabledOperators.contains(selectedOffer.Offer->OperatorId)) {
                return NTravelProto::NOfferCache::NApi::SR_None;
            }
            return NTravelProto::NOfferCache::NApi::SR_AuxFilter;
        });
    }
}

void TOfferBlender::FillFullStats() {
    ReadRequestProcessor.Stats.Info.SetFullResultCount(ReadRequestProcessor.Stats.Info.GetFullResultCount() + SelectedOffers.size());
    if (!SelectedOffers.empty()) {
        ++ReadRequestProcessor.Stats.TotalPermalinksWithPrices;
        size_t order = ReadRequestProcessor.PermalinkOrder[Permalink];
        if (order < 5) {
            ++ReadRequestProcessor.Stats.Top5PermalinksWithPrices;
        }
        if (order < 10) {
            ++ReadRequestProcessor.Stats.Top10PermalinksWithPrices;
        }
    }
    if (IsMainPermalink) {
        if (!SelectedOffers.empty()) {
            ReadRequestProcessor.Stats.MainPermalinkHasPrices = true;
        }
        ReadRequestProcessor.Stats.Info.SetMainPermalink_FullOfferCount(SelectedOffers.size());
        THashMap<EOperatorId, size_t> offerCountByOp;
        if (SelectedOffers) {
            TPriceVal minPrice = SelectedOffers.begin()->FinalPrice;
            TPriceVal maxPrice = minPrice;
            for (const TSelectedOffer& selectedOffer: SelectedOffers) {
                ++offerCountByOp[selectedOffer.Offer->OperatorId];
                minPrice = Min(minPrice, selectedOffer.FinalPrice);
                maxPrice = Max(maxPrice, selectedOffer.FinalPrice);
            }
            RespHotel->MutablePriceRange()->SetMinPrice(minPrice);
            RespHotel->MutablePriceRange()->SetMaxPrice(maxPrice);
        }
        for (auto it = offerCountByOp.begin(); it != offerCountByOp.end(); ++it) {
            (*ReadRequestProcessor.Stats.Info.MutableByOperators())[it->first].SetMainPermalink_FullOfferCount(it->second);
        }
    }

    if (!IsBrief && !Req.GetCompactResponseForCalendar()) {
        for (const auto& selectedOffer: SelectedOffers) {
            ++FullOperatorOfferCount[selectedOffer.Offer->OperatorId];
        }
        RespHotel->SetFullPriceCount(SelectedOffers.size());
        RespHotel->SetFullOperatorCount(FullOperatorOfferCount.size());
    }
}

void TOfferBlender::ReduceOffersByOperator() {
    // Нам надо получить по одной цене от каждого оператора, но не больше чем MinPriceCount
    // Если MinPriceCount больше количества операторов, добавим ещё предложений
    THashSet<EOperatorId> availableOperators;
    for (const TSelectedOffer& selectedOffer: SelectedOffers) {
        availableOperators.insert(selectedOffer.Offer->OperatorId);
    }
    int extraOfferCount = static_cast<int>(MinPriceCount) - availableOperators.size();
    THashSet<EOperatorId> addedOperators;
    SkipOffers([&extraOfferCount, &addedOperators](const TSelectedOffer& selectedOffer){
        EOperatorId offerOperatorId = selectedOffer.Offer->OperatorId;
        if (addedOperators.insert(offerOperatorId).second) {
            return NTravelProto::NOfferCache::NApi::SR_None;
        } else {
            if (extraOfferCount <= 0) {
                return NTravelProto::NOfferCache::NApi::SR_ReduceByOperator;
            }
            extraOfferCount--;
        }
        return NTravelProto::NOfferCache::NApi::SR_None;
    });
}

void TOfferBlender::SortOffers() {
    TVector<TSelectedOffer> offers(SelectedOffers.begin(), SelectedOffers.end());
    Sort(offers, [this](const TSelectedOffer& p1, const TSelectedOffer& p2) {
        return SelectedOfferLess(p1, p2);
    });
    SelectedOffers.clear();
    for (const auto& so: offers) {
        SelectedOffers.push_back(so);
    }
}

TMaybe<bool> TOfferBlender::SelectedOfferLessByComplexPrice(const TSelectedOffer& selectedOffer1, const TSelectedOffer& selectedOffer2) const {
    if (BoostedOfferId) {
        bool b1 = selectedOffer1.Offer->OfferId == BoostedOfferId;
        bool b2 = selectedOffer2.Offer->OfferId == BoostedOfferId;
        if (b1 != b2) {
            return b1 > b2;
        }
    }
    // Цены внутри каждого отеля должны быть отсортированы:
    // 1. По возрастанию цены
    // 2. При одинаковой цене - по приоритету операторов - HOTELS-2650
    // Цены должны выводиться с учетом скидки по ЯПлюс, если пользователь залогинин HOTELS-5580

    auto getPlusPoints = [](const TSelectedOffer &selectedOffer) {
        if (selectedOffer.PromoRsp.HasPlus()) {
            if (selectedOffer.PromoRsp.GetPlus().HasPoints()) {
                return selectedOffer.PromoRsp.GetPlus().GetPoints().value();
            }
        }
        return static_cast<TPriceVal>(0);
    };

    TPriceVal finalPrice1 = selectedOffer1.FinalPrice, finalPrice2 = selectedOffer2.FinalPrice;
    if (Req.GetSortOffersUsingPlus()) { // ВАЖНО! Это можно делать только на портале, не на серпе.
        finalPrice1 -= getPlusPoints(selectedOffer1);
        finalPrice2 -= getPlusPoints(selectedOffer2);
    }
    if (finalPrice1 != finalPrice2) {
        return finalPrice1 < finalPrice2;
    }

    return {};
}


bool TOfferBlender::SelectedOfferLess(const TSelectedOffer& selectedOffer1, const TSelectedOffer& selectedOffer2) const {
    auto byPrice = SelectedOfferLessByComplexPrice(selectedOffer1, selectedOffer2);
    if (byPrice.Defined()) {
        return byPrice.GetRef();
    }
    return TOfferUtils::OfferLessByOperatorAndOfferId(Service, *selectedOffer1.Offer, *selectedOffer2.Offer);
}

void TOfferBlender::FillDefaultOffer() {
    if (SelectedOffers.empty()) {
        return;
    }
    const auto& offerLabels = Service.Config().GetOther().GetDefaultOfferLabel();
    auto it = offerLabels.find(Req.GetExp3845());
    if (it != offerLabels.end()) {
        auto pbPrice = RespHotel->MutableDefaultPrice();
        FillPriceBase(*SelectedOffers.begin(), pbPrice, false);
        FillPricePartnerLink(*SelectedOffers.begin(), pbPrice);
        pbPrice->SetLabel(it->second);// другой label, текстовый
    }
}

void TOfferBlender::BumpOffers() {
    if (Req.GetBumpMirOffers()) {
        for (auto offerIt = SelectedOffers.begin(); offerIt != SelectedOffers.end(); ++offerIt) {
            if (offerIt->PromoRsp.GetMir().GetEligibility() == NTravelProto::NPromoService::ME_ELIGIBLE) {
                TSelectedOffer selectedOffer = *offerIt;
                SelectedOffers.erase(offerIt);
                SelectedOffers.push_front(selectedOffer);
                return;
            }
        }
    }
    if (Req.GetBumpBoYOffers()) {
        auto ops = Service.GetBoYOperators();
        for (auto opId: ops) {
            if (BumpOffersForOperator(opId)) {
                return; // BoY-оператор должен быть только один, можно выходить
            }
        }
    }
    if (Req.GetEnableBumpOperators()) {
        // Для Bumped операторов выносим по одному предложению в начало списка
        const auto& bumpedOperators = Service.GetBumpedOperators();
        for (auto opIt = bumpedOperators.rbegin(); opIt != bumpedOperators.rend(); ++opIt) {
            BumpOffersForOperator(*opIt);
        }
    }
}

bool TOfferBlender::BumpOffersForOperator(EOperatorId opId) {
    for (auto offerIt = SelectedOffers.begin(); offerIt != SelectedOffers.end(); ++offerIt) {
        if (offerIt->Offer->OperatorId == opId) {
            TSelectedOffer selectedOffer = *offerIt;
            SelectedOffers.erase(offerIt);
            SelectedOffers.push_front(selectedOffer);
            return true;
        }
    }
    return false;
}

void TOfferBlender::CheckMinPriceBadge() {
    if (!IsSingleOrgPermalink || !Service.Config().GetOther().HasMinPriceBadge()) {
        return;
    }

    MinPrice = std::numeric_limits<TPriceVal>::max();
    size_t offersWithMinPrice = 0;
    THashSet<EOperatorId> operators;

    for (const auto& selectedOffer: SelectedOffers) {
        const TOffer& offer = *(selectedOffer.Offer);
        operators.insert(offer.OperatorId);
        if (selectedOffer.FinalPrice < MinPrice) {
            MinPrice = selectedOffer.FinalPrice;
            offersWithMinPrice = 1;
        } else if (selectedOffer.FinalPrice == MinPrice) {
            offersWithMinPrice++;
        }
    }

    NeedMinPriceBadge = offersWithMinPrice == 1 && operators.size() >= 2;
}

void TOfferBlender::MaybeAddBadgeToSelectedOffer(bool badgeNeeded, const NTravelProto::NOfferCache::NApi::TBadge& badge, TSelectedOffer* selectedOffer) const {
    if (badge.HasId() && badgeNeeded && (selectedOffer->Badges.size() == 0 || Req.GetShowAllBadges())) {
        selectedOffer->Badges.push_back(badge);
    }
}

void TOfferBlender::FillMirCashbackAvailability() const {
    bool mirPromoAvailable = ReadRequestProcessor.IsMirPromoAvailable(Permalink);
    RespHotel->SetMirPromoAvailable(mirPromoAvailable);

    bool hasBoY = false;
    for (const TSelectedOffer& selectedOffer: SelectedOffers) {
        if (Service.IsBoYPartner(selectedOffer.Rec->Key.PreKey.HotelId.PartnerId)) {
            hasBoY = true;
            break;
        }
    }

    RespHotel->SetShowMirPromoBadge(mirPromoAvailable && hasBoY);

    if (IsBrief || Req.GetCompactResponseForCalendar()) {
        return;
    }
    RespHotel->SetSuggestMirPromoDateChange(false);

    if (!mirPromoAvailable) {
        // Промо недоступно
        return;
    }

    auto currentWaveSettings = Service.GetPromoService().GetMirCurrentWaveSettings(ReadRequestProcessor.Started);

    if (!currentWaveSettings) {
        return;
    }

    // Если всё хорошо, менять даты не предлагаем
    if (ReadRequestProcessor.SubKey.Nights >= currentWaveSettings->MinNights &&
        static_cast<int>(ReadRequestProcessor.SubKey.Date + ReadRequestProcessor.SubKey.Nights) <= currentWaveSettings->LastCheckOut &&
        ReadRequestProcessor.SubKey.Date >= currentWaveSettings->FirstCheckIn) {
        return;
    }

    // Если нет BoY-предложений, то не рискуем предлагать менять даты TRAVELBACK-1308
    if (!hasBoY) {
        return;
    }

    // В остальных случаях предлагаем: TRAVELBACK-2490
    RespHotel->SetSuggestMirPromoDateChange(true);
}

void TOfferBlender::FillWelcomePromocode() const {
    if (!ReadRequestProcessor.Exp.IsExp(1861)) {
        return;
    }
    bool hasBoY = false;
    for (const TSelectedOffer& selectedOffer: SelectedOffers) {
        if (Service.IsBoYPartner(selectedOffer.Rec->Key.PreKey.HotelId.PartnerId)) {
            hasBoY = true;
            break;
        }
    }
    if (!hasBoY) {
        return;
    }
    RespHotel->SetWelcomePromocodeAvailable(true);
}

void TOfferBlender::CalculatePromos() {
    TPromoService::TForOfferInternalReq promoReq;
    promoReq.Now = ReadRequestProcessor.Started;
    promoReq.Date = ReadRequestProcessor.SubKey.Date;
    promoReq.Nights = ReadRequestProcessor.SubKey.Nights;
    promoReq.UserInfo.SetIsPlus(ReadRequestProcessor.Attribution.GetIsPlusUser());
    promoReq.UserInfo.SetIsLoggedIn(ReadRequestProcessor.UserInfo.PassportUid.Defined());
    if (ReadRequestProcessor.UserInfo.PassportUid.Defined()) {
        promoReq.UserInfo.SetPassportId(ToString(ReadRequestProcessor.UserInfo.PassportUid.GetRef()));
    }
    promoReq.ExpInfo.SetStrikeThroughPrices(ReadRequestProcessor.Exp.IsExp(2713));
    promoReq.ExpInfo.MutableKVExperiments()->CopyFrom(ReadRequestProcessor.Req.GetKVExperiments());
    promoReq.WhiteLabelInfo.SetPartnerId(ReadRequestProcessor.Req.GetWhiteLabelPartnerId());

    for (TSelectedOffer& selectedOffer: SelectedOffers) {
        promoReq.HotelId = selectedOffer.Rec->Key.PreKey.HotelId;
        promoReq.PriceFromPartnerOffer = TPriceWithCurrency(selectedOffer.Rec->Key.PreKey.Currency, selectedOffer.FinalPrice);
        NTravelProto::NPromoService::TCalculateDiscountForOfferRsp calculateDiscountRsp;
        Service.GetPromoService().CalculateDiscountForOffer(promoReq, &calculateDiscountRsp);
        if (calculateDiscountRsp.GetDiscountInfo().GetDiscountApplied()) {
            selectedOffer.StrikethroughPrice = selectedOffer.FinalPrice;
            selectedOffer.FinalPrice = calculateDiscountRsp.GetDiscountInfo().GetPriceAfterDiscount().GetAmount();
            promoReq.PriceBeforePromoCodes = TPriceWithCurrency::FromProto(calculateDiscountRsp.GetDiscountInfo().GetPriceAfterDiscount());
        }
        Service.GetPromoService().DeterminePromosForOffer(promoReq, &selectedOffer.PromoRsp);
    }
}

void TOfferBlender::CalculateBadges() {
    if (Req.GetCompactResponseForCalendar()) {
        return;
    }
    for (TSelectedOffer& selectedOffer: SelectedOffers) {
        MaybeAddWhiteLabelBadge(&selectedOffer);

        const TOffer& offer = *(selectedOffer.Offer);
        auto showMirCashbackBadge = Service.Config().GetOther().HasMirCashbackBadge() && selectedOffer.PromoRsp.GetMir().GetEligibility() == NTravelProto::NPromoService::ME_ELIGIBLE;
        auto showTaxiPromocodeBadge = Service.Config().GetOther().HasTaxiPromocodeBadge() && selectedOffer.PromoRsp.GetTaxi2020().GetEligible();
        auto showMinPriceBadge = NeedMinPriceBadge && selectedOffer.FinalPrice == MinPrice;
        auto showWelcomePromocodeBadge =
            (Service.IsBoYOperator(offer.OperatorId) && (ReadRequestProcessor.UserInfo.ConfirmedHotelOrderCount == 0)) &&  // Common conditions
                (
                    ReadRequestProcessor.Exp.IsExp(1861) ||  // For Serp
                        (Service.Config().GetOther().GetEnableWelcomePromocodeBadgeByUtmTerm() && Req.GetShowAllBadges() && (Req.GetAttribution().GetUtmTerm() == g_WelcomePromocodeUtmTerm)) // For portal
                );
        auto showYandexPlusBadge = Service.Config().GetOther().HasYandexPlusBadge() && selectedOffer.PromoRsp.HasPlus() && selectedOffer.PromoRsp.GetPlus().HasPoints() && Req.GetShowAllBadges();
        auto showYandexEdaBadge = Service.Config().GetOther().HasYandexEda2022Badge() && selectedOffer.PromoRsp.HasYandexEda2022Status() && selectedOffer.PromoRsp.GetYandexEda2022Status().GetShowBadge() && Req.GetShowAllBadges();

        MaybeAddBadgeToSelectedOffer(showYandexEdaBadge, Service.Config().GetOther().GetYandexEda2022Badge(), &selectedOffer);
        MaybeAddBadgeToSelectedOffer(selectedOffer.PromoRsp.GetBlackFriday2021Status().GetShowBadge(),Service.Config().GetOther().GetBlackFriday2021Badge(), &selectedOffer);
        if (showYandexPlusBadge) {
            MaybeAddBadgeToSelectedOffer(true, FormatYandexPlusBadge(selectedOffer.PromoRsp.GetPlus()), &selectedOffer);
        }

        if (showMirCashbackBadge) {
            MaybeAddBadgeToSelectedOffer(true, FormatMirBadge(selectedOffer.PromoRsp.GetMir()), &selectedOffer);
        }

        MaybeAddBadgeToSelectedOffer(showWelcomePromocodeBadge, Service.Config().GetOther().GetWelcomePromocodeBadge(), &selectedOffer);
        MaybeAddBadgeToSelectedOffer(showTaxiPromocodeBadge, Service.Config().GetOther().GetTaxiPromocodeBadge(), &selectedOffer);
        MaybeAddBadgeToSelectedOffer(showMinPriceBadge, Service.Config().GetOther().GetMinPriceBadge(), &selectedOffer);
    }
}

void TOfferBlender::MaybeAddWhiteLabelBadge(TSelectedOffer* selectedOffer) {
    auto showWhiteLabelBadges = Service.Config().GetOther().HasWhiteLabelBadges() && selectedOffer->PromoRsp.HasWhiteLabelStatus() && selectedOffer->PromoRsp.GetWhiteLabelStatus().GetEligibility() == NTravelProto::NPromoService::WLE_ELIGIBLE;
    if (!showWhiteLabelBadges) {
        return;
    }

    auto whiteLabelBadges = Service.Config().GetOther().GetWhiteLabelBadges();
    auto whiteLabelStatus = selectedOffer->PromoRsp.GetWhiteLabelStatus();

    switch (whiteLabelStatus.GetPartnerId()) {
        case NTravelProto::NWhiteLabel::WL_S7:
            if (whiteLabelBadges.HasS7Badge()) {
                MaybeAddBadgeToSelectedOffer(true, FormatDefaultWLBadge(whiteLabelBadges.GetS7Badge(), whiteLabelStatus), selectedOffer);
            }
            break;
        default:
            break;
    }
}

NTravelProto::NOfferCache::NApi::TBadge TOfferBlender::FormatDefaultWLBadge(NTravelProto::NOfferCache::NApi::TBadge badge, const NTravelProto::NPromoService::TWhiteLabelStatus& whiteLabelStatus) {
    TStringBuilder textBuilder;
    textBuilder << whiteLabelStatus.GetPoints().GetAmount() << " "
                << whiteLabelStatus.GetPointsLinguistics().GetNameForNumeralNominative();
    badge.SetText(textBuilder);
    return badge;
}

void TOfferBlender::FilterNotFirstOffers() {
    bool first = true;
    SkipOffers([&first](const TSelectedOffer&){
        if (first) {
            first = false;
            return NTravelProto::NOfferCache::NApi::SR_None;
        }
        return NTravelProto::NOfferCache::NApi::SR_CompactResponseForSearch;
    });
}

void TOfferBlender::PutHotelBadgesToResp() {
    if (Req.GetCompactResponseForCalendar()) {
        return;
    }

    THashSet<TString> seenBadges;
    if (!SelectedOffers.empty()) {
        for (const auto& badge: SelectedOffers.front().Badges) {
            if (seenBadges.insert(badge.GetId()).second) {
                RespHotel->AddBadges()->CopyFrom(badge);
            }
        }
    }
}

void TOfferBlender::PutAggregatedOfferInfoToResp() {
    RespHotel->SetHasOffersWithFreeCancellation(false);
    RespHotel->SetHasOffersWithPansion(false);
    RespHotel->SetHasOffersWithBreakfast(false);
    RespHotel->SetHasBoyOffers(false);
    for (const auto& selectedOffer: SelectedOffers) {
        if (GetCurrentRefundRule(selectedOffer) == NTravelProto::ERefundType::RT_FULLY_REFUNDABLE) {
            RespHotel->SetHasOffersWithFreeCancellation(true);
        }
        if (!HasNoPansion(selectedOffer.Offer->PansionType)) {
            RespHotel->SetHasOffersWithPansion(true);
        }
        if (HasBreakfast(selectedOffer.Offer->PansionType)) {
            RespHotel->SetHasOffersWithBreakfast(true);
        }
        if (Service.IsBoYOperator(selectedOffer.Offer->OperatorId)) {
            RespHotel->SetHasBoyOffers(true);
        }
    }
}

void TOfferBlender::PutOffersToResp() {
    EPermalinkType pt = ReadRequestProcessor.GetPermalinkType(Permalink);
    bool first = true;
    for (const auto& so: SelectedOffers) {
        PutOfferToResp(so);
        auto& stats = OfferShowStats.Stats[static_cast<size_t>(pt)][static_cast<size_t>(so.Offer->OperatorId)];
        stats.NAnyOffer++;
        if (first) {
            first = false;
            stats.NFirstOffer++;
            bool isBoY, isBoYDirect;
            ReadRequestProcessor.CheckPermalinkSubHotels(Permalink, &isBoY, &isBoYDirect);
            if (isBoY) {
                stats.NBoYHotelFirstOffer++;
            }
            if (isBoYDirect) {
                stats.NDirectBoYHotelFirstOffer++;
            }
        }
    }
    if (Req.GetShowSkipped()) {
        Sort(SkippedOffers, [this](const TSelectedOffer& p1, const TSelectedOffer& p2) {
            return SelectedOfferLess(p1, p2);
        });
        for (const auto& so: SkippedOffers) {
            PutOfferToResp(so);
        }
    }
}

void TOfferBlender::PutOfferToResp(const TSelectedOffer& selectedOffer) {
    const TOffer& offer = *(selectedOffer.Offer);
    ReadRequestProcessor.Stats.Info.SetPriceCount(ReadRequestProcessor.Stats.Info.GetPriceCount() + 1);
    auto pbPrice = RespHotel->AddPrices();
    FillPriceBase(selectedOffer, pbPrice, false);
    ReadRequestProcessor.ActualOperators.insert(offer.OperatorId);
    if (!IsSimilarPermalink && !Req.GetCompactResponseForSearch() && !Req.GetCompactResponseForCalendar()) {
        FillPricePartnerLink(selectedOffer, pbPrice);
    }
    if (!IsBrief && !Req.GetCompactResponseForCalendar()) {
        pbPrice->SetOperatorOfferCount(FullOperatorOfferCount[offer.OperatorId]);// кому-то это вообще надо?
        FillRefundInfo(selectedOffer, pbPrice);
        FillYandexPlusInfo(selectedOffer, pbPrice);
        for (const auto& badge: selectedOffer.Badges) {
            pbPrice->AddBadges()->CopyFrom(badge);
        }
    }
}

NTravelProto::NOfferCache::NApi::TBadge TOfferBlender::FormatMirBadge(const NTravelProto::NPromoService::TMirPromoStatus& mirStatus) {
    bool isPortalRequest = Req.GetShowAllBadges(); // dirty ad-hoc

    TStringBuilder textBuilder;
    if (mirStatus.GetCashbackAmount().value() > Service.Config().GetOther().GetMaxMirCashbackForPercentBadge()) {
        auto cashbackAmount = ToString(mirStatus.GetCashbackAmount().value());
        auto specialSpace = " ";
        auto firstBlockSize = cashbackAmount.length() % 3;
        textBuilder << "Возврат ";
        textBuilder << cashbackAmount.substr(0, firstBlockSize) << specialSpace;
        for (size_t i = firstBlockSize; i < cashbackAmount.length(); i += 3) {
            textBuilder << cashbackAmount.substr(i, 3) << specialSpace;
        }
        textBuilder << "₽";
        if (!isPortalRequest) {
            textBuilder << " по карте «Мир»"; // https://st.yandex-team.ru/TRAVELBACK-1308#5f844dcd6d01c90513a51557
        }
    } else {
        textBuilder << "Возврат " << mirStatus.GetCashbackPercent().value() << "%";
        if (!isPortalRequest) {
            textBuilder << " по карте «Мир»"; // https://st.yandex-team.ru/TRAVELBACK-1308#5f844dcd6d01c90513a51557
        }
    }
    auto badge = Service.Config().GetOther().GetMirCashbackBadge();
    badge.SetText(textBuilder);
    return badge;
}

NTravelProto::NOfferCache::NApi::TBadge TOfferBlender::FormatYandexPlusBadge(NTravelProto::NPromoService::TYandexPlusStatus yandexPlusStatus) {
    auto plusPoints = yandexPlusStatus.MutablePoints()->value();
    auto firstDecimal = plusPoints % 10;
    auto secondDecimal = plusPoints % 100 / 10;
    TStringBuilder textBuilder;

    textBuilder << plusPoints << " ";

    if (secondDecimal == 1) {
        textBuilder << "баллов";
    } else if (firstDecimal == 1) {
        textBuilder << "балл";
    } else if (2 <= firstDecimal && firstDecimal <= 4) {
        textBuilder << "балла";
    } else {
        textBuilder << "баллов";
    }

    NTravelProto::NOfferCache::NApi::TBadge badge;
    auto promoInfo = yandexPlusStatus.GetPromoInfo();
    if (promoInfo.GetEventType() == NTravelProto::NPromoService::YPET_SPECIAL && Service.Config().GetOther().HasYandexPlusSpecialBadge()) {
        TStringBuilder additionalTextBuilder;
        additionalTextBuilder << "Кешбэк баллами Плюса " << promoInfo.GetDiscountPercent()
                              << "%. Персональное предложение";
        if (promoInfo.HasSpecialOfferEndUtc()) {
            auto endTime = NTravel::NProtobuf::TimestampToInstant(promoInfo.GetSpecialOfferEndUtc());
            auto endDate = endTime.FormatGmTime("%d.%m.%Y");
            additionalTextBuilder << " до " << endDate << " включительно";
        }
        additionalTextBuilder << ".\n";
        badge = Service.Config().GetOther().GetYandexPlusSpecialBadge();
        auto additionalInfo = badge.MutableAdditionalInfo();
        additionalTextBuilder << additionalInfo->GetText();
        additionalInfo->SetText(additionalTextBuilder);
    } else if (promoInfo.GetEventType() == NTravelProto::NPromoService::YPET_CULTURAL_DREAMS && Service.Config().GetOther().HasYandexPlusCulturalDreamsBadge()) {
        badge = Service.Config().GetOther().GetYandexPlusCulturalDreamsBadge();
    } else {
        badge = Service.Config().GetOther().GetYandexPlusBadge();
    }
    badge.SetText(textBuilder);

    return badge;
}

void TOfferBlender::FillPriceBase(const TSelectedOffer& selectedOffer, NTravelProto::NOfferCache::NApi::TPrice* pbPrice, bool forStat) const {
    const TOffer& offer = *(selectedOffer.Offer);
    pbPrice->SetOperatorId(offer.OperatorId);
    pbPrice->SetPrice(selectedOffer.FinalPrice);
    pbPrice->SetOfferId(ToString(offer.OfferId));
    auto apiPansion = PansionToOCFormat(offer.PansionType);
    pbPrice->SetPansion(apiPansion);
    pbPrice->SetBreakfastIncluded(HasBreakfast(offer.PansionType));
    if (offer.FreeCancellation != EFreeCancellationType::Unknown) {
        pbPrice->SetFreeCancellation(offer.FreeCancellation == EFreeCancellationType::Yes);
    }
    if (selectedOffer.StrikethroughPrice) {
        auto* stPrice = pbPrice->MutableStrikethroughPrice();
        stPrice->SetPrice(selectedOffer.StrikethroughPrice);
        stPrice->SetReason(selectedOffer.StrikethroughPriceReason);
    }
    if (!forStat) {
        ReadRequestProcessor.ActualPansions.insert(apiPansion);
    }
    if (IsBrief || Req.GetCompactResponseForCalendar()) {
        return;
    }
    if (selectedOffer.SkipReason != NTravelProto::NOfferCache::NApi::SR_None) {
        pbPrice->SetSkipReason(selectedOffer.SkipReason);
    }
    pbPrice->SetPartnerId(selectedOffer.Rec->Key.PreKey.HotelId.PartnerId);
    if (!Req.GetCompactResponseForSearch() && !Req.GetCompactResponseForCalendar()) {
        pbPrice->SetOperatorName(Service.GetOperator(offer.OperatorId)->GetName());
        pbPrice->SetOrigHotelId(selectedOffer.Rec->Key.PreKey.HotelId.OriginalId);
        auto decompressedTitle = selectedOffer.GetDecompressedTitle();
        pbPrice->SetRoomType(OfferTextPrefix ? OfferTextPrefix + decompressedTitle : decompressedTitle);
    }
    if (ShowPermarooms) {
        pbPrice->SetPermaroomId(ToString(selectedOffer.PermaroomId));
    }
    pbPrice->SetCatRoomMappingKey(selectedOffer.CatRoomMappingKey);
    pbPrice->SetRawPermaroomId(selectedOffer.RawPermaroomId);
    pbPrice->SetPermaroomVersion(selectedOffer.PermaroomVersion);
    if (offer.OfferRestrictions.RequiresMobile) {
        pbPrice->SetRequiresMobile(true);
    }
    if (offer.OfferRestrictions.RequiresRestrictedUser) {
        pbPrice->SetRequiresRestrictedUser(true);
    }
}

void TOfferBlender::FillPricePartnerLink(const TSelectedOffer& selectedOffer, NTravelProto::NOfferCache::NApi::TPrice* pbPrice) {
    TProfileTimer fixPriceTimer;
    Label.SetPermalink(Permalink);
    Label.SetPartnerId(selectedOffer.Rec->Key.PreKey.HotelId.PartnerId);
    Label.SetOperatorId(selectedOffer.Offer->OperatorId);
    Label.SetPrice(selectedOffer.FinalPrice);
    Label.SetCacheTimestamp(selectedOffer.Rec->Timestamp.Seconds());
    Label.SetOfferId(ToString(selectedOffer.Offer->OfferId));
    Label.SetSearcherReqId(selectedOffer.Rec->SearcherReqId);
    Label.SetOfferOriginOfferCacheClientId(selectedOffer.Rec->OfferCacheClientId);
    Label.SetOriginalHotelId(selectedOffer.Rec->Key.PreKey.HotelId.OriginalId);

    TString labelStr = Service.LabelCodec().Encode(Label, NLabel::TLabelCodec::TOpts().WithCheckSum(true).WithPrefix(false));
    Service.GetCounters().LabelSize.Update(labelStr.size());

    TUrl redirUrl(RedirUrlBase);
    redirUrl.SetCgiParam("OfferId", ToString(selectedOffer.Offer->OfferId));
    redirUrl.SetCgiParam("OfferIdHash", ToString(selectedOffer.Offer->OfferIdHash));
    redirUrl.SetCgiParam("ProtoLabel", labelStr);
    pbPrice->SetPartnerLink(redirUrl.ToString());

    ReadRequestProcessor.Stats.FixPriceDuration += fixPriceTimer.Get();
}

NTravelProto::ERefundType TOfferBlender::GetCurrentRefundRule(const TSelectedOffer& selectedOffer) const {
    if (selectedOffer.Offer->RefundRules.GetValue().empty()) {
        switch (selectedOffer.Offer->FreeCancellation) {
            case EFreeCancellationType::Unknown:
                return NTravelProto::ERefundType::RT_UNKNOWN;
            case EFreeCancellationType::No:
                return NTravelProto::ERefundType::RT_NON_REFUNDABLE;
            case EFreeCancellationType::Yes:
                return NTravelProto::ERefundType::RT_FULLY_REFUNDABLE;
        }
    } else {
        for (const auto& refundRule : selectedOffer.Offer->RefundRules.GetValue()) {
            if (refundRule.StartsAtTimestampSec <= ReadRequestProcessor.Started.Seconds() && ReadRequestProcessor.Started.Seconds() < refundRule.EndsAtTimestampSec) {
                return refundRule.Type;
            }
        }
        return NTravelProto::ERefundType::RT_NON_REFUNDABLE;
    }
}

void TOfferBlender::FillRefundInfo(const TSelectedOffer& selectedOffer, NTravelProto::NOfferCache::NApi::TPrice* pbPrice) {
    pbPrice->SetRefundType(GetCurrentRefundRule(selectedOffer));

    for (const auto& refundRule : selectedOffer.Offer->RefundRules.GetValue()) {
        if (refundRule.EndsAtTimestampSec > ReadRequestProcessor.Started.Seconds()) {
            auto pbRefundRule = pbPrice->AddRefundRule();
            pbRefundRule->SetType(refundRule.Type);
            if (refundRule.Type == NTravelProto::ERefundType::RT_REFUNDABLE_WITH_PENALTY) {
                pbRefundRule->SetPenalty(refundRule.Penalty);
            }
            if (refundRule.StartsAtTimestampSec != std::numeric_limits<decltype(refundRule.StartsAtTimestampSec)>::min()) {
                pbRefundRule->SetStartsAtTimestampSec(refundRule.StartsAtTimestampSec);
            }
            if (refundRule.EndsAtTimestampSec != std::numeric_limits<decltype(refundRule.EndsAtTimestampSec)>::max()) {
                pbRefundRule->SetEndsAtTimestampSec(refundRule.EndsAtTimestampSec);
            }
        }
    }
}

void TOfferBlender::FillYandexPlusInfo(const TSelectedOffer& selectedOffer, NTravelProto::NOfferCache::NApi::TPrice* pbPrice) {
    if (!selectedOffer.PromoRsp.HasPlus() || !selectedOffer.PromoRsp.GetPlus().HasPoints()) {
        return;
    }
    pbPrice->MutableYandexPlusInfo()->SetEligible(selectedOffer.PromoRsp.GetPlus().GetEligibility() == NTravelProto::NPromoService::EYandexPlusEligibility::YPE_ELIGIBLE);
    pbPrice->MutableYandexPlusInfo()->SetPoints(selectedOffer.PromoRsp.GetPlus().GetPoints().value());
}

void TOfferBlender::PutPartnerRefsToResp() {
    if (IsBrief || Req.GetCompactResponseForCalendar()) {
        return;
    }
    THashSet<EPartnerId> partnerIds;
    for (const auto& hotelId : ReadRequestProcessor.ComplexHotelIds[Permalink].HotelIds) {
        partnerIds.insert(hotelId.PartnerId);
    }
    for (EPartnerId pId: partnerIds) {
        RespHotel->AddAvailablePartnerIds(pId);
    }
}

void TOfferBlender::FillHasBoyPartner() const {
    bool hasBoYPartner = false;
    for (const auto& hotelId : ReadRequestProcessor.ComplexHotelIds[Permalink].HotelIds) {
        if (Service.IsBoYPartner(hotelId.PartnerId)) {
            hasBoYPartner = true;
            break;
        }
    }
    RespHotel->SetHasBoyPartner(hasBoYPartner);
    if (IsMainPermalink) {
        bool hasBoYOffers = false;
        for (const TSelectedOffer& selectedOffer: SelectedOffers) {
            if (Service.IsBoYPartner(selectedOffer.Rec->Key.PreKey.HotelId.PartnerId)) {
                hasBoYOffers = true;
                break;
            }
        }
        ReadRequestProcessor.Stats.Info.SetMainPermalink_HasBoYOffers(hasBoYOffers);
        ReadRequestProcessor.Stats.Info.SetMainPermalink_HasBoYPartner(hasBoYPartner);
    }
}

void TOfferBlender::FillIsPlusAvailable() const {
    bool isPlusAvailable = Service.GetPromoService().IsPlusAvailableForHotel(
        ReadRequestProcessor.Started,
        ReadRequestProcessor.ComplexHotelIds[Permalink]
    );
    RespHotel->SetIsPlusAvailable(isPlusAvailable);
}

void TOfferBlender::FillPerHotelPlusInfo() const {
    TMaybe<ui32> minPlusPoints;
    TMaybe<ui32> maxPlusCashbackPercent;

    for (const TSelectedOffer& selectedOffer: SelectedOffers) {
        if (selectedOffer.PromoRsp.HasPlus() && selectedOffer.PromoRsp.GetPlus().HasPoints()) {
            auto plusPoints = selectedOffer.PromoRsp.GetPlus().GetPoints();
            if (!minPlusPoints.Defined() || plusPoints.value() < minPlusPoints.GetRef()) {
                minPlusPoints = plusPoints.value();
            }

            auto currPercent = selectedOffer.PromoRsp.GetPlus().GetPromoInfo().GetDiscountPercent();
            if (maxPlusCashbackPercent.Empty() || maxPlusCashbackPercent.GetRef() < currPercent) {
                maxPlusCashbackPercent = currPercent;
            }
        }
    }

    if (ReadRequestProcessor.Exp.IsExp(5581) && minPlusPoints.Defined()) {
        RespHotel->SetMinPlusPoints(minPlusPoints.GetRef());
    }

    if (maxPlusCashbackPercent.Defined()) {
        RespHotel->SetMaxAvailablePlusCashbackPercent(maxPlusCashbackPercent.GetRef());
    }
}

void TOfferBlender::LogSkippedOffers() {
    for (const TSelectedOffer& skippedOffer: SkippedOffers) {
        auto& statsHotel = (*ReadRequestProcessor.Stats.Info.MutableHotels())[Permalink];
        FillPriceBase(skippedOffer, statsHotel.AddSkippedPrices(), true);
    }
}

void TOfferBlender::LogTime() {
    auto stats = (*ReadRequestProcessor.Stats.Info.MutableHotels())[Permalink].MutableBlendingStagesStats();
    TDuration total;
    for (const auto& [name, dur]: BlendingStageTimes.Times) {
        (*stats)[name + "Micros"] = dur.MicroSeconds();
        total += dur;
    }
    (*stats)["TotalMicros"] = total.MicroSeconds();
}

}// namespace NOfferCache
}// namespace NTravel
