#pragma once

#include "data.h"
#include "read_request_processor.h"
#include "promo_service.h"
#include "rooms/room_data.h"
#include "rooms/offer_to_room_key_mapper.h"

#include <travel/hotels/offercache/proto/reqans_logrecord.pb.h>
#include <travel/hotels/proto2/label.pb.h>

#include <travel/hotels/lib/cpp/http/http_service.h>
#include <travel/hotels/lib/cpp/util/url.h>

#include <util/generic/list.h>
#include <util/generic/vector.h>
#include <util/generic/ptr.h>
#include <util/generic/set.h>

namespace NTravel {
namespace NOfferCache {

class TService;
class THttpJob;

class TOfferBlender {
public:
    TOfferBlender(TReadRequestProcessor& readRequestProcessor,
                  TPermalink permalink,
                  const THashMap<THotelId, TVector<TCacheRecordRef>>& records,
                  const TUrl& redirUrl,
                  NTravelProto::TLabel& label,
                  NTravelProto::NOfferCache::NApi::TReadResp::THotel* hotel
    );

    void Blend();

    const TBlendingStageTimes& GetBlendingStageTimes() const;
    const TOfferShowStats& GetOfferShowStats() const;

private:
    // Оффер отобранный для показа
    class TSelectedOffer {
    public:
        const TOffer* Offer;
        TCacheRecordRef Rec;
        EOfferSkipReason SkipReason = NTravelProto::NOfferCache::NApi::SR_None;
        TPermaroomIdStr PermaroomId;
        TPermaroomVersion PermaroomVersion;
        TPermaroomIdStr RawPermaroomId;
        TString CatRoomMappingKey;
        TPriceVal FinalPrice = 0;
        TPriceVal StrikethroughPrice = 0;
        NTravelProto::NOfferCache::NApi::EStrikethroughPriceReason StrikethroughPriceReason = NTravelProto::NOfferCache::NApi::SPR_Unknown;
        TVector<NTravelProto::NOfferCache::NApi::TBadge> Badges;
        NTravelProto::NPromoService::TDeterminePromosForOfferRsp PromoRsp;

        bool SkippedDuringDeduplication = false;

        TString GetDecompressedTitle() const;

    private:
        mutable TMaybe<TString> DecompressedTitle;
    };

    struct TCatRoomStat {
        size_t OfferCountAll = 0;
        size_t OfferCountOther = 0;
        THashSet<TPermaroomIdStr> PermaroomIds;
    };

    using ECatRoomStatus = NTravelProto::NOfferCache::NApi::ECatRoomStatus;

    struct TStPricesSettings {
        bool DetermineForRestrictedOffers = false;
        bool DetermineForBlackFriday2021 = false;
    };

    TReadRequestProcessor& ReadRequestProcessor;
    TService& Service;
    const TString& OfferTextPrefix;
    const NTravelProto::NOfferCache::NApi::TReadReq& Req;
    const TPermalink Permalink;
    const bool IsMainPermalink;
    const bool IsSingleOrgPermalink;
    const bool IsSimilarPermalink;
    const THashMap<THotelId, TVector<TCacheRecordRef>>& Records;
    const TUrl RedirUrlBase;
    const TOfferId BoostedOfferId;

    NTravelProto::TLabel& Label;
    bool AllowPermarooms;
    bool IsBrief;
    size_t MinPriceCount;
    TList<TSelectedOffer> SelectedOffers;
    TVector<TSelectedOffer> SkippedOffers;
    NTravelProto::NOfferCache::NApi::TReadResp::THotel* RespHotel;
    bool ShowPermarooms;
    ECatRoomStatus CatRoomStatus;
    TCatRoomDataSourceIdStr SelectedCatRoomDataSourceId;

    THashMap<EOperatorId, size_t/*offerCount*/> FullOperatorOfferCount;

    // For MinPrice badge
    bool NeedMinPriceBadge;
    TPriceVal MinPrice;

    TOfferToRoomKeyMapper OfferToRoomKeyMapper;

    THashSet<EOperatorId> EnabledAfterFilteringOperators;

    TBlendingStageTimes BlendingStageTimes;
    TOfferShowStats OfferShowStats;

    void RearrangeOffers();
    void ReduceOffers();
    void FillRespAfterBlending();

    void GetAllOffers();
    void RemoveDisabledOperators();
    void RemoveExperiments();
    void RemoveMetaIfBoyAvailable();
    void RemoveBannedOfferTypes();
    void RemoveRestrictedOffers();
    void FilterOffersByMinPrice();

    template <class TDedupKey>
    void RemoveDuplicatedOffers(std::function<TDedupKey(const TSelectedOffer& selectedOffer)> keyGetter, const TStPricesSettings& stPricesSettings);
    void RemoveDuplicatedOffersBeforeScanPermarooms();
    void RemoveDuplicatedOffersAfterScanPermarooms();

    void RemoveStrikethroughPrices(std::function<bool (TSelectedOffer& selectedOffer)> shouldRemoveDiscountInfo);
    void RemoveMetaStrikethroughPricesForBoyHotels();

    ECatRoomStatus ScanPermarooms();
    TMaybe<ECatRoomStatus> ValidateCatRoomOtherPortionAndReportStats(
        const TCatRoomDataSourceIdStr& dsId,
        const TCatRoomStat& totalStat,
        const THashMap<EOperatorId, TCatRoomStat>& op2stat);

    void ProcessCatRoomStatus();
    void AddPermaroomToHotel(const TPermaroom& permaroom, bool showBlackFriday2021Badge);

    void SelectBestBoY();
    void SkipOffers(std::function<EOfferSkipReason (const TSelectedOffer& selectedOffer)> skipper);
    void BanOperators(const std::function<bool (EOperatorId operatorId)>& isBanned);

    // AuxFilters
    void UpdateTotalPriceRange();
    void ApplyAuxFilters();
    bool IsOfferGoodForAuxFilter(const TSelectedOffer& selectedOffer) const;

    // UserFilters
    void FillUserFiltersInfo();
    void ApplyUserFilters();

    void ApplyFilterPartnerIdAfterBoYSelection();

    bool IsOfferGoodForUserFilterByPansion(const TOffer& offer) const;
    bool IsOfferGoodForUserFilterByFreeCancellation(const TOffer& offer) const;
    bool IsOfferGoodForUserFilterByOperator(const TOffer& offer) const;

    // AuxPostUserFilters
    void ApplyAuxPostUserFilters();

    void FillFullStats();

    void ReduceOffersByOperator();
    void SortOffers();
    bool SelectedOfferLess(const TSelectedOffer& selectedOffer1, const TSelectedOffer& selectedOffer2) const;
    TMaybe<bool> SelectedOfferLessByComplexPrice(const TSelectedOffer& selectedOffer1, const TSelectedOffer& selectedOffer2) const;

    void FillDefaultOffer();
    void BumpOffers();
    bool BumpOffersForOperator(EOperatorId opId);

    void CheckMinPriceBadge();
    void MaybeAddBadgeToSelectedOffer(bool badgeNeeded, const NTravelProto::NOfferCache::NApi::TBadge& badge, TSelectedOffer* selectedOffer) const;

    void FillMirCashbackAvailability() const;
    void FillWelcomePromocode() const;

    void CalculatePromos();

    void CalculateBadges();
    void MaybeAddWhiteLabelBadge(TSelectedOffer* selectedOffer);
    NTravelProto::NOfferCache::NApi::TBadge FormatDefaultWLBadge(NTravelProto::NOfferCache::NApi::TBadge badge, const NTravelProto::NPromoService::TWhiteLabelStatus& whiteLabelStatus);

    void FilterNotFirstOffers();

    void PutHotelBadgesToResp();
    void PutAggregatedOfferInfoToResp();
    void PutOffersToResp();
    void PutOfferToResp(const TSelectedOffer& selectedOffer);
    NTravelProto::NOfferCache::NApi::TBadge FormatMirBadge(const NTravelProto::NPromoService::TMirPromoStatus& mirStatus);
    NTravelProto::NOfferCache::NApi::TBadge FormatYandexPlusBadge(const NTravelProto::NPromoService::TYandexPlusStatus yandexPlusStatus);
    void FillPriceBase(const TSelectedOffer& selectedOffer, NTravelProto::NOfferCache::NApi::TPrice* pbPrice, bool forStat) const;
    void FillPricePartnerLink(const TSelectedOffer& selectedOffer, NTravelProto::NOfferCache::NApi::TPrice* pbPrice);
    NTravelProto::ERefundType GetCurrentRefundRule(const TSelectedOffer& selectedOffer) const;
    void FillRefundInfo(const TSelectedOffer& selectedOffer, NTravelProto::NOfferCache::NApi::TPrice* pbPrice);
    void FillYandexHotelInfo();
    void FillYandexPlusInfo(const TSelectedOffer& selectedOffer, NTravelProto::NOfferCache::NApi::TPrice* pbPrice);

    void PutPartnerRefsToResp();

    void LogSkippedOffers();
    void LogTime();

    void FillHasBoyPartner() const;
    void FillIsPlusAvailable() const;
    void FillPerHotelPlusInfo() const;
};

}// namespace NOfferCache
}// namespace NTravel
