#include "outdated_offers_transmitter.h"
#include "data.h"
#include "service.h"

#include <library/cpp/digest/md5/md5.h>
#include <util/random/random.h>

namespace NTravel::NOfferCache {
    TOutdatedOffersTransmitter::TOutdatedOffersTransmitter(TYtQueueWriter& outdatedOfferWriter, TOutdatedRecordBuilder& outdatedRecordBuilder, const NTravelProto::NOfferCache::TConfig::TOutdatedOffersTransmitter& config)
        : Config_(config)
        , DeduplicationService_(TDuration::Seconds(config.GetDeduplicationExpirationPeriodSec()), true)
        , OutdatedOfferWriter_(outdatedOfferWriter)
        , OutdatedRecordBuilder_(outdatedRecordBuilder)
        , YpAutoResolver_(config.GetYpAutoResolver())
        , WriteDelaySec_(Config_.GetWriteDelayMinSec() + RandomNumber<ui32>(Config_.GetWriteDelayIntervalSec() * 10)) // default fallback while yp is not initialized
        , WriterThread_(nullptr) {

        YpAutoResolver_.SetCallback([this] (const TVector<TYpConnectionInfo>& endpoints) {
            TVector<TString> hosts(endpoints.size());
            std::transform(endpoints.begin(), endpoints.end(), hosts.begin(), [] (const auto& ep) { return ep.GetFqdn(); });
            std::sort(hosts.begin(), hosts.end());

            int writeDelaySec = Config_.GetWriteDelayMinSec();
            bool found = false;
            for (const auto& host : hosts) {
                if (host == FQDNHostName()) {
                    found = true;
                    break;
                }
                writeDelaySec += Config_.GetWriteDelayIntervalSec();
            }
            if (!found) {
                WARNING_LOG << "Not found current host (" << FQDNHostName() << ") in yp response (" << JoinStrings(hosts, ", ") << "). Using random fallback" << Endl;
                writeDelaySec = Config_.GetWriteDelayMinSec() + RandomNumber<ui32>(Config_.GetWriteDelayIntervalSec() * Max(static_cast<ui32>(10), static_cast<ui32>(hosts.size())));
            }
            WriteDelaySec_ = writeDelaySec;
            Counters_.DesiredDelaySec = writeDelaySec;
            INFO_LOG << "Write delay is " << writeDelaySec << " seconds" << Endl;
            IsReady_.Set();
        });
    }

    void TOutdatedOffersTransmitter::Start() {
        if (!Config_.GetEnabled()) {
            return;
        }
        if (IsStarted_.TrySet()) {
            YpAutoResolver_.Start();
            WriterThread_ = SystemThreadFactory()->Run([this]() {
                ConsumeRecords();
            });
        }
    }

    void TOutdatedOffersTransmitter::Stop() {
        if (!Config_.GetEnabled()) {
            return;
        }
        if (!IsStopping_.TrySet()) {
            return;
        }
        YpAutoResolver_.Stop();
        StopEvent_.Signal();
        if (WriterThread_ != nullptr) {
            WriterThread_->Join();
        }
    }

    bool TOutdatedOffersTransmitter::IsReady() const {
        if (!Config_.GetEnabled()) {
            return true;
        }
        return IsReady_;
    }

    void TOutdatedOffersTransmitter::RegisterCounters(NMonitor::TCounterSource& source) const {
        DeduplicationService_.RegisterCounters(source, "OutdatedOffersTransmitterDeduplication");
        source.RegisterSource(&Counters_, "OutdatedOffersTransmitter");
    }

    void TOutdatedOffersTransmitter::ProcessOutdatedOfferBusMessage(TInstant timestamp, const TString& messageId, const ru::yandex::travel::hotels::TSearcherMessage&) {
        if (!Config_.GetEnabled()) {
            return;
        }
        DeduplicationService_.CheckIfKeyNewAndRememberIt(messageId, timestamp);
    }

    void TOutdatedOffersTransmitter::ProcessOfferBusMessage(TInstant timestamp,
                                                            const TString& messageId,
                                                            const ru::yandex::travel::hotels::TSearcherMessage& message,
                                                            const TVector<TCacheRecordRef>& records) {
        if (!Config_.GetEnabled()) {
            return;
        }
        if (QueueSizeBytes_.load() > Config_.GetMaxQueueSizeBytes()) {
            Counters_.NNewMessagesDropped.Inc();
            return;
        }

        THashSet<TOfferId> outdatedOfferIds;
        for (const auto& rec: records) {
            TMaybe<TCacheRecordRef> outdatedRec = OutdatedRecordBuilder_.ConvertToOutdated(rec);
            if (outdatedRec.Defined()) {
                for (const auto& offer: outdatedRec.GetRef()->Offers) {
                    outdatedOfferIds.insert(offer.OfferId);
                }
            }
        }

        if (!message.GetResponse().HasOffers()) {
            return;
        }

        ru::yandex::travel::hotels::TSearcherMessage newMessage(message);
        auto ocClientId = newMessage.MutableRequest()->MutableAttribution()->GetOfferCacheClientId();
        newMessage.MutableRequest()->MutableAttribution()->Clear();
        newMessage.MutableRequest()->MutableAttribution()->SetOfferCacheClientId(ocClientId);
        newMessage.MutableRequest()->ClearId();

        newMessage.MutableResponse()->MutableOffers()->ClearOffer();
        for (const auto& offer: message.GetResponse().GetOffers().GetOffer()) {
            if (outdatedOfferIds.contains(TOfferId::FromString(offer.GetId()))) {
                auto newOffer = newMessage.MutableResponse()->MutableOffers()->AddOffer();
                newOffer->CopyFrom(offer);

                // These fields are not used in offercache
                newOffer->ClearAvailabilityGroupKey();
                newOffer->ClearExternalId();
                newOffer->ClearActualizationTime();
                newOffer->ClearWifiIncluded();

                // These fields are not used for outdated prices
                newOffer->ClearPartnerSpecificData();
                newOffer->ClearOriginalRoomId();
                newOffer->ClearRestrictions();
                newOffer->ClearDisplayedTitle();
            }
        }
        newMessage.MutableResponse()->MutableOffers()->ClearWarnings();

        auto newMessageId = TransformMessageId(messageId);
        if (DeduplicationService_.IsNewKey(newMessageId, timestamp)) {
            auto queueRecord = TQueueRecord{timestamp, timestamp, messageId, newMessage};
            with_lock (QueueMutex_) {
                Queue_.push(queueRecord);
                Counters_.QueueHeadDelaySec = (Queue_.front().WriteTime - Now()).Seconds();
            }
            OnQueueUpdate(queueRecord, 1);
        }
    }

    TString TOutdatedOffersTransmitter::TransformMessageId(const TString& messageId) const {
        auto md5 = MD5::Calc(messageId);
        auto res = TStringBuilder();
        res.reserve(36);
        res.append(md5, 0, 8).append('-');
        res.append(md5, 8, 4).append('-');
        res.append(md5, 12, 4).append('-');
        res.append(md5, 16, 4).append('-');
        res.append(md5,  20);
        return TString(res);
    }

    void TOutdatedOffersTransmitter::ConsumeRecords() {
        TVector<TQueueRecord> recordsToWrite{};
        while (!IsStopping_) {
            auto now = Now();
            auto timeBorder = now - TDuration::Seconds(WriteDelaySec_.load());
            bool hasMore = false;
            with_lock (QueueMutex_) {
                while (!Queue_.empty() && Queue_.front().WriteTime < timeBorder) {
                    if (recordsToWrite.size() >= Config_.GetWriterBatchSize()) {
                        hasMore = true;
                        break;
                    }
                    recordsToWrite.push_back(Queue_.front());
                    Queue_.pop();
                }
                if (Queue_.empty()) {
                    Counters_.QueueHeadDelaySec = 0;
                } else {
                    Counters_.QueueHeadDelaySec = (Queue_.front().WriteTime - now).Seconds();
                }
            }
            for (const auto& rec: recordsToWrite) {
                if (DeduplicationService_.IsNewKey(rec.MessageId, rec.Timestamp)) {
                    // todo (mpivko): lifetime should not exceed (checkin_time - rec.Timestamp)
                    OutdatedOfferWriter_.Write(rec.Message, rec.Timestamp, rec.MessageId, OutdatedRecordBuilder_.GetOutdatedPriceLifetime());
                }
                Counters_.RealDelaySec = (rec.WriteTime - now).Seconds();
                OnQueueUpdate(rec, -1);
            }
            recordsToWrite.clear();
            if (!hasMore) {
                StopEvent_.WaitT(TDuration::Seconds(Config_.GetWriterSleepTimeSec()));
            }
        }
    }

    void TOutdatedOffersTransmitter::OnQueueUpdate(const TQueueRecord& rec, int sign) {
        auto sz = TTotalByteSize<TQueueRecord>()(rec);
        QueueSizeBytes_ += sz * sign;
        Counters_.NQueueBytes += sz * sign;
        Counters_.NQueueRecords += sign;
    }

    void TOutdatedOffersTransmitter::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        ct->insert(MAKE_COUNTER_PAIR(DesiredDelaySec));
        ct->insert(MAKE_COUNTER_PAIR(RealDelaySec));
        ct->insert(MAKE_COUNTER_PAIR(QueueHeadDelaySec));
        ct->insert(MAKE_COUNTER_PAIR(NQueueRecords));
        ct->insert(MAKE_COUNTER_PAIR(NQueueBytes));
        ct->insert(MAKE_COUNTER_PAIR(NNewMessagesDropped));
    }

    size_t TOutdatedOffersTransmitter::TQueueRecord::CalcTotalByteSize() const {
        return sizeof(TQueueRecord) + TTotalByteSize<TString>()(MessageId) - sizeof(TString) + Message.ByteSizeLong();
    }
}
