#include "outdated_record_builder.h"
#include "data.h"
#include "service.h"
#include "offer_utils.h"


namespace NTravel::NOfferCache {
    TOutdatedRecordBuilder::TOutdatedRecordBuilder(TService& service, const NTravelProto::NOfferCache::TConfig::TOutdatedPrices& config)
        : Service_(service)
        , OutdatedPriceLifetime_(TDuration::Seconds(config.GetOutdatedPricesLifetimeSeconds()))
        , EnableOutdatedPrices_(config.GetEnableOutdatedPrices()) {
    }

    bool TOutdatedRecordBuilder::AreOutdatedPricesEnabled() const {
        return EnableOutdatedPrices_;
    }

    TDuration TOutdatedRecordBuilder::GetOutdatedPriceLifetime() const {
        return OutdatedPriceLifetime_;
    }

    TMaybe<TCacheRecordRef> TOutdatedRecordBuilder::ConvertToOutdated(const TCacheRecordRef& record) {
        if (record->ErrorCode != EErrorCode::EC_OK) {
            return {};
        }
        TCacheRecordRef newRec = new TCacheRecord();
        newRec->Key = record->Key;
        newRec->Timestamp = record->Timestamp;
        newRec->ExpireTimestamp = record->Timestamp + GetOutdatedPriceLifetime();
        newRec->SearcherReqId = record->SearcherReqId;
        newRec->SourceCounters = record->SourceCounters;
        newRec->OfferCacheClientId = record->OfferCacheClientId;
        newRec->UsageCount = 0;
        newRec->Offers = BuildOffers(record->Offers);
        newRec->ErrorCode = EErrorCode::EC_OK;
        newRec->SearchWarnings = nullptr;
        newRec->IsInvalidated = false;
        newRec->IsOutdated = true;
        return newRec;
    }

    TVector<TOffer> TOutdatedRecordBuilder::BuildOffers(const TVector<TOffer>& offers) {
        // Results of this method are persisted in yt for several days, so behavior here should be changed very carefully
        // Also, this method should be idempotent, because it can be applied several times: before writing to yt (for space-efficiency) and after reading from yt (for consistency)
        TVector<TOffer> sortedOffers(offers.begin(), offers.end());
        Sort(sortedOffers, [this](const TOffer& lhs, const TOffer& rhs) {
            if (lhs.PriceVal != rhs.PriceVal) {
                return lhs.PriceVal < rhs.PriceVal;
            }
            return TOfferUtils::OfferLessByOperatorAndOfferId(Service_, lhs, rhs);
        });
        for (const auto& offer: sortedOffers) {
            if (Service_.Config().GetPartnerDataFilter().GetEnablePartnerDataOfferFilter() &&
                Service_.PartnerDataOfferFilter().GetSkipReasonOrNone("", offer) != NTravelProto::NOfferCache::NApi::SR_None) {
                continue;
            }
            if (offer.OfferRestrictions.RequiresMobile || offer.OfferRestrictions.RequiresRestrictedUser) {
                continue;
            }
            return {ConvertOffer(offer)};
        }
        return {};
    }

    TOffer TOutdatedRecordBuilder::ConvertOffer(const TOffer& offer) {
        TOffer result = offer;
        result.PartnerSpecificOfferData = Service_.ObjectDeduplicator().Deduplicate(TCommonDeduplicatorKeys::PartnerSpecificOfferData,
                                                                                    TPartnerSpecificOfferData{TFakeOutdatedOfferPartnerData()});
        result.OfferRestrictions = {};
        result.TitleAndOriginalRoomId = Service_.ObjectDeduplicator().Deduplicate(TCommonDeduplicatorKeys::TitleAndOriginalRoomId,
                                                                                  TOfferTitleAndOriginalRoomId{TString(), TString()});
        return result;
    }
}
