#include "partner_data_offer_filter.h"

namespace NTravel::NOfferCache {
    TPartnerDataOfferFilter::TPartnerDataOfferFilter(const NTravelProto::NAppConfig::TYtTableCacheConfig& YtConfigTravellineRatePlans,
                                                     const NTravelProto::NAppConfig::TYtTableCacheConfig& YtConfigDolphinTour,
                                                     const NTravelProto::NAppConfig::TYtTableCacheConfig& YtConfigDolphinPansion,
                                                     const NTravelProto::NAppConfig::TYtTableCacheConfig& YtConfigDolphinRoom,
                                                     const NTravelProto::NAppConfig::TYtTableCacheConfig& YtConfigDolphinRoomCat,
                                                     const NTravelProto::NAppConfig::TYtTableCacheConfig& YtConfigBNovoRatePlans,
                                                     bool unkownTravellineItemIsBanned,
                                                     bool unkownDolphinItemIsBanned,
                                                     bool unkownBNovoItemIsBanned,
                                                     TServiceCounters& serviceCounters)
        : TravellineRatePlansConfig_("YtConfigTravellineRatePlans", YtConfigTravellineRatePlans)
        , DolphinTourConfig_("YtConfigDolphinTour", YtConfigDolphinTour)
        , DolphinPansionConfig_("YtConfigDolphinPansion", YtConfigDolphinPansion)
        , DolphinRoomConfig_("YtConfigDolphinRoom", YtConfigDolphinRoom)
        , DolphinRoomCatConfig_("YtConfigDolphinRoomCat", YtConfigDolphinRoomCat)
        , BNovoRatePlansConfig_("YtConfigBNovoRatePlans", YtConfigBNovoRatePlans)
        , UnkownTravellineItemIsBanned_(unkownTravellineItemIsBanned)
        , UnkownDolphinItemIsBanned_(unkownDolphinItemIsBanned)
        , UnkownBNovoItemIsBanned_(unkownBNovoItemIsBanned)
        , ServiceCounters_(serviceCounters)
    {
    }

    void TPartnerDataOfferFilter::RegisterCounters(NMonitor::TCounterSource& source) {
        TravellineRatePlansConfig_.RegisterCounters(source);
        DolphinTourConfig_.RegisterCounters(source);
        DolphinPansionConfig_.RegisterCounters(source);
        DolphinRoomConfig_.RegisterCounters(source);
        DolphinRoomCatConfig_.RegisterCounters(source);
        BNovoRatePlansConfig_.RegisterCounters(source);
    }

    void TPartnerDataOfferFilter::Start() {
        TravellineRatePlansConfig_.Start();
        DolphinTourConfig_.Start();
        DolphinPansionConfig_.Start();
        DolphinRoomConfig_.Start();
        DolphinRoomCatConfig_.Start();
        BNovoRatePlansConfig_.Start();
    }

    void TPartnerDataOfferFilter::Stop() {
        TravellineRatePlansConfig_.Stop();
        DolphinTourConfig_.Stop();
        DolphinPansionConfig_.Stop();
        DolphinRoomConfig_.Stop();
        DolphinRoomCatConfig_.Stop();
        BNovoRatePlansConfig_.Stop();
    }

    void TPartnerDataOfferFilter::SetIsReadyFlags(std::function<void(const TString& name, bool isReady)> setFlag) const {
        setFlag("TravellineRatePlans", TravellineRatePlansConfig_.IsReady());
        setFlag("DolphinTour", DolphinTourConfig_.IsReady());
        setFlag("DolphinPansion", DolphinPansionConfig_.IsReady());
        setFlag("DolphinRoom", DolphinRoomConfig_.IsReady());
        setFlag("DolphinRoomCat", DolphinRoomCatConfig_.IsReady());
        setFlag("BNovoRatePlans", BNovoRatePlansConfig_.IsReady());
    }

    NTravelProto::NOfferCache::NApi::ESkipReason TPartnerDataOfferFilter::GetSkipReasonOrNone(const TString& logPrefix, const TOffer& offer) const {
        auto partnerOfferData = offer.PartnerSpecificOfferData.GetValue().Data;
        if (std::holds_alternative<TFakeOutdatedOfferPartnerData>(partnerOfferData)) {
            // Outdated offers are not filtered by partner data
            // Filtering occurs for them only during convertation
            return NTravelProto::NOfferCache::NApi::SR_None;
        }
        if (offer.OperatorId == EOperatorId::OI_TRAVELLINE) {
            if (!std::holds_alternative<TTravellineData>(partnerOfferData)) {
                ServiceCounters_.NoPartnerSpecificDataInStoredOffer.Inc();
                ERROR_LOG << logPrefix << "No travelline data in offer (OfferId=" << offer.OfferId << ")" << Endl;
                return NTravelProto::NOfferCache::NApi::SR_BannedNoPartnerData;
            }
            auto travellineData = std::get<TTravellineData>(partnerOfferData);
            auto ratePlan = TravellineRatePlansConfig_.GetById(std::make_tuple(travellineData.HotelCode, travellineData.RatePlanCode));
            if (!ratePlan) {
                ServiceCounters_.UnknownTravellineRatePlan.Inc();
                WARNING_LOG << logPrefix << "Unknown travelline rate plan: [HotelCode=" << travellineData.HotelCode << "; RatePlanCode=" << travellineData.RatePlanCode << "]"
                          << " (OfferId=" << offer.OfferId << ")" << Endl;
                if (UnkownTravellineItemIsBanned_) {
                    return NTravelProto::NOfferCache::NApi::SR_BannedTravellineRatePlan;
                }
            }
            else if (!ratePlan->GetEnabled()) {
                return NTravelProto::NOfferCache::NApi::SR_BannedTravellineRatePlan;
            }
        } else if (offer.OperatorId == EOperatorId::OI_DOLPHIN) {
            if (!std::holds_alternative<TDolphinData>(partnerOfferData)) {
                ServiceCounters_.NoPartnerSpecificDataInStoredOffer.Inc();
                ERROR_LOG << logPrefix << "No dolphin data in offer (OfferId=" << offer.OfferId << ")" << Endl;
                return NTravelProto::NOfferCache::NApi::SR_BannedNoPartnerData;
            }
            auto dolphinData = std::get<TDolphinData>(partnerOfferData);
            for (const auto& [config, key, name, skipReason, counter] : {
                     std::make_tuple(&DolphinTourConfig_, dolphinData.Tour, "Tour", NTravelProto::NOfferCache::NApi::SR_BannedDolphinTour, &ServiceCounters_.UnknownDolphinTour),
                     std::make_tuple(&DolphinPansionConfig_, dolphinData.Pansion, "Pansion", NTravelProto::NOfferCache::NApi::SR_BannedDolphinPansion, &ServiceCounters_.UnknownDolphinPansion),
                     std::make_tuple(&DolphinRoomConfig_, dolphinData.Room, "Room", NTravelProto::NOfferCache::NApi::SR_BannedDolphinRoom, &ServiceCounters_.UnknownDolphinRoom),
                     std::make_tuple(&DolphinRoomCatConfig_, dolphinData.RoomCat, "RoomCat", NTravelProto::NOfferCache::NApi::SR_BannedDolphinRoomCat, &ServiceCounters_.UnknownDolphinRoomCat),
                 }) {
                auto configItem = config->GetById(key);
                if (!configItem) {
                    counter->Inc();
                    WARNING_LOG << logPrefix << "Unknown dolphin " << name << ": " << key << " (OfferId=" << offer.OfferId << ")" << Endl;
                    if (UnkownDolphinItemIsBanned_) {
                        return skipReason;
                    }
                }
                else if (!configItem->GetEnabled()) {
                    return skipReason;
                }
            }
        } else if (offer.OperatorId == EOperatorId::OI_BNOVO) {
            if (!std::holds_alternative<TBNovoData>(partnerOfferData)) {
                ServiceCounters_.NoPartnerSpecificDataInStoredOffer.Inc();
                ERROR_LOG << logPrefix << "No bnovo data in offer (OfferId=" << offer.OfferId << ")" << Endl;
                return NTravelProto::NOfferCache::NApi::SR_BannedNoPartnerData;
            }
            auto bnovoData = std::get<TBNovoData>(partnerOfferData);
            auto ratePlan = BNovoRatePlansConfig_.GetById(std::make_tuple(bnovoData.AccountId, bnovoData.RatePlanId));
            if (!ratePlan) {
                ServiceCounters_.UnknownBNovoRatePlan.Inc();
                WARNING_LOG << logPrefix << "Unknown bnovo rate plan: [AccountId=" << bnovoData.AccountId << "; RatePlanId=" << bnovoData.RatePlanId << "]"
                            << " (OfferId=" << offer.OfferId << ")" << Endl;
                if (UnkownBNovoItemIsBanned_) {
                    return NTravelProto::NOfferCache::NApi::SR_BannedBNovoRatePlan;
                }
            }
            else if (!ratePlan->GetEnabled()) {
                return NTravelProto::NOfferCache::NApi::SR_BannedBNovoRatePlan;
            }
        }

        return NTravelProto::NOfferCache::NApi::SR_None;
    }
}
