#include "promo_service.h"

#include "service.h"
#include <travel/hotels/lib/cpp/protobuf/tools.h>

namespace NTravel::NOfferCache {

bool TPromoService::TUserListKey::operator<(const TUserListKey& rhs) const {
    return PassportId < rhs.PassportId;
}

bool TPromoService::TUserListKey::operator==(const TUserListKey& rhs) const {
    return PassportId == rhs.PassportId;
}

size_t TPromoService::TUserListKey::Hash() const {
    return THash<TString>()(PassportId);
}

bool TPromoService::TUserWithOrderTypeListKey::operator<(const TUserWithOrderTypeListKey& rhs) const {
    if (PassportId != rhs.PassportId) {
        return PassportId < rhs.PassportId;
    }
    return OrderType < rhs.OrderType;
}

bool TPromoService::TUserWithOrderTypeListKey::operator==(const TUserWithOrderTypeListKey& rhs) const {
    return PassportId == rhs.PassportId && OrderType == rhs.OrderType;
}

size_t TPromoService::TUserWithOrderTypeListKey::Hash() const {
    return MultiHash(PassportId, OrderType);
}

TInstant ParseUtcTimeOrStartDate(const TString& str) {
    try {
        NOrdinalDate::TOrdinalDate dt = NOrdinalDate::FromString(str);
        return NOrdinalDate::ToInstant(dt);
    } catch (...) {
        return TInstant::ParseIso8601(str);
    }
}

TInstant ParseUtcTimeOrEndDate(const TString& str) {
    try {
        // Последняя миллисекунда дня
        NOrdinalDate::TOrdinalDate dt = NOrdinalDate::FromString(str);
        TInstant nextDayStart = NOrdinalDate::ToInstant(dt + 1);
        return TInstant::FromValue(nextDayStart.GetValue() - 1);
    } catch (...) {
        return TInstant::ParseIso8601(str);
    }
}

const TPriceWithCurrency& TPromoService::TForOfferInternalReq::GetLatestPrice() const {
    return PriceAfterPromoCodes.OrElse(PriceBeforePromoCodes).GetOrElse(PriceFromPartnerOffer);
}

bool TPromoService::TForOfferInternalReq::HasWhiteLabel() const {
    return WhiteLabelInfo.IsInitialized() && WhiteLabelInfo.GetPartnerId() != NTravelProto::NWhiteLabel::EWhiteLabelPartnerId::WL_UNKNOWN;
}


TPromoService::TPromoService(const TService& service, const NTravelProto::NOfferCache::TConfig::TPromoService& config)
  : Service(service)
  , WhiteLabelPromoService(config.GetWhiteLabel())
  , Taxi2020Config(MakeConfig(config.GetTaxi2020()))
  , MirConfig(MakeConfig(config.GetMir()))
  , PlusConfig(MakeConfig(config.GetPlus()))
  , DiscountConfig(MakeConfig(config.GetDiscount()))
  , BlackFriday2021Config(MakeConfig(config.GetBlackFriday2021()))
  , YandexEda2022Config(MakeConfig(config.GetYandexEda2022()))
  , YtConfigMirPromoWhitelist("YtConfigMirPromoWhitelist", config.GetMir().GetYtConfigWhitelist())
  , YtConfigBlackFriday2021Hotels("YtConfigBlackFriday2021Hotels", config.GetBlackFriday2021().GetYtConfigHotels())
{
}

TPromoService::TTaxi2020Config TPromoService::MakeConfig(const NTravelProto::NOfferCache::TConfig::TPromoService::TTaxi2020& pbCfg) const {
    TTaxi2020Config res;
    res.MinPrice = pbCfg.GetMinPrice();
    res.StartDate = NOrdinalDate::FromString(pbCfg.GetStartDate());
    res.LastDate = NOrdinalDate::FromString(pbCfg.GetLastDate());
    res.MaxCheckIn = NOrdinalDate::FromString(pbCfg.GetMaxCheckInDate());
    return res;
}

TPromoService::TMirConfig TPromoService::MakeConfig(const NTravelProto::NOfferCache::TConfig::TPromoService::TMir& pbCfg) const {
    TMirConfig res;
    for (const auto& waveConfig: {pbCfg.GetThirdWave(), pbCfg.GetFourthWave(), pbCfg.GetFifthWave(), pbCfg.GetSixthWave()}) {
        res.Waves.push_back(TMirWaveSettings{TInstant::ParseIso8601(waveConfig.GetStartUtc()),
                                      TInstant::ParseIso8601(waveConfig.GetEndUtc()),
                                      NOrdinalDate::FromString(waveConfig.GetFirstCheckIn()),
                                      NOrdinalDate::FromString(waveConfig.GetLastCheckOut()),
                                      waveConfig.GetMinNights(),
                                      waveConfig.GetCashbackPercent(),
                                      waveConfig.GetMaxCashback()});
    }
    return res;
}

TPromoService::TPlusConfig TPromoService::MakeConfig(const NTravelProto::NOfferCache::TConfig::TPromoService::TPlus& pbCfg) const {
    TPlusConfig res;
    res.Enabled = pbCfg.GetEnabled();
    res.OrdinalDiscount.PointsbackPercent = pbCfg.GetOrdinalDiscount().GetPointsbackPercent();
    res.OrdinalDiscount.MaxPointsback = pbCfg.GetOrdinalDiscount().GetMaxPointsback();
    if (pbCfg.HasYtConfigUserOrderCountersByType() && pbCfg.GetYtConfigUserOrderCountersByType().GetEnabled()) {
        res.YtConfigUserOrderCountersByType = std::make_shared<TYtPersistentConfig<TUserWithOrderTypeListKey, ru::yandex::travel::user_order_counters::TUserOrderCounterByType>>("YtConfigUserOrderCountersByType", pbCfg.GetYtConfigUserOrderCountersByType());
    }
    if (pbCfg.HasYtConfigAdditionalFee() && pbCfg.GetYtConfigAdditionalFee().GetEnabled()) {
        res.YtConfigAdditionalFee = std::make_shared<TYtMultiValuePersistentConfig<THotelId, NTravelProto::NOrders::NHotelsExtranet::THotelPlusAdditionalFeeItem>>("YtConfigAdditionalFee", pbCfg.GetYtConfigAdditionalFee());
    }
    for (const auto& event : pbCfg.GetPlusEvents()) {
        TPromoService::TPlusEvent plusEvent{};
        plusEvent.EventId = event.GetEventId();
        plusEvent.Discount.PointsbackPercent = event.GetDiscount().GetPointsbackPercent();
        plusEvent.Discount.MaxPointsback = event.GetDiscount().GetMaxPointsback();

        if (event.HasOrderTimeStartUtc()) {
            plusEvent.OrderTimeStartUtc = TInstant::ParseIso8601(event.GetOrderTimeStartUtc());
        }
        if (event.HasOrderTimeEndUtc()) {
            plusEvent.OrderTimeEndUtc = TInstant::ParseIso8601(event.GetOrderTimeEndUtc());
        }
        if (event.HasCheckInStart()) {
            plusEvent.CheckInStart = NOrdinalDate::FromString(event.GetCheckInStart());
        }
        if (event.HasCheckInEnd()) {
            plusEvent.CheckInEnd = NOrdinalDate::FromString(event.GetCheckInEnd());
        }
        if (event.HasCheckOutStart()) {
            plusEvent.CheckOutStart = NOrdinalDate::FromString(event.GetCheckOutStart());
        }
        if (event.HasCheckOutEnd()) {
            plusEvent.CheckOutEnd = NOrdinalDate::FromString(event.GetCheckOutEnd());
        }
        if (event.HasYtConfigUserWhitelist() && event.GetYtConfigUserWhitelist().GetEnabled()) {
            TString configName = "YtConfigUserWhitelist-" + event.GetEventId();
            plusEvent.YtConfigUserWhitelist = std::make_shared<TYtPersistentConfig<TUserListKey, NTravelProto::NPromoService::TUserListItem>>(configName, event.GetYtConfigUserWhitelist());
        }
        if (event.HasYtConfigUserBlacklist() && event.GetYtConfigUserBlacklist().GetEnabled()) {
            TString configName = "YtConfigUserBlacklist-" + event.GetEventId();
            plusEvent.YtConfigUserBlacklist = std::make_shared<TYtPersistentConfig<TUserListKey, NTravelProto::NPromoService::TUserListItem>>(configName, event.GetYtConfigUserBlacklist());
        }
        plusEvent.LoginRequired = event.GetLoginRequired();
        plusEvent.CheckFirstOrder = event.GetCheckFirstOrder();
        for (const auto& firstOrderType: event.GetFirstOrderTypes()) {
            plusEvent.FirstOrderTypes.insert(static_cast<NTravelProto::NOrderType::EOrderType>(firstOrderType));
        }
        if (event.HasRequiredExpName()) {
            plusEvent.RequiredExpName = event.GetRequiredExpName();
        }
        if (event.HasRequiredExpValue()) {
            plusEvent.RequiredExpValue = event.GetRequiredExpValue();
        }
        plusEvent.CheckHotelWhiteList = event.GetCheckHotelWhiteList();
        for (const auto& hotel: event.GetHotelWhiteList()) {
            plusEvent.HotelWhiteList.insert(THotelId::FromProto(hotel));
        }
        if (event.HasYtConfigHotelWhitelist() && event.GetYtConfigHotelWhitelist().GetEnabled()) {
            TString configName = "YtConfigHotelWhitelist-" + event.GetEventId();
            plusEvent.YtConfigHotelWhitelist = std::make_shared<TYtPersistentConfig<THotelId, NTravelProto::THotelId>>(configName, event.GetYtConfigHotelWhitelist());
        };
        if (event.HasYtConfigHotelBlacklist() && event.GetYtConfigHotelBlacklist().GetEnabled()) {
            TString configName = "YtConfigHotelBlacklist-" + event.GetEventId();
            plusEvent.YtConfigHotelBlacklist = std::make_shared<TYtPersistentConfig<THotelId, NTravelProto::THotelId>>(configName, event.GetYtConfigHotelBlacklist());
        };
        plusEvent.EventType = event.GetEventType();
        plusEvent.Priority = event.GetPriority();
        res.PlusEvents.push_back(plusEvent);
    }
    return res;
}

TPromoService::TDiscountConfig TPromoService::MakeConfig(const NTravelProto::NOfferCache::TConfig::TPromoService::TDiscount& pbCfg) const {
    TDiscountConfig res;
    res.Enabled = pbCfg.GetEnabled();
    for (const auto& entry: pbCfg.GetHotelDiscount()) {
        res.Hotels[THotelId::FromProto(entry.GetHotelId())] = entry.GetDiscountPercent();
    }
    return res;
}

TPromoService::TBlackFriday2021Config TPromoService::MakeConfig(const NTravelProto::NOfferCache::TConfig::TPromoService::TBlackFriday2021& pbCfg) const {
    TBlackFriday2021Config res;
    res.Enabled = pbCfg.GetEnabled();
    res.BadgeStart = TInstant::ParseIso8601(pbCfg.GetBadgeStartUtc());
    res.BadgeEnd = TInstant::ParseIso8601(pbCfg.GetBadgeEndUtc());
    return res;
}

TPromoService::TYandexEda2022Config TPromoService::MakeConfig(const NTravelProto::NOfferCache::TConfig::TPromoService::TYandexEda2022& pbCfg) const {
    TYandexEda2022Config res;
    res.Enabled = pbCfg.GetEnabled();
    res.EventStart = TInstant::ParseIso8601(pbCfg.GetEventStartUtc());
    res.EventEnd = TInstant::ParseIso8601(pbCfg.GetEventEndUtc());
    res.PromoCodeNominal = pbCfg.GetPromoCodeNominal();
    if (pbCfg.HasYtConfigHotelWhitelist() && pbCfg.GetYtConfigHotelWhitelist().GetEnabled()) {
        TString configName = "YtConfigEda2022Whitelist";
        res.YtConfigHotelWhitelist = std::make_shared<TYtPersistentConfig<THotelId, NTravelProto::THotelId>>(configName, pbCfg.GetYtConfigHotelWhitelist());
    }
    if (pbCfg.HasYtConfigHotelBlacklist() && pbCfg.GetYtConfigHotelBlacklist().GetEnabled()) {
        TString configName = "YtConfigEda2022Blacklist";
        res.YtConfigHotelBlacklist = std::make_shared<TYtPersistentConfig<THotelId, NTravelProto::THotelId>>(configName, pbCfg.GetYtConfigHotelBlacklist());
    }
    return res;
}

TPromoService::~TPromoService() {
    Stop();
}

bool TPromoService::IsReady() const {
    if (!YtConfigMirPromoWhitelist.IsReady() || !YtConfigBlackFriday2021Hotels.IsReady()) {
        return false;
    }
    if (PlusConfig.YtConfigUserOrderCountersByType.Defined() && !PlusConfig.YtConfigUserOrderCountersByType.GetRef()->IsReady()) {
        return false;
    }
    if (PlusConfig.YtConfigAdditionalFee.Defined() && !PlusConfig.YtConfigAdditionalFee.GetRef()->IsReady()) {
        return false;
    }
    for (const auto& event: PlusConfig.PlusEvents) {
        if (event.YtConfigUserWhitelist.Defined() && !event.YtConfigUserWhitelist.GetRef()->IsReady()) {
            return false;
        }
        if (event.YtConfigUserBlacklist.Defined() && !event.YtConfigUserBlacklist.GetRef()->IsReady()) {
            return false;
        }
        if (event.YtConfigHotelWhitelist.Defined() && !event.YtConfigHotelWhitelist.GetRef()->IsReady()) {
            return false;
        }
        if (event.YtConfigHotelBlacklist.Defined() && !event.YtConfigHotelBlacklist.GetRef()->IsReady()) {
            return false;
        }
    }
    if (YandexEda2022Config.YtConfigHotelWhitelist.Defined() && !YandexEda2022Config.YtConfigHotelWhitelist.GetRef()->IsReady()) {
        return false;
    }
    if (YandexEda2022Config.YtConfigHotelBlacklist.Defined() && !YandexEda2022Config.YtConfigHotelBlacklist.GetRef()->IsReady()) {
        return false;
    }
    return true;
}

void TPromoService::RegisterCounters(NMonitor::TCounterSource& source) {
    YtConfigMirPromoWhitelist.RegisterCounters(source);
    YtConfigBlackFriday2021Hotels.RegisterCounters(source);
    if (PlusConfig.YtConfigUserOrderCountersByType.Defined()) {
        PlusConfig.YtConfigUserOrderCountersByType.GetRef()->RegisterCounters(source);
    }
    if (PlusConfig.YtConfigAdditionalFee.Defined()) {
        PlusConfig.YtConfigAdditionalFee.GetRef()->RegisterCounters(source);
    }
    for (const auto& event: PlusConfig.PlusEvents) {
        if (event.YtConfigUserWhitelist.Defined()) {
            event.YtConfigUserWhitelist.GetRef()->RegisterCounters(source);
        }
        if (event.YtConfigUserBlacklist.Defined()) {
            event.YtConfigUserBlacklist.GetRef()->RegisterCounters(source);
        }
        if (event.YtConfigHotelWhitelist.Defined()) {
            event.YtConfigHotelWhitelist.GetRef()->RegisterCounters(source);
        }
        if (event.YtConfigHotelBlacklist.Defined()) {
            event.YtConfigHotelBlacklist.GetRef()->RegisterCounters(source);
        }
    }
    if (YandexEda2022Config.YtConfigHotelWhitelist.Defined()) {
        YandexEda2022Config.YtConfigHotelWhitelist.GetRef()->RegisterCounters(source);
    }
    if (YandexEda2022Config.YtConfigHotelBlacklist.Defined()) {
        YandexEda2022Config.YtConfigHotelBlacklist.GetRef()->RegisterCounters(source);
    }
}

void TPromoService::Start() {
    YtConfigMirPromoWhitelist.Start();
    YtConfigBlackFriday2021Hotels.Start();
    if (PlusConfig.YtConfigUserOrderCountersByType.Defined()) {
        PlusConfig.YtConfigUserOrderCountersByType.GetRef()->Start();
    }
    if (PlusConfig.YtConfigAdditionalFee.Defined()) {
        PlusConfig.YtConfigAdditionalFee.GetRef()->Start();
    }
    for (const auto& event: PlusConfig.PlusEvents) {
        if (event.YtConfigUserWhitelist.Defined()) {
            event.YtConfigUserWhitelist.GetRef()->Start();
        }
        if (event.YtConfigUserBlacklist.Defined()) {
            event.YtConfigUserBlacklist.GetRef()->Start();
        }
        if (event.YtConfigHotelWhitelist.Defined()) {
            event.YtConfigHotelWhitelist.GetRef()->Start();
        }
        if (event.YtConfigHotelBlacklist.Defined()) {
            event.YtConfigHotelBlacklist.GetRef()->Start();
        }
    }
    if (YandexEda2022Config.YtConfigHotelWhitelist.Defined()) {
        YandexEda2022Config.YtConfigHotelWhitelist.GetRef()->Start();
    }
    if (YandexEda2022Config.YtConfigHotelBlacklist.Defined()) {
        YandexEda2022Config.YtConfigHotelBlacklist.GetRef()->Start();
    }
}

void TPromoService::Stop() {
    YtConfigBlackFriday2021Hotels.Stop();
    YtConfigMirPromoWhitelist.Stop();
    if (PlusConfig.YtConfigUserOrderCountersByType.Defined()) {
        PlusConfig.YtConfigUserOrderCountersByType.GetRef()->Stop();
    }
    if (PlusConfig.YtConfigAdditionalFee.Defined()) {
        PlusConfig.YtConfigAdditionalFee.GetRef()->Stop();
    }
    for (const auto& event: PlusConfig.PlusEvents) {
        if (event.YtConfigUserWhitelist.Defined()) {
            event.YtConfigUserWhitelist.GetRef()->Stop();
        }
        if (event.YtConfigUserBlacklist.Defined()) {
            event.YtConfigUserBlacklist.GetRef()->Stop();
        }
        if (event.YtConfigHotelWhitelist.Defined()) {
            event.YtConfigHotelWhitelist.GetRef()->Stop();
        }
        if (event.YtConfigHotelBlacklist.Defined()) {
            event.YtConfigHotelBlacklist.GetRef()->Stop();
        }
    }
    if (YandexEda2022Config.YtConfigHotelWhitelist.Defined()) {
        YandexEda2022Config.YtConfigHotelWhitelist.GetRef()->Stop();
    }
    if (YandexEda2022Config.YtConfigHotelBlacklist.Defined()) {
        YandexEda2022Config.YtConfigHotelBlacklist.GetRef()->Stop();
    }
}

const TPromoService::TMirWaveSettings* TPromoService::GetMirCurrentWaveSettings(TInstant now) const {
    for (const auto& wave: MirConfig.Waves) {
        if (now >= wave.PromoStart && now <= wave.PromoEnd) {
            return &wave;
        }
    }
    return nullptr;
}

bool TPromoService::IsMirPromoAvailableForHotel(TInstant now, const TComplexHotelId& complexHotelId) const {
    if (!GetMirCurrentWaveSettings(now)) {
        return false;
    }
    for (const THotelId& hotelId: complexHotelId.HotelIds) {
        if (auto mirPromo = YtConfigMirPromoWhitelist.GetById(hotelId)) {
            if (mirPromo->GetEnabled()) {
                return true;
            }
        }
    }
    return false;
}

bool TPromoService::IsPlusAvailableForHotel(TInstant now, const TComplexHotelId& complexHotelId) const {
    Y_UNUSED(now);

    if (!PlusConfig.Enabled) {
        return false;
    }

    return HotelHasBoyPartner(complexHotelId);
}

bool TPromoService::ShowBlackFriday2021BadgeForHotel(TInstant now, const TComplexHotelId& complexHotelId) const {
    return IsBlackFriday2021Active(now) && HotelHasBoyPartner(complexHotelId);
}

bool TPromoService::HotelHasBoyPartner(const TComplexHotelId& complexHotelId) const {
    for (const THotelId& hotelId: complexHotelId.HotelIds) {
        if (Service.IsBoYPartner(hotelId.PartnerId)) {
            return true;
        }
    }
    return false;
}

void TPromoService::FillTaxi2020Status(const TForOfferInternalReq& req, NTravelProto::NPromoService::TTaxi2020Status* res) const {
    auto today = NOrdinalDate::FromInstant(req.Now);
    bool eligible = today >= Taxi2020Config.StartDate
            && today <= Taxi2020Config.LastDate
            && req.PriceBeforePromoCodes.GetOrElse(req.PriceFromPartnerOffer).Currency == NTravelProto::C_RUB
            && req.PriceBeforePromoCodes.GetOrElse(req.PriceFromPartnerOffer).Value >= Taxi2020Config.MinPrice
            && req.Date <= Taxi2020Config.MaxCheckIn
            && Service.IsBoYPartner(req.HotelId.PartnerId)
            && !req.HasWhiteLabel();
   res->SetEligible(eligible);
}


void TPromoService::FillMirPromoStatus(const TForOfferInternalReq& req, NTravelProto::NPromoService::TMirPromoStatus* res) const {
    auto currentWaveSettings = GetMirCurrentWaveSettings(req.Now);
    if (!currentWaveSettings) {
        res->SetEligibility(NTravelProto::NPromoService::ME_WRONG_BOOKING_DATE);
        return;
    }
    auto wlRec = YtConfigMirPromoWhitelist.GetById(req.HotelId);
    if (!wlRec || !wlRec->GetEnabled()) {
        res->SetEligibility(NTravelProto::NPromoService::ME_BLACKLISTED);
        return;
    }
    if (req.Nights < currentWaveSettings->MinNights) {
        res->SetEligibility(NTravelProto::NPromoService::ME_WRONG_LOS);
        return;
    }
    if (req.Date < currentWaveSettings->FirstCheckIn) {
        res->SetEligibility(NTravelProto::NPromoService::ME_WRONG_STAY_DATES);
        return;
    }
    if (static_cast<int>(req.Date + req.Nights) > currentWaveSettings->LastCheckOut) {
        res->SetEligibility(NTravelProto::NPromoService::ME_WRONG_STAY_DATES);
        return;
    }
    if (req.HasWhiteLabel()) {
        res->SetEligibility(NTravelProto::NPromoService::ME_CONFLICTS_WITH_WL);
        return;
    }
    res->SetEligibility(NTravelProto::NPromoService::ME_ELIGIBLE);
    double cashbackRate = currentWaveSettings->CashbackPercent * 0.01;
    auto cashbackAmountValue = Min(static_cast<ui32>(req.GetLatestPrice().Value * cashbackRate), currentWaveSettings->MaxCashback);
    res->MutableCashbackAmount()->set_value(cashbackAmountValue);
    res->MutableCashbackRate()->set_value(cashbackRate);
    res->MutableCashbackPercent()->set_value(currentWaveSettings->CashbackPercent);
    NTravel::NProtobuf::InstantToTimestamp(currentWaveSettings->PromoEnd, res->MutableExpiresAt());
    res->SetMirId(wlRec->GetMirId());
    res->MutableMaxCashbackAmount()->set_value(currentWaveSettings->MaxCashback);
}

void TPromoService::FillPlusStatus(const TForOfferInternalReq& req, NTravelProto::NPromoService::TYandexPlusStatus* res) const {
    if (!PlusConfig.Enabled) {
        res->SetEligibility(NTravelProto::NPromoService::YPE_PROMO_DISABLED);
        return;
    }
    if (req.HasWhiteLabel()) {
        res->SetEligibility(NTravelProto::NPromoService::YPE_CONFLICTS_WITH_WL);
        return;
    }
    if (!Service.IsBoYPartner(req.HotelId.PartnerId)) {
        res->SetEligibility(NTravelProto::NPromoService::YPE_WRONG_PARTNER);
        return;
    }
    TPlusDiscount plusDiscount = GetPlusDiscount(req);
    res->MutablePoints()->set_value(plusDiscount.Points);
    res->MutablePromoInfo()->SetEventId(plusDiscount.EventId);
    res->MutablePromoInfo()->SetMaxPointsback(plusDiscount.MaxPointsback);
    res->MutablePromoInfo()->SetDiscountPercent(plusDiscount.PointsbackPercent);
    res->MutablePromoInfo()->SetEventType(plusDiscount.EventType);
    res->MutableAdditionalFeeInfo()->SetEligibility(plusDiscount.AdditionalFeeEligibility);
    res->MutableAdditionalFeeInfo()->SetFeePercent(plusDiscount.AdditionalFeePercent);
    res->MutableAdditionalFeeInfo()->SetFeeValue(plusDiscount.AdditionalFeeValue);
    if (plusDiscount.SpecialOfferEndUtc.Defined()) {
        NTravel::NProtobuf::InstantToTimestamp(plusDiscount.SpecialOfferEndUtc.GetRef(), res->MutablePromoInfo()->MutableSpecialOfferEndUtc());
    }
    if (!req.UserInfo.GetIsLoggedIn()) {
        res->SetEligibility(NTravelProto::NPromoService::YPE_USER_NOT_LOGGED_IN);
        return;
    }
    if (!req.UserInfo.GetIsPlus()) {
        res->SetEligibility(NTravelProto::NPromoService::YPE_USER_NOT_PLUS);
        return;
    }
    res->SetEligibility(NTravelProto::NPromoService::YPE_ELIGIBLE);
}

void TPromoService::FillBlackFriday2021Status(const TForOfferInternalReq& req, NTravelProto::NPromoService::TBlackFriday2021Status* res) const {
    if (!BlackFriday2021Config.Enabled) {
        return;
    }
    if (!Service.IsBoYPartner(req.HotelId.PartnerId)) {
        return;
    }
    if (req.HasWhiteLabel()) {
        return;
    }
    res->SetShowBadge(IsBlackFriday2021Active(req.Now));
    if (auto bfHotel = YtConfigBlackFriday2021Hotels.GetById(req.HotelId)) {
        try {
            TInstant start = ParseUtcTimeOrStartDate(bfHotel->GetStartAt());
            TInstant end = ParseUtcTimeOrEndDate(bfHotel->GetEndAt());
            if (start <= req.Now && req.Now <= end) {
                res->SetHotelHasSpecialTariff(true);
            }
        } catch (...) {
            ERROR_LOG << "Error while parsing black friday time for hotel " << req.HotelId << Endl;
        }
    }
}

void TPromoService::FillYandexEda2022Status(const TForOfferInternalReq& req, NTravelProto::NPromoService::TYandexEda2022Status* res) const {
    if (!YandexEda2022Config.Enabled) {
        res->SetEligibility(NTravelProto::NPromoService::YEE_PROMO_DISABLED);
        return;
    }
    if (req.HasWhiteLabel()) {
        res->SetEligibility(NTravelProto::NPromoService::YEE_CONFLICTS_WITH_WL);
        return;
    }
    if (req.Now < YandexEda2022Config.EventStart || req.Now >= YandexEda2022Config.EventEnd) {
        res->SetEligibility(NTravelProto::NPromoService::YEE_WRONG_BOOKING_DATE);
        return;
    }
    if (YandexEda2022Config.YtConfigHotelWhitelist.Defined() && !YandexEda2022Config.YtConfigHotelWhitelist.GetRef()->GetById(req.HotelId)) {
        res->SetEligibility(NTravelProto::NPromoService::YEE_WL_MISS_MATCH);
        return;
    }
    if (YandexEda2022Config.YtConfigHotelBlacklist.Defined() && YandexEda2022Config.YtConfigHotelBlacklist.GetRef()->GetById(req.HotelId)) {
        res->SetEligibility(NTravelProto::NPromoService::YEE_BLACKLISTED);
        return;
    }
    res->SetEligibility(NTravelProto::NPromoService::YEE_ELIGIBLE);
    auto promoInfo = res->MutablePromoInfo();
    int lastDate = req.Date + req.Nights;
    lastDate = Min(lastDate, NOrdinalDate::FromInstant(YandexEda2022Config.EventEnd) + 1);
    int promoCodeCount = lastDate - req.Date;
    promoInfo->SetPromoCodeNominal(YandexEda2022Config.PromoCodeNominal);
    promoInfo->SetPromoCodeCount(promoCodeCount);
    promoInfo->SetFirstDate(NOrdinalDate::ToString(req.Date + 1));
    promoInfo->SetLastDate(NOrdinalDate::ToString(lastDate));
    res->SetShowBadge(true);
}

void TPromoService::FillWhiteLabelStatus(const TForOfferInternalReq &req, NTravelProto::NPromoService::TWhiteLabelStatus *res) const {
    TWhiteLabelPromoService::TForOfferInternalReq iReq = {
            .Now = req.Now,
            .LatestPrice = req.GetLatestPrice(),
            .UserInfo = req.UserInfo,
            .ExpInfo = req.ExpInfo,
    };
    if (req.HasWhiteLabel()) {
        iReq.WhiteLabelInfo = req.WhiteLabelInfo;
    }
    WhiteLabelPromoService.FillStatus(iReq, res);
}

TPromoService::TPlusDiscount TPromoService::GetPlusDiscount(const TForOfferInternalReq& req) const {
    TUserListKey userListKey = {req.UserInfo.GetPassportId() == "" ? "No passport id" : req.UserInfo.GetPassportId()};
    TPassportUid parsedPassportId;
    if (req.UserInfo.GetPassportId() == "" || !TryFromString(req.UserInfo.GetPassportId(), parsedPassportId)) {
        parsedPassportId = -1;
    }
    TPlusAdditionalFeeInfo additionalFeeInfo = GetPlusAdditionalFeeInfo(req);

    TPlusDiscount bestDiscount  = {
            .EventId = "",
            .Points = GetPlusPoints(req.GetLatestPrice().Value, PlusConfig.OrdinalDiscount, additionalFeeInfo),
            .PointsbackPercent = static_cast<ui32>(PlusConfig.OrdinalDiscount.PointsbackPercent + additionalFeeInfo.PointsPercent),
            .MaxPointsback = PlusConfig.OrdinalDiscount.MaxPointsback,
            .SpecialOfferEndUtc = {},
            .AdditionalFeePercent = additionalFeeInfo.FeePercent,
            .AdditionalFeeValue = GetPlusAdditionalFeeValue(req.GetLatestPrice().Value, additionalFeeInfo),
            .AdditionalFeeEligibility = additionalFeeInfo.Eligibility,
            .EventType = NTravelProto::NPromoService::YPET_COMMON,
            .Priority = 0,
    };

    for (const auto& event: PlusConfig.PlusEvents) {
        if (!IsPlusEventApplicable(req, userListKey, parsedPassportId, event)) {
            continue;
        }

        ui32 currentPoints = GetPlusPoints(req.GetLatestPrice().Value, event.Discount, additionalFeeInfo);
        if (currentPoints > bestDiscount.Points || (currentPoints == bestDiscount.Points && event.Priority > bestDiscount.Priority)) {
            bestDiscount = {
                    .EventId = event.EventId,
                    .Points = currentPoints,
                    .PointsbackPercent = static_cast<ui32>(event.Discount.PointsbackPercent + additionalFeeInfo.PointsPercent),
                    .MaxPointsback = event.Discount.MaxPointsback,
                    .SpecialOfferEndUtc = event.OrderTimeEndUtc,
                    .AdditionalFeePercent = additionalFeeInfo.FeePercent,
                    .AdditionalFeeValue = GetPlusAdditionalFeeValue(req.GetLatestPrice().Value, additionalFeeInfo),
                    .AdditionalFeeEligibility = additionalFeeInfo.Eligibility,
                    .EventType = additionalFeeInfo.Eligibility == NTravelProto::NPromoService::PAFE_ELIGIBLE
                            ? NTravelProto::NPromoService::YPET_SPECIAL
                            : event.EventType,
                    .Priority = event.Priority,
            };
        }
    }
    return bestDiscount;
}

TPromoService::TPlusAdditionalFeeInfo TPromoService::GetPlusAdditionalFeeInfo(const TForOfferInternalReq& req) const {
    if (!PlusConfig.YtConfigAdditionalFee.Defined()) {
        return {NTravelProto::NPromoService::PAFE_PROMO_DISABLED, 0, 0};
    }
    for (const auto& item: PlusConfig.YtConfigAdditionalFee.GetRef()->GetById(req.HotelId)) {
        if (TInstant::Seconds(item.GetStartsAtSeconds()) <= req.Now && req.Now < TInstant::Seconds(item.GetEndsAtSeconds())) {
            return {
                NTravelProto::NPromoService::PAFE_ELIGIBLE,
                item.GetAdditionalPointsPercent(),
                item.GetAdditionalFeePercent(),
            };
        }
        ERROR_LOG << "skipped " << item << Endl;
    }
    return {NTravelProto::NPromoService::PAFE_HOTEL_MISMATCH, 0, 0};
}

bool TPromoService::IsPlusEventApplicable(const TForOfferInternalReq& req, TUserListKey userListKey, TPassportUid parsedPassportId, const TPlusEvent& event) const {
    if (event.OrderTimeStartUtc.Defined() && req.Now < event.OrderTimeStartUtc.GetRef()) {
        return false;
    }
    if (event.OrderTimeEndUtc.Defined() && req.Now >= event.OrderTimeEndUtc.GetRef()) {
        return false;
    }
    if (event.CheckInStart.Defined() && req.Date < event.CheckInStart.GetRef()) {
        return false;
    }
    if (event.CheckInEnd.Defined() && req.Date > event.CheckInEnd.GetRef()) {
        return false;
    }
    if (event.CheckOutStart.Defined() && static_cast<int>(req.Date + req.Nights) < event.CheckOutStart.GetRef()) {
        return false;
    }
    if (event.CheckOutEnd.Defined() && static_cast<int>(req.Date + req.Nights) > event.CheckOutEnd.GetRef()) {
        return false;
    }
    if (event.LoginRequired && !req.UserInfo.GetIsLoggedIn()) {
        return false;
    }
    if (event.YtConfigUserWhitelist.Defined() && !event.YtConfigUserWhitelist.GetRef()->GetById(userListKey)) {
        return false;
    }
    if (event.YtConfigUserBlacklist.Defined() && event.YtConfigUserBlacklist.GetRef()->GetById(userListKey)) {
        return false;
    }
    if (event.CheckFirstOrder && !IsUserEligibleByFirstOrderTypes(parsedPassportId, event.FirstOrderTypes, req.UserInfo.GetUseExistingOrderTypes(), req.UserInfo.GetExistingOrderTypes())) {
        return false;
    }
    if (event.CheckHotelWhiteList && !event.HotelWhiteList.contains(req.HotelId)) {
        return false;
    }
    if (event.YtConfigHotelWhitelist.Defined() && !event.YtConfigHotelWhitelist.GetRef()->GetById(req.HotelId)) {
        return false;
    }
    if (event.YtConfigHotelBlacklist.Defined() && event.YtConfigHotelBlacklist.GetRef()->GetById(req.HotelId)) {
        return false;
    }
    if (event.RequiredExpName.Defined()) {
        bool expMatched = false;
        for (const auto& expItem: req.ExpInfo.GetKVExperiments()) {
            if (event.RequiredExpName.GetRef() != expItem.GetKey()) {
                continue;
            }
            if (event.RequiredExpValue.Defined() && event.RequiredExpValue.GetRef() != expItem.GetValue()) {
                continue;
            }
            expMatched = true;
            break;
        }
        if (!expMatched) {
            return false;
        }
    }
    return true;
}

bool TPromoService::IsUserEligibleByFirstOrderTypes(TPassportUid passportId,
                                                    const THashSet<NTravelProto::NOrderType::EOrderType>& orderTypesToCheck,
                                                    bool useUserOrderTypesFromRequest,
                                                    const ::google::protobuf::RepeatedField<int>& userOrderTypesFromRequest) const {
    if (useUserOrderTypesFromRequest) {
        for (const auto& userOrderTypeFromRequest : userOrderTypesFromRequest) {
            if (orderTypesToCheck.contains(static_cast<NTravelProto::NOrderType::EOrderType>(userOrderTypeFromRequest))) {
                return false;
            }
        }
        return true;
    }

    if (!PlusConfig.YtConfigUserOrderCountersByType.Defined()) {
        return true;
    }
    for (const auto& orderType: orderTypesToCheck) {
        auto userOrderCounter = PlusConfig.YtConfigUserOrderCountersByType.GetRef()->GetById({passportId, orderType});
        if (userOrderCounter && userOrderCounter->GetOrderCount() > 0) {
            return false;
        }
    }
    return true;
}

ui32 TPromoService::GetPlusAdditionalFeeValue(TPriceVal value, TPlusAdditionalFeeInfo additionalFeeInfo) {
    return static_cast<int>(lround(value * additionalFeeInfo.FeePercent / 100.0));
}

ui32 TPromoService::GetPlusPoints(TPriceVal value, TPlusDiscountConfig discount, TPlusAdditionalFeeInfo additionalFeeInfo) {
    ui32 points = static_cast<int>(lround(value * discount.PointsbackPercent / 100.0));
    points = Min(points, discount.MaxPointsback);
    ui32 additionalPoints = static_cast<int>(lround(value * additionalFeeInfo.PointsPercent / 100.0));
    return points + additionalPoints;
}

void TPromoService::FillTaxi2020Params(TInstant now, NTravelProto::NPromoService::TTaxi2020PromoParams* res) const {
    auto today = NOrdinalDate::FromInstant(now);
    res->SetActive(today >= Taxi2020Config.StartDate && today <= Taxi2020Config.LastDate);
}

void TPromoService::FillMirPromoParams(TInstant now, NTravelProto::NPromoService::TMirPromoParams* res) const {
    auto currentWaveSettings = GetMirCurrentWaveSettings(now);
    if (!currentWaveSettings) {
        res->SetActive(false);
        return;
    }

    res->SetActive(true);
    res->SetFirstCheckIn(NOrdinalDate::ToString(currentWaveSettings->FirstCheckIn));
    res->SetLastCheckOut(NOrdinalDate::ToString(currentWaveSettings->LastCheckOut));
    res->SetMinNights(currentWaveSettings->MinNights);
    res->SetCashbackPercent(currentWaveSettings->CashbackPercent);
    res->SetMaxCashback(currentWaveSettings->MaxCashback);
}

void TPromoService::FillPlusParams(TInstant, NTravelProto::NPromoService::TYandexPlusPromoParams* res) const {
    res->SetActive(PlusConfig.Enabled);
}

void TPromoService::FillBlackFridayParams(TInstant now, NTravelProto::NPromoService::TBlackFridayParams* res) const {
    res->SetActive(IsBlackFriday2021Active(now));
}

void TPromoService::DeterminePromosForOffer(const NTravelProto::NPromoService::TDeterminePromosForOfferReq& req,
                                                  NTravelProto::NPromoService::TDeterminePromosForOfferRsp* rsp) const {
    TForOfferInternalReq iReq;
    iReq.Now = NTravel::NProtobuf::TimestampToInstant(req.GetNow());
    iReq.HotelId = THotelId::FromProto(req.GetOfferInfo().GetHotelId());
    NOrdinalDate::TOrdinalDate checkIn = NOrdinalDate::FromString(req.GetOfferInfo().GetCheckInDate());
    NOrdinalDate::TOrdinalDate checkOut = NOrdinalDate::FromString(req.GetOfferInfo().GetCheckOutDate());
    iReq.Date = checkIn;
    iReq.Nights = checkOut - checkIn;
    if (req.GetOfferInfo().HasPriceFromPartnerOffer()) {
        iReq.PriceFromPartnerOffer = TPriceWithCurrency::FromProto(req.GetOfferInfo().GetPriceFromPartnerOffer());
    } else {
        // Tmp for migration
        if (!req.GetOfferInfo().HasPriceBeforePromocodes()) {
            throw yexception() << "Malformed request: both PriceFromPartnerOffer and PriceBeforePromoCodes are missing";
        }
        iReq.PriceFromPartnerOffer = TPriceWithCurrency::FromProto(req.GetOfferInfo().GetPriceBeforePromocodes());
    }
    if (req.GetOfferInfo().HasPriceBeforePromocodes()) {
        iReq.PriceBeforePromoCodes = TPriceWithCurrency::FromProto(req.GetOfferInfo().GetPriceBeforePromocodes());
    }
    if (req.GetOfferInfo().HasPriceAfterPromocodes()) {
        iReq.PriceAfterPromoCodes = TPriceWithCurrency::FromProto(req.GetOfferInfo().GetPriceAfterPromocodes());
    }
    iReq.UserInfo = req.GetUserInfo();
    iReq.ExpInfo = req.GetExperimentInfo();
    if (req.HasWhiteLabelInfo()) {
        iReq.WhiteLabelInfo = req.GetWhiteLabelInfo();
    }

    DeterminePromosForOffer(iReq, rsp);
}

void TPromoService::DeterminePromosForOffer(const TForOfferInternalReq& req,
                                                  NTravelProto::NPromoService::TDeterminePromosForOfferRsp* rsp) const {
    FillTaxi2020Status(req, rsp->MutableTaxi2020());
    FillMirPromoStatus(req, rsp->MutableMir());
    FillPlusStatus(req, rsp->MutablePlus());
    FillBlackFriday2021Status(req, rsp->MutableBlackFriday2021Status());
    FillYandexEda2022Status(req, rsp->MutableYandexEda2022Status());
    FillWhiteLabelStatus(req, rsp->MutableWhiteLabelStatus());
}

bool TPromoService::IsBlackFriday2021Active(TInstant now) const {
    if (!BlackFriday2021Config.Enabled) {
        return false;
    }
    return BlackFriday2021Config.BadgeStart <= now && now < BlackFriday2021Config.BadgeEnd;
}

void TPromoService::GetActivePromos(const NTravelProto::NPromoService::TGetActivePromosReq& req,
                                    NTravelProto::NPromoService::TGetActivePromosRsp* rsp) const {
    auto now = NTravel::NProtobuf::TimestampToInstant(req.GetNow());
    FillTaxi2020Params(now, rsp->MutableTaxi2020());
    FillMirPromoParams(now, rsp->MutableMir());
    FillPlusParams(now, rsp->MutablePlus());
    FillBlackFridayParams(now, rsp->MutableBlackFriday());
}

void TPromoService::CalculateDiscountForOffer(const NTravelProto::NPromoService::TCalculateDiscountForOfferReq& req,
                                              NTravelProto::NPromoService::TCalculateDiscountForOfferRsp* rsp) const {
    TForOfferInternalReq iReq;
    iReq.Now = NTravel::NProtobuf::TimestampToInstant(req.GetNow());
    iReq.HotelId = THotelId::FromProto(req.GetOfferInfo().GetHotelId());
    NOrdinalDate::TOrdinalDate checkIn = NOrdinalDate::FromString(req.GetOfferInfo().GetCheckInDate());
    NOrdinalDate::TOrdinalDate checkOut = NOrdinalDate::FromString(req.GetOfferInfo().GetCheckOutDate());
    iReq.Date = checkIn;
    iReq.Nights = checkOut - checkIn;
    if (req.GetOfferInfo().HasPriceFromPartnerOffer()) {
        iReq.PriceFromPartnerOffer = TPriceWithCurrency::FromProto(req.GetOfferInfo().GetPriceFromPartnerOffer());
    } else {
        throw yexception() << "Malformed request: both PriceFromPartnerOffer is missing";
    }
    if (req.GetOfferInfo().HasPriceBeforePromocodes()) {
        throw yexception() << "Malformed request: PriceBeforePromocodes is present";
    }
    if (req.GetOfferInfo().HasPriceAfterPromocodes()) {
        throw yexception() << "Malformed request: PriceAfterPromocodes is present";
    }
    iReq.UserInfo = req.GetUserInfo();
    iReq.ExpInfo = req.GetExperimentInfo();

    CalculateDiscountForOffer(iReq, rsp);
}

void TPromoService::CalculateDiscountForOffer(const TForOfferInternalReq& req,
                                              NTravelProto::NPromoService::TCalculateDiscountForOfferRsp* rsp) const {
    NTravelProto::NPromoService::TDiscountInfo* res = rsp->MutableDiscountInfo();
    auto* priceAfterDiscount = res->MutablePriceAfterDiscount();
    req.PriceFromPartnerOffer.ToProto(priceAfterDiscount);
    res->SetDiscountPercent(0);
    res->SetStatus(NTravelProto::NPromoService::DS_NOT_APPLIED);
    res->SetDiscountApplied(false);

    if (DiscountConfig.Enabled && req.ExpInfo.GetStrikeThroughPrices()) {
        if (const ui32 *discountPct = DiscountConfig.Hotels.FindPtr(req.HotelId)) {
            res->SetDiscountPercent(*discountPct);
            res->SetStatus(NTravelProto::NPromoService::DS_YANDEX_HOTEL);
            res->SetDiscountApplied(true);
            int discount = static_cast<int>(lround(priceAfterDiscount->GetAmount() * (*discountPct) / 100.0));
            priceAfterDiscount->SetAmount(priceAfterDiscount->GetAmount() - discount);
        }
    }
}

void TPromoService::GetWhiteLabelPointsProps(const NTravelProto::NPromoService::TGetWhiteLabelPointsPropsReq& req,
                                             NTravelProto::NPromoService::TGetWhiteLabelPointsPropsRsp* rsp) const {
    WhiteLabelPromoService.FillPointsLinguistics(req.GetAmount(), req.GetPointsType(), rsp->MutablePointsLinguistics());
}

}//namespace NTravel::NOfferCache

template <>
void Out<NTravel::NOfferCache::TPromoService::TUserListKey>(IOutputStream& out, const NTravel::NOfferCache::TPromoService::TUserListKey& key) {
    out << "PassportId:" << key.PassportId;
}

template <>
NTravel::NOfferCache::TPromoService::TUserListKey NTravel::NProtobuf::GetIdentifier<NTravel::NOfferCache::TPromoService::TUserListKey, NTravelProto::NPromoService::TUserListItem>(const NTravelProto::NPromoService::TUserListItem& data) {
    return {data.GetPassportId()};
}

template <>
void Out<NTravel::NOfferCache::TPromoService::TUserWithOrderTypeListKey>(IOutputStream& out, const NTravel::NOfferCache::TPromoService::TUserWithOrderTypeListKey& key) {
    out << "PassportId:" << key.PassportId << ", OrderType: " << NTravelProto::NOrderType::EOrderType_Name(key.OrderType);
}

template <>
NTravel::NOfferCache::TPromoService::TUserWithOrderTypeListKey NTravel::NProtobuf::GetIdentifier<NTravel::NOfferCache::TPromoService::TUserWithOrderTypeListKey, ru::yandex::travel::user_order_counters::TUserOrderCounterByType>(const ru::yandex::travel::user_order_counters::TUserOrderCounterByType& data) {
    return {data.GetPassportId(), data.GetOrderType()};
}

template <>
NTravel::THotelId NTravel::NProtobuf::GetIdentifier<NTravel::THotelId, NTravelProto::NOrders::NHotelsExtranet::THotelPlusAdditionalFeeItem>(const NTravelProto::NOrders::NHotelsExtranet::THotelPlusAdditionalFeeItem& data) {
    return {data.GetPartnerId(), data.GetOriginalId()};
}
