#pragma once

#include "data.h"
#include "white_label_promo_service.h"

#include <travel/hotels/offercache/proto/config.pb.h>
#include <travel/hotels/proto/data_config/promo/mir_white_list.pb.h>
#include <travel/hotels/proto/data_config/promo/black_friday_2021_hotel.pb.h>
#include <travel/hotels/proto/extranet_api/hotel_plus_additional_fee.pb.h>
#include <travel/hotels/proto/promo_service/promo_service.pb.h>

#include <travel/hotels/lib/cpp/yt/multi_value_persistent_config.h>
#include <travel/hotels/lib/cpp/yt/persistent_config.h>
#include <travel/hotels/lib/cpp/mon/counter.h>

#include <travel/proto/order_type/order_type.pb.h>
#include <travel/proto/user_order_counters/user_order_counters.pb.h>

#include <util/datetime/base.h>
#include <util/generic/ptr.h>

namespace NTravel::NOfferCache {

class TService;

class TPromoService {
public:
    struct TUserListKey {
        TString PassportId;
        bool operator<(const TUserListKey& rhs) const;
        bool operator==(const TUserListKey& rhs) const;
        size_t Hash() const;
    };

    struct TUserWithOrderTypeListKey {
        TPassportUid PassportId;
        NTravelProto::NOrderType::EOrderType OrderType;
        bool operator<(const TUserWithOrderTypeListKey& rhs) const;
        bool operator==(const TUserWithOrderTypeListKey& rhs) const;
        size_t Hash() const;
    };

    struct TMirWaveSettings {
        TInstant PromoStart;
        TInstant PromoEnd;
        NOrdinalDate::TOrdinalDate FirstCheckIn;
        NOrdinalDate::TOrdinalDate LastCheckOut;
        ui32 MinNights;

        ui32 CashbackPercent;
        ui32 MaxCashback;
    };

    struct TForOfferInternalReq {
        TInstant Now;
        THotelId HotelId;
        NOrdinalDate::TOrdinalDate Date;
        TNights Nights;
        TPriceWithCurrency PriceFromPartnerOffer;
        TMaybe<TPriceWithCurrency> PriceBeforePromoCodes;
        TMaybe<TPriceWithCurrency> PriceAfterPromoCodes;
        NTravelProto::NPromoService::TUserInfo UserInfo;
        NTravelProto::NPromoService::TExperimentInfo ExpInfo;
        NTravelProto::NPromoService::TWhiteLabelInfo WhiteLabelInfo;

        const TPriceWithCurrency& GetLatestPrice() const;
        bool HasWhiteLabel() const;
    };

    TPromoService(const TService& service, const NTravelProto::NOfferCache::TConfig::TPromoService& config);
    ~TPromoService();

    bool IsReady() const;
    void RegisterCounters(NMonitor::TCounterSource& source);

    void Start();
    void Stop();

    const TMirWaveSettings* GetMirCurrentWaveSettings(TInstant now) const;

    bool IsMirPromoAvailableForHotel(TInstant now, const TComplexHotelId& complexHotelId) const;

    bool IsPlusAvailableForHotel(TInstant now, const TComplexHotelId& complexHotelId) const;

    bool ShowBlackFriday2021BadgeForHotel(TInstant now, const TComplexHotelId& complexHotelId) const;

    void DeterminePromosForOffer(const NTravelProto::NPromoService::TDeterminePromosForOfferReq& req,
         NTravelProto::NPromoService::TDeterminePromosForOfferRsp* rsp) const;

    void DeterminePromosForOffer(const TForOfferInternalReq& req,
         NTravelProto::NPromoService::TDeterminePromosForOfferRsp* rsp) const;

    void GetActivePromos(const NTravelProto::NPromoService::TGetActivePromosReq& req,
                         NTravelProto::NPromoService::TGetActivePromosRsp* rsp) const;

    void CalculateDiscountForOffer(const NTravelProto::NPromoService::TCalculateDiscountForOfferReq& req,
                                   NTravelProto::NPromoService::TCalculateDiscountForOfferRsp* rsp) const;

    void CalculateDiscountForOffer(const TForOfferInternalReq& req,
                                   NTravelProto::NPromoService::TCalculateDiscountForOfferRsp* rsp) const;

    void GetWhiteLabelPointsProps(const NTravelProto::NPromoService::TGetWhiteLabelPointsPropsReq& req,
                                  NTravelProto::NPromoService::TGetWhiteLabelPointsPropsRsp* rsp) const;

private:
    struct TTaxi2020Config {
        TPriceVal MinPrice;
        NOrdinalDate::TOrdinalDate StartDate;
        NOrdinalDate::TOrdinalDate LastDate;
        NOrdinalDate::TOrdinalDate MaxCheckIn;
    };

    struct TMirConfig {
        TVector<TMirWaveSettings> Waves;
    };

    struct TPlusDiscount {
        TString EventId;
        ui32 Points;
        ui32 PointsbackPercent;
        ui32 MaxPointsback;
        TMaybe<TInstant> SpecialOfferEndUtc;
        double AdditionalFeePercent;
        ui32 AdditionalFeeValue;
        NTravelProto::NPromoService::EPlusAdditionalFeeEligibility AdditionalFeeEligibility;
        NTravelProto::NPromoService::EYandexPlusEventType EventType;
        i32 Priority;
    };

    struct TPlusDiscountConfig {
        ui32 PointsbackPercent;
        ui32 MaxPointsback;
    };

    struct TPlusAdditionalFeeInfo {
        NTravelProto::NPromoService::EPlusAdditionalFeeEligibility Eligibility;
        double PointsPercent;
        double FeePercent;
    };

    struct TPlusEvent {
        TString EventId;
        TPlusDiscountConfig Discount;
        TMaybe<TInstant> OrderTimeStartUtc;
        TMaybe<TInstant> OrderTimeEndUtc;
        TMaybe<NOrdinalDate::TOrdinalDate> CheckInStart;
        TMaybe<NOrdinalDate::TOrdinalDate> CheckInEnd;
        TMaybe<NOrdinalDate::TOrdinalDate> CheckOutStart;
        TMaybe<NOrdinalDate::TOrdinalDate> CheckOutEnd;
        TMaybe<std::shared_ptr<TYtPersistentConfig<TUserListKey, NTravelProto::NPromoService::TUserListItem>>> YtConfigUserWhitelist;
        TMaybe<std::shared_ptr<TYtPersistentConfig<TUserListKey, NTravelProto::NPromoService::TUserListItem>>> YtConfigUserBlacklist;
        bool LoginRequired;
        bool CheckFirstOrder;
        THashSet<NTravelProto::NOrderType::EOrderType> FirstOrderTypes;
        TMaybe<TString> RequiredExpName;
        TMaybe<TString> RequiredExpValue;
        bool CheckHotelWhiteList;
        THashSet<THotelId> HotelWhiteList;
        TMaybe<std::shared_ptr<TYtPersistentConfig<THotelId, NTravelProto::THotelId>>> YtConfigHotelWhitelist;
        TMaybe<std::shared_ptr<TYtPersistentConfig<THotelId, NTravelProto::THotelId>>> YtConfigHotelBlacklist;
        NTravelProto::NPromoService::EYandexPlusEventType EventType;
        i32 Priority;
    };

    struct TPlusConfig {
        bool Enabled;
        TPlusDiscountConfig OrdinalDiscount;
        TVector<TPlusEvent> PlusEvents;
        TMaybe<std::shared_ptr<TYtPersistentConfig<TUserWithOrderTypeListKey, ru::yandex::travel::user_order_counters::TUserOrderCounterByType>>> YtConfigUserOrderCountersByType;
        TMaybe<std::shared_ptr<TYtMultiValuePersistentConfig<THotelId, NTravelProto::NOrders::NHotelsExtranet::THotelPlusAdditionalFeeItem>>> YtConfigAdditionalFee;
    };

    struct TDiscountConfig {
        bool Enabled;
        THashMap<THotelId, ui32/*Discount*/> Hotels;
    };

    struct TBlackFriday2021Config {
        bool Enabled;
        TInstant BadgeStart;
        TInstant BadgeEnd;
    };

    struct TYandexEda2022Config {
        bool Enabled;
        TInstant EventStart;
        TInstant EventEnd;
        ui32 PromoCodeNominal;
        TMaybe<std::shared_ptr<TYtPersistentConfig<THotelId, NTravelProto::THotelId>>> YtConfigHotelWhitelist;
        TMaybe<std::shared_ptr<TYtPersistentConfig<THotelId, NTravelProto::THotelId>>> YtConfigHotelBlacklist;
    };

    const TService& Service;
    TWhiteLabelPromoService WhiteLabelPromoService;

    const TTaxi2020Config Taxi2020Config;
    const TMirConfig MirConfig;
    const TPlusConfig PlusConfig;
    const TDiscountConfig DiscountConfig;
    const TBlackFriday2021Config BlackFriday2021Config;
    const TYandexEda2022Config YandexEda2022Config;

    TYtPersistentConfig<THotelId, NTravelProto::NConfig::TMirWhitelistRecord> YtConfigMirPromoWhitelist;
    TYtPersistentConfig<THotelId, NTravelProto::NConfig::TBlackFriday2021Hotel> YtConfigBlackFriday2021Hotels;

    TTaxi2020Config MakeConfig(const NTravelProto::NOfferCache::TConfig::TPromoService::TTaxi2020& pbCfg) const;
    TMirConfig MakeConfig(const NTravelProto::NOfferCache::TConfig::TPromoService::TMir& pbCfg) const;
    TPlusConfig MakeConfig(const NTravelProto::NOfferCache::TConfig::TPromoService::TPlus& pbCfg) const;
    TDiscountConfig MakeConfig(const NTravelProto::NOfferCache::TConfig::TPromoService::TDiscount& pbCfg) const;
    TBlackFriday2021Config MakeConfig(const NTravelProto::NOfferCache::TConfig::TPromoService::TBlackFriday2021& pbCfg) const;
    TYandexEda2022Config MakeConfig(const NTravelProto::NOfferCache::TConfig::TPromoService::TYandexEda2022& pbCfg) const;

    void FillTaxi2020Status(const TForOfferInternalReq& req, NTravelProto::NPromoService::TTaxi2020Status* res) const;
    void FillMirPromoStatus(const TForOfferInternalReq& req, NTravelProto::NPromoService::TMirPromoStatus* res) const;
    void FillPlusStatus(const TForOfferInternalReq& req, NTravelProto::NPromoService::TYandexPlusStatus* res) const;
    void FillBlackFriday2021Status(const TForOfferInternalReq& req, NTravelProto::NPromoService::TBlackFriday2021Status* res) const;
    void FillYandexEda2022Status(const TForOfferInternalReq& req, NTravelProto::NPromoService::TYandexEda2022Status* res) const;
    void FillWhiteLabelStatus(const TForOfferInternalReq& req, NTravelProto::NPromoService::TWhiteLabelStatus* res) const;
    TPlusDiscount GetPlusDiscount(const TForOfferInternalReq& req) const;
    TPlusAdditionalFeeInfo GetPlusAdditionalFeeInfo(const TForOfferInternalReq& req) const;
    bool IsPlusEventApplicable(const TForOfferInternalReq& req, TUserListKey userListKey, TPassportUid parsedPassportId, const TPlusEvent& event) const;

    void FillTaxi2020Params(TInstant now, NTravelProto::NPromoService::TTaxi2020PromoParams* res) const;
    void FillMirPromoParams(TInstant now, NTravelProto::NPromoService::TMirPromoParams* res) const;
    void FillPlusParams(TInstant now, NTravelProto::NPromoService::TYandexPlusPromoParams* res) const;
    void FillBlackFridayParams(TInstant now, NTravelProto::NPromoService::TBlackFridayParams* res) const;

    bool IsUserEligibleByFirstOrderTypes(TPassportUid passportId,
                                         const THashSet<NTravelProto::NOrderType::EOrderType>& orderTypesToCheck,
                                         bool useUserOrderTypesFromRequest,
                                         const ::google::protobuf::RepeatedField<int>& userOrderTypesFromRequest) const;

    static ui32 GetPlusPoints(TPriceVal value, TPlusDiscountConfig discount, TPlusAdditionalFeeInfo additionalFeeInfo);

    static ui32 GetPlusAdditionalFeeValue(TPriceVal value, TPlusAdditionalFeeInfo additionalFeeInfo);

    bool HotelHasBoyPartner(const TComplexHotelId& complexHotelId) const;

    bool IsBlackFriday2021Active(TInstant now) const;
};


}//namespace NTravel::NOfferCache

template <>
struct THash<NTravel::NOfferCache::TPromoService::TUserListKey> {
    inline size_t operator() (const NTravel::NOfferCache::TPromoService::TUserListKey& v) const { return v.Hash(); }
};

template <>
struct THash<NTravel::NOfferCache::TPromoService::TUserWithOrderTypeListKey> {
    inline size_t operator() (const NTravel::NOfferCache::TPromoService::TUserWithOrderTypeListKey& v) const { return v.Hash(); }
};
