#include "read_request_processor.h"

#include "service.h"
#include "offer_blender.h"

#include <travel/hotels/proto/redir_add_info/redir_add_info.pb.h>

#include <travel/hotels/proto2/bus_messages.pb.h>

#include <travel/hotels/lib/cpp/protobuf/tools.h>
#include <travel/hotels/lib/cpp/util/host_filter.h>

#include <util/digest/city.h>
#include <util/generic/set.h>
#include <util/generic/maybe.h>
#include <util/generic/algorithm.h>
#include <util/string/vector.h>

#include <type_traits>
#include <cmath>

#define JOB_LOG LogPrefix

namespace NTravel {
namespace NOfferCache {

template<typename T>
T GetOrElse(T v) {
  return v;
}

template<typename T, typename... Args>
T GetOrElse(T first, Args... args) {
    return first ? first : GetOrElse(args...);
}

NOrdinalDate::TOrdinalDate GetMinAllowedDate(TInstant after) {
    return NOrdinalDate::FromInstant(after + TDuration::Hours(12));
}

NOrdinalDate::TOrdinalDate GetDefaultDateTomorrow(TInstant now) {
    // Завтра по москве
    return NOrdinalDate::FromInstant(now) + 1;
}

TReadRequestProcessor::TReadRequestProcessor(const TAtomicSharedPtr<TReadJobStats>& statsPtr, TService& svc, TInstant started, const TString& logPrefix, const TMaybe<TString>& rawHttpRequest)
  : Service(svc)
  , Started(started)
  , Today(NOrdinalDate::FromInstant(Started))
  , LogPrefix(logPrefix)
  , RawHttpRequest(rawHttpRequest)
  , MinAllowedDate(GetMinAllowedDate(Started))
  , StatsPtr(statsPtr)
  , Stats(*StatsPtr)
{
    Stats.Info.MutableProcessingStagesStats()->SetInitMicros(StagesTimer.Step().MicroSeconds());
    Service.GetCounters().NHttpJobs.Inc();
}

TReadRequestProcessor::~TReadRequestProcessor() {
    WriteReqAnsLog();
    Service.GetCounters().NHttpJobs.Dec();
}

void TReadRequestProcessor::Parse(const NTravelProto::NOfferCache::NApi::TReadReq& req) {
    Req = req;
    Exp.Init(Req.GetExp(), Req.GetStrExp(), Service.Config().GetExperiments().GetForceEnableExp(), Service.Config().GetExperiments().GetForceDisableExp());

    ParseAttribution();
    SourceCounters = Service.GetSourceCounters(Attribution.GetOfferCacheClientId());
    SourceCounters->NRequests.Inc();
    if (Req.GetUseSearcher() && !Req.HasRequestId()) {
        throw yexception() << "When UseSearcher is set RequestId should be given";
    }
    if (!Req.GetUseCache() && !Req.GetUseSearcher()) {
        throw yexception() << "UseCache or UseSearcher should be set";
    }
    ParseSubKey();
    ParseSubKeyRangeAndDefaultSubKey();
    ConvertSHotelIds();
    ParseEnabledOperatorsAndPartners();
    size_t subHotelCount = 0;
    ParseHotelIds(&subHotelCount);
    InitProgress();
    DetermineRequestType();
    InitUserInfo();
    AdjustSubKeyForUser();
    SourceCounters->RequestHotelCount.Update(ComplexHotelIds.size());
    SourceCounters->RequestSubHotelCount.Update(subHotelCount);
}

NOrdinalDate::TOrdinalDate TReadRequestProcessor::CropDate(NOrdinalDate::TOrdinalDate dt) const {
    // Сделать дату не меньше MinDate
    return (dt < MinAllowedDate) ? MinAllowedDate : dt;
}

TPermalink TReadRequestProcessor::GetClusterPermalink(TPermalink permalink) const {
    if (auto* clusterPermalink = ClusterPermalinks.FindPtr(permalink)) {
        return *clusterPermalink;
    }
    return permalink;
}

void TReadRequestProcessor::ParseAttribution() {
    auto& attribution = Req.GetAttribution();
    Attribution.SetUtmSource(attribution.GetUtmSource());
    Attribution.SetUtmMedium(attribution.GetUtmMedium());
    Attribution.SetUtmCampaign(attribution.GetUtmCampaign());
    Attribution.SetUtmTerm(attribution.GetUtmTerm());
    Attribution.SetUtmContent(attribution.GetUtmContent());
    Attribution.SetYandexUid(attribution.GetYandexUid());
    Attribution.SetSerpReqId(GetOrElse(attribution.GetSerpReqId_Override(), attribution.GetSerpReqId()));
    Attribution.SetSearchQuery(attribution.GetSearchQuery());
    Attribution.SetPassportUid(attribution.GetPassportUid());
    Attribution.SetUuid(attribution.GetUuid());
    ParseIntList("TestIds", attribution.GetTestIds(), Attribution.MutableIntTestIds());
    ParseIntList("TestBuckets", attribution.GetTestBuckets(), Attribution.MutableIntTestBuckets());
    ParseIntList("PortalTestIds", attribution.GetPortalTestIds(), Attribution.MutableIntPortalTestIds());
    ParseIntList("PortalTestBuckets", attribution.GetPortalTestBuckets(), Attribution.MutableIntPortalTestBuckets());
    Attribution.SetRequestRegion(attribution.GetRequestRegion());
    Attribution.SetUserRegion(GetOrElse(attribution.GetUserRegion(), -1));
    Attribution.SetICookie(attribution.GetICookie());
    Attribution.SetGeoClientId(attribution.GetGeoClientId());
    Attribution.SetGeoOrigin(attribution.GetGeoOrigin());
    auto ocClient = Service.FindHttpOfferCacheClient(attribution.GetGeoOrigin(), attribution.GetGeoClientId());
    Attribution.SetOfferCacheClientId(ocClient.GetOfferCacheClientId());
    Attribution.SetSurface(ocClient.GetSurface());
    Attribution.SetUserDevice(attribution.GetUserDevice());
    Attribution.SetGclid(attribution.Getgclid());
    Attribution.SetYaTravelReqId(attribution.GetYaTravelReqId());
    Attribution.SetYaTravelDebugId(attribution.GetYaTravelDebugId());
    Attribution.SetYtpReferer(attribution.GetYtpReferer());
    Attribution.SetYclid(attribution.GetYclid());
    Attribution.SetFBclid(attribution.GetFBclid());
    Attribution.SetMetrikaClientId(attribution.GetMetrikaClientId());
    Attribution.SetMetrikaUserId(attribution.GetMetrikaUserId());
    Attribution.SetIsStaffUser(attribution.GetIsStaffUser());
    Attribution.SetClid(attribution.GetClid());
    Attribution.SetAffiliateClid(attribution.GetAffiliateClid());
    Attribution.SetAdmitadUid(attribution.GetAdmitadUid());
    Attribution.SetTravelpayoutsUid(attribution.GetTravelpayoutsUid());
    Attribution.SetVid(attribution.GetVid());
    Attribution.SetAffiliateVid(attribution.GetAffiliateVid());
    Attribution.SetIsPlusUser(attribution.GetIsPlusUser());
    Attribution.SetSearchPagePollingId(attribution.GetSearchPagePollingId());
    Attribution.SetReferralPartnerId(attribution.GetReferralPartnerId());
    Attribution.SetReferralPartnerRequestId(attribution.GetReferralPartnerRequestId());
    Stats.Info.SetOfferCacheClientId(Attribution.GetOfferCacheClientId());
}

void TReadRequestProcessor::ParseIntList(const TString& name, const TString& src, ::google::protobuf::RepeatedField< ::google::protobuf::int64 >* dst) const {
    TStringBuf srcBuf(src);
    TString restInts;
    while (!srcBuf.empty()) {
        TStringBuf origPart = srcBuf.NextTok(',');
        TStringBuf part = origPart;
        if (part.StartsWith('"') && part.EndsWith('"')) {
            part.Skip(1);
            part.Chop(1);
        }
        i64 v;
        if (TryFromString(part, v)) {
            dst->Add(v);
        } else {
            if (restInts) {
                restInts += ",";
            }
            restInts += origPart;
        }
    }
    if (restInts) {
        WARNING_LOG << JOB_LOG << "Not parsed " << name << ": " << restInts << Endl;
    }
}

void TReadRequestProcessor::ParseSubKey() {
    bool haveDateOrNights = Req.HasDate() || Req.HasNights();
    bool haveCheckInOrOut = Req.HasCheckInDate() || Req.HasCheckOutDate();
    if (haveDateOrNights && haveCheckInOrOut) {
        throw yexception() << "Date/Nights cannot be specified along with CheckInDate/CheckOutDate";
    }
    if (haveDateOrNights) {
        if (Req.HasDate()) {
            SubKey.Date = NOrdinalDate::FromString(Req.GetDate());
        }
        if (Req.HasNights()) {
            if (Req.GetNights() > Max<TNights>()) {
                WARNING_LOG << JOB_LOG << "Too much nights: " << Req.GetNights() << Endl;
                SourceCounters->NRequestsWrongNights.Inc();
                SubKey.Nights = 1;
            }  else if (Req.GetNights() < 0) {
                WARNING_LOG << JOB_LOG << "Negative nights: " << Req.GetNights() << Endl;
                SourceCounters->NRequestsWrongNights.Inc();
                SubKey.Nights = 1;
            } else {
                SubKey.Nights = Req.GetNights();
            }
        }
    }
    if (haveCheckInOrOut) {
        if (Req.HasCheckInDate()){
            SubKey.Date = NOrdinalDate::FromString(Req.GetCheckInDate());
        }
        if (Req.HasCheckOutDate()) {
            auto checkOut = NOrdinalDate::FromString(Req.GetCheckOutDate());
            if (!Req.HasCheckInDate()) {
                // Мы не умеем искать с жёстко заданным checkOut, но без checkIn,
                // поэтому будем считать, что checkIn - за день до checkOut
                SubKey.Date = checkOut - 1;
            }
            if (checkOut <= SubKey.Date) {
                throw yexception() << "CheckInDate should be before CheckOutDate";
            }
            auto diff = checkOut - SubKey.Date;
            if (diff > Max<TNights>()) {
                throw yexception() << "Too big distance between CheckInDate and CheckOutDate";
            }
            SubKey.Nights = diff;
        }
    }
    // Cropping
    if (!Req.GetAllowPastDates() && SubKey.Date != NOrdinalDate::g_DateZero) {
        if (SubKey.Nights != g_NightsZero) {
            if (NOrdinalDate::TOrdinalDate(SubKey.Date + SubKey.Nights) < Today) {
                SubKey.Date = NOrdinalDate::g_DateZero;
                SourceCounters->NRequestsCheckInAndOutInPast.Inc();
            }
        } else {
            if (SubKey.Date < Today) {
                SubKey.Date = NOrdinalDate::g_DateZero;
                SourceCounters->NRequestsCheckInInPast.Inc();
            }
        }
    }
    // For logging purposes, fill all possible fields
    if (SubKey.Date != NOrdinalDate::g_DateZero) {
        Req.SetDate(NOrdinalDate::ToString(SubKey.Date));
        Req.SetCheckInDate(Req.GetDate());
        if (SubKey.Nights != g_NightsZero) {
            Req.SetCheckOutDate(NOrdinalDate::ToString(SubKey.Date + SubKey.Nights));
        }
    }
    if (SubKey.Nights != g_NightsZero) {
        Req.SetNights(SubKey.Nights);
    }
    // Ages
    if (Req.HasAges()) {
        if (Req.GetAges().Contains("NaN")) {
            WARNING_LOG << JOB_LOG << "Got NaN in ages: " << Req.GetAges() << Endl;
            SourceCounters->NRequestsWrongAges.Inc();
            SubKey.Ages = TAges::FromOccupancyString("2");
        } else {
            SubKey.Ages = TAges::FromAgesString(Req.GetAges(), true); // See https://st.yandex-team.ru/SERP-77787#5c4ee71dbef18f002125d7f7
        }
    } else {
        SubKey.Ages = TAges::Empty();
    }
    IsInitialRequest = !SubKey.IsComplete();
}

void TReadRequestProcessor::ParseSubKeyRangeAndDefaultSubKey() {
    const NTravelProto::NOfferCache::TConfig::TOther::TDetermineKeyRange& keyRangeConfig = Service.Config().GetOther().GetDetermineKeyRange();
    if (SubKey.Date == NOrdinalDate::g_DateZero) {
        if (Req.HasRestrictKeyRelDateFrom()) {
            SubKeyRange.DateFrom = CropDate(Today + Req.GetRestrictKeyRelDateFrom());
        } else if (keyRangeConfig.HasRelDateFrom()) {
            SubKeyRange.DateFrom = CropDate(Today + keyRangeConfig.GetRelDateFrom());
        } else {
            SubKeyRange.DateFrom = CropDate(Today);
        }
        if (Req.HasRestrictKeyRelDateTo()) {
            SubKeyRange.DateTo = CropDate(Today + Req.GetRestrictKeyRelDateTo());
        } else if (keyRangeConfig.HasRelDateTo()) {
            SubKeyRange.DateTo = CropDate(Today + keyRangeConfig.GetRelDateTo());
        }
    } else {
        SubKeyRange.DateFrom = SubKey.Date;
        SubKeyRange.DateTo = SubKey.Date;
    }
    if (SubKey.Nights == g_NightsZero) {
        if (Req.HasRestrictKeyNightsFrom()) {
            SubKeyRange.NightsFrom = Req.GetRestrictKeyNightsFrom();
        } else if (keyRangeConfig.HasNightsFrom()) {
            SubKeyRange.NightsFrom = keyRangeConfig.GetNightsFrom();
        }
        if (Req.HasRestrictKeyNightsTo()) {
            SubKeyRange.NightsTo = Req.GetRestrictKeyNightsTo();
        } else if (keyRangeConfig.HasNightsTo()) {
            SubKeyRange.NightsTo = keyRangeConfig.GetNightsTo();
        }
    } else {
        SubKeyRange.NightsFrom = SubKey.Nights;
        SubKeyRange.NightsTo = SubKey.Nights;
    }
    if (SubKey.Ages.IsEmpty()) {
        if (Req.HasRestrictKeyAdultsFrom()) {
            SubKeyRange.AdultsFrom = Req.GetRestrictKeyAdultsFrom();
        } else if (keyRangeConfig.HasAdultsFrom()) {
            SubKeyRange.AdultsFrom = keyRangeConfig.GetAdultsFrom();
        }
        if (Req.HasRestrictKeyAdultsTo()) {
            SubKeyRange.AdultsTo = Req.GetRestrictKeyAdultsTo();
        } else if (keyRangeConfig.HasAdultsTo()) {
            SubKeyRange.AdultsTo = keyRangeConfig.GetAdultsTo();
        }
        if (Req.HasRestrictKeyAllowChildren()) {
            SubKeyRange.AllowChildren = Req.GetRestrictKeyAllowChildren();
        } else if (keyRangeConfig.HasAllowChildren()) {
            SubKeyRange.AllowChildren = keyRangeConfig.GetAllowChildren();
        }
    } else {
        SubKeyRange.ExactAges = SubKey.Ages;
    }

    DefaultSubKey.Date = GetDefaultDateTomorrow(Started + TDuration::Hours(Service.Config().GetOther().GetDateSwitchHours()));
    size_t adultCount = 2;
    DefaultSubKey.Nights = 1;

    // Adjust default subkey for mir promo
    auto currentWaveSettings = Service.GetPromoService().GetMirCurrentWaveSettings(Started);
    if (currentWaveSettings) {
        if (Req.GetAdjustDefaultSubKeyForMirPromoAlways() ||
            (Req.GetAdjustDefaultSubKeyForMirPromo() && IsMirPromoAvailable(MainPermalink)) ||
            (Req.GetAdjustDefaultSubKeyForMirPromoTop10() && IsMirPromoAvailableTop10())) {
            SubKeyRange.DateFrom = currentWaveSettings->FirstCheckIn;
            SubKeyRange.DateTo = currentWaveSettings->LastCheckOut - currentWaveSettings->MinNights;
            SubKeyRange.NightsFrom = currentWaveSettings->MinNights;
            SubKeyRange.NightsTo = SubKeyRange.NightsFrom;
            DefaultSubKey.Nights = SubKeyRange.NightsFrom;
        }
    }

    // Adjust default subkey to match within SubKeyRange
    if (SubKeyRange.DateFrom != NOrdinalDate::g_DateZero) {
        DefaultSubKey.Date = Max(DefaultSubKey.Date, SubKeyRange.DateFrom);
    }
    if (SubKeyRange.DateTo != NOrdinalDate::g_DateZero) {
        DefaultSubKey.Date = Min(DefaultSubKey.Date, SubKeyRange.DateTo);
    }
    if (SubKeyRange.NightsFrom != 0) {
        DefaultSubKey.Nights = Max(DefaultSubKey.Nights, SubKeyRange.NightsFrom);
    }
    if (SubKeyRange.NightsTo != 0) {
        DefaultSubKey.Nights = Min(DefaultSubKey.Nights, SubKeyRange.NightsTo);
    }
    if (SubKeyRange.AdultsFrom != 0) {
        adultCount = Max(adultCount, SubKeyRange.AdultsFrom);
    }
    if (SubKeyRange.AdultsTo != 0) {
        adultCount = Min(adultCount, SubKeyRange.AdultsTo);
    }
    DefaultSubKey.Ages = TAges::FromAdultCount(adultCount);
}


void TReadRequestProcessor::ConvertSHotelIds() {
    for (const TString& sHotelIdStr: Req.GetSHotelId()) {
        // Сконвертируем SHotelId в обычные HotelId, это полезно для удобства чтения reqans лога
        TStringBuf sHotelId = sHotelIdStr;
        ::NTravelProto::NOfferCache::NApi::TComplexHotelId* pbCHotelId = Req.AddHotelId();
        pbCHotelId->SetPermalink(FromString(sHotelId.NextTok('~')));

        while (sHotelId) {
            TStringBuf field = sHotelId.NextTok('~');
            if (field.empty()) {
                continue;
            }
            if (field.StartsWith("ytravel")) {
                // Partner subhotel id
                auto p = pbCHotelId->AddPartners();
                p->SetPartnerCode(TString(field.NextTok('.')));
                p->SetOriginalId(TString(field));
            } else if (field.StartsWith("f.")) {
                // Flag
                field.NextTok('.');
                if (field == "similar"sv) {
                    pbCHotelId->SetIsSimilar(true);
                } else if (field == "single_org"sv) {
                    pbCHotelId->SetIsSingleOrg(true);
                } else {
                    SourceCounters->NSubReqWrongField.Inc();
                    ERROR_LOG << JOB_LOG << "Unknown flag '" << field << "' in SHotelId=" << sHotelIdStr << Endl;
                }
            } else {
                SourceCounters->NSubReqWrongField.Inc();
                ERROR_LOG << JOB_LOG << "Unknown field '" << field << "' in SHotelId=" << sHotelIdStr << Endl;
            }
        }
    }
}

double TReadRequestProcessor::GetMirPromoAvailableTop10Ratio() const {
    int cntMir = 0;
    int cntSeen = 0;
    for (const auto& [permalink, pos]: PermalinkOrder) {
        if (!SimilarPermalinks.contains(permalink) && pos < 10) {
            ++cntSeen;
            if (IsMirPromoAvailable(permalink)) {
                ++cntMir;
            }
        }
    }
    return cntSeen == 0 ? 0 : static_cast<double>(cntMir) / cntSeen;
}

bool TReadRequestProcessor::IsMirPromoAvailableTop10() const {
    return GetMirPromoAvailableTop10Ratio() > 0;
}

void TReadRequestProcessor::ParseHotelIds(size_t* subHotelCount) {
    const bool testModeCacheHit = Req.GetTestMode() == NTravelProto::NOfferCache::NApi::TTestMode::CacheHit;
    MainPermalink = 0;
    MainPermalinkIsBlacklisted = false;
    MainPermalinkIsGreylistedNoOffers = false;
    for (const auto& pbCHotelId: Req.GetHotelId()) {
        if (!pbCHotelId.HasPermalink()) {
            WARNING_LOG << JOB_LOG << "Permalink should be specified for each hotel!" << Endl;
            continue;
        }
        const TPermalink permalink = pbCHotelId.GetPermalink();
        const bool duplicate = ComplexHotelIds.contains(permalink);
        auto& hotelIds = ComplexHotelIds[permalink].HotelIds;// Уже мог существовать, бывают дубли
        const auto clusterPermalink = Service.PermalinkToClusterMapper().GetClusterPermalink(permalink);
        ClusterPermalinks.emplace(permalink, clusterPermalink);
        PermalinkOrder.insert(std::make_pair(permalink, PermalinkOrder.size()));// Дубли пропускаются
        // В similar добавляем только если все дубликаты similar
        if (!duplicate && pbCHotelId.GetIsSimilar()) {
            SimilarPermalinks.insert(permalink);
        }
        if (duplicate && !pbCHotelId.GetIsSimilar()) {
            SimilarPermalinks.erase(permalink);
        }
        if (pbCHotelId.GetIsSingleOrg()) {
            if (!MainPermalink) {
                MainPermalink = permalink;
            }
            SingleOrgPermalinks.insert(permalink);
        }
        bool recovered = false;
        bool permalinkIsFullyBlacklisted = false;
        bool permalinkIsPartiallyBlacklisted = false;
        bool permalinkWizardBanned = false;

        if (testModeCacheHit) {
            THotelId hId;
            hId.OriginalId = ToString(permalink) + "-HOTELS-4427";
            for (auto pId: EnabledPartners) {
                hId.PartnerId = pId;
                hotelIds.insert(hId);
            }
        } else if (Service.IsWizardBanned(clusterPermalink)) {
            permalinkWizardBanned = true;
        } else {
            for (const auto& p: pbCHotelId.GetPartners()) {
                THotelId hId;
                if (!HotelIdFromApiProto(p, &hId)) {
                    continue;
                }
                hotelIds.insert(hId);
            }
            if (pbCHotelId.PartnersSize() == 0) {
                // No hotelIds - try to restore by permalink
                if (const auto mapping = Service.PermalinkToOriginalIdsMapper().GetMapping(clusterPermalink)) {
                    const auto& partnerIds = mapping->PartnerIds;
                    Y_ASSERT(!partnerIds.empty());
                    hotelIds.insert(std::cbegin(partnerIds), std::cend(partnerIds));
                    recovered = true;
                }
            }
            if (!hotelIds.empty() && !Req.GetIgnoreBlacklist()) {
                if (auto blacklistedOriginalIds = Service.HotelsBlacklist().GetMapping(clusterPermalink)) {
                    permalinkIsFullyBlacklisted = true;
                    for (auto hIt = hotelIds.begin(); hIt != hotelIds.end(); ) {
                        if (blacklistedOriginalIds->Contains(*hIt)) {
                            SourceCounters->NSubReqBlacklisted.Inc();
                            permalinkIsPartiallyBlacklisted = true;
                            hotelIds.erase(hIt++);
                        } else {
                            permalinkIsFullyBlacklisted = false;
                            ++hIt;
                        }
                    }
                }
            }
            if (!hotelIds.empty() && !Req.GetIgnoreGreylist()) {
                if (auto greylistedOriginalIds = Service.HotelsGreylist().GetMapping(clusterPermalink)) {
                    bool permalinkIsFullyGreylisted = true;
                    for (auto hIt = hotelIds.begin(); hIt != hotelIds.end(); ++hIt) {
                        if (!greylistedOriginalIds->Contains(*hIt)) {
                            permalinkIsFullyGreylisted = false;
                            break;
                        }
                    }
                    if (permalinkIsFullyGreylisted) {
                        Stats.Info.AddGreylistedPermalinks(permalink);
                        SourceCounters->NPermalinksGreylisted.Inc();
                        FullyGreylistedPermalinks.insert(permalink);
                    }
                }
            }
        }
        if (permalinkWizardBanned) {
            ComplexHotelIds.erase(permalink); // hotelIds is now invalid
            Stats.Info.AddWizardBannedPermalinks(permalink);
            SourceCounters->NPermalinksWizardBanned.Inc();
            continue;
        }
        if (permalinkIsFullyBlacklisted) {
            if (permalink == MainPermalink) {
                MainPermalinkIsBlacklisted = true;
            }
            Stats.Info.AddBlacklistedPermalinks(permalink);
            SourceCounters->NPermalinksBlacklisted.Inc();
            continue;
        } else if (permalinkIsPartiallyBlacklisted) {
            SourceCounters->NPermalinksPartiallyBlacklisted.Inc();
        } else if (hotelIds.empty()) {
            SourceCounters->NPermalinksEmpty.Inc();
        } else {
            SourceCounters->NPermalinksOK.Inc();
        }

        if (!Req.GetIgnoreWhitelist() && !testModeCacheHit) {
            Service.HotelsWhitelist().FilterPartnerHotels(clusterPermalink, &hotelIds);
        }

        if (hotelIds.empty()) {
            SourceCounters->NHotelNoPartnerIdsNoTravelId.Inc();
        } else {
            if (recovered) {
                SourceCounters->NHotelNoPartnerIdsNoTravelIdRecovered.Inc();
            } else {
                SourceCounters->NHotelNoTravelId.Inc();
            }
        }
        SourceCounters->NSubReq += hotelIds.size();// Возможны искажения от дублей пермалинков,ну и ладно
        (*subHotelCount) += hotelIds.size();
    }
    if (!Req.GetForManyOrg() && SingleOrgPermalinks.empty()) {
        // Если в запрос нет спец флага, и не указано ни одного "1орг" пермалинка, то все, кроме "похожих" - 1орг
        for (auto pIt = ComplexHotelIds.begin(); pIt != ComplexHotelIds.end(); ++pIt) {
            if (!SimilarPermalinks.contains(pIt->first)) {
                SingleOrgPermalinks.insert(pIt->first);
            }
        }
    }
}

void TReadRequestProcessor::ParseEnabledOperatorsAndPartners() {
    if (Req.GetCompactResponseForCalendar() && Req.EnablePartnerIdSize() == 0 && Req.EnableOpIdSize() == 0) {
        EnabledPartners.insert(NTravelProto::PI_TRAVELLINE);
        EnabledOperators = Service.GetOperatorsForPartners(EnabledPartners);
        return;
    }

    if (Req.EnablePartnerIdSize() > 0) {
        for (auto pId: Req.GetEnablePartnerId()) {
            EnabledPartners.insert((EPartnerId)pId);
        }
        EnabledOperators = Service.GetOperatorsForPartners(EnabledPartners);
        FilterOnlyBoYOperatorsAndPartners();
        return;
    }

    if (Req.EnableOpIdSize() > 0) {
        for (auto opId: Req.GetEnableOpId()) {
            EnabledOperators.insert((EOperatorId)opId);
        }
    } else {
        EnabledOperators = Service.GetEnabledOperatorsDefault();
        for (auto opId: Req.GetAddOpId()) {
            EnabledOperators.insert((EOperatorId)opId);
        }
        for (auto opId: Req.GetBanOpId()) {
            EnabledOperators.erase((EOperatorId)opId);
        }
    }
    EnabledPartners = Service.GetPartnersForOperators(EnabledOperators);

    BoYMode = Req.HasBoYPartner();
    if (BoYMode) {
        EPartnerId partnerId = (EPartnerId)Req.GetBoYPartner();
        if (EnabledPartners.contains(partnerId)) {
            EnabledPartnersBoY.insert(partnerId);
            EnabledOperatorsBoY = Service.GetOperatorsForPartners(EnabledPartnersBoY);
        } else {
            WARNING_LOG << JOB_LOG << "BoY partner " << partnerId << " is requested, but it is not enabled" << Endl;
        }
    }
    FilterOnlyBoYOperatorsAndPartners();
}

void TReadRequestProcessor::FilterOnlyBoYOperatorsAndPartners() {
    if (!Req.GetOnlyBoYOffers()) {
        return;
    }
    if (BoYMode) {
        throw yexception() << "Can't use OnlyBoYOffers and BoYMode together";
    }

    THashSet<EPartnerId> filteredPartners;
    for (auto partner: EnabledPartners) {
        if (Service.IsBoYPartner(partner)) {
            filteredPartners.insert(partner);
        }
    }
    EnabledPartners = filteredPartners;
    EnabledOperators = Service.GetOperatorsForPartners(EnabledPartners);
}

void TReadRequestProcessor::InitProgress() {
    THashSet<EPartnerId> allPartnerIds;
    for (auto cIt = ComplexHotelIds.begin(); cIt != ComplexHotelIds.end(); ++cIt) {
        for (auto hIt = cIt->second.HotelIds.begin(); hIt != cIt->second.HotelIds.end(); ++hIt) {
            allPartnerIds.insert(hIt->PartnerId);
        }
    }
    TotalPartnerCount = 0;
    for (const EPartnerId& pid: allPartnerIds) {
        if (EnabledPartners.contains(pid)) {
            ++TotalPartnerCount;
        }
    }
}

void TReadRequestProcessor::DetermineRequestType() {
    if (Req.GetTestMode() != NTravelProto::NOfferCache::NApi::TTestMode::None) {
        RequestType = ERequestType::Test;
        return;// No counters!
    }

    if (Req.GetUseCache()) {
        if (Req.GetUseSearcher()) {
            if (Req.GetFull()) {
                RequestType = ERequestType::SearcherFull;
                SourceCounters->NRequestsCacheSearcherFull.Inc();
            } else {
                RequestType = ERequestType::Searcher;
                SourceCounters->NRequestsCacheSearcher.Inc();
            }
        } else {
            if (Req.GetFull()) {
                RequestType = ERequestType::Full;
                SourceCounters->NRequestsCacheFull.Inc();
            } else {
                RequestType = ERequestType::Initial;
                SourceCounters->NRequestsCache.Inc();
            }
        }
    } else {
        Y_ASSERT(Req.GetUseSearcher());
        RequestType = ERequestType::OnlySearcher;
        SourceCounters->NRequestsSearcher.Inc();
    }
}

void TReadRequestProcessor::InitUserInfo() {
    TYandexUid yandexUid;
    if (Req.GetAttribution().HasYandexUid() && TryFromString(Req.GetAttribution().GetYandexUid(), yandexUid)) {
        UserInfo.YandexUid = yandexUid;
    }
    TPassportUid passportUid;
    if (Req.GetAttribution().HasPassportUid() && TryFromString(Req.GetAttribution().GetPassportUid(), passportUid)) {
        UserInfo.PassportUid = passportUid;
        UserInfo.ConfirmedHotelOrderCount = Service.GetUserConfirmedHotelOrderCount(passportUid);
    }
    UserInfo.UserIdentifier = TUserIdentifier::Create(UserInfo.YandexUid, UserInfo.PassportUid);
}

EPermalinkType TReadRequestProcessor::GetPermalinkType(TPermalink permalink) const {
    if (permalink == MainPermalink) {
        return EPermalinkType::Main;
    }
    if (SimilarPermalinks.contains(permalink)) {
        return EPermalinkType::Similar;
    }
    return EPermalinkType::Usual;
}

void TReadRequestProcessor::CheckPermalinkSubHotels(TPermalink permalink, bool* isBoY, bool* isBoYDirect) const {
    *isBoY = false;
    *isBoYDirect = false;
    auto it = ComplexHotelIds.find(permalink);
    if (it == ComplexHotelIds.end()) {
        return;
    }
    for (const THotelId& hotelId: it->second.HotelIds) {
        if (Service.IsBoYPartner(hotelId.PartnerId)) {
            *isBoY = true;
            if (Service.IsBoYDirectPartner(hotelId.PartnerId)) {
                *isBoYDirect = true;
            }
        }
    }
}

bool TReadRequestProcessor::IsMirPromoAvailable(TPermalink permalink) const {
    auto it = ComplexHotelIds.find(permalink);
    if (it == ComplexHotelIds.end()) {
        return false;
    }
    return Service.GetPromoService().IsMirPromoAvailableForHotel(Started, it->second);
}

void TReadRequestProcessor::Process(const TReadRequestProcessor::TOnResponse& onResp) {
    if (Req.GetUseCache()) {
        DEBUG_LOG << JOB_LOG << "Searching in cache" << Endl;
        SearchInCache();
    } else {
        SubKey.Merge(DefaultSubKey);
    }
    Stats.Info.MutableProcessingStagesStats()->SetSearchInCacheMicros(StagesTimer.Step().MicroSeconds());
    bool delayedReply = Req.GetDebug() && Req.GetDebugDelayedReply();
    if (!delayedReply) {
        Reply(onResp);
    }
    if (Req.GetUseSearcher()) {
        DoSearcherRequest();
    }
    Stats.Info.MutableProcessingStagesStats()->SetDoSearcherRequestMicros(StagesTimer.Step().MicroSeconds());
    if (delayedReply) {
        Reply(onResp);
    }
}

void TReadRequestProcessor::Reply(const TReadRequestProcessor::TOnResponse& onResp) {
    Resp.SetCurrency(Req.GetCurrency());
    Resp.SetDate(NOrdinalDate::ToString(SubKey.Date));
    Resp.SetNights(SubKey.Nights);
    Resp.SetCheckOutDate(NOrdinalDate::ToString(SubKey.Date + SubKey.Nights));
    if (Req.HasAges()) {
        // See https://st.yandex-team.ru/SERP-77787
        Resp.SetAges(Req.GetAges());
    } else {
        Resp.SetAges(SubKey.Ages.ToAgesString());
    }
    if (!Req.GetUseSearcher() || IncompletePartners.empty()) {
        Resp.SetIsFinished(true);
        Resp.MutableProgress()->SetOperatorsComplete(TotalPartnerCount);
        Resp.MutableProgress()->SetOperatorsTotal(TotalPartnerCount);

    } else {
        Resp.SetIsFinished(false);
        Resp.MutableProgress()->SetOperatorsComplete((TotalPartnerCount >= IncompletePartners.size()) ?  (TotalPartnerCount - IncompletePartners.size()) : 0);
        Resp.MutableProgress()->SetOperatorsTotal(TotalPartnerCount);
    }
    for (const auto& partner: IncompletePartners) {
        Resp.MutableProgress()->AddPendingPartners(partner);
    }
    for (const auto& partner: CompletePartners) {
        Resp.MutableProgress()->AddFinishedPartners(partner);
    }
    Resp.SetShowEmptyManyOrgForm(!(Req.HasDate() || Req.HasNights() || Req.HasAges()));
    TProfileTimer profileTimer;
    InitHotelsInResp();
    Stats.Info.MutableProcessingStagesStats()->SetReplyFieldsBeforePutRecordsToRespMicros(StagesTimer.Step().MicroSeconds());
    auto totalBlendingStageTimes = PutRecordsToResp();
    Stats.Info.SetPutRecordsDurationMicros((profileTimer.Step() - Stats.FixPriceDuration).MicroSeconds());
    Stats.Info.MutableProcessingStagesStats()->SetReplyFieldsPutRecordsToRespMicros(StagesTimer.Step().MicroSeconds());
    PutSearchPropsToResp();
    PutOperatorInfosToResp();
    PutPansionInfosToResp();
    PutPartnerInfosToResp();
    PutTotalPriceRangeToResp();
    if (Exp.IsExp(1861) && (UserInfo.ConfirmedHotelOrderCount == 0)) {
        Resp.SetForceUtmTerm(g_WelcomePromocodeUtmTerm);
    }
    if (Req.GetDebug()) {
        Resp.MutableDebug()->MutableCacheStats()->CopyFrom(Stats.Info.GetCacheStats());
        Resp.MutableDebug()->MutableSearcherStats()->CopyFrom(Stats.Info.GetSearcherStats());
        Resp.MutableDebug()->MutableProcessingStagesStats()->CopyFrom(Stats.Info.GetProcessingStagesStats());
        for (const auto& pair: Stats.Info.GetHotels()) {
            Resp.MutableDebug()->MutableHotelExtras()->insert(pair);
        }
    }
    auto totalDuration = LifespanTimer.Get();
    Resp.MutableDebug()->MutableCommon()->SetFixPriceDurationMicros(Stats.FixPriceDuration.MicroSeconds());
    Resp.MutableDebug()->MutableCommon()->SetTotalDurationMicros(totalDuration.MicroSeconds());
    Resp.MutableDebug()->MutableCommon()->SetHostName(FQDNHostName());
    Stats.Info.MutableProcessingStagesStats()->SetReplyFieldsAfterPutRecordsToRespMicros(StagesTimer.Step().MicroSeconds());
    auto respSize = onResp(Resp);
    Stats.Info.MutableProcessingStagesStats()->SetReplyResponseCbMicros(StagesTimer.Step().MicroSeconds());

    // Далее можно не торопиться
    if (!Req.GetRobotRequest()) {
        if (IncompletePartners.empty()) {
            if (Resp.GetWasFound()) {
                if (Stats.Info.GetFullResultCount() > 0) {
                    SourceCounters->NRespCacheHit.Inc();
                } else {
                    SourceCounters->NRespCacheHitEmpty.Inc();
                }
            } else {
                SourceCounters->NRespCacheMiss.Inc();
            }
        } else {
            SourceCounters->NRespInProgress.Inc();
        }
    }

    auto d = Stats.FullDuration.MicroSeconds();
    size_t respSizeKb = respSize / 1024;
    ERequestSize size;
    if (Req.HotelIdSize() < Service.Config().GetCache().GetNHotelsMid()) {
        Service.GetCounters().NCacheTimeXus.Update(d);
        Service.GetCounters().RespSizeXKb.Update(respSizeKb);
        size = ERequestSize::Small;
    } else if (Req.HotelIdSize() < Service.Config().GetCache().GetNHotelsBig()) {
        Service.GetCounters().NCacheTimeXusMid.Update(d);
        Service.GetCounters().RespSizeXKbMid.Update(respSizeKb);
        size = ERequestSize::Medium;
    } else if (Req.HotelIdSize() < Service.Config().GetCache().GetNHotelsLarge()) {
        Service.GetCounters().NCacheTimeXusBig.Update(d);
        Service.GetCounters().RespSizeXKbBig.Update(respSizeKb);
        size = ERequestSize::Big;
    } else if (Req.HotelIdSize() < Service.Config().GetCache().GetNHotelsExtraLarge()) {
        Service.GetCounters().NCacheTimeXusLarge.Update(d);
        Service.GetCounters().RespSizeXKbLarge.Update(respSizeKb);
        size = ERequestSize::Large;
    } else {
        Service.GetCounters().NCacheTimeXusExtraLarge.Update(d);
        Service.GetCounters().RespSizeXKbExtraLarge.Update(respSizeKb);
        size = ERequestSize::ExtraLarge;
    }

#define REPORT(_S_) Service.GetCountersPerStage(size, ERequestStage::_S_, Req.GetUseSearcher())->TimeMicroseconds += Stats.Info.GetProcessingStagesStats().Get##_S_##Micros();
    REPORT(Init)
    REPORT(Start)
    REPORT(Parse)
    REPORT(SearchInCache)
    REPORT(DoSearcherRequest)
    REPORT(ReplyFieldsBeforePutRecordsToResp)
    REPORT(ReplyFieldsPutRecordsToResp)
    REPORT(ReplyFieldsAfterPutRecordsToResp)
    REPORT(ReplyBuildJson)
    REPORT(ReplyResponseCb)
#undef REPORT

    for (const auto& [stage, dur]: totalBlendingStageTimes.Times) {
        Service.GetCountersPerBlendingStage(Req.GetRespMode(), Req.GetFull(), stage)->TimeMicroseconds += dur.MicroSeconds();
    }

    WriteToOfferReqBus();
    ReportBaseFiltersStats();
    WriteInteractiveSearchEvent();
}

void TReadRequestProcessor::WriteToOfferReqBus() const {
    if (SubKey.Date < MinAllowedDate && SubKey.Nights <= 0 || SubKey.Ages.IsEmpty()) {
        // поднимаем популярность только разумным запросам
        return;
    }
    if (Req.GetRobotRequest()) {
        return; // Роботы не влияют на популярность https://st.yandex-team.ru/TRAVELBACK-1313#5f7efbe62f585043f53a0ad9
    }

    bool allowDateShift = false;
    if (!Req.HasDate() && (SubKey.Date == DefaultSubKey.Date)) {
        // Если дата в запросе не указана, и была выбрана дата "завтра"
        allowDateShift = true;
    }

    ru::yandex::travel::hotels::TOfferCacheReq request;
    request.SetCheckInDate(NOrdinalDate::ToString(SubKey.Date));
    request.SetCheckOutDate(NOrdinalDate::ToString(SubKey.Date + SubKey.Nights));
    request.SetOccupancy(SubKey.Ages.ToOccupancyString());
    request.SetCurrency(CurrencyFromOCFormat(Req.GetCurrency()));
    request.SetAllowDateShift(allowDateShift);
    request.SetIsFull(Req.GetFull());
    request.SetIsUseSearcher(Req.GetUseSearcher());
    request.SetIsDefaultSubKey(false);

    auto similar = request;
    request.SetIsSimilar(false);
    similar.SetIsSimilar(true);

    for (const auto& [permalink, complexHotelId]: ComplexHotelIds) {
        auto& req = SimilarPermalinks.contains(permalink) ? similar : request;
        for (const auto& hi: complexHotelId.HotelIds) {
            if (Service.IsSearchSubKeyAllowed(Today, hi, SubKey)) {
                auto hip = req.AddHotelIdsWithPermalink();
                hi.ToProto(hip->MutableHotelId());
                hip->SetPermalink(permalink);
            }
        }
    }
    if (request.HotelIdsWithPermalinkSize() > 0) {
        Service.OfferReqBus().Write(request);
    }
    if (similar.HotelIdsWithPermalinkSize() > 0) {
        Service.OfferReqBus().Write(similar);
    }
}

void TReadRequestProcessor::WriteReqAnsLog() const {
    TInstant now = ::Now();
    NTravelProto::NOfferCache::NReqAnsLog::TReqAnsLogRecord log;
    log.MutableInfo()->CopyFrom(Stats.Info);
    log.set_unixtime(now.Seconds());
    log.set_local_time(now.ToIsoStringLocal());
    if (Req.GetTestMode() == NTravelProto::NOfferCache::NApi::TTestMode::None) {
        log.SetEnvironment(Service.GetEnvironment());
    } else {
        log.SetEnvironment(Service.GetEnvironment() + "-testmode");
    }
    log.MutableReq()->CopyFrom(Req);
    log.MutableResp()->CopyFrom(Resp);
    if (Service.Config().GetOther().GetHideUrlsInLog()) {
        auto* h = log.MutableResp()->MutableHotels();
        for (auto hIt = h->begin(); hIt != h->end(); ++hIt) {
            for (auto& price: *(hIt->second.MutablePrices())) {
                price.ClearPartnerLink();
            }
        }
    }
    if (auto i = log.MutableInfo()) {
        i->SetStartedAt(Started.MicroSeconds());
        i->SetDurationMicros(Stats.FullDuration.MicroSeconds());
        i->SetHostName(FQDNHostName());
        if (RawHttpRequest.Defined()) {
            i->SetRawHttpRequest(RawHttpRequest.GetRef());
        }
    }
    Service.ReqAnsLogger().AddRecord(log);
}

void TReadRequestProcessor::ReportBaseFiltersStats() const {
    if (Resp.GetSearchProps().GetSearchStatus() == NTravelProto::NOfferCache::NApi::Finished) {// TODO it is bad to rely on searchprops!
        int countTotal = 0;
        int countHasOffers = 0;
        for (const auto&[permalink, hotel]: Resp.GetHotels()) {
            if (SimilarPermalinks.contains(permalink)) {
                continue;
            }
            auto statsHotel = Stats.Info.GetHotels().find(permalink);
            if (hotel.PricesSize() > 0) {
                Service.GetCounters().BaseFiltersHasOffers.Inc();
                countHasOffers++;
            } else if (statsHotel != Stats.Info.GetHotels().end() && statsHotel->second.SkippedPricesSize() > 0) {
                auto skippedByUser = false;
                for (const auto& skippedPrice: statsHotel->second.GetSkippedPrices()) {
                    if (skippedPrice.GetSkipReason() == NTravelProto::NOfferCache::NApi::SR_UserFilter) {
                        skippedByUser = true;
                        break;
                    }
                }
                if (skippedByUser) {
                    Service.GetCounters().BaseFiltersSkippedByUserFilter.Inc();
                } else {
                    Service.GetCounters().BaseFiltersSkippedOther.Inc();
                }
            } else {
                Service.GetCounters().BaseFiltersNoOffers.Inc();
            }
            countTotal++;
        }
        if (countTotal > 0) {
            Service.GetCounters().BaseFiltersWasteHotelsPct.Update((countTotal - countHasOffers) * 100 / countTotal);
        }
    }
}

bool TReadRequestProcessor::HotelIdFromApiProto(const NTravelProto::NOfferCache::NApi::TComplexHotelId::TPartnerCodeAndHotelId& pb, THotelId* hotelId) const {
    if (!Service.GetPartnerIdByCode(pb.GetPartnerCode(), &hotelId->PartnerId)) {
        DEBUG_LOG << JOB_LOG << "Ignoring unknown PartnerCode '" << pb.GetPartnerCode() << "'" << Endl;
        return false;
    }
    hotelId->OriginalId = pb.GetOriginalId();
    return true;
}

void TReadRequestProcessor::HotelIdToProto(const THotelId& hotelId, NTravelProto::THotelId* pb) const {
    pb->SetPartnerId(hotelId.PartnerId);
    pb->SetOriginalId(hotelId.OriginalId);
}

void TReadRequestProcessor::SearchInCache() {
    // Кэш ничего не знает про permalink-и, знает только про HotelId-ы

    // Мы делаем несколько поисков в кэше
    // Если BoYMode:
    //   1. поиск обычных отелей
    // Иначе
    //   1. Поиск обычных отелей, но только для BoY-партнёра
    //
    // 2. Поиск похожих отелей (всех сразу)

    THashMap<TPreKey, THashSet<TPermalink>> preKeysForMainSearch;// для (1)

    THashMap<TPreKey, THashSet<TPermalink>> preKeysForSimilarCommonSearch; // для (2) если для похожих общие даты

    TPreKey preKey;
    preKey.Currency = CurrencyFromOCFormat(Req.GetCurrency());
    for (const auto& [permalink, complexHotelId]: ComplexHotelIds) {
        for (const auto& hi: complexHotelId.HotelIds) {
            preKey.HotelId = hi;
            if (SimilarPermalinks.contains(permalink)) {
                // "Похожие" отели не влияют на выбор подключа HOTELS-3505
                preKeysForSimilarCommonSearch[preKey].insert(permalink);
            } else {
                if (!BoYMode || EnabledPartnersBoY.contains(preKey.HotelId.PartnerId)) {
                    preKeysForMainSearch[preKey].insert(permalink);
                }
            }
        }
    }
    // Внимание! Эта функция возвращает в том числе записи,у которых запрещенный оператор.
    // Причина: возвращается целая CacheRecord, в которой сразу все операторы
    // (1) поиск основых отелей
    DoSearchInCache(preKeysForMainSearch, DefaultSubKey, &SubKey);
    // (2) поиск похожих отелей
    if (!preKeysForSimilarCommonSearch.empty()) {
        Y_ASSERT(SubKey.IsComplete());// После поиска обязан быть Complete
        TSearchSubKey altKey = SubKey;
        DoSearchInCache(preKeysForSimilarCommonSearch, DefaultSubKey, &altKey);
    }
    if (IsInitialRequest) {
        for (const auto& [permalink, subHotels]: Records) {
            for (const auto& [hotelId, records]: subHotels) {
                for (const auto& record: records) {
                    record->SourceCounters->NInitialCacheHitContrib.Inc();
                }
            }
        }
    }
    DEBUG_LOG << JOB_LOG << "Req to cache done in " << CacheStat.Duration
              << ", waitLockDuration: " << CacheStat.WaitLockDuration
              << ", preDuration: " << CacheStat.PreDuration
              << ", Scanned: " << CacheStat.RecordsScanned
              << ", PreGood: " << CacheStat.RecordsPreGood
              << ", Good: " << CacheStat.RecordsGood << Endl;
    auto cacheDebug = Stats.Info.MutableCacheStats();
    cacheDebug->SetDurationMicros(CacheStat.Duration.MicroSeconds());
    cacheDebug->SetWaitLockDurationMicros(CacheStat.WaitLockDuration.MicroSeconds());
    cacheDebug->SetPreDurationMicros(CacheStat.PreDuration.MicroSeconds());
    cacheDebug->SetRecordsScanned(CacheStat.RecordsScanned);
    cacheDebug->SetRecordsGood(CacheStat.RecordsGood);
    cacheDebug->SetRecordsPreGood(CacheStat.RecordsPreGood);

    THashSet<EPartnerId> allPartners;
    // посчитаем FullyFoundPermalinks и PartiallyFoundPermalinks, а так же счетчики и Stats.Info
    for (auto itPermalink = ComplexHotelIds.begin(); itPermalink != ComplexHotelIds.end(); ++itPermalink) {
        const TPermalink permalink = itPermalink->first;
        const EPermalinkType pt = GetPermalinkType(permalink);

        TCacheHitCountersRef cacheHitCounters = Service.GetCacheHitCounters(RequestType, pt, Attribution.GetOfferCacheClientId());
        NTravelProto::NOfferCache::NApi::THotelExtra& hotelExtra = (*Stats.Info.MutableHotels())[permalink];

        THashMap<THotelId, TVector<TCacheRecordRef>>& permalinkRecords = Records[permalink];

        size_t cntFound = 0;
        size_t cntEnabled = 0;
        bool isEmpty = true;
        THashMap<EPartnerId, NTravelProto::NOfferCache::NApi::EHotelCacheStatus> partner2CacheStatus;
        for (const THotelId& hotelId: itPermalink->second.HotelIds) {
            if (BoYMode && (pt != EPermalinkType::Similar) && !EnabledPartnersBoY.contains(hotelId.PartnerId)) {
                // Не трогаем поотельные счетчики для запрещенных партнёров
                continue;
            }
            if (!EnabledPartners.contains(hotelId.PartnerId)) {
                // Не трогаем поотельные счетчики для запрещенных партнёров
                continue;
            }
            if (Req.GetRobotRequest()) {
                // Не трогаем поотельные счетчики для запросов роботов
                continue;
            }
            NTravelProto::NOfferCache::NApi::THotelExtra::TSubHotelInfo* subHotelInfo = GetSubHotelInfo(hotelId, hotelExtra);
            FillCacheStatus(hotelId, permalinkRecords[hotelId], subHotelInfo);
            NTravelProto::NOfferCache::NApi::EHotelCacheStatus cacheStatus = subHotelInfo->GetCacheStatus();
            auto it = partner2CacheStatus.find(hotelId.PartnerId);
            if (it == partner2CacheStatus.end()) {
                partner2CacheStatus[hotelId.PartnerId] = cacheStatus;
            } else {
                it->second = CombineCacheStatuses(it->second, cacheStatus);
            }
            switch (cacheStatus) {
                case NTravelProto::NOfferCache::NApi::HCS_Miss:
                    cacheHitCounters->NCacheMiss.Inc();
                    ++cntEnabled;
                    IncompletePartners.insert(hotelId.PartnerId);
                    IncompletePermalinks.insert(permalink);
                    break;
                case NTravelProto::NOfferCache::NApi::HCS_Found:
                    cacheHitCounters->NCacheHit.Inc();
                    ++cntEnabled;
                    ++cntFound;
                    isEmpty = false;
                    break;
                case NTravelProto::NOfferCache::NApi::HCS_Empty:
                    cacheHitCounters->NCacheHitEmpty.Inc();
                    ++cntEnabled;
                    ++cntFound;
                    break;
                case NTravelProto::NOfferCache::NApi::HCS_Error:
                    ++cntEnabled;
                    ++cntFound;
                    break;
                case NTravelProto::NOfferCache::NApi::HCS_RestrictedBySearchSubKey:
                    break;
            }
            allPartners.insert(hotelId.PartnerId);
        }
        if (cntEnabled == 0) {
            // Не трогаем по-пермалинковые счетчики для пермалинков, состоящих только из запрещенных партнёров или restricted ключей
            continue;
        }
        if (cntFound == 0) {
            cacheHitCounters->NPermalinkCacheMiss.Inc();
        } else if (cntEnabled == cntFound) {
            FullyFoundPermalinks.insert(itPermalink->first);
            if (isEmpty) {
                cacheHitCounters->NPermalinkCacheHitFullEmpty.Inc();
            } else {
                cacheHitCounters->NPermalinkCacheHitFull.Inc();
            }
        } else {
            PartiallyFoundPermalinks.insert(itPermalink->first);
            if (isEmpty) {
                cacheHitCounters->NPermalinkCacheHitPartialEmpty.Inc();
            } else {
                cacheHitCounters->NPermalinkCacheHitPartial.Inc();
            }
        }
        for (const auto [pId, currentCacheStatus]: partner2CacheStatus) {
            TMaybe<NTravelProto::NOfferCache::NApi::EHotelCacheStatus> otherCacheStatuses;
            for (const auto [otherPId, otherCacheStatus]: partner2CacheStatus) {
                if (otherPId != pId) {
                    if (otherCacheStatuses.Defined()) {
                        otherCacheStatuses = CombineCacheStatuses(otherCacheStatuses.GetRef(), otherCacheStatus);
                    } else {
                        otherCacheStatuses = otherCacheStatus;
                    }
                }
            }
            // Если других партнёров нет, то считаем, что статус такой же как у текущего
            Service.GetCountersPerPartner(pId)->IncCacheHitCounter(currentCacheStatus, otherCacheStatuses.GetOrElse(currentCacheStatus));
        }
    }
    for (auto partner: allPartners) {
        if (!IncompletePartners.contains(partner)) {
            CompletePartners.insert(partner);
        }
    }
    if (IS_LOG_ACTIVE(TLOG_DEBUG)) {
        DEBUG_LOG << JOB_LOG << "Best key: " << SubKey << Endl;
    }
}

void TReadRequestProcessor::FillCacheStatus(const THotelId& hotelId, const TVector<TCacheRecordRef>& records, NTravelProto::NOfferCache::NApi::THotelExtra::TSubHotelInfo* subHotelInfo) const {
    TString restriction;
    if (!Service.IsSearchSubKeyAllowed(Today, hotelId, SubKey, &restriction)) {
        subHotelInfo->SetCacheStatus(NTravelProto::NOfferCache::NApi::HCS_RestrictedBySearchSubKey);
        subHotelInfo->SetRestriction(restriction);
        return;
    }

    EErrorCode errorCode = NTravelProto::EC_OK;
    bool foundNonEmpty = false;
    for (const TCacheRecordRef& record: records) {
        NTravelProto::NOfferCache::NApi::THotelExtra::TSubHotelInfo::TCacheRecordInfo* crInfo = subHotelInfo->AddCacheRecords();
        crInfo->SetOfferCacheClientId(record->OfferCacheClientId);
        crInfo->SetSearcherReqId(record->SearcherReqId);
        crInfo->SetCachedTimestamp(record->Timestamp.Seconds());
        crInfo->SetCachedAgoSeconds((Started - record->Timestamp).Seconds());
        crInfo->SetExpireTimestamp(record->ExpireTimestamp.Seconds());
        crInfo->SetExpireInSeconds((record->ExpireTimestamp - Started).Seconds());
        if (record->ErrorCode != NTravelProto::EC_OK) {
            crInfo->SetCacheStatus(NTravelProto::NOfferCache::NApi::HCS_Error);
            crInfo->SetErrorCode(record->ErrorCode);
            errorCode = record->ErrorCode;
        } else {
            if (record->Offers.empty()) {
                crInfo->SetCacheStatus(NTravelProto::NOfferCache::NApi::HCS_Empty);
            } else {
                crInfo->SetCacheStatus(NTravelProto::NOfferCache::NApi::HCS_Found);
                foundNonEmpty = true;
            }
        }
        if (record->SearchWarnings) {
            auto searchWarnings = crInfo->MutableSearchWarnings();
            for (auto& [code, count] : record->SearchWarnings->WarningCounts) {
                auto warning = searchWarnings->AddWarnings();
                warning->SetCode(code);
                warning->SetCount(count);
            }
        }
    }
    if (records.empty()) {
        subHotelInfo->SetCacheStatus(NTravelProto::NOfferCache::NApi::HCS_Miss);
    } else if (foundNonEmpty) {
        subHotelInfo->SetCacheStatus(NTravelProto::NOfferCache::NApi::HCS_Found);
    } else if (errorCode != NTravelProto::EC_OK) {
        subHotelInfo->SetCacheStatus(NTravelProto::NOfferCache::NApi::HCS_Error);
        subHotelInfo->SetCacheErrorCode(errorCode);
    } else {
        subHotelInfo->SetCacheStatus(NTravelProto::NOfferCache::NApi::HCS_Empty);
    }
}

void TReadRequestProcessor::DoSearchInCache(
                     const THashMap<TPreKey, THashSet<TPermalink>>& preKeysForSearch,
                     const TSearchSubKey& defaultSubKey,
                     TSearchSubKey* subKey) {
    if (Req.GetTestMode() == NTravelProto::NOfferCache::NApi::TTestMode::CacheMiss) {
        *subKey = defaultSubKey;
        return;
    }
    if (Req.GetTestMode() == NTravelProto::NOfferCache::NApi::TTestMode::CacheHit) {
        *subKey = defaultSubKey;
        TCacheSubKey cacheSubKey;
        cacheSubKey.Capacity = TCapacity::FromAges(subKey->Ages);
        cacheSubKey.Date = subKey->Date;
        cacheSubKey.Nights = subKey->Nights;
        for (const auto& [preKey, permalinks]: preKeysForSearch) {
            TCacheRecordRef rec = new TCacheRecord;
            rec->Key.PreKey = preKey;
            rec->Key.SubKey = cacheSubKey;
            rec->OfferCacheClientId = "serp";
            rec->SourceCounters = Service.GetSourceCounters(rec->OfferCacheClientId);
            for (auto opId: EnabledOperators) {
                TOffer& offer = rec->Offers.emplace_back();
                offer.OfferId = TOfferId::FromString("ed6090e2-bd10-7486-2217-aa27ae3d4229");
                offer.PriceVal = 4427;
                offer.OperatorId = opId;
                offer.TitleAndOriginalRoomId = Service.ObjectDeduplicator().Deduplicate(TCommonDeduplicatorKeys::TitleAndOriginalRoomId, TOfferTitleAndOriginalRoomId{"Test room for HOTELS-4427", ""});
                offer.FreeCancellation = EFreeCancellationType::Yes;
                offer.PansionType = NTravelProto::PT_AI;
            }
            for (TPermalink permalink: permalinks) {
                Records[permalink][preKey.HotelId].push_back(rec);
            }
        }
        return;
    }
    Service.Cache().ComplexSearch(EnabledOperators, preKeysForSearch,
                                  Started, SubKeyRange, defaultSubKey, UserSubKey,
                                  subKey, &Records, &CacheStat, Req.GetAllowOutdated(),
                                  Req.GetAllowOutdatedForKeyDetectionByUserDates() && Req.GetAllowOutdated(),
                                  Req.GetAllowOutdatedForKeyDetectionByAnyDate() && Req.GetAllowOutdated());
}

void TReadRequestProcessor::DoSearcherRequest() {
    if (Req.GetTestMode() != NTravelProto::NOfferCache::NApi::TTestMode::None) {
        return;
    }
    TProfileTimer started;
    NTravelProto::TSearchOffersRpcReq rpcReq;
    PrepareSearcherRequest(&rpcReq);
    Stats.Info.MutableSearcherStats()->SetPrepareDurationMicros(started.Step().MicroSeconds());
    if (rpcReq.SubrequestSize() > 0) {
        DEBUG_LOG << JOB_LOG << "Doing request to searcher" << Endl;
        TIntrusivePtr<TReadRequestProcessor> job = this; // Save myself
        Service.DoSearcherRequest(LogPrefix, rpcReq, NGrpc::TClientMetadata(),
                          [job, started, rpcReq](const NTravelProto::TSearchOffersRpcRsp& rpcResp) {
            job->OnSearcherRequestFinished(started.Get(), rpcReq, rpcResp);
        });
    } else {
        DEBUG_LOG << JOB_LOG << "Skip searcher request due to no subrequests" << Endl;
    }
}

void TReadRequestProcessor::OnSearcherRequestFinished(TDuration dur, const NTravelProto::TSearchOffersRpcReq& rpcReq,
                                                      const NTravelProto::TSearchOffersRpcRsp& rpcResp) {
    INFO_LOG << JOB_LOG << "Searcher request finished in " << dur << ", request ids: " << GetSearchOffersRpcReqIds(rpcReq) << Endl;
    TStringBuilder error;
    for (size_t pos = 0; pos < rpcResp.SubresponseSize(); ++pos) {
        const auto& resp = rpcResp.GetSubresponse(pos);
        if (resp.HasError()) {
            TString requestId;
            if (pos < rpcReq.SubrequestSize()) {
                requestId = rpcReq.GetSubrequest(pos).GetId();
            }
            ERROR_LOG << JOB_LOG << "Searcher response has error in subresponse #" << pos << ", RequestId: " << requestId << ": " << resp.GetError() << Endl;
            if (error) {
                error << "; ";
            }
            error << "SubReq #" << pos << " error: " << ToString(resp.GetError());
        }
    }
    Stats.Info.MutableSearcherStats()->SetError(error);
    Stats.Info.MutableSearcherStats()->SetResponseDurationMicros(dur.MicroSeconds());
}

void TReadRequestProcessor::PrepareSearcherRequest(NTravelProto::TSearchOffersRpcReq* rpcReq) {
    // https://st.yandex-team.ru/HOTELS-2997#1532694988000
    // В сёрчер идём если запрос не делался давно
    rpcReq->SetSync(false);
    rpcReq->SetIncludeDebug(Req.GetDebug());
    THashMap<THotelId, TPermalink> requestedHotels;
    for (const auto& [permalink, hotelIds] : ComplexHotelIds) {
        for (const THotelId& hotelId: hotelIds.HotelIds) {
            auto& hotelExtra = (*Stats.Info.MutableHotels())[permalink];
            NTravelProto::NOfferCache::NApi::THotelExtra::TSubHotelInfo* subHotelInfo = GetSubHotelInfo(hotelId, hotelExtra);

            auto it = requestedHotels.find(hotelId);
            if (it != requestedHotels.end()) {
                // Отель уже запрашивался,возможно - в другом пермалинке
                auto& otherHotelExtra = (*Stats.Info.MutableHotels())[it->second];
                NTravelProto::NOfferCache::NApi::THotelExtra::TSubHotelInfo* otherSubHotelInfo = GetSubHotelInfo(hotelId, otherHotelExtra);
                subHotelInfo->SetSearcherStatus(otherSubHotelInfo->GetSearcherStatus());
                if (otherSubHotelInfo->HasRestriction()) {
                    subHotelInfo->SetRestriction(otherSubHotelInfo->GetRestriction());
                }
                if (otherSubHotelInfo->HasSentSearcherReqId()) {
                    subHotelInfo->SetSentSearcherReqId(otherSubHotelInfo->GetSentSearcherReqId());
                }
                continue;
            }
            requestedHotels[hotelId] = permalink;

            TString restrictionReason;
            if (!Service.IsSearchSubKeyAllowed(Today, hotelId, SubKey, &restrictionReason)) {
                subHotelInfo->SetSearcherStatus(NTravelProto::NOfferCache::NApi::HSS_RestrictedBySearchSubKey);
                subHotelInfo->SetRestriction(restrictionReason);
                Service.GetCountersPerPartner(hotelId.PartnerId)->NSearchSubKeyRestricted.Inc();
                continue;
            }
            NTravelProto::TSearchOffersReq req;
            req.SetId(CreateGuidAsString());
            hotelId.ToProto(req.MutableHotelId());
            req.SetCheckInDate(NOrdinalDate::ToString(SubKey.Date));
            req.SetCheckOutDate(NOrdinalDate::ToString(SubKey.Date + SubKey.Nights));
            req.SetOccupancy(SubKey.Ages.ToOccupancyString());
            req.SetCurrency(CurrencyFromOCFormat(Req.GetCurrency()));
            req.SetRequestClass(Req.GetRequestClass());
            req.SetPermalink(permalink);
            req.MutableAttribution()->CopyFrom(Attribution);

            TString existingReqId;
            TReqCache::EReqState state = Service.ReqCache().GetRequestState(req, &existingReqId);
            if (state == TReqCache::EReqState::Started) {
                // Запрос уже послан
                subHotelInfo->SetSearcherStatus(NTravelProto::NOfferCache::NApi::HSS_AlreadyInProgress);
                subHotelInfo->SetSentSearcherReqId(existingReqId);
                continue;
            }
            if (state == TReqCache::EReqState::Finished) {
                // Запрос уже завершен
                subHotelInfo->SetSearcherStatus(NTravelProto::NOfferCache::NApi::HSS_AlreadyFinished);
                subHotelInfo->SetSentSearcherReqId(existingReqId);
                continue;
            }
            SearcherRequests[req.GetId()] = hotelId;
            subHotelInfo->SetSearcherStatus(NTravelProto::NOfferCache::NApi::HSS_Sent);
            subHotelInfo->SetSentSearcherReqId(req.GetId());
            rpcReq->AddSubrequest()->Swap(&req);
        }
    }
}

void TReadRequestProcessor::InitHotelsInResp() {
    Resp.SetWasFound(false);
    for (const auto& complexHotelId: ComplexHotelIds) {
        auto& hotel = (*Resp.MutableHotels())[complexHotelId.first];
        hotel.SetWasFound(false);// Default value, overriden in PutRecordsToResp
        hotel.SetIsFinished(!(Req.GetUseSearcher() && IncompletePermalinks.contains(complexHotelId.first)));
    }
}

void TReadRequestProcessor::PrepareLabel(NTravelProto::TLabel* label) const {
    label->SetSource(Attribution.GetUtmSource());
    label->SetMedium(Attribution.GetUtmMedium());
    label->SetCampaign(Attribution.GetUtmCampaign());
    label->SetContent(Attribution.GetUtmContent());
    label->SetTerm(Attribution.GetUtmTerm());
    label->SetYandexUid(Attribution.GetYandexUid());
    label->SetQuery(Attribution.GetSearchQuery());
    label->SetSerpReqId(Attribution.GetSerpReqId());
    label->SetPassportUid(Attribution.GetPassportUid());
    label->SetUuid(Attribution.GetUuid());
    label->SetCheckInDate(NOrdinalDate::ToString(SubKey.Date));
    label->SetNights(SubKey.Nights);
    label->SetOccupancy(SubKey.Ages.ToOccupancyString());
    label->SetRequestRegion(Attribution.GetRequestRegion());
    label->SetUserRegion(Attribution.GetUserRegion());
    label->SetICookie(Attribution.GetICookie());
    label->SetGeoClientId(Attribution.GetGeoClientId());
    label->SetGeoOrigin(Attribution.GetGeoOrigin());
    label->MutableIntTestIds()->CopyFrom(Attribution.GetIntTestIds());
    label->MutableIntTestBuckets()->CopyFrom(Attribution.GetIntTestBuckets());
    label->SetSurface(Attribution.GetSurface());
    label->SetUserDevice(Attribution.GetUserDevice());
    label->SetGclid(Attribution.GetGclid());
    label->SetYaTravelReqId(Attribution.GetYaTravelReqId());
    label->SetYtpReferer(Attribution.GetYtpReferer());
    label->SetYclid(Attribution.GetYclid());
    label->SetFBclid(Attribution.GetFBclid());
    label->SetMetrikaClientId(Attribution.GetMetrikaClientId());
    label->SetMetrikaUserId(Attribution.GetMetrikaUserId());
    label->MutableIntPortalTestIds()->CopyFrom(Attribution.GetIntPortalTestIds());
    label->MutableIntPortalTestBuckets()->CopyFrom(Attribution.GetIntPortalTestBuckets());
    label->SetIsStaffUser(Attribution.GetIsStaffUser());
    label->SetOfferCacheClientId(Attribution.GetOfferCacheClientId());
    label->SetClid(Attribution.GetClid());
    label->SetAffiliateClid(Attribution.GetAffiliateClid());
    label->SetAdmitadUid(Attribution.GetAdmitadUid());
    label->SetTravelpayoutsUid(Attribution.GetTravelpayoutsUid());
    label->SetVid(Attribution.GetVid());
    label->SetAffiliateVid(Attribution.GetAffiliateVid());
    label->SetIsPlusUser(Attribution.GetIsPlusUser());
    label->SetSearchPagePollingId(Attribution.GetSearchPagePollingId());
    label->SetReferralPartnerId(Attribution.GetReferralPartnerId());
    label->SetReferralPartnerRequestId(Attribution.GetReferralPartnerRequestId());
}

TString TReadRequestProcessor::PrepareRedirAddInfo() const {
    NTravelProto::NRedirAddInfo::TRedirAddInfo info;
    info.MutableExp()->CopyFrom(Req.GetExp());
    info.MutableStrExp()->CopyFrom(Req.GetStrExp());
    TString serialized = info.SerializeAsString();
    if (serialized) {
        serialized = Service.RedirAddInfoCodec().Encode(serialized);
    }
    return serialized;
}

TBlendingStageTimes TReadRequestProcessor::PutRecordsToResp() {
    NTravelProto::TLabel label;
    PrepareLabel(&label);
    TString redirAddInfo = PrepareRedirAddInfo();

    TBlendingStageTimes totalBlendingStageTimes;
    TOfferShowStats totalOfferShowStats;

    TString redirHost = GetAllowedHostOrDefault(Req.GetDebugPortalHost(), Service.Config().GetOther().GetAllowedRedirHost(), Service.Config().GetOther().GetDefaultRedirHost());

    TUrl redirUrl;
    redirUrl.SetScheme(NUri::TScheme::SchemeHTTPS);
    redirUrl.SetHost(redirHost);
    redirUrl.SetPath("/redir");
    if (IsHostAllowed(Req.GetDebugPortalHost(), Service.Config().GetOther().GetAllowedPortalHost())) {
        redirUrl.SetCgiParam("DebugPortalHost", Req.GetDebugPortalHost());
    }
    if (redirAddInfo) {
        redirUrl.SetCgiParam("AddInfo", redirAddInfo);
    }
    if (Req.GetLabelHash()) {
        redirUrl.SetCgiParam("LabelHash", Req.GetLabelHash());
    }

    //HOTELS-2604
    for (auto permalinkIt = Records.begin(); permalinkIt != Records.end(); ++permalinkIt) {
        const TPermalink permalink = permalinkIt->first;
        const auto& records = permalinkIt->second;

        auto& hotel = (*Resp.MutableHotels())[permalink];

        TOfferBlender blender(*this, permalink, records, redirUrl, label, &hotel);
        blender.Blend();

        totalBlendingStageTimes += blender.GetBlendingStageTimes();
        totalOfferShowStats += blender.GetOfferShowStats();

        if (permalink == MainPermalink) {
            if (hotel.PricesSize() == 0 && FullyGreylistedPermalinks.contains(permalink)) {
                MainPermalinkIsGreylistedNoOffers = true;
                hotel.SetHideSearchForm(true);
                SourceCounters->NMainPermalinksSearchFormHidden.Inc();
            } else {
                SourceCounters->NMainPermalinksOK.Inc();
            }
        }

        if (hotel.GetWasFound()) {
            Resp.SetWasFound(true);
        }
    }

    for (size_t i = 0; i < GetEnumItemsCount<EPermalinkType>(); i++) {
        auto permalinkType = EPermalinkType(i);
        for (size_t j = 0; j < NTravelProto::EOperatorId_ARRAYSIZE; j++) {
            if (!NTravelProto::EOperatorId_IsValid(j)) {
                continue;
            }
            auto operatorId = NTravelProto::EOperatorId(j);
            auto counters = Service.GetOfferShowCounters(RequestType, permalinkType, operatorId);
            auto& stats = totalOfferShowStats.Stats[i][j];
            counters->NAnyOffer += stats.NAnyOffer;
            counters->NFirstOffer += stats.NFirstOffer;
            counters->NBoYHotelFirstOffer += stats.NBoYHotelFirstOffer;
            counters->NDirectBoYHotelFirstOffer += stats.NDirectBoYHotelFirstOffer;
        }
    }

    return totalBlendingStageTimes;
}

void TReadRequestProcessor::PutSearchPropsToResp() {
    if (Req.GetCompactResponseForCalendar()) {
        return;
    }
    auto& searchProps = *Resp.MutableSearchProps();
    searchProps.SetMainPermalink(MainPermalink);

    NTravelProto::NOfferCache::NApi::ESearchProps_SearchStatus searchStatus;
    if (Req.GetUseSearcher()) {
        searchStatus = Resp.GetIsFinished() ? NTravelProto::NOfferCache::NApi::Finished : NTravelProto::NOfferCache::NApi::InProgress;
    } else {
        searchStatus = NTravelProto::NOfferCache::NApi::NotStarted;
    }
    searchProps.SetSearchStatus(searchStatus);

    NTravelProto::NOfferCache::NApi::ESearchProps_CacheStatus cacheStatus;
    if (MainPermalink != 0) {
        if (MainPermalinkIsBlacklisted) {
            cacheStatus = NTravelProto::NOfferCache::NApi::Blacklisted;
        } else if (MainPermalinkIsGreylistedNoOffers) {
            cacheStatus = NTravelProto::NOfferCache::NApi::GreylistedNoOffers;
        } else if (FullyFoundPermalinks.contains(MainPermalink)) {
            cacheStatus = NTravelProto::NOfferCache::NApi::FullHit;
        } else if (PartiallyFoundPermalinks.contains(MainPermalink)) {
            cacheStatus = NTravelProto::NOfferCache::NApi::PartialHit;
        } else {
            cacheStatus = NTravelProto::NOfferCache::NApi::Miss;
        }
        searchProps.SetMainCacheStatus(cacheStatus);
        searchProps.SetMainAvailability(Stats.MainPermalinkHasPrices ? 1 : 0);
    }

    if (FullyFoundPermalinks.size() == ComplexHotelIds.size()) {
        cacheStatus = NTravelProto::NOfferCache::NApi::FullHit;
    } else if (!PartiallyFoundPermalinks.empty()) {
        cacheStatus = NTravelProto::NOfferCache::NApi::PartialHit;
    } else {
        cacheStatus = NTravelProto::NOfferCache::NApi::Miss;
    }
    searchProps.SetTotalCacheStatus(cacheStatus);
    searchProps.SetTotalAvailability((float)Stats.TotalPermalinksWithPrices / Max(1UL, ComplexHotelIds.size()));
    searchProps.SetTop5Availability((float)Stats.Top5PermalinksWithPrices / Max(1UL, Min(5UL, ComplexHotelIds.size())));
    searchProps.SetTop10Availability((float)Stats.Top10PermalinksWithPrices / Max(1UL, Min(10UL, ComplexHotelIds.size())));

    if (Req.GetDebug()) {
        TString res;
        auto pmIt = ComplexHotelIds.find(MainPermalink);
        if (pmIt != ComplexHotelIds.end()) {
            const TComplexHotelId& cHotelId = pmIt->second;
            for (auto it = cHotelId.HotelIds.begin(); it != cHotelId.HotelIds.end(); ++it) {
                if (res) {
                    res += "~";
                }
                res += Service.GetPartnerCode(it->PartnerId) + "." + it->OriginalId;
            }
        }
        searchProps.SetMainOriginalIds(res);
    }
    searchProps.SetMainCatRoomPossible(Stats.Info.GetMainPermalink_CatRoomShown());
    if (Stats.Info.HasMainPermalink_CatRoomDSId()) {
        searchProps.SetMainCatRoomDataSourceId(Stats.Info.GetMainPermalink_CatRoomDSId());
    }

    searchProps.SetMainCatRoomShown(Stats.Info.GetMainPermalink_CatRoomShown());
    searchProps.SetMainCatRoomAvailable(Stats.Info.GetMainPermalink_CatRoomAvailable());
    searchProps.SetMainCatRoomTooManyOther(Stats.Info.GetMainPermalink_CatRoomTooManyOther());

    searchProps.SetBlacklistedCount(Stats.Info.BlacklistedPermalinksSize());

    searchProps.SetDate(Resp.GetDate());
    searchProps.SetNights(Resp.GetNights());
    searchProps.SetAges(Resp.GetAges());
    searchProps.SetHorizon(SubKey.Date - Today);

    searchProps.SetUserConfirmedHotelOrderCount(UserInfo.ConfirmedHotelOrderCount);

    searchProps.SetMainMir(IsMirPromoAvailable(MainPermalink) ? 1 : 0);
    searchProps.SetTop10MirRatio(static_cast<float>(GetMirPromoAvailableTop10Ratio()));
    searchProps.SetExp2216Effect(Stats.Exp2216Effect);

    searchProps.SetMainHasBoYOffers(Stats.Info.GetMainPermalink_HasBoYOffers());
    searchProps.SetMainHasBoYPartner(Stats.Info.GetMainPermalink_HasBoYPartner());
}

void TReadRequestProcessor::PutOperatorInfosToResp() {
    if (Req.GetCompactResponseForCalendar()) {
        return;
    }
    const THashSet<EOperatorId>& ops = Req.GetShowAllOperators() ? EnabledOperators : ActualOperators;
    for (EOperatorId opId: ops) {
        auto opInfo = Service.GetOperator(opId);
        auto& pbOpInfo = (*Resp.MutableOperators())[opId];
        pbOpInfo.SetName(opInfo->GetName());
        pbOpInfo.SetFaviconUrl(opInfo->GetFaviconUrl());
        pbOpInfo.SetPngPattern(opInfo->GetPngPattern());
        pbOpInfo.SetGreenUrl(opInfo->GetGreenUrl());
        pbOpInfo.SetIsBookOnYandex(Service.IsBoYOperator(opId));
    }
}

void TReadRequestProcessor::PutPartnerInfosToResp() {
    if (Req.GetCompactResponseForCalendar()){
        return;
    }
    for (EPartnerId pId: EnabledPartners) {
        auto pInfo = Service.GetPartner(pId);
        auto& pbPInfo = (*Resp.MutablePartners())[pId];
        pbPInfo.SetIsBookOnYandex(pInfo->GetIsBoY());
    }
}

void TReadRequestProcessor::PutPansionInfosToResp() {
    if (Req.GetCompactResponseForCalendar()){
        return;
    }
    if (Req.GetShowAllPansions()) {
        for (int pansionInt = NTravelProto::NOfferCache::NApi::TOCPansion::EPansion_MIN; pansionInt <= NTravelProto::NOfferCache::NApi::TOCPansion::EPansion_MAX; ++pansionInt) {
            if (NTravelProto::NOfferCache::NApi::TOCPansion::EPansion_IsValid(pansionInt)) {
                PutPansionInfoToResp(EOCPansion(pansionInt));
            }
        }
    } else {
        for (EOCPansion pansion: ActualPansions) {
            PutPansionInfoToResp(pansion);
        }
    }
}

void TReadRequestProcessor::PutPansionInfoToResp(EOCPansion pansion) {
    auto& pansionPb = (*Resp.MutablePansions())[ToString(pansion)];
    pansionPb.SetName(Service.GetPansionDisplayName(pansion));
}

void TReadRequestProcessor::PutTotalPriceRangeToResp() {
    if (Req.GetCompactResponseForCalendar()){
        return;
    }
    if (TotalPriceValRange.IsValid) {
        auto pbPriceRange = Resp.MutableTotalPriceRange();
        pbPriceRange->SetMinPrice(TotalPriceValRange.MinPriceVal);
        pbPriceRange->SetMaxPrice(TotalPriceValRange.MaxPriceVal);
    }
}

NTravelProto::NOfferCache::NApi::THotelExtra::TSubHotelInfo* TReadRequestProcessor::GetSubHotelInfo(const THotelId& hotelId, NTravelProto::NOfferCache::NApi::THotelExtra& extra) const {
    for (size_t i = 0; i < extra.SubHotelsSize(); ++i) {
        NTravelProto::NOfferCache::NApi::THotelExtra::TSubHotelInfo* subHotel = extra.MutableSubHotels(i);
        if (subHotel->GetHotelId().GetPartnerId() == hotelId.PartnerId && subHotel->GetHotelId().GetOriginalId() == hotelId.OriginalId) {
            return subHotel;
        }
    }
    NTravelProto::NOfferCache::NApi::THotelExtra::TSubHotelInfo* subHotel = extra.AddSubHotels();
    HotelIdToProto(hotelId, subHotel->MutableHotelId());
    return subHotel;
}

void TReadRequestProcessor::WriteInteractiveSearchEvent() const {
    if (!Req.GetUseSearcher()) {
        return;
    }
    if (!Attribution.GetOfferCacheClientId().StartsWith("serp")) {
        // Temp hotfix for exp
        return;
    }
    if (!UserInfo.YandexUid.Defined() && !UserInfo.PassportUid.Defined()) {
        return;
    }
    if (!Service.UserEventsStorage().IsEnabled()) {
        return;
    }
    NTravelProto::NOfferCache::TInteractiveSearchEvent ise;
    if (UserInfo.YandexUid.Defined()) {
        ise.SetYandexUid(UserInfo.YandexUid.GetRef());
    }
    if (UserInfo.PassportUid.Defined()) {
        ise.SetPassportUid(UserInfo.PassportUid.GetRef());
    }
    ise.SetDate(SubKey.Date);
    ise.SetNights(SubKey.Nights);
    ise.SetAges(SubKey.Ages.ToAgesString());
    Service.ICBusWriter().Write(ise);
}

void TReadRequestProcessor::AdjustSubKeyForUser() {
    if (!IsInitialRequest || !UserInfo.UserIdentifier) {
        return;
    }
    TMaybe<TSearchSubKey> userSubKey = Service.UserEventsStorage().FindSearchSubKey(UserInfo.UserIdentifier.GetRef());
    if (!userSubKey) {
        return;
    }
    Stats.Exp2216Effect = true;

    bool useAsForced = Exp.IsExp(2216); // TRAVELBACK-2216
    bool useAsDefault = Exp.IsExp(2217); // fake key
    if (!useAsForced && !useAsDefault) {
        return;
    }
    if (userSubKey.GetRef().Date < Today) {
        userSubKey.GetRef().Date = MinAllowedDate;
    }

    if (useAsForced) {
        SubKey.Merge(userSubKey.GetRef());
    } else {
        UserSubKey = userSubKey.GetRef();
    }
}

}// namespace NOfferCache
}// namespace NTravel

