#include "req_cache.h"

#include <travel/hotels/lib/cpp/util/sizes.h>
#include <travel/hotels/lib/cpp/scheduler/scheduler.h>

#include <library/cpp/logger/global/global.h>
#include <util/string/builder.h>

namespace NTravel {
namespace NOfferCache {

void TReqCache::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NReqStarted));
    ct->insert(MAKE_COUNTER_PAIR(NReqFinished));
    ct->insert(MAKE_COUNTER_PAIR(NReqStartedExpired));
    ct->insert(MAKE_COUNTER_PAIR(NBytes));
}

size_t TReqCache::TRequestInfo::GetAllocSize() const {
    return TTotalByteSize<TString>()(Id) - sizeof(Id);
}

TReqCache::TReqCache(const NTravelProto::NOfferCache::TConfig::TReqCache& cfg, const TCacheInvalidationService& cacheInvalidationService)
    : Enabled_(cfg.GetEnabled())
    , CleanupPeriod_(TDuration::MilliSeconds(cfg.GetMemoryCleanupPeriodMSec()))
    , MaxAgeRequestStarted_(TDuration::Seconds(cfg.GetMaxAgeRequestStartedSec()))
    , MaxAgeRequestStartedBackground_(TDuration::Seconds(cfg.GetMaxAgeRequestStartedBackgroundSec()))
    , RelMaxAgeRequestFinished_(TDuration::Seconds(cfg.GetRelativeMaxAgeRequestFinishedSec()))
    , MaxAgeRequestFinishedError_(TDuration::Seconds(cfg.GetMaxAgeRequestFinishedErrorSec()))
    , NewCleanupMode_(cfg.GetNewCleanupMode())
    , CacheInvalidationService(cacheInvalidationService)
{
}

TReqCache::~TReqCache() {
}

void TReqCache::RegisterCounters(NMonitor::TCounterSource& counters) {
    counters.RegisterSource(&Counters_, "ReqCache");
}

TReqCache::EReqState TReqCache::GetRequestState(const NTravelProto::TSearchOffersReq& req, TString* reqId) const {
    TInstant now = ::Now();
    auto key = TSearcherRequestKey::FromRequest(req);

    // Классы в порядке убывания приоритета: INTERACTIVE - самый приоритетный
    // Приоритет влияет на то, какие классы запросов в статусе Started нас интересуют
    // Запросы с любым приоритетом будут считаться завершенными, если завершен любой запрос с любым другим приоритетом
    // Запросы с большим приоритетом будут считаться начавшимися, только если они начались сами
    // Запросы с меньшим приоритетом будут считаться начавшимися, если начался любой запрос с более высоким или равным приоритетом
    // так например
    // Для interactive запросов - если вдруг background или calendar finished, то finished
    // но другие состояния background и calendar запросов нас не интересуют, ориентируемся на состояние interactive.
    // Для запросов с меньшим приоритетом ищем самый продвинутый запрос среди всех
    // В итоге это влияет на то, что запрос в searcher для interactive запросов отправится, даже если
    // уже отправляли запрос для background, но не наоборот
    TVector<NTravelProto::ERequestClass> classPriorities{
        NTravelProto::RC_INTERACTIVE,
        NTravelProto::RC_CALENDAR,
        NTravelProto::RC_BACKGROUND,
    };

    TVector<std::pair<TReqCache::EReqState, TString>> requestStates;
    for (const auto& clz : classPriorities) {
        key.RequestClass = clz;
        requestStates.push_back(GetRequestStateExact(key, now));
    }

    auto requestClass = req.GetRequestClass();
    auto requestClassIndex = -1;
    for (int i = 0; i < static_cast<int>(classPriorities.size()); i++) {
        if (requestClass == classPriorities[i]) {
            requestClassIndex = i;
            break;
        }
    }
    if (requestClassIndex < 0) {
        return EReqState::None;
    }

    // Для начала находим любой finished запрос
    for (const auto& requestState : requestStates) {
        if (requestState.first == EReqState::Finished) {
            if (reqId) {
                *reqId = requestState.second;
            }
            return EReqState::Finished;
        }
    }
    // Если нет finished, ищем первый started запрос от класса с высоким приоритетом до класса с низким
    for (int i = 0; i <= requestClassIndex; i++) {
        if (requestStates[i].first == EReqState::Started) {
            if (reqId) {
                *reqId = requestStates[i].second;
            }
            return EReqState::Started;
        }
    }
    return EReqState::None;
}

std::pair<TReqCache::EReqState, TString> TReqCache::GetRequestStateExact(const TSearcherRequestKey& key, TInstant now) const {
    auto& bucket = Requests_.GetBucketForKey(key);
    TReadGuard g(bucket.GetMutex());
    auto itR = bucket.GetMap().find(key);
    if (itR == bucket.GetMap().end()) {
        return {EReqState::None, ""};
    }
    const TRequestInfo& ri = itR->second;
    if (ri.ExpTime < now) {
        return {EReqState::None, ""};
    }
    if (CacheInvalidationService.IsInvalidated(key.SearchKey.PreKey, key.SearchKey.SubKey.Date,
                                               key.SearchKey.SubKey.Date + key.SearchKey.SubKey.Nights,
                                               ri.Timestamp)) {
        return {EReqState::None, ""};
    }
    return {ri.State, ri.Id};
}

void TReqCache::OnRequestStarted(const NTravelProto::TSearchOffersReq& req, TInstant ts) {
    auto maxAge = (req.GetRequestClass() == NTravelProto::RC_BACKGROUND) ? MaxAgeRequestStartedBackground_ : MaxAgeRequestStarted_;
    Put(req, ts, ts + maxAge, EReqState::Started);
}

void TReqCache::OnRequestFinished(const NTravelProto::TSearchOffersReq& req, TInstant ts, TInstant expireTimestamp) {
    Put(req, ts, expireTimestamp - RelMaxAgeRequestFinished_, EReqState::Finished);
}

void TReqCache::OnRequestFinishedError(const NTravelProto::TSearchOffersReq& req, TInstant ts) {
    Put(req, ts, ts + MaxAgeRequestFinishedError_, EReqState::Finished);
}

void TReqCache::Put(const NTravelProto::TSearchOffersReq& req, TInstant timestamp, TInstant expTime, EReqState newState) {
    if (!Enabled_) {
        return;
    }
    bool expired = expTime < Now();
    // Нельзя просто не обрабатывать expired - т.к. они могут вытеснять более старые записи из кэша

    auto key = TSearcherRequestKey::FromRequest(req);
    // Надо выровнять время очистки на расписание очисток
    TInstant cleanupTime = expTime + TDuration::MicroSeconds(CleanupPeriod_.MicroSeconds() - (expTime.MicroSeconds() % CleanupPeriod_.MicroSeconds()));
    bool isCleanupScheduleRequired;
    {
        auto& bucket = Requests_.GetBucketForKey(key);
        {
            TWriteGuard g(bucket.GetMutex());
            auto itR = bucket.GetMap().find(key);
            if (itR == bucket.GetMap().end()) {
                if (expired) {
                    return;
                }
                itR = bucket.GetMap().insert({key, {}}).first;
                Counters_.NBytes += sizeof *itR + itR->first.GetAllocSize();

                TRequestInfo& ri = itR->second;
                ri.Id = req.GetId();
                ri.Timestamp = timestamp;
                ri.ExpTime = expTime;
                ri.State = newState;
                OnRecordAdded(ri);
            } else {
                TRequestInfo& ri = itR->second;
                if (ri.Id == req.GetId()) {
                    // Речь о том же запросе, что у нас есть
                    if (newState < ri.State) {
                        // Состояние может только прогрессировать, запретим деградацию
                        return;
                    }
                    if (newState == ri.State && expTime <= ri.ExpTime) {
                        // Если состояние не поменялось, то нельзя уменьшать время expTime
                        // А увеличивать можно и нужно!
                        return;
                    }
                    OnRecordRemoved(ri, false);
                    ri.Timestamp = timestamp;
                    ri.ExpTime = expTime;
                    ri.State = newState;
                    OnRecordAdded(ri);
                } else {
                    // Новый запрос нас интересует, только если он строго новее
                    // ! Важно! Это условие должно быть таким же как в TCache::AddBulk
                    if (ri.Timestamp >= timestamp) {
                        return;
                    }
                    OnRecordRemoved(ri, false);
                    ri.Id = req.GetId();
                    ri.Timestamp = timestamp;
                    ri.ExpTime = expTime;
                    ri.State = newState;
                    OnRecordAdded(ri);
                }
            }
        }
        {
            auto& cleanupBucket = CleanupSchedule_.GetBucketForKey(cleanupTime);
            TWriteGuard g(cleanupBucket.GetMutex());
            auto& bucketPtrsToCleanup = cleanupBucket.GetMap()[cleanupTime];
            isCleanupScheduleRequired = bucketPtrsToCleanup.empty();
            if (bucketPtrsToCleanup.insert(&bucket).second) {
                Counters_.NBytes += sizeof &bucket;
            }
        }
    }
    if (isCleanupScheduleRequired) {
        TScheduler::Instance().Enqueue(cleanupTime, [this, cleanupTime]() {ExecuteCleanup(cleanupTime);});
    }
    DEBUG_LOG << "Done Put to ReqCache " << req.GetId() << ", newState " << (int)newState << ", expire at " << expTime << Endl;
    if (newState == EReqState::Finished) {
        // Закончившийся запрос вышибает из кэша аналогичный запрос с противоположным классом
        // Причина: из основного кэша данных запись вымывается, так и тут.
        // иначе возможна ситуация:
        //  ts=X: закончен INTERACTIVE запрос, TTL=3600
        //  ts=X+100: закончен BACKGROUND запрос (проверка pricechecker-ом), TTL = 180 (ошибка, например)
        //  ts=X+280: что тут? Получаем Miss в кэше данных, но Hit + Finished в ReqCache, и запрос в сёрчер не делаем
        // https://st.yandex-team.ru/TRAVELBACK-866#5f2bfb3e1dbadc3f1cfe9e0a

        auto originalRequestClass = key.RequestClass;

        for (int i = NTravelProto::ERequestClass_MIN; i <= NTravelProto::ERequestClass_MAX; i++) {
            auto removeRequestClass = static_cast<NTravelProto::ERequestClass>(i);
            if (removeRequestClass == originalRequestClass) {
                continue;
            }
            key.RequestClass = removeRequestClass;
            auto& bucket = Requests_.GetBucketForKey(key);
            {
                TWriteGuard g(bucket.GetMutex());
                auto itR = bucket.GetMap().find(key);
                if (itR != bucket.GetMap().end()) {
                    OnRecordRemoved(itR->second, false);
                    Counters_.NBytes -= sizeof *itR + itR->first.GetAllocSize();
                    bucket.GetMap().erase(itR);
                }
            }
        }
    }
}

void TReqCache::OnRecordAdded(const TRequestInfo& ri) const {
    Counters_.NBytes += sizeof ri + ri.GetAllocSize();
    switch (ri.State) {
        case EReqState::None:
            Y_FAIL("Impossible");
            break;
        case EReqState::Started:
            Counters_.NReqStarted.Inc();
            break;
        case EReqState::Finished:
            Counters_.NReqFinished.Inc();
            break;
    }
}

void TReqCache::OnRecordRemoved(const TRequestInfo& ri, bool expired) const {
    switch (ri.State) {
        case EReqState::None:
            return;
        case EReqState::Started:
            Counters_.NReqStarted.Dec();
            if (expired) {
                Counters_.NReqStartedExpired.Inc();
            }
            break;
        case EReqState::Finished:
            Counters_.NReqFinished.Dec();
            break;
    }
    Counters_.NBytes -= sizeof ri + ri.GetAllocSize();
}

void TReqCache::ExecuteCleanup(TInstant cleanupTime) {
    TProfileTimer started;
    TDuration maxLock = TDuration::Zero();
    if (!NewCleanupMode_) {
        THashSet<TKeyToRequestsMap::TBucket*> bucketPtrsToCleanup;
        {
            auto& cleanupBucket = CleanupSchedule_.GetBucketForKey(cleanupTime);
            TWriteGuard g(cleanupBucket.GetMutex());
            TProfileTimer lockStarted;
            auto itB = cleanupBucket.GetMap().find(cleanupTime);
            if (itB == cleanupBucket.GetMap().end()) {
                return;
            }
            itB->second.swap(bucketPtrsToCleanup);
            Counters_.NBytes -= sizeof *itB;
            cleanupBucket.GetMap().erase(itB);
            maxLock = Max(maxLock, lockStarted.Get());
        }
        for (auto bucketPtr : bucketPtrsToCleanup) {
            TDuration dur = CleanupBucket(bucketPtr, cleanupTime);
            maxLock = Max(maxLock, dur);
        }
        INFO_LOG << "Cleanup done in " << started.Get() << ", cleaned " << bucketPtrsToCleanup.size() << " buckets, maxLock: " << maxLock << Endl;
    } else {
        for (auto& bucket: Requests_.Buckets) {
            TDuration dur = CleanupBucket(&bucket, cleanupTime);
            maxLock = Max(maxLock, dur);
        }
        INFO_LOG << "Cleanup done in " << started.Get() << ", cleaned " << Requests_.Buckets.size() << " buckets, maxLock: " << maxLock << Endl;
    }
}

TDuration TReqCache::CleanupBucket(TKeyToRequestsMap::TBucket* bucketPtr, TInstant expTime) {
    TProfileTimer started;
    TWriteGuard g(bucketPtr->GetMutex());
    for (auto itR = bucketPtr->GetMap().begin(); itR != bucketPtr->GetMap().end();) {
        if (itR->second.ExpTime < expTime) {
            OnRecordRemoved(itR->second, true);
            Counters_.NBytes -= sizeof *itR + itR->first.GetAllocSize();
            bucketPtr->GetMap().erase(itR++);
        } else {
            ++itR;
        }
    }
    return started.Get();
}

}// namespace NOfferCache
}// namespace NTravel
