#pragma once

#include "travel/hotels/lib/cpp/data/data.h"
#include <travel/hotels/lib/cpp/offer_cache_invalidation/cache_invalidation_service.h>
#include <travel/hotels/offercache/proto/config.pb.h>
#include <travel/hotels/lib/cpp/util/flag.h>
#include <travel/hotels/lib/cpp/util/profiletimer.h>
#include <travel/hotels/lib/cpp/mon/tools.h>

#include <library/cpp/containers/concurrent_hash/concurrent_hash.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/vector.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/event.h>
#include <util/system/rwlock.h>

namespace NTravel {
namespace NOfferCache {

class TReqCache {
public:
    enum class EReqState {
        None,
        Started,  // Searcher начал исполнение
        Finished  // Searcher выполнил (возможно, с ошибкой)
    };

    TReqCache(const NTravelProto::NOfferCache::TConfig::TReqCache& cfg, const TCacheInvalidationService& cacheInvalidationService);
    ~TReqCache();

    void RegisterCounters(NMonitor::TCounterSource& counters);

    // Если ни одного запроса нет - то None
    // Если есть хоть один запрос в Started, то возвращает Started
    // Если все Finished - то Finished
    // Реализует дополнительную логику взаимного влияния interactive и background запросов
    EReqState GetRequestState(const NTravelProto::TSearchOffersReq& req, TString* reqId = nullptr) const;

    // Устанавливает время жизни записи в MaxAgeRequestStarted
    void OnRequestStarted(const NTravelProto::TSearchOffersReq& req, TInstant ts);

    // Устанавливает время жизни записи в MaxAgeRequestFinished
    void OnRequestFinished(const NTravelProto::TSearchOffersReq& req, TInstant ts, TInstant expireTimestamp);

    // Устанавливает время жизни записи в MaxAgeRequestFinishedError
    void OnRequestFinishedError(const NTravelProto::TSearchOffersReq& req, TInstant ts);
private:
    struct TCounters : public NMonitor::TCounterSource {
        NMonitor::TCounter      NReqStarted;        // Запросов в состоянии "Started"
        NMonitor::TCounter      NReqFinished;       // Запросов в состоянии "Finished"
        NMonitor::TDerivCounter NReqStartedExpired; // Запросов, которые истекли в состоянии "Started"
        NMonitor::TCounter      NBytes;             // Количество памяти, отданной под запросы

        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    static constexpr size_t BucketCount_ = 256;

    struct TRequestInfo {
        TString   Id;
        TInstant  Timestamp;
        TInstant  ExpTime;
        EReqState State;

        size_t GetAllocSize() const;
    };

    const bool Enabled_;
    const TDuration CleanupPeriod_;
    const TDuration MaxAgeRequestStarted_;
    const TDuration MaxAgeRequestStartedBackground_;
    const TDuration RelMaxAgeRequestFinished_;
    const TDuration MaxAgeRequestFinishedError_;
    const bool NewCleanupMode_;

    mutable TCounters Counters_;

    using TKeyToRequestsMap = TConcurrentHashMap<TSearcherRequestKey, TRequestInfo, BucketCount_, TRWMutex>;
    TKeyToRequestsMap Requests_;
    TConcurrentHashMap<TInstant, THashSet<TKeyToRequestsMap::TBucket*>, BucketCount_, TRWMutex> CleanupSchedule_;

    const TCacheInvalidationService& CacheInvalidationService;

    std::pair<TReqCache::EReqState, TString> GetRequestStateExact(const TSearcherRequestKey& key, TInstant now) const;

    void Put(const NTravelProto::TSearchOffersReq& req, TInstant timestamp, TInstant expTime, EReqState newState);
    void OnRecordAdded(const TRequestInfo& ri) const;
    void OnRecordRemoved(const TRequestInfo& ri, bool expired) const;

    void ExecuteCleanup(TInstant cleanupTime);
    TDuration CleanupBucket(TKeyToRequestsMap::TBucket* bucketPtr, TInstant expTime);
};

} // namespace NOfferCache
} // namespace NTravel
