#include "offer_to_room_key_mapper.h"

#include <travel/hotels/offercache/service/data.h>

#include <travel/hotels/proto2/hotels.pb.h>

#include <library/cpp/logger/global/global.h>

#include <util/string/strip.h>

namespace NTravel::NOfferCache {
    const re2::RE2 TOfferToRoomKeyMapper::TrailingBracesPattern = re2::RE2("^(.*)\\(.*\\)\\s*$");
    const re2::RE2 TOfferToRoomKeyMapper::AfterHyphenPattern = re2::RE2("^(.*) - .*\\s*$");

    TOfferToRoomKeyMapper::TOfferToRoomKeyMapper(TServiceCounters& counters)
        : Counters(counters)
    {
    }

    TString TOfferToRoomKeyMapper::GetCatRoomMappingKey(const TOffer& offer, const TString& decompressedTitle) const {
        switch (offer.OperatorId) {
            case EOperatorId::OI_OSTROVOK:
            case EOperatorId::OI_HTC_ONSITE:
            case EOperatorId::OI_ACCORHOTELSCOM:
            case EOperatorId::OI_HOTELINFO:
            case EOperatorId::OI_AGODA:
            case EOperatorId::OI_AMOMA:
            case EOperatorId::OI_PRESTIGIACOM:
            case EOperatorId::OI_OTELCOM:
            case EOperatorId::OI_TRAVELLINE_ONSITE:
                return PrepareKey(decompressedTitle);

            case EOperatorId::OI_BOOKING:
            case EOperatorId::OI_EXPEDIA:
            case EOperatorId::OI_TRAVELLINE:
            case EOperatorId::OI_DOLPHIN:
            case EOperatorId::OI_BNOVO:
            case EOperatorId::OI_TVIL:
                return PrepareKey(offer.TitleAndOriginalRoomId.GetValue().OriginalRoomId);

            case EOperatorId::OI_HOTELS101:
                return PrepareKey(CutTrailingBraces(decompressedTitle));
            case EOperatorId::OI_HOTELSCOM:
                return PrepareKey(CutAfterHyphen(CutTrailingBraces(decompressedTitle)));

            default:
                ERROR_LOG << "Unknown operator id " << offer.OperatorId << " OfferId: " << offer.OfferId << Endl;
                Counters.UnknownOperatorForCatRoomMapping.Inc();
                return PrepareKey(decompressedTitle);
        }
    }

    TString TOfferToRoomKeyMapper::PrepareKey(const TString& s) {
        return ToLowerUTF8(Strip(s));
    };

    TString TOfferToRoomKeyMapper::CutTrailingBraces(const TString& s) const {
        TString result;
        if (re2::RE2::FullMatch(s, TrailingBracesPattern, &result)) {
            return result;
        }
        return s;
    };
    TString TOfferToRoomKeyMapper::CutAfterHyphen(const TString& s) const {
        TString result;
        if (re2::RE2::FullMatch(s, AfterHyphenPattern, &result)) {
            return result;
        }
        return s;
    };
}
