#include "room_data.h"

#include <library/cpp/logger/global/global.h>

#include <util/digest/multi.h>
#include <util/stream/format.h>
#include <util/string/builder.h>

namespace NTravel::NOfferCache {
    const TString TDeduplicatorKeys::PhotoUrlPrefix = "PhotoUrlPrefix";
    const TString TDeduplicatorKeys::HotelId = "HotelId";
    const TString TDeduplicatorKeys::PermaroomMappingSubKey = "PermaroomMappingSubKey";
    const TString TDeduplicatorKeys::PermaroomVersion = "PermaroomVersion";
    const TString TDeduplicatorKeys::PermaroomId = "PermaroomId";
    const TString TDeduplicatorKeys::PhotoSizes = "PhotoSizes";
    const TString TDeduplicatorKeys::Photos = "Photos";
    const TString TDeduplicatorKeys::BinaryFeatures = "BinaryFeatures";
    const TString TDeduplicatorKeys::EnumFeatures = "EnumFeatures";
    const TString TDeduplicatorKeys::IntegerFeatures = "IntegerFeatures";
    const TString TDeduplicatorKeys::FloatFeatures = "FloatFeatures";
    const TString TDeduplicatorKeys::StringFeatures = "StringFeatures";
    const TString TDeduplicatorKeys::PermaroomInner = "PermaroomInner";
    const TString TDeduplicatorKeys::BedGroups = "BedGroups";
    const TString TDeduplicatorKeys::Name = "Name";
    const TString TDeduplicatorKeys::Description = "Description";

    TPermaroomMappingKey::TPermaroomMappingKey(EOperatorId operatorId, TString originalId, TString mappingKey)
        : OperatorId(operatorId)
        , OriginalId(std::move(originalId))
        , MappingKey(std::move(mappingKey))
    {
    }

    bool TPermaroomMappingKey::operator==(const TPermaroomMappingKey& rhs) const {
        return (OperatorId == rhs.OperatorId && OriginalId == rhs.OriginalId && MappingKey == rhs.MappingKey);
    }

    size_t TPermaroomMappingKey::Hash() const {
        return MultiHash(OperatorId, OriginalId, MappingKey);
    }

    size_t TPermaroomMappingKey::CalcTotalByteSize() const {
        return GetTotalByteSizeMulti(OperatorId, OriginalId, MappingKey);
    }

    TPermaroomMappingSubKey::TPermaroomMappingSubKey(EOperatorId operatorId, TString mappingKey)
        : OperatorId(operatorId)
        , MappingKey(std::move(mappingKey))
    {
    }

    bool TPermaroomMappingSubKey::operator==(const TPermaroomMappingSubKey& rhs) const {
        return (OperatorId == rhs.OperatorId && MappingKey == rhs.MappingKey);
    }

    size_t TPermaroomMappingSubKey::Hash() const {
        return MultiHash(OperatorId, MappingKey);
    }

    size_t TPermaroomMappingSubKey::CalcTotalByteSize() const {
        return GetTotalByteSizeMulti(OperatorId, MappingKey);
    }

    TPhoto::TSize::TSize(ui32 height, ui32 width, const TString& size)
        : Height(height)
        , Width(width)
        , Size(size)
    {
    }

    bool TPhoto::TSize::operator==(const TPhoto::TSize& rhs) const {
        return Height == rhs.Height && Width == rhs.Width && Size == rhs.Size;
    }

    size_t TPhoto::TSize::Hash() const {
        return MultiHash(Height, Width, Size);
    }

    size_t TPhoto::TSize::CalcTotalByteSize() const {
        return GetTotalByteSizeMulti(Height, Width, Size);
    }

    bool TPhoto::operator==(const TPhoto& rhs) const {
        return UrlTemplate == rhs.UrlTemplate && Sizes == rhs.Sizes;
    }

    size_t TPhoto::Hash() const {
        return MultiHash(Sizes, UrlTemplate);
    }

    TPhotoInner::TUrl::TUrl()
        : Prefix()
        , FirstPart(0)
        , SecondPart()
    {
    }

    TMaybe<TPhotoInner::TUrl> TPhotoInner::TUrl::TryFromString(const TString& value, TObjectDeduplicator& objectDeduplicator) {
        TVector<TString> parts;
        Split(value, "/", parts);
        TUrl url;
        url.Prefix = objectDeduplicator.Deduplicate(TDeduplicatorKeys::PhotoUrlPrefix, parts[0] + "//" + parts[1] + "/" + parts[2]);
        if (!::TryFromString(parts[3], url.FirstPart)) {
            DEBUG_LOG << "Failed to parse photo url. Bad first part: " << parts[3] << Endl;
            return {};
        }
        if (parts[4].length() != SecondPartBytes * 2) {
            DEBUG_LOG << "Failed to parse photo url. Bad second part: " << parts[4] << Endl;
            return {};
        }
        for (auto& i : url.SecondPart) {
            i = 0;
        }
        for (size_t i = 0; i < SecondPartBytes * 2; i++) {
            auto ch = AsciiToLower(parts[4][i]);
            int curr = 0;
            if ('0' <= ch && ch <= '9') {
                curr = ch - '0';
            } else if ('a' <= ch && ch <= 'f') {
                curr = ch - 'a' + 10;
            } else {
                DEBUG_LOG << "Failed to parse photo url. Bad second part: " << parts[4] << Endl;
                return {};
            }
            url.SecondPart[i / 2] += curr * (i % 2 == 0 ? 16 : 1);
        }
        if (url.ToString() != value) {
            DEBUG_LOG << "Failed to parse photo url. Bad result: " << url.ToString() << " != " << value << Endl;
            return {};
        }
        return url;
    }

    TString TPhotoInner::TUrl::ToString() const {
        Y_ENSURE(!Prefix.IsEmpty(), "Tried to decode empty TPhotoInner::TUrl");
        TStringBuilder builder;
        builder << Prefix.GetValue() << "/" << FirstPart << "/";
        for (auto part : SecondPart) {
            builder << Hex(part, HF_FULL);
        }
        builder << "/%s";
        TString res = builder;
        res.to_lower();
        return res;
    }

    bool TPhotoInner::TUrl::operator==(const TPhotoInner::TUrl& rhs) const {
        return Prefix == rhs.Prefix && FirstPart == rhs.FirstPart
            && std::equal(std::begin(SecondPart), std::end(SecondPart), std::begin(rhs.SecondPart), std::end(rhs.SecondPart));
    }

    size_t TPhotoInner::TUrl::Hash() const {
        size_t res = MultiHash(Prefix, FirstPart);
        for (auto x: SecondPart) {
            res = CombineHashes(res, THash<decltype(SecondPart[0])>()(x));
        }
        return res;
    }

    size_t TPhotoInner::TUrl::CalcTotalByteSize() const {
        return sizeof(TPhotoInner::TUrl);
    }

    bool TPhotoInner::operator==(const TPhotoInner& rhs) const {
        return UrlTemplate == rhs.UrlTemplate && Sizes == rhs.Sizes;
    }

    size_t TPhotoInner::Hash() const {
        return MultiHash(UrlTemplate, Sizes);
    }

    size_t TPhotoInner::CalcTotalByteSize() const {
        return GetTotalByteSizeMulti(UrlTemplate, Sizes);
    }

    bool TFeatureCommons::operator==(const TFeatureCommons& rhs) const {
        return Id == rhs.Id && Name == rhs.Name && TopFeatureImportance == rhs.TopFeatureImportance && Category == rhs.Category && CategoryName == rhs.CategoryName
            && DisplayValue == rhs.DisplayValue && IconId == rhs.IconId;
    }

    size_t TFeatureCommons::Hash() const {
        return MultiHash(Id, Name, TopFeatureImportance, Category, CategoryName, DisplayValue, IconId);
    }

    size_t TFeatureCommons::CalcTotalByteSize() const {
        return GetTotalByteSizeMulti(Id, Name, TopFeatureImportance, Category, CategoryName, DisplayValue, IconId);
    }

    bool TBinaryFeature::operator==(const TBinaryFeature& rhs) const {
        return Commons == rhs.Commons && Value == rhs.Value;
    }

    size_t TBinaryFeature::Hash() const {
        return MultiHash(Commons, Value);
    }

    size_t TBinaryFeature::CalcTotalByteSize() const {
        return GetTotalByteSizeMulti(Commons, Value);
    }

    bool TEnumFeature::operator==(const TEnumFeature& rhs) const {
        return Commons == rhs.Commons && ValueId == rhs.ValueId && ValueName == rhs.ValueName;
    }

    size_t TEnumFeature::Hash() const {
        return MultiHash(Commons, ValueId, ValueName);
    }

    size_t TEnumFeature::CalcTotalByteSize() const {
        return GetTotalByteSizeMulti(Commons, ValueId, ValueName);
    }

    bool TIntegerFeature::operator==(const TIntegerFeature& rhs) const {
        return Commons == rhs.Commons && Value == rhs.Value;
    }

    size_t TIntegerFeature::Hash() const {
        return MultiHash(Commons, Value);
    }

    size_t TIntegerFeature::CalcTotalByteSize() const {
        return GetTotalByteSizeMulti(Commons, Value);
    }

    bool TFloatFeature::operator==(const TFloatFeature& rhs) const {
        return Commons == rhs.Commons && Value == rhs.Value;
    }

    size_t TFloatFeature::Hash() const {
        return MultiHash(Commons, Value);
    }

    size_t TFloatFeature::CalcTotalByteSize() const {
        return GetTotalByteSizeMulti(Commons, Value);
    }

    bool TStringFeature::operator==(const TStringFeature& rhs) const {
        return Commons == rhs.Commons && Value == rhs.Value;
    }

    size_t TStringFeature::Hash() const {
        return MultiHash(Commons, Value);
    }

    size_t TStringFeature::CalcTotalByteSize() const {
        return GetTotalByteSizeMulti(Commons, Value);
    }

    bool TBedGroup::operator==(const TBedGroup& rhs) const {
        return Id == rhs.Id && Configuration == rhs.Configuration;
    }

    size_t TBedGroup::Hash() const {
        return MultiHash(Id, Configuration);
    }

    size_t TBedGroup::CalcTotalByteSize() const {
        return GetTotalByteSizeMulti(Id, Configuration);
    }

    bool TBedGroup::TConfigurationItem::operator==(const TBedGroup::TConfigurationItem& rhs) const {
        return Type == rhs.Type && Quantity == rhs.Quantity && NominativeCaseSingular == rhs.NominativeCaseSingular
            && GenitiveCaseSingular == rhs.GenitiveCaseSingular && GenitiveCasePlural == rhs.GenitiveCasePlural;
    }

    size_t TBedGroup::TConfigurationItem::Hash() const {
        return MultiHash(Type, Quantity, NominativeCaseSingular, GenitiveCaseSingular, GenitiveCasePlural);
    }

    size_t TBedGroup::TConfigurationItem::CalcTotalByteSize() const {
        return GetTotalByteSizeMulti(Type, Quantity, NominativeCaseSingular, GenitiveCaseSingular, GenitiveCasePlural);
    }

    bool TPermaroomInner::operator==(const TPermaroomInner& rhs) const {
        return Permalink == rhs.Permalink &&
               PermaroomId == rhs.PermaroomId &&
               NameHash == rhs.NameHash && // Comparing hashes instead of real values to avoid complex and slow decompression logic here
               DescriptionHash == rhs.DescriptionHash &&
               Photos == rhs.Photos &&
               BinaryFeatures == rhs.BinaryFeatures &&
               EnumFeatures == rhs.EnumFeatures &&
               IntegerFeatures == rhs.IntegerFeatures &&
               FloatFeatures == rhs.FloatFeatures &&
               StringFeatures == rhs.StringFeatures &&
               BedGroups == rhs.BedGroups;
    }

    size_t TPermaroomInner::Hash() const {
        // Comparing hashes instead of real values to avoid complex and slow decompression logic here
        return MultiHash(Permalink, PermaroomId, NameHash, DescriptionHash, Photos, BinaryFeatures,
                         EnumFeatures, IntegerFeatures, FloatFeatures, StringFeatures, BedGroups);
    }

    size_t TPermaroomInner::CalcTotalByteSize() const {
        auto res = sizeof(TPermaroomInner);
        res += Photos.capacity() * sizeof(decltype(Photos)::value_type);
        res += BinaryFeatures.capacity() * sizeof(decltype(BinaryFeatures)::value_type);
        res += EnumFeatures.capacity() * sizeof(decltype(EnumFeatures)::value_type);
        res += IntegerFeatures.capacity() * sizeof(decltype(IntegerFeatures)::value_type);
        res += FloatFeatures.capacity() * sizeof(decltype(FloatFeatures)::value_type);
        res += StringFeatures.capacity() * sizeof(decltype(StringFeatures)::value_type);
        res += BedGroups.capacity() * sizeof(decltype(BedGroups)::value_type);
        return res;
    }
}
