#pragma once

#include <travel/hotels/offercache/service/data.h>

#include <travel/hotels/lib/cpp/util/sizes.h>
#include <travel/hotels/lib/cpp/util/string_compressor.h>
#include <travel/hotels/lib/cpp/util/object_deduplicator.h>

#include <util/generic/guid.h>

namespace NTravel::NOfferCache {
    struct TDeduplicatorKeys {
        static const TString PhotoUrlPrefix;
        static const TString HotelId;
        static const TString PermaroomMappingSubKey;
        static const TString PermaroomVersion;
        static const TString PermaroomId;
        static const TString PhotoSizes;
        static const TString Photos;
        static const TString BinaryFeatures;
        static const TString EnumFeatures;
        static const TString IntegerFeatures;
        static const TString FloatFeatures;
        static const TString StringFeatures;
        static const TString PermaroomInner;
        static const TString BedGroups;
        static const TString Name;
        static const TString Description;
    };

    template <class T>
    using TUnique = TObjectDeduplicator::TUnique<T>;

    struct TPermaroomMappingKey {
        TPermaroomMappingKey(EOperatorId operatorId, TString originalId, TString mappingKey);

        EOperatorId OperatorId;
        TString OriginalId;
        TString MappingKey;

        bool operator==(const TPermaroomMappingKey& rhs) const;
        size_t Hash() const;
        size_t CalcTotalByteSize() const;
    };

    struct TPermaroomMappingSubKey {
        TPermaroomMappingSubKey(EOperatorId operatorId, TString mappingKey);

        EOperatorId OperatorId;
        TString MappingKey;

        bool operator==(const TPermaroomMappingSubKey& rhs) const;
        size_t Hash() const;
        size_t CalcTotalByteSize() const;
    };

    struct TPhoto {
        struct TSize {
            TSize(ui32 height, ui32 width, const TString& size);

            ui32 Height;
            ui32 Width;
            TString Size;

            bool operator==(const TSize& rhs) const;
            size_t Hash() const;
            size_t CalcTotalByteSize() const;
        };

        TVector<TSize> Sizes;
        TString UrlTemplate;

        bool operator==(const TPhoto& rhs) const;
        size_t Hash() const;
    };

    struct TPhotoInner {
        struct TUrl {
            constexpr static int SecondPartBytes = 18;
            // https://avatars.mds.yandex.net/get-altay/374295/2a0000015b1dade88e53c818ef0951922ba0/%s
            //                                          ^^^^^^ ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
            //                                        FirstPart               SecondPart
            TUnique<TString> Prefix;
            ui32 FirstPart;
            ui8 SecondPart[SecondPartBytes];

            TUrl();

            static TMaybe<TUrl> TryFromString(const TString& value, TObjectDeduplicator& objectDeduplicator);
            TString ToString() const;

            bool operator==(const TUrl& rhs) const;
            size_t Hash() const;
            size_t CalcTotalByteSize() const;
        };

        TUnique<TVector<TPhoto::TSize>> Sizes;
        TUrl UrlTemplate;

        bool operator==(const TPhotoInner& rhs) const;
        size_t Hash() const;
        size_t CalcTotalByteSize() const;
    };

    struct TFeatureCommons {
        TString Id;
        TString Name;
        TMaybe<int> TopFeatureImportance;
        TString Category;
        TString CategoryName;
        TString DisplayValue;
        TString IconId;

        bool operator==(const TFeatureCommons& rhs) const;
        size_t Hash() const;
        size_t CalcTotalByteSize() const;
    };

    struct TBinaryFeature {
        TFeatureCommons Commons;
        bool Value;

        bool operator==(const TBinaryFeature& rhs) const;
        size_t Hash() const;
        size_t CalcTotalByteSize() const;
    };

    struct TEnumFeature {
        TFeatureCommons Commons;
        TString ValueId;
        TString ValueName;

        bool operator==(const TEnumFeature& rhs) const;
        size_t Hash() const;
        size_t CalcTotalByteSize() const;
    };

    struct TIntegerFeature {
        TFeatureCommons Commons;
        int Value;

        bool operator==(const TIntegerFeature& rhs) const;
        size_t Hash() const;
        size_t CalcTotalByteSize() const;
    };

    struct TFloatFeature {
        TFeatureCommons Commons;
        double Value;

        bool operator==(const TFloatFeature& rhs) const;
        size_t Hash() const;
        size_t CalcTotalByteSize() const;
    };

    struct TStringFeature {
        TFeatureCommons Commons;
        TString Value;

        bool operator==(const TStringFeature& rhs) const;
        size_t Hash() const;
        size_t CalcTotalByteSize() const;
    };

    struct TBedGroup {
        struct TConfigurationItem {
            TString Type;
            ui32 Quantity;
            TString NominativeCaseSingular;
            TString GenitiveCaseSingular;
            TString GenitiveCasePlural;

            bool operator==(const TConfigurationItem& rhs) const;
            size_t Hash() const;
            size_t CalcTotalByteSize() const;
        };

        TString Id;
        TVector<TConfigurationItem> Configuration;

        bool operator==(const TBedGroup& rhs) const;
        size_t Hash() const;
        size_t CalcTotalByteSize() const;
    };

    struct TPermaroom {
        TPermaroomIdStr PermaroomId;
        TPermaroomVersion PermaroomVersion;
        TPermalink Permalink;
        TString Name;
        TString Description;
        TVector<TPhoto> Photos;
        TVector<TBinaryFeature> BinaryFeatures;
        TVector<TEnumFeature> EnumFeatures;
        TVector<TIntegerFeature> IntegerFeatures;
        TVector<TFloatFeature> FloatFeatures;
        TVector<TStringFeature> StringFeatures;
        TVector<TBedGroup> BedGroups;
    };

    struct TPermaroomInner {
        TPermalink Permalink;
        TUnique<TPermaroomIdStr> PermaroomId;
        TUnique<TString> Name;
        TUnique<TString> Description;
        size_t NameHash;
        size_t DescriptionHash;
        TVector<TUnique<TPhotoInner>> Photos;
        TVector<TUnique<TBinaryFeature>> BinaryFeatures;
        TVector<TUnique<TEnumFeature>> EnumFeatures;
        TVector<TUnique<TIntegerFeature>> IntegerFeatures;
        TVector<TUnique<TFloatFeature>> FloatFeatures;
        TVector<TUnique<TStringFeature>> StringFeatures;
        TVector<TUnique<TBedGroup>> BedGroups;

        bool operator==(const TPermaroomInner& rhs) const;
        size_t Hash() const;
        size_t CalcTotalByteSize() const;
    };

    template <class T>
    TVector<T> ExtractVectorUniqueValues(const TVector<TUnique<T>>& values) {
        TVector<T> result;
        std::transform(values.begin(), values.end(), std::back_inserter(result),
                       [](TUnique<T> value) { return value.GetValue(); });
        return result;
    }
}

template <>
struct NTravel::TTotalByteSize<TVector<NTravel::NOfferCache::TPhoto::TSize>, void> {
    inline size_t operator()(const TVector<NTravel::NOfferCache::TPhoto::TSize>& item) const {
        auto res = sizeof(item);
        for (const auto& x : item) {
            res += TTotalByteSize<NTravel::NOfferCache::TPhoto::TSize>()(x);
        }
        res += (item.capacity() - item.size()) * sizeof(NTravel::NOfferCache::TPhoto::TSize);
        return res;
    }
};

template <>
struct NTravel::TTotalByteSize<TVector<NTravel::NOfferCache::TBedGroup::TConfigurationItem>, void> {
    inline size_t operator()(const TVector<NTravel::NOfferCache::TBedGroup::TConfigurationItem>& item) const {
        auto res = sizeof(item);
        for (const auto& x : item) {
            res += TTotalByteSize<NTravel::NOfferCache::TBedGroup::TConfigurationItem>()(x);
        }
        res += (item.capacity() - item.size()) * sizeof(NTravel::NOfferCache::TBedGroup::TConfigurationItem);
        return res;
    }
};

template <>
struct THash<TVector<NTravel::NOfferCache::TPhoto::TSize>>: public TSimpleRangeHash {};

template <>
struct THash<TVector<NTravel::NOfferCache::TBedGroup::TConfigurationItem>>: public TSimpleRangeHash {};

template <>
struct THash<TVector<NTravel::TObjectDeduplicator::TUnique<NTravel::NOfferCache::TBedGroup>>>: public TSimpleRangeHash {};

template <>
struct THash<TVector<NTravel::TObjectDeduplicator::TUnique<NTravel::NOfferCache::TPhotoInner>>>: public TSimpleRangeHash {};

template <>
struct THash<TVector<NTravel::TObjectDeduplicator::TUnique<NTravel::NOfferCache::TBinaryFeature>>>: public TSimpleRangeHash {};

template <>
struct THash<TVector<NTravel::TObjectDeduplicator::TUnique<NTravel::NOfferCache::TEnumFeature>>>: public TSimpleRangeHash {};

template <>
struct THash<TVector<NTravel::TObjectDeduplicator::TUnique<NTravel::NOfferCache::TIntegerFeature>>>: public TSimpleRangeHash {};

template <>
struct THash<TVector<NTravel::TObjectDeduplicator::TUnique<NTravel::NOfferCache::TFloatFeature>>>: public TSimpleRangeHash {};

template <>
struct THash<TVector<NTravel::TObjectDeduplicator::TUnique<NTravel::NOfferCache::TStringFeature>>>: public TSimpleRangeHash {};

template <>
struct THash<NTravel::NOfferCache::TPermaroomMappingKey> {
    inline size_t operator()(const NTravel::NOfferCache::TPermaroomMappingKey& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TPermaroomMappingSubKey> {
    inline size_t operator()(const NTravel::NOfferCache::TPermaroomMappingSubKey& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TPhoto::TSize> {
    inline size_t operator()(const NTravel::NOfferCache::TPhoto::TSize& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TPhoto> {
    inline size_t operator()(const NTravel::NOfferCache::TPhoto& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TPhotoInner::TUrl> {
    inline size_t operator()(const NTravel::NOfferCache::TPhotoInner::TUrl& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TPhotoInner> {
    inline size_t operator()(const NTravel::NOfferCache::TPhotoInner& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TFeatureCommons> {
    inline size_t operator()(const NTravel::NOfferCache::TFeatureCommons& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TBinaryFeature> {
    inline size_t operator()(const NTravel::NOfferCache::TBinaryFeature& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TEnumFeature> {
    inline size_t operator()(const NTravel::NOfferCache::TEnumFeature& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TIntegerFeature> {
    inline size_t operator()(const NTravel::NOfferCache::TIntegerFeature& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TFloatFeature> {
    inline size_t operator()(const NTravel::NOfferCache::TFloatFeature& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TStringFeature> {
    inline size_t operator()(const NTravel::NOfferCache::TStringFeature& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TBedGroup::TConfigurationItem> {
    inline size_t operator()(const NTravel::NOfferCache::TBedGroup::TConfigurationItem& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TBedGroup> {
    inline size_t operator()(const NTravel::NOfferCache::TBedGroup& v) const {
        return v.Hash();
    }
};

template <>
struct THash<NTravel::NOfferCache::TPermaroomInner> {
    inline size_t operator()(const NTravel::NOfferCache::TPermaroomInner& v) const {
        return v.Hash();
    }
};
