#include "room_service.h"

#include "offer_to_room_key_mapper.h"

#include <library/cpp/protobuf/json/json2proto.h>

#include <util/string/strip.h>

namespace NTravel::NOfferCache {
    TRoomService::TRoomService(const NTravelProto::NOfferCache::TConfig::TRoomService& roomServiceConfig, std::function<EPartnerId(EOperatorId)> getPartnerByOperator)
        : Config_(roomServiceConfig)
        , Objects_("RoomServiceObjectDeduplicator")
        , GetPartnerByOperator_(std::move(getPartnerByOperator))
    {
        for (const auto& catRoomDs : Config_.GetCatRoomDataSources()) {
            DataSources_[catRoomDs.GetId()] = std::make_unique<TCatRoomData>("CatRoomDataSource-" + catRoomDs.GetId(), catRoomDs.GetData());
        }
        InitDataTables();
    }

    TRoomService::TRoomServiceResult TRoomService::GetPermarooms(const TCatRoomDataSourceIdStr& dsId,
                                                                 const TVector<TGetHotelPermaroomsRequest>& perHotelRequests) {
        if (!DataSources_.contains(dsId)) {
            Counters_.UnknownDSId.Inc();
            ERROR_LOG << "Unknown DSId: '" << dsId << "'. Returning no permaroom." << Endl;
            TRoomService::TRoomServiceResult result{};
            for (const auto& req : perHotelRequests) {
                for (const auto& mappingKey : req.MappingKeys) {
                    result.Mapping[mappingKey] = {};
                }
            }
            return result;
        }
        const auto& dataSource = DataSources_.at(dsId);
        TAtomicSharedPtr<TDataType> data;
        with_lock (dataSource->Mutex) {
            data = dataSource->Data;
        }
        TRoomService::TRoomServiceResult result{};
        for (const auto& req : perHotelRequests) {
            auto innerDataIt = data->find(Objects_.Deduplicate(TDeduplicatorKeys::HotelId, req.HotelId));
            if (innerDataIt == data->end()) {
                for (const auto& mappingKey: req.MappingKeys) {
                    result.Mapping[mappingKey] = {};
                }
                continue;
            }
            for (const auto& [_, dataItem]: innerDataIt->second) {
                const auto& permaroomInner = dataItem.PermaroomInner.GetValue();
                auto permaroomId = permaroomInner.PermaroomId.GetValue();
                if (!result.Permarooms.contains(permaroomId)) {
                    TPermaroom permaroom;
                    permaroom.PermaroomId = permaroomId;
                    permaroom.PermaroomVersion = dataItem.PermaroomVersion.GetValue();
                    permaroom.Permalink = permaroomInner.Permalink;
                    permaroom.Name = permaroomInner.Name.GetValue();
                    permaroom.Description = permaroomInner.Description.GetValue();
                    permaroom.Photos.reserve(permaroomInner.Photos.size());
                    for (const auto& innerPhoto : permaroomInner.Photos) {
                        auto& photo = permaroom.Photos.emplace_back();
                        auto innerPhotoValue = innerPhoto.GetValue();
                        photo.Sizes = innerPhotoValue.Sizes.GetValue();
                        photo.UrlTemplate = innerPhotoValue.UrlTemplate.ToString();
                    }
                    permaroom.BinaryFeatures = ExtractVectorUniqueValues(permaroomInner.BinaryFeatures);
                    permaroom.EnumFeatures = ExtractVectorUniqueValues(permaroomInner.EnumFeatures);
                    permaroom.IntegerFeatures = ExtractVectorUniqueValues(permaroomInner.IntegerFeatures);
                    permaroom.FloatFeatures = ExtractVectorUniqueValues(permaroomInner.FloatFeatures);
                    permaroom.StringFeatures = ExtractVectorUniqueValues(permaroomInner.StringFeatures);
                    permaroom.BedGroups = ExtractVectorUniqueValues(permaroomInner.BedGroups);
                    result.Permarooms[permaroomId] = permaroom;
                }
            }
            for (const auto& mappingKey: req.MappingKeys) {
                auto mappingSubKey = TPermaroomMappingSubKey(mappingKey.OperatorId, mappingKey.MappingKey);
                auto permaroomIt = innerDataIt->second.find(Objects_.Deduplicate(TDeduplicatorKeys::PermaroomMappingSubKey, mappingSubKey));
                if (permaroomIt == innerDataIt->second.end()) {
                    result.Mapping[mappingKey] = {};
                } else {
                    auto permaroomId = permaroomIt->second.PermaroomInner.GetValue().PermaroomId.GetValue();
                    result.Mapping[mappingKey] = permaroomId;
                    Y_ENSURE(result.Permarooms.contains(permaroomId), "PermaroomId found by mappingSubKey, but not found by hotelId");
                }
            }
        }
        return result;
    }

    void TRoomService::RegisterCounters(NMonitor::TCounterSource& source) {
        source.RegisterSource(&Counters_, "RoomService");
        Objects_.RegisterCounters(source);
        for (auto& [_, dataSource] : DataSources_) {
            dataSource->DataTable.RegisterCounters(source, dataSource->Name);
        }
    }

    void TRoomService::Start() {
        if (Started_.TrySet()) {
            for (auto& [_, dataSource] : DataSources_) {
                dataSource->DataTable.Start();
            }
        } else {
            WARNING_LOG << "Duplicate call of TRoomService::Start()" << Endl;
        }
    }

    void TRoomService::Stop() {
        if (Started_) {
            for (auto& [_, dataSource] : DataSources_) {
                dataSource->DataTable.Stop();
            }
        } else {
            WARNING_LOG << "Call of TRoomService::Stop() before Start()" << Endl;
        }
    }

    bool TRoomService::IsReady() const {
        return RoomDataReady_;
    }

    void TRoomService::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        ct->insert(MAKE_COUNTER_PAIR(NOldDataBytes));
        ct->insert(MAKE_COUNTER_PAIR(NOldDataRecords));
        ct->insert(MAKE_COUNTER_PAIR(NDataBytes));
        ct->insert(MAKE_COUNTER_PAIR(NDataRecords));
        ct->insert(MAKE_COUNTER_PAIR(NNewDataBytes));
        ct->insert(MAKE_COUNTER_PAIR(NNewDataRecords));
        ct->insert(MAKE_COUNTER_PAIR(UrlParseFailure));
        ct->insert(MAKE_COUNTER_PAIR(UnknownDSId));
        ct->insert(MAKE_COUNTER_PAIR(TooBigPhoto));
    }

    void TRoomService::InitDataTables() {
        for (auto& kv : DataSources_) {
            TCatRoomData& dataSource = *kv.second;
            auto conv = [](const NYT::TNode& node, NTravelProto::NOfferCache::TRoomDataRec* proto) {
                *proto = NProtobufJson::Json2Proto<NTravelProto::NOfferCache::TRoomDataRec>(
                    node["Permaroom"].AsString(),
                    NProtobufJson::TJson2ProtoConfig().SetCastRobust(true));
            };
            auto data = [this, &dataSource](const NTravelProto::NOfferCache::TRoomDataRec& proto) {
                TPermaroomInner permaroom{};
                auto permaroomVersion = Objects_.Deduplicate(TDeduplicatorKeys::PermaroomVersion, proto.GetPermaroomVersion());
                permaroom.PermaroomId = Objects_.Deduplicate(TDeduplicatorKeys::PermaroomId, proto.GetPermaroomId());
                permaroom.Permalink = proto.GetPermalink();
                permaroom.Name = Objects_.Deduplicate(TDeduplicatorKeys::Name, proto.GetName());
                permaroom.NameHash = THash<TString>()(proto.GetName());
                permaroom.Description = Objects_.Deduplicate(TDeduplicatorKeys::Description, proto.GetDescription());
                permaroom.DescriptionHash = THash<TString>()(proto.GetDescription());

                permaroom.Photos.reserve(proto.PhotosSize());
                for (const auto& protoPhoto : proto.GetPhotos()) {
                    TVector<TPhoto::TSize> sizes;
                    sizes.reserve(protoPhoto.SizesSize());
                    auto skip = false;
                    for (const auto& protoSize : protoPhoto.GetSizes()) {
                        if (protoSize.GetHeight() > std::numeric_limits<decltype(TPhoto::TSize::Height)>::max() ||
                            protoSize.GetWidth() > std::numeric_limits<decltype(TPhoto::TSize::Width)>::max())
                        {
                            Counters_.TooBigPhoto.Inc();
                            ERROR_LOG << "Too big photo size: " << protoSize.GetHeight() << "x" << protoSize.GetWidth()
                                      << " (PermaroomId=" << proto.GetPermaroomId() << "). Skipping this photo." << Endl;
                            skip = true;
                            break;
                        }
                        sizes.push_back(TPhoto::TSize(protoSize.GetHeight(), protoSize.GetWidth(), protoSize.GetSize()));
                    }
                    Sort(sizes.begin(), sizes.end(), [](const TPhoto::TSize& lhs, const TPhoto::TSize& rhs) {
                        return lhs.Size < rhs.Size;
                    });
                    if (skip) {
                        continue;
                    }
                    auto urlInner = TPhotoInner::TUrl::TryFromString(protoPhoto.GetUrlTemplate(), Objects_);
                    if (urlInner.Empty()) {
                        Counters_.UrlParseFailure.Inc();
                        ERROR_LOG << "Failed to parse photo url: '" << protoPhoto.GetUrlTemplate() << "' (PermaroomId="
                                  << proto.GetPermaroomId() << "). Skipping this photo." << Endl;
                        continue;
                    }
                    TPhotoInner photo{};
                    photo.Sizes = Objects_.Deduplicate(TDeduplicatorKeys::PhotoSizes, sizes);
                    photo.UrlTemplate = *urlInner.Get();
                    permaroom.Photos.push_back(Objects_.Deduplicate(TDeduplicatorKeys::Photos, photo));
                }

                auto fillCommons = [](const NTravelProto::NOfferCache::TRoomDataRec::TFeatureCommons& pbCommons, TFeatureCommons* commons) {
                    commons->Id = pbCommons.GetId();
                    commons->Name = pbCommons.GetName();
                    if (pbCommons.HasTopFeatureImportance()) {
                        commons->TopFeatureImportance = pbCommons.GetTopFeatureImportance();
                    }
                    commons->Category = pbCommons.GetCategory();
                    commons->CategoryName = pbCommons.GetCategoryName();
                    commons->DisplayValue = pbCommons.GetDisplayValue();
                    commons->IconId = pbCommons.GetIconId();
                };

                permaroom.BinaryFeatures.reserve(proto.BinaryFeaturesSize());
                for (const auto& protoFeature : proto.GetBinaryFeatures()) {
                    TBinaryFeature feature{};
                    fillCommons(protoFeature.GetCommons(), &feature.Commons);
                    feature.Value = protoFeature.GetValue();
                    permaroom.BinaryFeatures.push_back(Objects_.Deduplicate(TDeduplicatorKeys::BinaryFeatures, feature));
                }

                permaroom.EnumFeatures.reserve(proto.EnumFeaturesSize());
                for (const auto& protoFeature : proto.GetEnumFeatures()) {
                    TEnumFeature feature{};
                    fillCommons(protoFeature.GetCommons(), &feature.Commons);
                    feature.ValueId = protoFeature.GetValueId();
                    feature.ValueName = protoFeature.GetValueName();
                    permaroom.EnumFeatures.push_back(Objects_.Deduplicate(TDeduplicatorKeys::EnumFeatures, feature));
                }

                permaroom.IntegerFeatures.reserve(proto.IntegerFeaturesSize());
                for (const auto& protoFeature : proto.GetIntegerFeatures()) {
                    TIntegerFeature feature{};
                    fillCommons(protoFeature.GetCommons(), &feature.Commons);
                    feature.Value = protoFeature.GetValue();
                    permaroom.IntegerFeatures.push_back(Objects_.Deduplicate(TDeduplicatorKeys::IntegerFeatures, feature));
                }

                permaroom.FloatFeatures.reserve(proto.FloatFeaturesSize());
                for (const auto& protoFeature : proto.GetFloatFeatures()) {
                    TFloatFeature feature{};
                    fillCommons(protoFeature.GetCommons(), &feature.Commons);
                    feature.Value = protoFeature.GetValue();
                    permaroom.FloatFeatures.push_back(Objects_.Deduplicate(TDeduplicatorKeys::FloatFeatures, feature));
                }

                permaroom.StringFeatures.reserve(proto.StringFeaturesSize());
                for (const auto& protoFeature : proto.GetStringFeatures()) {
                    TStringFeature feature{};
                    fillCommons(protoFeature.GetCommons(), &feature.Commons);
                    feature.Value = protoFeature.GetValue();
                    permaroom.StringFeatures.push_back(Objects_.Deduplicate(TDeduplicatorKeys::StringFeatures, feature));
                }

                permaroom.BedGroups.reserve(proto.BedGroupsSize());
                for (const auto& protoBedGroup : proto.GetBedGroups()) {
                    TBedGroup bedGroup{};
                    bedGroup.Id = protoBedGroup.GetId();
                    bedGroup.Configuration.reserve(protoBedGroup.ConfigurationSize());
                    for (const auto& protoConfigurationItem : protoBedGroup.GetConfiguration()) {
                        TBedGroup::TConfigurationItem bedConfigurationItem{};
                        bedConfigurationItem.Type = protoConfigurationItem.GetType();
                        bedConfigurationItem.NominativeCaseSingular = protoConfigurationItem.GetNominativeCaseSingular();
                        bedConfigurationItem.GenitiveCaseSingular = protoConfigurationItem.GetGenitiveCaseSingular();
                        bedConfigurationItem.GenitiveCasePlural = protoConfigurationItem.GetGenitiveCasePlural();
                        bedConfigurationItem.Quantity = protoConfigurationItem.GetQuantity();
                        bedGroup.Configuration.push_back(bedConfigurationItem);
                    }
                    permaroom.BedGroups.push_back(Objects_.Deduplicate(TDeduplicatorKeys::BedGroups, bedGroup));
                }

                for (const auto& protoMapping : proto.GetMappings()) {
                    auto hotelId = THotelId{GetPartnerByOperator_(protoMapping.GetOperatorId()), protoMapping.GetOriginalId()};
                    auto key = TPermaroomMappingSubKey(protoMapping.GetOperatorId(), TOfferToRoomKeyMapper::PrepareKey(protoMapping.GetMappingKey()));
                    TDataItem dataItem;
                    dataItem.PermaroomInner = Objects_.Deduplicate(TDeduplicatorKeys::PermaroomInner, permaroom);
                    dataItem.PermaroomVersion = permaroomVersion;
                    auto [innerDataIt, innerDataInserted] = dataSource.NewData->insert({Objects_.Deduplicate(TDeduplicatorKeys::HotelId, hotelId), TInnerDataType()});
                    if (innerDataInserted) {
                        Counters_.NNewDataBytes += sizeof(TDataType::value_type); // sic! only value_type because it's pair<K, V> inside
                    }
                    auto [it, inserted] = innerDataIt->second.insert({Objects_.Deduplicate(TDeduplicatorKeys::PermaroomMappingSubKey, key), dataItem});
                    if (inserted) {
                        Counters_.NNewDataRecords++;
                        Counters_.NNewDataBytes += sizeof(TInnerDataType::value_type); // sic! only value_type because it's pair<K, V> inside
                    }
                }
            };
            auto finish = [this, &dataSource](bool ok, bool /*initial*/) {
                if (ok) {
                    TAtomicSharedPtr<TDataType> oldData;
                    with_lock (dataSource.Mutex) {
                        oldData = dataSource.Data;
                        std::swap(dataSource.NewData, dataSource.Data);

                        Counters_.NOldDataBytes = Counters_.NDataBytes; // will become so after update of dataSource.OldData several lines later
                        Counters_.NOldDataRecords = Counters_.NDataRecords;

                        Counters_.NDataBytes = Counters_.NNewDataBytes;
                        Counters_.NDataRecords = Counters_.NNewDataRecords;

                        Counters_.NNewDataBytes = 0; // will become so after update of dataSource.NewData several lines later
                        Counters_.NNewDataRecords = 0;
                    }

                    // We store old buckets to call destructor here on the next iteration
                    // If we just drop data here, one unlucky user request will be executing destructor of old data, which can be time-consuming
                    // So instead we hold old data for some time (till the next iteration) to call destructors here, on not time critical path
                    // Destructor can take long time, so don't do it under lock!
                    dataSource.OldData = oldData;

                    if (dataSource.Ready.TrySet()) {
                        auto allReady = true;
                        for (auto& [_, dataSource] : DataSources_) {
                            if (!dataSource->Ready) {
                                allReady = false;
                            }
                        }
                        if (allReady) {
                            Counters_.IsReady = 1;
                            RoomDataReady_.Set();
                        }
                    }
                }
                dataSource.NewData = MakeAtomicShared<TDataType>(); // sic! Not Clear(), because its dataSource.Data (after swap), so it can be still used in running requests
                Objects_.RemoveUnusedRecords();
            };
            dataSource.DataTable.SetCallbacks(conv, data, finish);
        }
        if (DataSources_.empty()) {
            Counters_.IsReady = 1;
            RoomDataReady_.Set();
        }
    }
}
