#include "searcher_client.h"

#include <util/generic/set.h>

namespace NTravel {
namespace NOfferCache {

#define JOB_LOG job->LogPrefix << ": "

TSearcherClient::THost::THost(const THostAddr& addr, const NTravelProto::NAppConfig::TConfigGrpcClient& cfg)
    : Address(addr)
    , GrpcClient(cfg)
    , IsReady(false)
{
}

TSearcherClient::THost::~THost() {
    INFO_LOG << "Destroy host" << Endl;
}

void TSearcherClient::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NTotalHosts));
    ct->insert(MAKE_COUNTER_PAIR(NReadyHosts));
    ct->insert(MAKE_COUNTER_PAIR(NRequests));
    ct->insert(MAKE_COUNTER_PAIR(NInFly));
    ct->insert(MAKE_COUNTER_PAIR(NGrpcErrors));
    ct->insert(MAKE_COUNTER_PAIR(NOverLimitRequests));
    ct->insert(MAKE_COUNTER_PAIR(NJobsSize));
}

TSearcherClient::TSearcherClient(const NTravelProto::NOfferCache::TConfig::TSearcher& cfg)
    : Timeout_(TDuration::Seconds(cfg.GetTimeoutSec()))
    , PingPeriod_(TDuration::Seconds(cfg.GetPingPeriodSec()))
    , PingTimeout_(TDuration::Seconds(cfg.GetPingTimeoutSec()))
    , Attempts_(cfg.GetAttempts())
    , JobInFlyLimit_(cfg.GetJobInFlyLimit())
    , YpAutoResolver_(cfg.GetYpAutoResolver())
    , ReadyHostCount_(0)
    , ActiveJobCount_(0)
{
    YpAutoResolver_.SetCallback([this] (const TVector<TYpConnectionInfo>& endpoints) {
        TVector<TString> addresses(endpoints.size());
        std::transform(endpoints.begin(), endpoints.end(), addresses.begin(), &ConvertToAddress);
        SetHostAddresses(addresses);
    });
}

TSearcherClient::~TSearcherClient() {
    Stop();
}

void TSearcherClient::SetPartners(const TVector<EPartnerId>& partners) {
    TWriteGuard g(Lock_);
    if (partners == Partners_) {
        return;
    }
    Partners_ = partners;
    DistributePartnersByHostsUnlocked();
}

void TSearcherClient::SetHostAddresses(const TVector<TString>& addresses) {
    THashMap<THostAddr, THostRef> oldHosts;
    {
        TWriteGuard g(Lock_);
        Hosts_.swap(oldHosts);
        NTravelProto::NAppConfig::TConfigGrpcClient grpcConfig;
        grpcConfig.SetAttempts(1);
        grpcConfig.SetTimeoutSec(Timeout_.Seconds());
        for (auto address : addresses) {
            auto it = oldHosts.find(address);
            if (it == oldHosts.end()) {
                grpcConfig.SetAddress(address);
                Hosts_[address] = new THost(address, grpcConfig);
            } else {
                Hosts_.insert(*it);
                oldHosts.erase(it);
            }
        }
        AtomicSet(ReadyHostCount_, 0);
        Counters_.NTotalHosts = Hosts_.size();
        Counters_.NReadyHosts = 0;
        DistributePartnersByHostsUnlocked();
    }
    for (auto it = oldHosts.begin(); it != oldHosts.end(); ++it) {
        it->second->GrpcClient.Shutdown();
    }
}

void TSearcherClient::DistributePartnersByHostsUnlocked() {
    Partner2HostAddr_.clear();
    if (Hosts_.empty()) {
        return;
    }
    TSet<THostAddr> hostAddrs;
    for (auto it = Hosts_.begin(); it != Hosts_.end(); ++it) {
        hostAddrs.insert(it->first);
    }
    for (auto pId : Partners_) {
        size_t hostNr = Partner2HostAddr_.size() % hostAddrs.size();
        auto it = hostAddrs.begin();
        while (hostNr > 0) {
            Y_ASSERT(it != hostAddrs.end());
            ++it;
            --hostNr;
        }
        THostAddr hostAddr = *it;
        Partner2HostAddr_[pId] = hostAddr;
        INFO_LOG << "Partner " << pId << " will be bound to searcher at " << hostAddr << Endl;
    }
    WakeUp_.Signal();// Do ping right now!
}

void TSearcherClient::RegisterCounters(NMonitor::TCounterSource& counters) {
    counters.RegisterSource(&Counters_, "SearcherClient");
}

void TSearcherClient::Start() {
    Thread_ = SystemThreadFactory()->Run(this);
    YpAutoResolver_.Start();
}

void TSearcherClient::Stop() {
    YpAutoResolver_.Stop();
    StopFlag_.Set();
    WakeUp_.Signal();
    if (Thread_) {
        Thread_->Join();
        Thread_.Reset();
    }
    SetHostAddresses({});
}

bool TSearcherClient::IsReady() const {
    return AtomicGet(ReadyHostCount_) > 0;
}

bool TSearcherClient::IsFlushed() const {
    return AtomicGet(ActiveJobCount_) == 0;
}


void TSearcherClient::Request(const TString& logPrefix, const NTravelProto::TSearchOffersRpcReq& req, const NGrpc::TClientMetadata& meta, TOnResponse onResponse) {
    if (req.SubrequestSize() == 0) {
        onResponse(NTravelProto::TSearchOffersRpcRsp());
        return;
    }

    if (AtomicGet(ActiveJobCount_) >= JobInFlyLimit_) {
        ERROR_LOG << logPrefix << ": Limit searcher request exceeded" << Endl;
        Counters_.NOverLimitRequests.Inc();
        onResponse(NTravelProto::TSearchOffersRpcRsp());
        return;
    }

    AtomicIncrement(ActiveJobCount_);
    TJobRef job = new TJob;
    job->LogPrefix = logPrefix;
    job->OnResponse = onResponse;
    job->Meta = meta;
    {
        TReadGuard g(Lock_);
        // Йохохо... у нас запросы по партнёрам вперемешку, а надо слать от каждого на свой searcher
        for (size_t pos = 0; pos < req.SubrequestSize(); ++pos) {
            auto& subReq = req.GetSubrequest(pos);
            THostAddr hostAddr = GetHostForPartnerUnlocked(subReq.GetHotelId().GetPartnerId());
            job->StateByDesiredHost[hostAddr].OriginalPositions.push_back(pos);
            job->ReqByDesiredHost[hostAddr].AddSubrequest()->CopyFrom(subReq);
            job->Resp.AddSubresponse();
        }
        Counters_.NJobsSize += GetTotalByteSize(*job);
    }

    DEBUG_LOG << JOB_LOG << "Started job with " << job->StateByDesiredHost.size() << " per host requests" << Endl;
    for (auto it = job->ReqByDesiredHost.begin(); it != job->ReqByDesiredHost.end(); ++it) {
        auto& hostReq = it->second;
        hostReq.SetSync(req.GetSync());
        hostReq.SetIncludeDebug(req.GetIncludeDebug());
        DoSingleHostRequest(job, it->first);
    }
}

void TSearcherClient::DoSingleHostRequest(const TJobRef& job, const THostAddr& desiredHostAddr) {
    THostAddr actualHostAddr;
    if (THostRef host = SelectReadyHost(desiredHostAddr, &actualHostAddr)) {
        Counters_.NRequests.Inc();
        Counters_.NInFly.Inc();
        host->GrpcClient.Request<NTravelProto::TSearchOffersRpcReq, NTravelProto::TSearchOffersRpcRsp,
                &ru::yandex::travel::hotels::OfferSearchServiceV1::Stub::AsyncSearchOffers>
                (job->ReqByDesiredHost[desiredHostAddr], job->Meta, [this, job, desiredHostAddr, actualHostAddr]
                 (const TString& grpcError, const TString& remoteFQDN, const NTravelProto::TSearchOffersRpcRsp& rpcResp) {
            Y_UNUSED(remoteFQDN);
            Counters_.NInFly.Dec();
            if (grpcError) {
                Counters_.NGrpcErrors.Inc();
            }
            ProcessSearcherResponse(job, desiredHostAddr, actualHostAddr, grpcError, rpcResp);
        });
    } else {
        ProcessSearcherResponse(job, desiredHostAddr, "", "No alive hosts", NTravelProto::TSearchOffersRpcRsp());
    }
}

void TSearcherClient::ProcessSearcherResponse(const TJobRef& job, const THostAddr& desiredHostAddr, const THostAddr& actualHostAddr,
                                              const TString& grpcError, const NTravelProto::TSearchOffersRpcRsp& resp) {
    if (!actualHostAddr.empty() && grpcError) {
        ChangeReady(actualHostAddr, false);
        bool retry = false;
        with_lock (job->Lock_) {
            auto& state = job->StateByDesiredHost[desiredHostAddr];
            ++state.Attempts;
            retry = state.Attempts <= Attempts_;
        }
        if (retry) {
            DoSingleHostRequest(job, desiredHostAddr);
            return;
        }
    }
    with_lock (job->Lock_) {
        size_t prevJobSize = GetTotalByteSize(*job);
        auto it = job->StateByDesiredHost.find(desiredHostAddr);
        if (it == job->StateByDesiredHost.end()) {
            ERROR_LOG << JOB_LOG << "Unexpected response for desired host " << desiredHostAddr << Endl;
            return;
        }
        TVector<size_t> origPositions;
        origPositions.swap(it->second.OriginalPositions);
        job->StateByDesiredHost.erase(it);
        for (size_t pos = 0; pos < origPositions.size(); ++pos) {
            auto* subResp = job->Resp.MutableSubresponse(origPositions[pos]);
            if (grpcError) {
                subResp->MutableError()->SetCode(NTravelProto::EC_GENERAL_ERROR);
                subResp->MutableError()->SetMessage("General Searcher gRPC error: " + grpcError);
            } else {
                if (pos < resp.SubresponseSize()) {
                    subResp->CopyFrom(resp.GetSubresponse(pos));
                } else {
                    subResp->MutableError()->SetCode(NTravelProto::EC_GENERAL_ERROR);
                    subResp->MutableError()->SetMessage("SubResponse count mismatch");
                }
            }
        }
        i64 curJobSize = static_cast<i64>(GetTotalByteSize(*job)) - prevJobSize;
        Counters_.NJobsSize += curJobSize;
        if (job->StateByDesiredHost.empty()) {
            DEBUG_LOG << JOB_LOG << "Got last per-host response, respond to caller" << Endl;
            job->OnResponse(job->Resp);
            AtomicDecrement(ActiveJobCount_);
            Counters_.NJobsSize -= GetTotalByteSize(*job);
        } else {
            DEBUG_LOG << JOB_LOG << "Got per-host response, waiting for " << job->StateByDesiredHost.size() << " other" << Endl;
        }
    }
}

void TSearcherClient::DoExecute() {
    TInstant t = Now();
    while (!StopFlag_) {
        {
            TReadGuard g(Lock_);
            for (auto it = Hosts_.begin(); it != Hosts_.end(); ++it) {
                PingHost(it->second);
            }
        }
        TInstant tNext = t + PingPeriod_;
        if (!WakeUp_.WaitD(tNext)) {
            t = tNext;
        }
    }
}

void TSearcherClient::PingHost(const THostRef& host) {
    NTravelProto::TPingRpcReq req;
    host->GrpcClient.Request<NTravelProto::TPingRpcReq, NTravelProto::TPingRpcRsp, &ru::yandex::travel::hotels::OfferSearchServiceV1::Stub::AsyncPing>
            (req, NGrpc::TClientMetadata(), [this, host](const TString& grpcError, const TString& remoteFQDN, const NTravelProto::TPingRpcRsp& resp) {
        Y_UNUSED(remoteFQDN);
        ChangeReady(host->Address, !grpcError && resp.GetIsReady());
    }, PingTimeout_, 1);
}

void TSearcherClient::ChangeReady(const THostAddr& hostAddr, bool isReady) {
    TReadGuard g(Lock_);
    auto it = Hosts_.find(hostAddr);
    if (it == Hosts_.end()) {
        return;// Outdated host
    }
    const THostRef& host = it->second;
    if (host->IsReady == isReady) {
        if (isReady) {
            DEBUG_LOG << "Host " << host->Address << " is still ready" << Endl;
        } else {
            DEBUG_LOG << "Host " << host->Address << " is still NOT ready" << Endl;
        }
    } else {
        host->IsReady = isReady;
        if (isReady) {
            AtomicIncrement(ReadyHostCount_);
            Counters_.NReadyHosts.Inc();
            INFO_LOG << "Host " << host->Address << " is ready now" << Endl;
        } else {
            AtomicDecrement(ReadyHostCount_);
            Counters_.NReadyHosts.Dec();
            INFO_LOG << "Host " << host->Address << " is NOT ready now" << Endl;
        }
    }
}

TSearcherClient::THostAddr TSearcherClient::GetHostForPartnerUnlocked(EPartnerId pId) const {
    auto it = Partner2HostAddr_.find(pId);
    if (it == Partner2HostAddr_.end()) {
        return "";
    }
    return it->second;
}

TSearcherClient::THostRef TSearcherClient::SelectReadyHost(const THostAddr& desiredHostAddr, THostAddr* actualHostAddr) const {
    TReadGuard g(Lock_);
    if (Hosts_.empty()) {
        return nullptr;
    }
    auto firstHostIt = Hosts_.find(desiredHostAddr);
    if (firstHostIt == Hosts_.end()) {
        firstHostIt = Hosts_.begin();
    }
    auto it = firstHostIt;
    while (true) {
        if (it->second->IsReady) {
            *actualHostAddr = it->first;
            return it->second;
        }
        ++it;
        if (it == Hosts_.end()) {
            it = Hosts_.begin();
        }
        if (it == firstHostIt) { // Вернулись туда, откуда начали
            return nullptr;
        }
    }
}

size_t TSearcherClient::TJob::CalcTotalByteSize() const {
    // Не учитываем полный размер OnResponse, эта лямбда удерживает значительное количество памяти
    size_t result = sizeof(TJob);
    result += GetByteSizeWithoutSizeof(LogPrefix) + TTotalByteSize<TString>()(Meta.CallId)
            + TTotalByteSize<TString>()(Meta.ForwardedFor);

    result += GetHashMapByteSizeWithoutElementAllocations(ReqByDesiredHost) - sizeof(decltype(ReqByDesiredHost)) - sizeof(ReqByDesiredHost);
    for (const auto& [k, v] : ReqByDesiredHost) {
        result += GetByteSizeWithoutSizeof(k);
        result += v.ByteSizeLong();
    }

    result += GetHashMapByteSizeWithoutElementAllocations(StateByDesiredHost) - sizeof(decltype(StateByDesiredHost)) - sizeof(StateByDesiredHost);
    for (const auto& [k, v] : StateByDesiredHost) {
        result += GetByteSizeWithoutSizeof(k);
        result += GetByteSizeWithoutSizeof(v);
    }

    if (Resp.IsInitialized()) {
        result += Resp.ByteSizeLong();
    }

    return result;
}

size_t TSearcherClient::TJob::THostState::CalcTotalByteSize() const {
    return sizeof(Attempts) + GetVectorByteSizeWithoutElementAllocations(OriginalPositions);
}
}// namespace NOfferCache
}// namespace NTravel
