#pragma once

#include <travel/hotels/lib/cpp/data/data.h>

#include <travel/hotels/proto2/hotels.pb.h>
#include <travel/hotels/proto2/offer_search_service.grpc.pb.h>

#include <travel/hotels/offercache/proto/config.pb.h>
#include <travel/hotels/lib/cpp/util/flag.h>
#include <travel/hotels/lib/cpp/util/sizes.h>
#include <travel/hotels/lib/cpp/mon/tools.h>
#include <travel/hotels/lib/cpp/grpc/grpc_async_client.h>
#include <travel/hotels/lib/cpp/yp_auto_resolver/yp_auto_resolver.h>

#include <util/thread/factory.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/event.h>
#include <util/system/mutex.h>

namespace NTravel {
namespace NOfferCache {

class TService;

class TSearcherClient: public IThreadFactory::IThreadAble {
    using TSearcherGrpcClient = NGrpc::TAsyncClient<ru::yandex::travel::hotels::OfferSearchServiceV1>;
public:
    using TOnResponse = std::function<void (const NTravelProto::TSearchOffersRpcRsp& resp)>;

    TSearcherClient(const NTravelProto::NOfferCache::TConfig::TSearcher& cfg);
    ~TSearcherClient();

    void SetPartners(const TVector<EPartnerId>& partners);

    void RegisterCounters(NMonitor::TCounterSource& counters);

    void Start();
    void Stop();

    bool IsReady() const;
    bool IsFlushed() const;

    void Request(const TString& logPrefix, const NTravelProto::TSearchOffersRpcReq& req, const NGrpc::TClientMetadata& meta, TOnResponse onResponse);
private:
    struct TCounters : public NMonitor::TCounterSource {
        NMonitor::TCounter      NTotalHosts;
        NMonitor::TCounter      NReadyHosts;
        NMonitor::TDerivCounter NRequests;
        NMonitor::TCounter      NInFly;
        NMonitor::TDerivCounter NGrpcErrors;
        NMonitor::TDerivCounter NOverLimitRequests;
        NMonitor::TCounter      NJobsSize;

        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    using THostAddr = TString;

    struct THost : public TThrRefBase {
        THostAddr Address;
        TSearcherGrpcClient GrpcClient;
        TAtomicFlag IsReady;

        THost(const THostAddr& addr, const NTravelProto::NAppConfig::TConfigGrpcClient& cfg);
        ~THost();
    };
    using THostRef = TIntrusivePtr<THost>;

    struct TJob: public TThrRefBase {
        struct THostState {
            TVector<size_t> OriginalPositions;
            size_t Attempts = 1;
            size_t CalcTotalByteSize() const;
        };

        TString LogPrefix;
        NGrpc::TClientMetadata Meta;
        TOnResponse OnResponse;

        THashMap<THostAddr, NTravelProto::TSearchOffersRpcReq> ReqByDesiredHost;// Not guarder

        TMutex Lock_;
        NTravelProto::TSearchOffersRpcRsp Resp;
        THashMap<THostAddr, THostState> StateByDesiredHost;

        size_t CalcTotalByteSize() const;
    };
    using TJobRef = TIntrusivePtr<TJob>;

    const TDuration Timeout_;
    const TDuration PingPeriod_;
    const TDuration PingTimeout_;
    const size_t    Attempts_;
    const int       JobInFlyLimit_;

    TAutoPtr<IThreadFactory::IThread> Thread_;
    TAtomicFlag StopFlag_;
    TAutoEvent WakeUp_;

    TCounters Counters_;

    TYpAutoResolver YpAutoResolver_;

    TRWMutex Lock_;
    TVector<EPartnerId> Partners_;
    THashMap<EPartnerId, THostAddr> Partner2HostAddr_;
    THashMap<THostAddr, THostRef> Hosts_;
    TAtomic ReadyHostCount_;
    TAtomic ActiveJobCount_;

    void DoExecute() override;
    void PingHost(const THostRef& host);
    void ChangeReady(const THostAddr& hostAddr, bool isReady);

    void SetHostAddresses(const TVector<TString>& addresses);
    void DistributePartnersByHostsUnlocked();

    void DoSingleHostRequest(const TJobRef& job, const THostAddr& desiredHostAddr);
    void ProcessSearcherResponse(const TJobRef& job, const THostAddr& desiredHostAddr,
                                 const THostAddr& actualHostAddr,
                                 const TString& grpcError, const NTravelProto::TSearchOffersRpcRsp& resp);

    THostAddr GetHostForPartnerUnlocked(EPartnerId pId) const;
    THostRef SelectReadyHost(const THostAddr& desiredHostAddr, THostAddr* actualHostAddr) const;
};

}// namespace NOfferCache
}// namespace NTravel
