#include "service.h"

#include "data.h"
#include "cache_heatmap.h"

#include <travel/hotels/proto2/hotels.pb.h>
#include <travel/hotels/offercache/proto/interconnect_bus.pb.h>

#include <travel/hotels/lib/cpp/scheduler/scheduler.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/http/client/client.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/yaml/as/tstring.h>
#include <library/cpp/resource/resource.h>

#include <util/datetime/base.h>
#include <util/charset/wide.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/deque.h>
#include <util/generic/algorithm.h>
#include <util/generic/guid.h>
#include <util/generic/ymath.h>
#include <util/system/hostname.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/string_utils/quote/quote.h>
#include <library/cpp/digest/crc32c/crc32c.h>

namespace NTravel {
namespace NOfferCache {

TService::TService(const NTravelProto::NOfferCache::TConfig& pbCfg, const TString& environment)
    : Config_(pbCfg)
    , Environment_(environment)
    , DummyPayload_(1024ULL * 1024ULL * Config_.GetOther().GetDummyPayloadSizeMBytes(), 'Z')
    , YtConfigOperators_("YtConfigOperators", pbCfg.GetYtConfigOperators())
    , YtConfigPartners_("YtConfigPartners", pbCfg.GetYtConfigPartners())
    , YtConfigOfferCacheClients_("YtConfigOfferCacheClients", pbCfg.GetYtConfigOfferCacheClients())
    , YtConfigSearchKeyRestrictions_("YtConfigSearchKeyRestrictions", pbCfg.GetYtConfigSearchKeyRestrictions())
    , YtConfigHotelWizardBan_("YtConfigHotelWizardBan", pbCfg.GetYtConfigHotelWizardBan())
    , YtConfigUserOrderCounters_("YtConfigUserOrderCounters", pbCfg.GetYtConfigUserOrderCounters())
    , IsPingEnabled_(true)
    , LastJobId_(0)
    , ServiceCounters_(*this)
    , CountersPerSource_({"source"})
    , CountersPerOperator_({"operator"})
    , CountersPerPartner_({"partner"})
    , CountersCacheHit_({"req_type", "permalink_type", "source"})
    , CountersCatRoom_({"datasource_id", "partner_offers_matching"})
    , CountersOfferShow_({"req_type", "permalink_type", "operator"})
    , CountersPerStage_({"size", "stage", "use_searcher"})
    , CountersPerBlendingStage_({"resp_mode", "full", "stage"})
    , RedirAddInfoCodec_(Config_.GetOther().GetRedirAddInfoKeyPath())
    , OutdatedRecordBuilder_(*this, Config_.GetOutdatedPrices())
    , CacheInvalidationService_(Config_.GetCacheInvalidationService())
    , Cache_(Config_.GetCache(), CacheInvalidationService_, OutdatedRecordBuilder_)
    , CacheHeatmap_(Cache_, Config_.GetCacheHeatmap())
    , ReqCache_(Config_.GetReqCache(), CacheInvalidationService_)
    // Более новые записи вытесняют более старые.
    // Если более новая запись оказалась с коротким TTL, её всё равно надо прочитать, чтобы узнать, что старая запись невалидна.
    // https://st.yandex-team.ru/TRAVELBACK-866#5f05a3b10234bc593a755fa5
    , CacheFiller_(Config_.GetOfferBusReader(), "CacheFiller", TYtQueueReaderOptions().SetCheckExpireTimestamp(false))
    , OutdatedOfferBusReader_(Config_.GetOutdatedOfferBusReader(), "OutdatedOfferBusReader", TYtQueueReaderOptions().SetCheckExpireTimestamp(false))
    , OutdatedOfferBusWriter_(Config_.GetOutdatedOfferBusWriter(), "OutdatedOfferBusWriter")
    , SearcherClient_(Config_.GetSearcher())
    , OfferReqBus_(Config_.GetOfferReqBusWriter(), "OfferReqBus")
    , ReqAnsLogger_("ReqAnsLogger", Config_.GetReqAnsLogger())
    , GrpcReqAnsLogger_("GrpcReqAnsLogger", Config_.GetGrpcReqAnsLogger())
    , PermalinkToOriginalIdsMapper_("PermalinkToOriginalIdsMapper", Config_.GetPermalinkToOriginalIdsMapper())
    , HotelsWhitelist_(Config_.GetHotelsWhitelist())
    , HotelsGreylist_("HotelsGreylist", Config_.GetHotelsGreylist())
    , HotelsBlacklist_("HotelsBlacklist", Config_.GetHotelsBlacklist())
    , PermalinkToClusterMapper_("PermalinkToClusterMapper", Config_.GetPermalinkToClusterMapper())
    , OfferCacheGrpcServer_(NGrpc::TAsyncServerConfig(Config_.GetOfferCacheGrpcServer().GetBindAddress(), Config_.GetOfferCacheGrpcServer().GetReplyThreads()))
    , RestartDetector_(Config_.GetOther().GetRestartDetectorStateFile())
    , OfferCacheClientDeduplicator_(Config_.GetOfferCacheClientDedup())
    , PartnerDataOfferFilter_(Config_.GetYtConfigTravellineRatePlans(),
                              Config_.GetYtConfigDolphinTour(),
                              Config_.GetYtConfigDolphinPansion(),
                              Config_.GetYtConfigDolphinRoom(),
                              Config_.GetYtConfigDolphinRoomCat(),
                              Config_.GetYtConfigBNovoRatePlans(),
                              Config_.GetPartnerDataFilter().GetUnkownTravellineItemIsBanned(),
                              Config_.GetPartnerDataFilter().GetUnkownDolphinItemIsBanned(),
                              Config_.GetPartnerDataFilter().GetUnkownBNovoItemIsBanned(),
                              ServiceCounters_)
    , ICBusReader_(Config_.GetICBusReader(), "ICBusReader")
    , ICBusWriter_(Config_.GetICBusWriter(), "ICBusWriter")
    , RoomService_(Config_.GetRoomService(), [this](EOperatorId operatorId) {
        return YtConfigOperators_.GetById(operatorId)->GetPartnerId();
    })
    , UserEventsStorage_(Config_.GetUserEventsStorage())
    , ObjectDeduplicator_("CommonObjectDeduplicator")
    , PromoService_(*this, Config_.GetPromoService())
    , PromoServiceGrpcServer_(NGrpc::TAsyncServerConfig(Config_.GetPromoServiceGrpcServer().GetBindAddress(), Config_.GetPromoServiceGrpcServer().GetReplyThreads()))
    , OutdatedOffersTransmitter_(OutdatedOfferBusWriter_, OutdatedRecordBuilder_, Config_.GetOutdatedOffersTransmitter())
{
    ServiceCounters_.DummyPayloadSize = TTotalByteSize<TString>()(DummyPayload_);

    YtConfigOperators_.SetOnUpdateHandler([this](bool first) {
        Y_UNUSED(first);
        TWriteGuard g(OperatorPartnerLock_);
        EnabledOperatorsDefault_.clear();
        auto operators = YtConfigOperators_.GetAll();
        for (auto it: *operators) {
            if (it.second.GetEnabled()) {
                EnabledOperatorsDefault_.insert(it.first);
            }
            GetCountersPerOperator(it.first);
        }
        UpdateBoYOperatorsUnlocked();
        if (first) {
            RoomService_.Start();
        }
    });

    YtConfigPartners_.SetOnUpdateHandler([this](bool first) {
        auto partners = YtConfigPartners_.GetAll();
        {
            TWriteGuard g(OperatorPartnerLock_);
            PartnerIdByCode_.clear();
            for (auto it: *partners) {
                PartnerIdByCode_[it.second.GetCode()] = it.first;
            }
            PermalinkToOriginalIdsMapper_.SetPartnerIdByCode(PartnerIdByCode_);
            HotelsWhitelist_.SetPartnerIdByCode(PartnerIdByCode_);
            HotelsGreylist_.SetPartnerIdByCode(PartnerIdByCode_);
            HotelsBlacklist_.SetPartnerIdByCode(PartnerIdByCode_);
            UpdateBoYOperatorsUnlocked();
        }

        TVector<EPartnerId> partnerKeys(partners->size());
        Transform(partners->begin(), partners->end(), partnerKeys.begin(), [] (const auto& element) { return element.first; });
        SearcherClient_.SetPartners(partnerKeys);
        if (first) {
            PermalinkToOriginalIdsMapper_.Start();
            HotelsWhitelist_.Start();
            HotelsGreylist_.Start();
            HotelsBlacklist_.Start();
        }
    });

    YtConfigOfferCacheClients_.SetOnUpdateHandler([this](bool first) {
        Y_UNUSED(first);
        InitSources();
    });

    for (const auto& bmp: Config_.GetExperiments().GetBumpedOperator()) {
        BumpedOperators_.push_back(bmp.GetOperator());
    }

    for (const auto& client: Config_.GetOfferCacheGrpcClients().GetClient()) {
        if (client.GetIsExact()) {
            GrpcExactClientIds_.insert(client.GetOfferCacheClientId());
        } else {
            GrpcPrefixClientIds_.push_back(client.GetOfferCacheClientId());
        }
    }

    {
        YAML::Node translations = YAML::Load(NResource::Find("resources/translations.yaml"));
        const TStringBuf pansionPrefix = "PANSION_";
        for (const auto& item: translations["keys"]) {
            const TString& key = item.first.as<TString>();
            if (!key.StartsWith(pansionPrefix)) {
                continue;
            }
            const auto id = key.substr(pansionPrefix.size());
            NTravelProto::EPansionType pansion;
            Y_VERIFY(NTravelProto::EPansionType_Parse(id, &pansion));
            for (const auto& locale: item.second) {
                if (locale.first.as<TString>() == "ru") {
                    const TString& value = locale.second.as<TString>();
                    Y_VERIFY(Pansions_.emplace(PansionToOCFormat(pansion), value).second);
                }
            }
        }
        size_t pansionsSize = 0;
        for (int pansion = NTravelProto::NOfferCache::NApi::TOCPansion::EPansion_MIN; pansion <= NTravelProto::NOfferCache::NApi::TOCPansion::EPansion_MAX; ++pansion) {
            if (NTravelProto::NOfferCache::NApi::TOCPansion::EPansion_IsValid(pansion)) {
                ++pansionsSize;
            }
        }
        Y_VERIFY(pansionsSize == Pansions_.size());
    }

    Http_.AddHandler("/read",            NHttp::ExternalWithoutTvm(), this, &TService::OnRead);
    Http_.AddHandler("/ping",            NHttp::ExternalWithoutTvm(), this, &TService::OnPing);
    Http_.AddHandler("/ping_nanny",      NHttp::ExternalWithoutTvm(), this, &TService::OnPingNanny);
    Http_.AddHandler("/ping/disable",    NHttp::Local(),    this, &TService::OnPingDisable);
    Http_.AddHandler("/shutdown",        NHttp::Local(),    this, &TService::OnShutdown);
    Http_.AddHandler("/setlog",          NHttp::Local(),    this, &TService::OnSetLogLevel);
    Http_.AddHandler("/reopen-ra-log",   NHttp::Local(),    this, &TService::OnReopenReqAnsLog);
    Http_.AddHandler("/reopen-gra-log",  NHttp::Local(),    this, &TService::OnReopenGrpcReqAnsLog);
    Http_.AddHandler("/reopen-main-log", NHttp::Local(),    this, &TService::OnReopenMainLog);
    Http_.AddHandler("/reopen-cu-log",   NHttp::Local(),    this, &TService::OnReopenCacheUsageLog);
    Http_.AddHandler("/heatmap",         NHttp::ExternalWithoutTvm(), this, &TService::OnHeatmap);
    Http_.AddHandler("/permalink",       NHttp::ExternalWithoutTvm(), this, &TService::OnGetPermalink);
    // For autotests only
    Http_.AddHandler("/wait-flush",      NHttp::Local(),    this, &TService::OnWaitFlush);
    // Adhoc
    Http_.AddHandler("/emergency-partner-disable", NHttp::Local(), this, &TService::OnEmergencyPartnerDisable);

    REGISTER_GRPC_HANDLER(OfferCacheGrpcServer_, SearchOffers, this, &TService::OnGrpcSearchOffers);
    REGISTER_GRPC_HANDLER(OfferCacheGrpcServer_, Read, this, &TService::OnGrpcRead);
    REGISTER_GRPC_HANDLER(OfferCacheGrpcServer_, Ping, this, &TService::OnGrpcPing);

    REGISTER_GRPC_HANDLER(PromoServiceGrpcServer_, Ping, this, &TService::OnGrpcPing);
    REGISTER_GRPC_HANDLER(PromoServiceGrpcServer_, DeterminePromosForOffer, this, &TService::OnPromoServiceGrpcDeterminePromosForOffer);
    REGISTER_GRPC_HANDLER(PromoServiceGrpcServer_, GetActivePromos, this, &TService::OnPromoServiceGrpcGetActivePromos);
    REGISTER_GRPC_HANDLER(PromoServiceGrpcServer_, CalculateDiscountForOffer, this, &TService::OnPromoServiceGrpcCalculateDiscountForOffer);
    REGISTER_GRPC_HANDLER(PromoServiceGrpcServer_, GetWhiteLabelPointsProps, this, &TService::OnPromoServiceGrpcGetWhiteLabelPointsProps);

    CacheFiller_.Ignore(ru::yandex::travel::hotels::TPingMessage());
    CacheFiller_.Subscribe(ru::yandex::travel::hotels::TSearcherMessage(), this, &TService::ProcessOfferBusSearcherMessage);
    CacheFiller_.Subscribe(NTravelProto::NOfferBus::TOfferInvalidationMessage(), [this](const TYtQueueMessage& busMessage) {
        NTravelProto::NOfferBus::TOfferInvalidationMessage message;
        if (!message.ParseFromString(busMessage.Bytes)) {
            throw yexception() << "Failed to parse TCacheInvalidationMessage record";
        }
        return CacheInvalidationService_.ProcessCacheInvalidationMessage(message);
    });
    CacheFiller_.Ignore(NTravelProto::TTravellineCacheEvent());

    OutdatedOfferBusReader_.Subscribe(ru::yandex::travel::hotels::TSearcherMessage(), this, &TService::ProcessOutdatedOfferBusSearcherMessage);

    ICBusReader_.Subscribe(NTravelProto::NOfferCache::TInteractiveSearchEvent(), [this](const TYtQueueMessage& busMessage) {
        NTravelProto::NOfferCache::TInteractiveSearchEvent message;
        if (!message.ParseFromString(busMessage.Bytes)) {
            throw yexception() << "Failed to parse TInteractiveSearchEvent record";
        }
        return UserEventsStorage_.OnInteractiveSearchEvent(message, busMessage.Timestamp);
    });

    OutdatedOfferBusReader_.SetReadinessNotifier([this]() {
        INFO_LOG << "OutdatedOfferBusReader is ready, starting CacheFiller" << Endl;
        InitialOutdatedOfferBusReadDone_.Set();
        CacheFiller_.Start();
    });

    CacheFiller_.SetReadinessNotifier([this]() {
        INFO_LOG << "CacheFiller is ready, starting Cache periodical jobs" << Endl;
        Cache_.StartPeriodicalJobs();
    });

    CountersPage_.RegisterSource(&ServiceCounters_, "Server");
    CountersPage_.RegisterSource(&MemUsageCounters_, "MemUsage");
    CountersPage_.RegisterSource(&CountersPerOperator_, "PerOperator");
    CountersPage_.RegisterSource(&CountersPerPartner_, "PerPartner");
    CountersPage_.RegisterSource(&CountersPerSource_, "Source");
    CountersPage_.RegisterSource(&CountersCacheHit_, "CacheHit");
    CountersPage_.RegisterSource(&CountersCatRoom_, "CatRoom");
    CountersPage_.RegisterSource(&CountersOfferShow_, "OfferShow");
    CountersPage_.RegisterSource(&CountersPerStage_, "PerStage");
    CountersPage_.RegisterSource(&CountersPerBlendingStage_, "PerBlendingStage");
    YtConfigOperators_.RegisterCounters(CountersPage_);
    YtConfigPartners_.RegisterCounters(CountersPage_);
    YtConfigOfferCacheClients_.RegisterCounters(CountersPage_);
    YtConfigSearchKeyRestrictions_.RegisterCounters(CountersPage_);
    YtConfigHotelWizardBan_.RegisterCounters(CountersPage_);
    YtConfigUserOrderCounters_.RegisterCounters(CountersPage_);
    Cache_.RegisterCounters(CountersPage_);
    CacheFiller_.RegisterCounters(CountersPage_);
    OutdatedOfferBusReader_.RegisterCounters(CountersPage_);
    OutdatedOfferBusWriter_.RegisterCounters(CountersPage_);
    ReqCache_.RegisterCounters(CountersPage_);
    Http_.RegisterCounters(CountersPage_);
    OfferCacheGrpcServer_.RegisterCounters(CountersPage_, "GrpcServer");
    SearcherClient_.RegisterCounters(CountersPage_);
    OfferReqBus_.RegisterCounters(CountersPage_);
    PermalinkToOriginalIdsMapper_.RegisterCounters(CountersPage_);
    HotelsWhitelist_.RegisterCounters(CountersPage_);
    HotelsGreylist_.RegisterCounters(CountersPage_);
    HotelsBlacklist_.RegisterCounters(CountersPage_);
    PermalinkToClusterMapper_.RegisterCounters(CountersPage_);
    RestartDetector_.RegisterCounters(CountersPage_);
    PartnerDataOfferFilter_.RegisterCounters(CountersPage_);
    TScheduler::Instance().RegisterCounters(CountersPage_);
    ICBusReader_.RegisterCounters(CountersPage_);
    ICBusWriter_.RegisterCounters(CountersPage_);
    RoomService_.RegisterCounters(CountersPage_);
    UserEventsStorage_.RegisterCounters(CountersPage_);
    ReqAnsLogger_.RegisterCounters(CountersPage_);
    GrpcReqAnsLogger_.RegisterCounters(CountersPage_);
    CacheInvalidationService_.RegisterCounters(CountersPage_);
    PromoService_.RegisterCounters(CountersPage_);
    PromoServiceGrpcServer_.RegisterCounters(CountersPage_, "PromoServiceGrpcServer");
    OutdatedOffersTransmitter_.RegisterCounters(CountersPage_);
    ObjectDeduplicator_.RegisterCounters(CountersPage_);

    CountersPage_.AddToHttpService(MonWebService_);

    {
        auto permalinkMapper = [this](NPermalinkMappers::TPermalinkToOriginalIdsMapper::TKey* key) {
            *key = PermalinkToClusterMapper_.GetClusterPermalink(*key);
        };
        PermalinkToOriginalIdsMapper_.SetMappingFilters(permalinkMapper, NPermalinkMappers::TPermalinkToOriginalIdsMappingRecJoiner::Join);
        HotelsWhitelist_.SetMappingFilters(permalinkMapper, NPermalinkMappers::TPermalinkToOriginalIdsMappingRecJoiner::Join);
        HotelsGreylist_.SetMappingFilters(permalinkMapper, NPermalinkMappers::TPermalinkToOriginalIdsMappingRecJoiner::Join);
        HotelsBlacklist_.SetMappingFilters(permalinkMapper, NPermalinkMappers::TPermalinkToOriginalIdsMappingRecJoiner::Join);
    }

    auto onNewPermalinkMapping = [this]() {
        INFO_LOG << "Started reloading of mappings" << Endl;
        PermalinkToOriginalIdsMapper_.Reload();
        HotelsWhitelist_.Reload();
        HotelsGreylist_.Reload();
        HotelsBlacklist_.Reload();
    };
    PermalinkToClusterMapper_.SetOnFinishHandler(onNewPermalinkMapping);
}

void TService::Start() {
    TScheduler::Instance().Start();
    MonWebService_.Start(Config_.GetOther().GetMonitoringPort());
    YtConfigOperators_.Start();
    YtConfigPartners_.Start();
    YtConfigOfferCacheClients_.Start();
    YtConfigSearchKeyRestrictions_.Start();
    YtConfigHotelWizardBan_.Start();
    YtConfigUserOrderCounters_.Start();
    CacheInvalidationService_.Start();
    OutdatedOffersTransmitter_.Start();
    Cache_.Start();
    OutdatedOfferBusReader_.Start();
    OutdatedOfferBusWriter_.Start();
    SearcherClient_.Start();
    OfferReqBus_.Start();
    ReqAnsLogger_.Start();
    GrpcReqAnsLogger_.Start();
    ICBusReader_.Start();
    ICBusWriter_.Start();
    PromoService_.Start();

    PermalinkToClusterMapper_.Start();
    PartnerDataOfferFilter_.Start();

    Http_.Start(Config_.GetHttp());
    OfferCacheGrpcServer_.Start();
    PromoServiceGrpcServer_.Start();

    ObjectDeduplicatorGcThread_ = SystemThreadFactory()->Run([this]() {
        while (!ShuttingDownEvent_.WaitT(TDuration::Seconds(Config_.GetOther().GetCommonObjectDeduplicatorGcIntervalSec()))) {
            TProfileTimer timer;
            ObjectDeduplicator_.RemoveUnusedRecords();
            INFO_LOG << "ObjectDeduplicator cleanup done in " << timer.Get() << Endl;
        }
    });
}

void TService::Wait() {
    StopEvent_.WaitI();
}

void TService::Stop() {
    ShuttingDownEvent_.Signal();
    INFO_LOG << "Stop called" << Endl;
    RestartDetector_.ReportShutdown();
    INFO_LOG << "Stopping Scheduler..." << Endl;
    TScheduler::Instance().Stop();
    INFO_LOG << "Stopping ObjectDeduplicatorGcThread..." << Endl;
    if (ObjectDeduplicatorGcThread_) {
        ObjectDeduplicatorGcThread_->Join();
    }
    INFO_LOG << "Stopping PromoServiceGrpcServer..." << Endl;
    PromoServiceGrpcServer_.Stop();
    INFO_LOG << "Stopping OfferCacheGrpcServer..." << Endl;
    OfferCacheGrpcServer_.Stop();
    INFO_LOG << "Stopping Http..." << Endl;
    Http_.Stop();
    INFO_LOG << "Stopping PartnerDataOfferFilter..." << Endl;
    PartnerDataOfferFilter_.Stop();
    INFO_LOG << "Stopping PermalinkToClusterMapper..." << Endl;
    PermalinkToClusterMapper_.Stop();
    INFO_LOG << "Stopping HotelsWhitelist..." << Endl;
    HotelsWhitelist_.Stop();
    INFO_LOG << "Stopping HotelsGreylist..." << Endl;
    HotelsGreylist_.Stop();
    INFO_LOG << "Stopping HotelsBlacklist..." << Endl;
    HotelsBlacklist_.Stop();
    INFO_LOG << "Stopping PermalinkToOriginalIdsMapper..." << Endl;
    PermalinkToOriginalIdsMapper_.Stop();
    INFO_LOG << "Stopping ReqAnsLogger..." << Endl;
    ReqAnsLogger_.Stop();
    INFO_LOG << "Stopping RoomService_..." << Endl;
    RoomService_.Stop();
    INFO_LOG << "Stopping ICBusWriter_..." << Endl;
    ICBusWriter_.Stop();
    INFO_LOG << "Stopping ICBusReader_..." << Endl;
    ICBusReader_.Stop();
    INFO_LOG << "Stopping GrpcReqAnsLogger..." << Endl;
    GrpcReqAnsLogger_.Stop();
    INFO_LOG << "Stopping OfferReqBus..." << Endl;
    OfferReqBus_.Stop();
    INFO_LOG << "Stopping SearcherClient..." << Endl;
    SearcherClient_.Stop();
    INFO_LOG << "Stopping CacheFiller..." << Endl;
    CacheFiller_.Stop();
    INFO_LOG << "Stopping OutdatedOfferBusReader..." << Endl;
    OutdatedOfferBusReader_.Stop();
    INFO_LOG << "Stopping OutdatedOfferBusWriter..." << Endl;
    OutdatedOfferBusWriter_.Stop();
    INFO_LOG << "Stopping Cache..." << Endl;
    Cache_.Stop();
    INFO_LOG << "Stopping OutdatedOffersTransmitter..." << Endl;
    OutdatedOffersTransmitter_.Stop();
    INFO_LOG << "Stopping CacheInvalidationService..." << Endl;
    CacheInvalidationService_.Stop();
    INFO_LOG << "Stopping PromoService..." << Endl;
    PromoService_.Stop();
    INFO_LOG << "Stopping YtConfigHotelWizardBan..." << Endl;
    YtConfigHotelWizardBan_.Stop();
    INFO_LOG << "Stopping YtConfigUserOrderCounters..." << Endl;
    YtConfigUserOrderCounters_.Stop();
    INFO_LOG << "Stopping YtConfigSearchKeyRestrictions..." << Endl;
    YtConfigSearchKeyRestrictions_.Stop();
    INFO_LOG << "Stopping YtConfigOfferCacheClients..." << Endl;
    YtConfigOfferCacheClients_.Stop();
    INFO_LOG << "Stopping YtConfigPartners..." << Endl;
    YtConfigPartners_.Stop();
    INFO_LOG << "Stopping YtConfigOperators..." << Endl;
    YtConfigOperators_.Stop();
    INFO_LOG << "Stopping MonWebService..." << Endl;
    MonWebService_.Stop();
    INFO_LOG << "Stop finished" << Endl;
    StopEvent_.Signal();
}

TService::~TService() {
}

void TService::OnPing(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    with_lock (LastPingLock_) {
        LastPing_.Reset();
    }
    SeenBalancerPing_.Set();
    TString text;
    bool isReady = IsReady(&text);
    responseCb(NHttp::TResponse::CreateText(text, isReady ? HTTP_OK : HTTP_BAD_GATEWAY));
}

void TService::OnPingNanny(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    TString text;
    bool isReady = IsReady(&text);
    // Если в конфиге неверные адреса сёрчера (все dead) - то не отвечаем няне и предотвращаем раскатку HOTELS-3304
    AddReadyFlag(SearcherClient_.IsReady(), "SearcherClient", &isReady, &text);
    // Если нам не приходил пинг от балансера, не отвечаем няне, чтобы она дождалась, пока балансер узнает про нас TRAVELBACK-132
    AddReadyFlag(SeenBalancerPing_, "BalancerPing", &isReady, &text);
    responseCb(NHttp::TResponse::CreateText(text, isReady ? HTTP_OK : HTTP_BAD_GATEWAY));
}

void TService::OnPingDisable(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    IsPingEnabled_.Clear();
    INFO_LOG << "Ping disabled by request" << Endl;
    responseCb(NHttp::TResponse::CreateText("Ping disabled\n"));
}

void TService::OnShutdown(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    if (IsShuttingDown_) {
        responseCb(NHttp::TResponse::CreateText("Already shutting down\n"));
        return;
    }
    IsShuttingDown_.Set();
    INFO_LOG << "Shutting down!" << Endl;
    SystemThreadFactory()->Run([this](){
        TDuration dur = TDuration::Seconds(12);
        IsPingEnabled_.Clear();
        INFO_LOG << "Ping disabled, wait " << dur << Endl;
        StopEvent_.WaitT(dur);// Даём время, чтобы балансер увидел, что сюда ходить не надо

        Http_.Shutdown();
        OfferCacheGrpcServer_.Shutdown();
        PromoServiceGrpcServer_.Shutdown();
        dur = TDuration::Seconds(7);
        INFO_LOG << "Http & gRPC Shutdown called,  wait " << dur << Endl;
        StopEvent_.WaitT(dur);// Даём время завершить текущие соединения

        Stop(); // Завершаемся, закрывая все соединения
    });
    responseCb(NHttp::TResponse::CreateText("Shutdown procedure started\n"));
}

void TService::OnSetLogLevel(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    if (auto v = httpReq.Query().Get("level")) {
        auto p = FromString<ELogPriority>(v);
        DoInitGlobalLog(CreateLogBackend(Config_.GetOther().GetMainLogFile(), p, true));
        responseCb(NHttp::TResponse::CreateText("Log level was changed to " + ToString(v) + "\n"));
    } else {
        responseCb(THttpResponse(HTTP_BAD_REQUEST));
    }
}

void TService::OnReopenReqAnsLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    INFO_LOG << "Reopening ReqAns log" << Endl;
    ReqAnsLogger_.Reopen();
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnReopenGrpcReqAnsLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    INFO_LOG << "Reopening GrpcReqAns log" << Endl;
    GrpcReqAnsLogger_.Reopen();
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnReopenCacheUsageLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    INFO_LOG << "Reopening CacheClean log" << Endl;
    Cache_.ReopenCacheUsageLog();
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnReopenMainLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    INFO_LOG << "Reopening main log" << Endl;
    TLoggerOperator<TGlobalLog>::Log().ReopenLog();
    responseCb(THttpResponse(HTTP_OK));
}

bool TService::IsReady(TString* reason) const {
    bool ready = true;
    AddReadyFlag(YtConfigOperators_.IsReady(), "YtConfigOperators", &ready, reason);
    AddReadyFlag(YtConfigPartners_.IsReady(), "YtConfigPartners", &ready, reason);
    AddReadyFlag(YtConfigOfferCacheClients_.IsReady(), "YtConfigOfferCacheClients", &ready, reason);
    AddReadyFlag(YtConfigSearchKeyRestrictions_.IsReady(), "YtConfigSearchKeyRestrictions", &ready, reason);
    AddReadyFlag(YtConfigHotelWizardBan_.IsReady(), "YtConfigHotelWizardBan", &ready, reason);
    AddReadyFlag(YtConfigUserOrderCounters_.IsReady(), "YtConfigUserOrderCounters", &ready, reason);
    AddReadyFlag(CacheFiller_.IsReady(), "CacheFiller", &ready, reason);
    AddReadyFlag(OutdatedOfferBusReader_.IsReady(), "OutdatedOfferBusReader", &ready, reason);
    AddReadyFlag(PermalinkToOriginalIdsMapper_.IsReady(), "PermalinkToOriginalIdsMapper", &ready, reason);
    AddReadyFlag(HotelsWhitelist_.IsReady(), "HotelsWhitelist", &ready, reason);
    AddReadyFlag(HotelsGreylist_.IsReady(), "HotelsGreylist", &ready, reason);
    AddReadyFlag(HotelsBlacklist_.IsReady(), "HotelsBlacklist", &ready, reason);
    PartnerDataOfferFilter_.SetIsReadyFlags([this, &ready, &reason](const TString& name, bool isReady) {
        AddReadyFlag(isReady, "PartnerDataOfferFilter-" + name, &ready, reason);
    });
    AddReadyFlag(IsPingEnabled_, "IsPingEnabled", &ready, reason);
    AddReadyFlag(ICBusReader_.IsReady(), "ICBusReader", &ready, reason);
    AddReadyFlag(RoomService_.IsReady(), "RoomService", &ready, reason);
    AddReadyFlag(CacheInvalidationService_.IsReady(), "CacheInvalidationService", &ready, reason);
    AddReadyFlag(PromoService_.IsReady(), "PromoService", &ready, reason);
    AddReadyFlag(OutdatedOffersTransmitter_.IsReady(), "OutdatedOffersTransmitter", &ready, reason);
    return ready;
}

void TService::AddReadyFlag(bool subReady, const TString& name, bool* ready, TString* reason) const {
    *ready = *ready && subReady;
    if (reason) {
        *reason += name;
        *reason += subReady ? ": Ready\n" : ": NOT ready\n";
    }
}

bool TService::IsPinged() const {
    with_lock (LastPingLock_) {
        return LastPing_.Get() < TDuration::Seconds(30);
    }
}

void TService::OnRead(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    THttpReadJobRef job = new THttpReadJob(*this, responseCb, GetFakeableNow(), AtomicAdd(LastJobId_, 1));
    job->Start(httpReq);
}

void TService::DoSearcherRequest(const TString& logPrefix, const NTravelProto::TSearchOffersRpcReq& rpcReq, const NGrpc::TClientMetadata& meta, TSearcherClient::TOnResponse respCb) {
    for (const auto& subReq: rpcReq.GetSubrequest()) {
        auto sourceCounters = GetSourceCounters(subReq);
        sourceCounters->NSearcherSubReq.Inc();
        sourceCounters->NSearcherSubReqInfly.Inc();
    }
    ServiceCounters_.NSearcherRequests.Inc();
    ServiceCounters_.NSearcherInFly.Inc();
    SearcherClient_.Request(logPrefix, rpcReq, meta, [this, logPrefix, respCb, rpcReq](const NTravelProto::TSearchOffersRpcRsp& rpcResp) {
        ServiceCounters_.NSearcherInFly.Dec();
        for (const auto& subReq: rpcReq.GetSubrequest()) {
            auto sourceCounters = GetSourceCounters(subReq);
            sourceCounters->NSearcherSubReqInfly.Dec();
        }

        if (rpcResp.SubresponseSize() != rpcReq.SubrequestSize()) {
            ERROR_LOG << logPrefix << ": Searcher request returned invalid response: "
                      << rpcResp.SubresponseSize() << " subresps for " << rpcReq.SubrequestSize() << " subrequests" << Endl;
        }
        for (size_t pos = 0; pos < rpcResp.SubresponseSize() && pos < rpcReq.SubrequestSize(); ++pos) {
            const auto& subReq = rpcReq.GetSubrequest(pos);
            const auto& subResp = rpcResp.GetSubresponse(pos);
            auto sourceCounters = GetSourceCounters(subReq);
            if (subResp.HasError()) {
                sourceCounters->NSearcherSubRespError.Inc();
            } else {
                sourceCounters->NSearcherSubRespOK.Inc();
            }
        }
        if (respCb) {
            respCb(rpcResp);
        }
    });
}

const TVector<EOperatorId>& TService::GetBumpedOperators() const {
    return BumpedOperators_;
}

size_t TService::GetOperatorOrder(EOperatorId opId) const {
    auto op = YtConfigOperators_.GetById(opId);
    if (op) {
        return op->GetOrder();
    }
    return YtConfigOperators_.Size() + opId;// Порядок для этого оператора не задан явно, будет определяться на основании его Id
}

std::shared_ptr<const NTravelProto::NConfig::TOperator> TService::GetOperator(EOperatorId opId) const {
    auto op = YtConfigOperators_.GetById(opId);
    if (op) {
        return op;
    }
    // TODO get rid of it (?)
    static auto def = std::make_shared<NTravelProto::NConfig::TOperator>();
    return def;
}

THashSet<EOperatorId> TService::GetEnabledOperatorsDefault() const {
    TReadGuard g(OperatorPartnerLock_);
    return EnabledOperatorsDefault_;
}

bool TService::IsBoYPartner(EPartnerId pId) const {
    auto p = YtConfigPartners_.GetById(pId);
    return p && p->GetIsBoY();
}

bool TService::IsBoYDirectPartner(EPartnerId pId) const {
    auto p = YtConfigPartners_.GetById(pId);
    return p && p->GetIsBoYDirect();

}

bool TService::IsBoYOperator(EOperatorId opId) const {
    TReadGuard g(OperatorPartnerLock_);
    return BoYOperators_.contains(opId);
}

bool TService::IsBoYDirectOperator(EOperatorId opId) const {
    TReadGuard g(OperatorPartnerLock_);
    return BoYDirectOperators_.contains(opId);
}

THashSet<EOperatorId> TService::GetBoYOperators() const {
    TReadGuard g(OperatorPartnerLock_);
    return BoYOperators_;
}

bool TService::GetPartnerIdByCode(const TString& code, EPartnerId* pId) const {
    TReadGuard g(OperatorPartnerLock_);
    auto it = PartnerIdByCode_.find(code);
    if (it == PartnerIdByCode_.end()) {
        return false;
    };
    *pId = it->second;
    return true;
}

THashSet<EPartnerId> TService::GetPartnersForOperators(const THashSet<EOperatorId>& opIds) const {
    THashSet<EPartnerId> result;
    for (auto opId: opIds) {
        auto op = YtConfigOperators_.GetById(opId);
        if (op) {
            result.insert(op->GetPartnerId());
        }
    }
    return result;
}

THashSet<EOperatorId> TService::GetOperatorsForPartners(const THashSet<EPartnerId>& partnerIds) const {
    THashSet<EOperatorId> result;
    auto operators = YtConfigOperators_.GetAll();
    for (const auto& [opId, op]: *operators) {
        if (partnerIds.contains(op.GetPartnerId())) {
            result.insert(opId);
        }
    }
    return result;
}


TString TService::GetPartnerCode(EPartnerId pId) const {
    auto p = YtConfigPartners_.GetById(pId);
    if (p) {
        return p->GetCode();
    }
    return "";
}

std::shared_ptr<const NTravelProto::NConfig::TPartner> TService::GetPartner(EPartnerId pId) const {
    auto p = YtConfigPartners_.GetById(pId);
    if (p) {
        return p;
    }
    // TODO get rid of it (?)
    static auto def = std::make_shared<NTravelProto::NConfig::TPartner>();
    return def;
}

void TService::OnGrpcSearchOffers(const NTravelProto::TSearchOffersRpcReq& origReq, const NGrpc::TServerReqMetadata& srvMeta, const TOfferCacheGrpcServer::TResponseCb<NTravelProto::TSearchOffersRpcRsp>& responseCb) {
    TGrpcJobRef job = new TGrpcJob(*this, AtomicAdd(LastJobId_, 1), responseCb, GetFakeableNow());
    job->Start(origReq, srvMeta);
}

void TService::OnGrpcPing(const NTravelProto::TPingRpcReq& req, const NGrpc::TServerReqMetadata& srvMeta, const TOfferCacheGrpcServer::TResponseCb<NTravelProto::TPingRpcRsp>& responseCb) {
    Y_UNUSED(req, srvMeta);
    NTravelProto::TPingRpcRsp resp;
    resp.SetIsReady(IsReady());
    responseCb(resp, NGrpc::TServerRespMetadata::BuildOk());
}

void TService::OnGrpcRead(const NTravelProto::NOfferCache::NApi::TReadReq& req, const NGrpc::TServerReqMetadata& srvMeta, const TOfferCacheGrpcServer::TResponseCb<NTravelProto::NOfferCache::NApi::TReadResp>& responseCb) {
    TIntrusivePtr<TGrpcReadJob> job = new TGrpcReadJob(*this, responseCb, GetFakeableNow(), AtomicAdd(LastJobId_, 1));
    job->Start(req, srvMeta);
}

void TService::OnPromoServiceGrpcDeterminePromosForOffer(const NTravelProto::NPromoService::TDeterminePromosForOfferReq& req,
                                                         const NGrpc::TServerReqMetadata& srvMeta,
                                                         const TPromoServiceGrpcServer::TResponseCb<NTravelProto::NPromoService::TDeterminePromosForOfferRsp>& responseCb) {
    Y_UNUSED(srvMeta);
    try {
        NTravelProto::NPromoService::TDeterminePromosForOfferRsp rsp;
        PromoService_.DeterminePromosForOffer(req, &rsp);
        responseCb(rsp, NGrpc::TServerRespMetadata::BuildOk());
    } catch (...) {
        ERROR_LOG << "Bad DeterminePromosForOffer request: " << CurrentExceptionMessage() << Endl;
        responseCb(
            NTravelProto::NPromoService::TDeterminePromosForOfferRsp(),
            NGrpc::TServerRespMetadata::BuildFailed(grpc::Status(grpc::StatusCode::INVALID_ARGUMENT, CurrentExceptionMessage()))
        );
    }
}

void TService::OnPromoServiceGrpcGetActivePromos(const NTravelProto::NPromoService::TGetActivePromosReq& req,
                                                 const NGrpc::TServerReqMetadata& srvMeta,
                                                 const TPromoServiceGrpcServer::TResponseCb<NTravelProto::NPromoService::TGetActivePromosRsp>& responseCb) {
    Y_UNUSED(srvMeta);
    try {
        NTravelProto::NPromoService::TGetActivePromosRsp rsp;
        PromoService_.GetActivePromos(req, &rsp);
        responseCb(rsp, NGrpc::TServerRespMetadata::BuildOk());
    } catch (...) {
        ERROR_LOG << "Bad GetActivePromos request: " << CurrentExceptionMessage() << Endl;
        responseCb(
            NTravelProto::NPromoService::TGetActivePromosRsp(),
            NGrpc::TServerRespMetadata::BuildFailed(grpc::Status(grpc::StatusCode::INVALID_ARGUMENT, CurrentExceptionMessage()))
        );
    }
}

void TService::OnPromoServiceGrpcCalculateDiscountForOffer(const NTravelProto::NPromoService::TCalculateDiscountForOfferReq& req,
                                                 const NGrpc::TServerReqMetadata& srvMeta,
                                                 const TPromoServiceGrpcServer::TResponseCb<NTravelProto::NPromoService::TCalculateDiscountForOfferRsp>& responseCb) {
    Y_UNUSED(srvMeta);
    try {
        NTravelProto::NPromoService::TCalculateDiscountForOfferRsp rsp;
        PromoService_.CalculateDiscountForOffer(req, &rsp);
        responseCb(rsp, NGrpc::TServerRespMetadata::BuildOk());
    } catch (...) {
        ERROR_LOG << "Bad CalculateDiscountForOffer request: " << CurrentExceptionMessage() << Endl;
        responseCb(
            NTravelProto::NPromoService::TCalculateDiscountForOfferRsp(),
            NGrpc::TServerRespMetadata::BuildFailed(grpc::Status(grpc::StatusCode::INVALID_ARGUMENT, CurrentExceptionMessage()))
        );
    }
}

void TService::OnPromoServiceGrpcGetWhiteLabelPointsProps(const NTravelProto::NPromoService::TGetWhiteLabelPointsPropsReq& req,
                                                          const NGrpc::TServerReqMetadata& srvMeta,
                                                          const TPromoServiceGrpcServer::TResponseCb<NTravelProto::NPromoService::TGetWhiteLabelPointsPropsRsp>& responseCb) {
    Y_UNUSED(srvMeta);
    try {
        NTravelProto::NPromoService::TGetWhiteLabelPointsPropsRsp rsp;
        PromoService_.GetWhiteLabelPointsProps(req, &rsp);
        responseCb(rsp, NGrpc::TServerRespMetadata::BuildOk());
    } catch (...) {
        ERROR_LOG << "Bad GetWhiteLabelPointsProps request: " << CurrentExceptionMessage() << Endl;
        responseCb(
                NTravelProto::NPromoService::TGetWhiteLabelPointsPropsRsp(),
                NGrpc::TServerRespMetadata::BuildFailed(grpc::Status(grpc::StatusCode::INVALID_ARGUMENT, CurrentExceptionMessage()))
        );
    }
}

bool TService::ProcessOfferBusSearcherMessage(const TYtQueueMessage& busMessage) {
    TProfileTimer timer;
    ru::yandex::travel::hotels::TSearcherMessage message;
    if (!message.ParseFromString(busMessage.Bytes)) {
        throw yexception() << "Failed to parse TSearcherMessage record";
    }
    ServiceCounters_.MsgParseNs += timer.Step().NanoSeconds();
    TProcessingStat stat;
    ProcessOfferBusSearcherMessageParsed(busMessage.Timestamp, busMessage.ExpireTimestamp, busMessage.MessageId, message, &stat);
    ServiceCounters_.MsgProcessCacheNs += stat.CacheTime.NanoSeconds();
    ServiceCounters_.MsgProcessReqCacheNs += stat.ReqCacheTime.NanoSeconds();
    ServiceCounters_.MsgProcessEncodingNs += stat.EncodingTime.NanoSeconds();
    ServiceCounters_.MsgProcessOtherNs += (timer.Get() - stat.CacheTime -
                                           stat.ReqCacheTime - stat.EncodingTime).NanoSeconds();
    return true;
}

void TService::ProcessOfferBusSearcherMessageParsed(TInstant timestamp, TInstant messageExpireTimestamp, const TString& messageId, const ru::yandex::travel::hotels::TSearcherMessage& message, TProcessingStat* stat) {
    auto now = Now();
    auto records = BuildAndCacheRecordsFromSearcherMessage(timestamp, messageExpireTimestamp, message, now, stat, true,
                                                           [this, now](const TCacheRecordRef& rec) -> TMaybe<TCacheRecordRef> {
                                                               if (rec->ExpireTimestamp < now && OutdatedRecordBuilder_.AreOutdatedPricesEnabled()) {
                                                                   auto outdatedRec = OutdatedRecordBuilder_.ConvertToOutdated(rec);
                                                                   if (outdatedRec.Defined()) {
                                                                       return outdatedRec;
                                                                   }
                                                               }
                                                               return rec;
                                                           }, false);

    OutdatedOffersTransmitter_.ProcessOfferBusMessage(timestamp, messageId, message, records);
}

bool TService::ProcessOutdatedOfferBusSearcherMessage(const TYtQueueMessage& busMessage) {
    ru::yandex::travel::hotels::TSearcherMessage message;
    if (!message.ParseFromString(busMessage.Bytes)) {
        throw yexception() << "Failed to parse TSearcherMessage record";
    }
    if (OutdatedRecordBuilder_.AreOutdatedPricesEnabled() && !InitialOutdatedOfferBusReadDone_) {
        TProcessingStat stat;
        auto now = Now();
        BuildAndCacheRecordsFromSearcherMessage(busMessage.Timestamp, busMessage.ExpireTimestamp, message, now, &stat, false,
                                                [this](const TCacheRecordRef& rec) -> TMaybe<TCacheRecordRef> {
                                                    return OutdatedRecordBuilder_.ConvertToOutdated(rec);
                                                }, true);
    }
    OutdatedOffersTransmitter_.ProcessOutdatedOfferBusMessage(busMessage.Timestamp, busMessage.MessageId, message);
    return true;
}

TVector<TCacheRecordRef> TService::BuildAndCacheRecordsFromSearcherMessage(TInstant timestamp,
                                                                           TInstant messageExpireTimestamp,
                                                                           const ru::yandex::travel::hotels::TSearcherMessage& message,
                                                                           TInstant now,
                                                                           TProcessingStat* stat,
                                                                           bool updateReqCacheInfo,
                                                                           std::function<TMaybe<TCacheRecordRef>(const TCacheRecordRef&)> preprocessRecord,
                                                                           bool fromOutdatedOffersBus) {
    TProfileTimer episodicTimer;
    const NTravelProto::TSearchOffersReq& req = message.GetRequest();
    const NTravelProto::TSearchOffersRsp& resp = message.GetResponse();

    TCacheKey key;
    key.PreKey.HotelId = THotelId::FromProto(req.GetHotelId());
    key.SubKey.Date = NOrdinalDate::FromString(req.GetCheckInDate());
    key.SubKey.Nights = NOrdinalDate::FromString(req.GetCheckOutDate()) - key.SubKey.Date;

    THashMap<TCacheKey, TCacheRecordRef> key2rec;
    auto ensureRecord = [this, &key2rec, timestamp, &key, &req]
        (const TCapacity& cap, const NTravelProto::ECurrency cur, TInstant expireTimestamp) {
        key.PreKey.Currency = cur;
        key.SubKey.Capacity = cap;
        auto it = key2rec.find(key);
        if (it == key2rec.end()) {
            TCacheRecordRef rec = new TCacheRecord;
            rec->Key = key;
            rec->Timestamp = timestamp;
            rec->ExpireTimestamp = expireTimestamp;
            rec->SearcherReqId = req.GetId();
            rec->UsageCount = 0;
            rec->SourceCounters = GetSourceCounters(req);
            rec->OfferCacheClientId = OfferCacheClientDeduplicator_.Deduplicate(req.GetAttribution().GetOfferCacheClientId());
            rec->ErrorCode = NTravelProto::EC_OK;
            rec->SearchWarnings = nullptr;
            rec->IsOutdated = false;
            it = key2rec.insert(std::make_pair(key, rec)).first;
        };
        return it->second;
    };
    const auto reqOccupancy = TAges::FromOccupancyString(req.GetOccupancy());
    const auto reqCapacity = TCapacity::FromAges(reqOccupancy);

    if (resp.HasError()) {
        // Ошибка при исполнении запроса, но исполнение закончено, данных не получено
        episodicTimer.Reset();
        if (updateReqCacheInfo) {
            ReqCache_.OnRequestFinishedError(req, timestamp);
        }
        stat->ReqCacheTime += episodicTimer.Get();
        // Даже с ошибкой, надо положить в кэш пустую запись - ведь мы пытались...
        ensureRecord(reqCapacity, req.GetCurrency(), messageExpireTimestamp)->ErrorCode = resp.GetError().GetCode();
    } else if (resp.HasPlaceholder()) {
        // Запрос начался, данных еще нет
        episodicTimer.Reset();
        if (updateReqCacheInfo) {
            ReqCache_.OnRequestStarted(req, timestamp);
        }
        stat->ReqCacheTime += episodicTimer.Get();
    } else if (resp.HasOffers()) {
        // Запрос закончился, данные есть
        episodicTimer.Reset();
        stat->ReqCacheTime += episodicTimer.Get();
        if (messageExpireTimestamp >= now) {
            const auto& offers = resp.GetOffers();
            // У офферов могут быть разные Capacity, Currency, которые должны попасть в разные ключи
            for (auto& offer: offers.GetOffer()) {
                TCapacity cap;
                try {
                    cap = TCapacity::FromCapacityString(offer.GetCapacity());
                } catch (...) {
                    WARNING_LOG << "Error while processing capacity " << offer.GetCapacity() << ": " << CurrentExceptionMessage()
                                << ", OfferId: " << offer.GetId() << ", ReqId: " << req.GetId() << Endl;
                    continue;
                }
                if (!cap.Matches(reqOccupancy)) {
                    continue;
                }
                if ((offer.GetOperatorId() == EOperatorId::OI_HOTELS101) && cap.HasChildren()) {
                    // HOTELS-3626. 101 отель шлют офферы с детьми с некорректными лендингами, выкидываем
                    continue;
                }
                AddOfferToRecord(ensureRecord(reqCapacity, offer.GetPrice().GetCurrency(), messageExpireTimestamp), offer, stat, fromOutdatedOffersBus);
            }
            if (!offers.GetWarnings().GetWarnings().empty()) {
                THashSet<NTravelProto::ESearchWarningCode> seenWarningCodes;
                auto record = ensureRecord(reqCapacity, req.GetCurrency(), messageExpireTimestamp);
                record->SearchWarnings = MakeIntrusive<TSearchWarnings>();
                for (auto& warning: offers.GetWarnings().GetWarnings()) {
                    if (seenWarningCodes.insert(warning.GetCode()).second) {
                        record->SearchWarnings->WarningCounts.emplace_back(warning.GetCode(), warning.GetCount());
                    } else {
                        WARNING_LOG << "Duplicate warning code: " << NTravelProto::ESearchWarningCode_Name(warning.GetCode()) << ", request id is " << req.GetId() << Endl;
                    }
                }
            }
        }
        // А capacity полученная из Occupancy в запросе тоже должна попасть в кэш, см HOTELS-2998
        // Если ровно такая capacity не встречалась, то запомним, что про неё ничего нет (пустой список офферов)
        ensureRecord(reqCapacity, req.GetCurrency(), messageExpireTimestamp);
        if (updateReqCacheInfo) {
            ReqCache_.OnRequestFinished(req, timestamp, messageExpireTimestamp);
        }
    } else {
        throw yexception() << "Something strange in response - no error, no placeholder, no offers, request id is " << req.GetId();
    }
    if (key2rec.empty()) {
        return {};
    }
    TVector<TCacheRecordRef> records;
    records.reserve(key2rec.size());
    for (const auto& [key, rec]: key2rec) {
        TMaybe<TCacheRecordRef> preprocessedRecord = preprocessRecord(rec);
        if (preprocessedRecord.Defined()) {
            auto& newRec = preprocessedRecord.GetRef();
            newRec->Offers.shrink_to_fit();
            DEBUG_LOG << "Adding record with key " << newRec->Key << ", Offers: " << newRec->Offers.size() << Endl;
            records.push_back(newRec);
        }
    }
    episodicTimer.Reset();
    Cache_.AddBulk(records, now, message.GetRequest().GetId());
    stat->CacheTime += episodicTimer.Get();
    DEBUG_LOG << "Cache AddBulk " << records.size() << " records done in " << stat->CacheTime << Endl;
    episodicTimer.Reset();
    return records;
}

void TService::AddOfferToRecord(const TCacheRecordRef& rec, const NTravelProto::TOffer& pbOffer, TProcessingStat* stat, bool fromOutdatedOffersBus) {
    rec->Offers.emplace_back();
    TOffer& offer = rec->Offers.back();
    offer.OfferId = TOfferId::FromString(pbOffer.GetId());
    TString OfferIdWithSalt = Config_.GetOther().GetOfferIdHashSalt() + pbOffer.GetId();
    offer.OfferIdHash = Crc32c(OfferIdWithSalt.data(), OfferIdWithSalt.length()); // It's not for security, so it's ok to use crc32
    offer.PriceVal = pbOffer.GetPrice().GetAmount();
    offer.OperatorId = pbOffer.GetOperatorId();
    TProfileTimer episodicTimer;
    stat->EncodingTime += episodicTimer.Get();
    offer.PansionType = pbOffer.GetPansion();
    offer.TitleAndOriginalRoomId = ObjectDeduplicator_.Deduplicate(TCommonDeduplicatorKeys::TitleAndOriginalRoomId,
                                                                   TOfferTitleAndOriginalRoomId{pbOffer.GetDisplayedTitle().value(), pbOffer.GetOriginalRoomId()});
    auto maxRoomCount = std::numeric_limits<decltype(TOffer::RoomCount)>::max();
    if (pbOffer.GetRoomCount() > static_cast<ui32>(maxRoomCount)) {
        ERROR_LOG << "Too big RoomCount, limiting to max possible (" << maxRoomCount << "). OfferId: " << pbOffer.GetId() << Endl;
        offer.RoomCount = maxRoomCount;
    } else {
        offer.RoomCount = pbOffer.GetRoomCount();
    }
    TString singleRoomCap = pbOffer.GetSingleRoomCapacity();
    if (!singleRoomCap) {
        singleRoomCap = pbOffer.GetCapacity();
    }
    auto maxSingleRoomAdultCount = std::numeric_limits<decltype(TOffer::SingleRoomAdultCount)>::max();
    auto realSingleRoomAdultCount = TCapacity::FromCapacityString(singleRoomCap).GetAdultCount();
    if (realSingleRoomAdultCount > static_cast<ui32>(maxSingleRoomAdultCount)) {
        ERROR_LOG << "Too big SingleRoomAdultCount, limiting to max possible (" << maxSingleRoomAdultCount << "). OfferId: " << pbOffer.GetId() << Endl;
        offer.SingleRoomAdultCount = maxSingleRoomAdultCount;
    } else {
        offer.SingleRoomAdultCount = realSingleRoomAdultCount;
    }
    if (pbOffer.HasFreeCancellation()) {
        offer.FreeCancellation = pbOffer.GetFreeCancellation().value() ? EFreeCancellationType::Yes : EFreeCancellationType::No;
    } else {
        offer.FreeCancellation = EFreeCancellationType::Unknown;
    }
    if (!ProcessPartnerSpecificData(pbOffer, rec->Key.PreKey.HotelId.PartnerId, &offer, fromOutdatedOffersBus)) {
        ERROR_LOG << "Failed to process partner specific data, dropping offer. OfferId: " << pbOffer.GetId() << Endl;
        rec->Offers.pop_back();
        return;
    }
    if (pbOffer.HasRestrictions()) {
        const auto& restrictions = pbOffer.GetRestrictions();
        offer.OfferRestrictions = TOfferRestrictions{restrictions.GetRequiresMobile(), restrictions.GetRequiresRestrictedUser()};
    }
    ProcessRefundRules(pbOffer, &offer, pbOffer.GetPrice().GetCurrency());
    ProcessPromo(pbOffer, &offer, pbOffer.GetPrice().GetCurrency());
}

void TService::ProcessPromo(const NTravelProto::TOffer&, TOffer*, NTravelProto::ECurrency) const {
}

void TService::ProcessRefundRules(const NTravelProto::TOffer& pbOffer, TOffer* offer, NTravelProto::ECurrency offerCurrency) {
    TVector<TRefundRule> refundRules{};
    bool error = false;
    bool warning = false;
    refundRules.reserve(pbOffer.RefundRuleSize());
    for (const auto& pbRefundRule : pbOffer.GetRefundRule()) {
        auto& refundRule = refundRules.emplace_back();
        refundRule.Type = pbRefundRule.GetType();
        if (pbRefundRule.HasPenalty()) {
            if (pbRefundRule.GetType() != NTravelProto::ERefundType::RT_REFUNDABLE_WITH_PENALTY) {
                ERROR_LOG << "Found refund rule of type " << NTravelProto::ERefundType_Name(pbRefundRule.GetType()) << " with penalty. OfferId: " << pbOffer.GetId() << Endl;
                error = true;
            }
            if (pbRefundRule.GetPenalty().GetCurrency() != offerCurrency) {
                ERROR_LOG << "Refund rule penalty has non-equal to offer currency (penalty currency is "
                          << NTravelProto::ECurrency_Name(pbRefundRule.GetPenalty().GetCurrency())
                          << ", offer currency is " << NTravelProto::ECurrency_Name(offerCurrency)
                          <<"). OfferId: " << pbOffer.GetId() << Endl;
                error = true;
            }
            auto precisionPow = Power(10, pbRefundRule.GetPenalty().GetPrecision());
            auto amount = pbRefundRule.GetPenalty().GetAmount();
            if (amount % precisionPow != 0) {
                DEBUG_LOG << "Refund rule penalty is not divisible by 10^precision (penalty=" << amount << ", precision="
                          << pbRefundRule.GetPenalty().GetPrecision() << "). OfferId: " << pbOffer.GetId() << Endl;
                ServiceCounters_.FractionalRefundRulePenalty.Inc();
            }
            refundRule.Penalty = (pbRefundRule.GetPenalty().GetAmount() + precisionPow - 1) / precisionPow; // rounding up
        } else {
            if (pbRefundRule.GetType() == NTravelProto::ERefundType::RT_REFUNDABLE_WITH_PENALTY) {
                ERROR_LOG << "Found refund rule of type " << NTravelProto::ERefundType_Name(pbRefundRule.GetType()) << " without penalty. OfferId: " << pbOffer.GetId() << Endl;
                error = true;
            }
        }
        refundRule.StartsAtTimestampSec = pbRefundRule.HasStartsAt() ? pbRefundRule.GetStartsAt().seconds() : std::numeric_limits<decltype(refundRule.StartsAtTimestampSec)>::min();
        refundRule.EndsAtTimestampSec = pbRefundRule.HasEndsAt() ? pbRefundRule.GetEndsAt().seconds() : std::numeric_limits<decltype(refundRule.EndsAtTimestampSec)>::max();
    }
    refundRules.shrink_to_fit();
    Sort(refundRules, [](const TRefundRule& lhs, const TRefundRule& rhs) {
        return lhs.StartsAtTimestampSec < rhs.StartsAtTimestampSec;
    });
    for (size_t i = 0; i < refundRules.size(); i++) {
        if (i > 0 && refundRules[i - 1].EndsAtTimestampSec > refundRules[i].StartsAtTimestampSec) {
            ERROR_LOG << "Refund rules are intersecting. OfferId: " << pbOffer.GetId() << Endl;
            error = true;
        }
        if (refundRules[i].StartsAtTimestampSec >= refundRules[i].EndsAtTimestampSec) {
            ERROR_LOG << "Refund rule end should be after its start. OfferId: " << pbOffer.GetId() << Endl;
            error = true;
        }
        if (i > 0 && refundRules[i - 1].EndsAtTimestampSec != refundRules[i].StartsAtTimestampSec) {
            WARNING_LOG << "Refund rules have gaps. OfferId: " << pbOffer.GetId() << Endl;
            warning = true;
        }
    }
    if (error) {
        ERROR_LOG << "Failed to process refund rules, dropping them. OfferId: " << pbOffer.GetId() << Endl;
        ServiceCounters_.InvalidRefundRulesError.Inc();
        return;
    } else if (warning) {
        ERROR_LOG << "Refund rules have some acceptable problems. OfferId: " << pbOffer.GetId() << Endl;
        ServiceCounters_.InvalidRefundRulesWarn.Inc();
    }
    offer->RefundRules = ObjectDeduplicator_.Deduplicate(TCommonDeduplicatorKeys::RefundRules, refundRules);
}

bool TService::ProcessPartnerSpecificData(const NTravelProto::TOffer& pbOffer, EPartnerId partnerId, TOffer* offer, bool isOutdated) {
    auto resultPartnerSpecificOfferData = TPartnerSpecificOfferData();
    if (isOutdated) {
        if (pbOffer.HasPartnerSpecificData()) {
            ServiceCounters_.PartnerSpecificDataInOutdatedRecord.Inc();
            ERROR_LOG << "Outdated record has offer with partner-specific data. OfferId: " << pbOffer.GetId() << Endl;
            return false;
        }
        resultPartnerSpecificOfferData = TPartnerSpecificOfferData{TFakeOutdatedOfferPartnerData()};
    } else if (partnerId == NTravelProto::PI_TRAVELLINE) {
        Y_ENSURE(pbOffer.GetOperatorId() == EOperatorId::OI_TRAVELLINE);
        if (!pbOffer.HasPartnerSpecificData() || !pbOffer.GetPartnerSpecificData().HasTravellineData()) {
            ServiceCounters_.NoTravellinePartnerSpecificData.Inc();
            ERROR_LOG << "Offer of partner PI_TRAVELLINE has no travelline data. OfferId: " << pbOffer.GetId() << Endl;
            return false;
        }
        const auto& travellineData = pbOffer.GetPartnerSpecificData().GetTravellineData();
        resultPartnerSpecificOfferData = TPartnerSpecificOfferData{TTravellineData{travellineData.GetHotelCode(), travellineData.GetRatePlanCode()}};
    } else if (partnerId == NTravelProto::PI_DOLPHIN) {
        Y_ENSURE(pbOffer.GetOperatorId() == EOperatorId::OI_DOLPHIN);
        if (!pbOffer.HasPartnerSpecificData() || !pbOffer.GetPartnerSpecificData().HasDolphinData()) {
            ServiceCounters_.NoDolphinPartnerSpecificData.Inc();
            ERROR_LOG << "Offer of partner PI_DOLPHIN has no dolphin data. OfferId: " << pbOffer.GetId() << Endl;
            return false;
        }
        const auto& dolphinData = pbOffer.GetPartnerSpecificData().GetDolphinData();
        resultPartnerSpecificOfferData = TPartnerSpecificOfferData{TDolphinData{dolphinData.GetTour(), dolphinData.GetPansion(), dolphinData.GetRoom(), dolphinData.GetRoomCat()}};
    } else if (partnerId == NTravelProto::PI_BNOVO) {
        Y_ENSURE(pbOffer.GetOperatorId() == EOperatorId::OI_BNOVO);
        if (!pbOffer.HasPartnerSpecificData() || !pbOffer.GetPartnerSpecificData().HasBNovoData()) {
            ServiceCounters_.NoBNovoPartnerSpecificData.Inc();
            ERROR_LOG << "Offer of partner PI_BNOVO has no bnovo data. OfferId: " << pbOffer.GetId() << Endl;
            return false;
        }
        const auto& bnovoData = pbOffer.GetPartnerSpecificData().GetBNovoData();
        resultPartnerSpecificOfferData = TPartnerSpecificOfferData{TBNovoData{bnovoData.GetAccountId(), bnovoData.GetRatePlanId()}};
    } else {
        if (pbOffer.HasPartnerSpecificData()) {
            ServiceCounters_.ExtraPartnerSpecificData.Inc();
            ERROR_LOG << "Offer has unexpected partner-specific data. OfferId: " << pbOffer.GetId() << ", partnerId: " << partnerId << Endl;
            return false;
        }
    }
    offer->PartnerSpecificOfferData = ObjectDeduplicator_.Deduplicate(TCommonDeduplicatorKeys::PartnerSpecificOfferData, resultPartnerSpecificOfferData);
    return true;
}

void TService::OnHeatmap(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    CacheHeatmap_.BuildCacheHeatmap(httpReq, responseCb);
}

void TService::OnGetPermalink(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    NTravelProto::NOfferCache::NApi::TPermalinkReq req;
    try {
        NTravel::NProtobuf::ParseCgiRequest(httpReq.Query(), &req);
    } catch (...) {
        ERROR_LOG << "Bad Permalink request: " << CurrentExceptionMessage() << Endl;
        responseCb(NHttp::TResponse::CreateText(CurrentExceptionMessage() + "\n", HTTP_BAD_REQUEST));
        return;
    }
    NTravelProto::NOfferCache::NApi::TPermalinkResp resp;
    resp.SetPermalink(req.GetPermalink());
    TPermalink clusterPermalink = PermalinkToClusterMapper_.GetClusterPermalink(req.GetPermalink());
    resp.MutableCluster()->SetClusterPermalink(clusterPermalink);
    for (const auto& p: PermalinkToClusterMapper_.GetCluster(clusterPermalink)) {
        resp.MutableCluster()->AddPermalinks(p);
    }
    if (auto partnerHotelIds = PermalinkToOriginalIdsMapper_.GetMapping(clusterPermalink)) {
        auto blacklist = HotelsBlacklist_.GetMapping(clusterPermalink);
        auto greylist = HotelsGreylist_.GetMapping(clusterPermalink);
        for (const auto& partnerHotelId: partnerHotelIds->PartnerIds) {
            if (auto pbHotelInfo = resp.AddPartnerHotels()) {
                pbHotelInfo->SetPartnerId(partnerHotelId.PartnerId);
                pbHotelInfo->SetOriginalId(partnerHotelId.OriginalId);
                if (greylist && greylist->Contains(partnerHotelId)) {
                    pbHotelInfo->SetIsGreylisted(true);
                }
                if (blacklist && blacklist->Contains(partnerHotelId)) {
                    pbHotelInfo->SetIsBlacklisted(true);
                }
                if (HotelsWhitelist_.IsInWhitelist(clusterPermalink, partnerHotelId)) {
                    pbHotelInfo->SetIsWhitelisted(true);
                } else {
                    if (HotelsWhitelist_.IsWhitelistedPartner(partnerHotelId.PartnerId)) {
                        pbHotelInfo->SetIsWhitelisted(false);
                    }
                }
            }
        }
    }
    if (IsWizardBanned(clusterPermalink)) {
        resp.SetIsWizardBanned(true);
    }
    responseCb(NHttp::TResponse::CreateJson(resp, HTTP_OK, true));
}

void TService::OnWaitFlush(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    while (true) {
        INFO_LOG << "Waiting for data to be flushed" << Endl;
        if (SearcherClient_.IsFlushed()) {
            break;
        }
        Sleep(TDuration::MilliSeconds(10));
    }
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnEmergencyPartnerDisable(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    auto active = IsIn({"1", "true"}, httpReq.Query().Get("active"));
    TVector<EPartnerId> partnersToDisable{};
    if (httpReq.Query().Has("partners")) {
        for (const auto& partner: httpReq.Query().Range("partners")) {
            EPartnerId parsedPartner;
            if (NTravelProto::EPartnerId_Parse(partner, &parsedPartner)) {
                partnersToDisable.push_back(parsedPartner);
            }
        }
    } else if (YtConfigPartners_.IsReady()) {
        for (const auto& [partnerId, partner]: *YtConfigPartners_.GetAll()) {
            if (!partner.GetIsBoY()) {
                partnersToDisable.push_back(partnerId);
            }
        }
    }

    if (!active) {
        partnersToDisable = TVector<EPartnerId>();
    }

    {
        TWriteGuard guard(EmergencyDisabledPartnersLock_);
        EmergencyDisabledPartners_ = partnersToDisable;
    }

    if (!active) {
        responseCb(NHttp::TResponse::CreateText("Emergency disable is not active, all partners are enabled\n"));
    } else {
        TVector<TString> partners;
        for (auto partner: partnersToDisable) {
            partners.push_back(NTravelProto::EPartnerId_Name(partner));
        }
        responseCb(NHttp::TResponse::CreateText("Emergency disable is active, partners "
                                                    + JoinStrings(partners, ", ")
                                                    + " are disabled\n"));
    }
}

TServiceCountersPerOperatorRef TService::GetCountersPerOperator(EOperatorId operatorId) {
    return CountersPerOperator_.GetOrCreate({NTravelProto::EOperatorId_Name(operatorId)});
}

TServiceCountersPerPartnerRef TService::GetCountersPerPartner(EPartnerId pId) {
    return CountersPerPartner_.GetOrCreate({NTravelProto::EPartnerId_Name(pId)});
}

bool TService::IsGrpcClientIdKnown(const TString& grpcClientId) const {
    if (GrpcExactClientIds_.contains(grpcClientId)) {
        return true;
    }
    for (const auto& prefix: GrpcPrefixClientIds_) {
        if (grpcClientId.StartsWith(prefix)) {
            return true;
        }
    }
    return false;
}

bool EqualsWithTrailingWildcard(const TString& pattern, const TString& value) {
    if (pattern.EndsWith("*")) {
        TStringBuf patternPart(pattern, 0, pattern.size() - 1);
        TStringBuf valuePart(value, 0, pattern.size() - 1);
        return patternPart == valuePart;
    } else {
        return pattern == value;
    }
}

NTravelProto::NConfig::TOfferCacheClient TService::FindHttpOfferCacheClient(const TString& geoOrigin, const TString& geoClientId) const {
    auto all = YtConfigOfferCacheClients_.GetAll();
    for (const auto& [key, value]: *all) {
        if (EqualsWithTrailingWildcard(key.first, geoOrigin) && EqualsWithTrailingWildcard(key.second, geoClientId)) {
            return value;
        }
    }
    static NTravelProto::NConfig::TOfferCacheClient def;
    return def;
}

void TService::InitSources() {
    GetSourceCounters("");
    auto clients = YtConfigOfferCacheClients_.GetAll();
    for (auto it: *clients) {
        if (it.second.GetOfferCacheClientId()) {
            GetSourceCounters(it.second.GetOfferCacheClientId());
        }
    }
    for (const auto& name: GrpcExactClientIds_) {
        GetSourceCounters(name);
    }
    // Вот бы еще GrpcPrefixClientIds_ создать, но мы же знаем только префиксы
}

void TService::UpdateBoYOperatorsUnlocked() {
    BoYOperators_.clear();
    BoYDirectOperators_.clear();
    auto operators = YtConfigOperators_.GetAll();
    for (const auto& [opId, op]: *operators) {
        if (IsBoYPartner(op.GetPartnerId())) {
            BoYOperators_.insert(opId);
        }
        if (IsBoYDirectPartner(op.GetPartnerId())) {
            BoYDirectOperators_.insert(opId);
        }
    }
}

TSourceCountersRef TService::GetSourceCounters(const TString& name) {
    if (name == "") {
        return GetSourceCounters("_UNKNOWN_");
    }
    return CountersPerSource_.GetOrCreate({name});
}

TSourceCountersRef TService::GetSourceCounters(const NTravelProto::TSearchOffersReq& subReq) {
    return GetSourceCounters(subReq.GetAttribution().GetOfferCacheClientId());
}

TCacheHitCountersRef TService::GetCacheHitCounters(ERequestType rt, EPermalinkType pt, const TString& ocClientId) {
    return CountersCacheHit_.GetOrCreate({ToString(rt), ToString(pt), ocClientId ? ocClientId : "_UNKNOWN_"});
}

TCatRoomCountersPerDataSourceRef TService::GetCatRoomCounters(TCatRoomDataSourceIdStr dsId, bool partnerOffersMatching) {
    return CountersCatRoom_.GetOrCreate({dsId, ToString(partnerOffersMatching)});
}

TOfferShowCountersRef TService::GetOfferShowCounters(ERequestType rt, EPermalinkType pt, EOperatorId operatorId) {
    return CountersOfferShow_.GetOrCreate({ToString(rt), ToString(pt), NTravelProto::EOperatorId_Name(operatorId)});
}

TServiceCountersPerStageRef TService::GetCountersPerStage(ERequestSize size, ERequestStage stage, bool useSearcher) {
    return CountersPerStage_.GetOrCreate({ToString(size), ToString(stage), ToString(useSearcher)});
}

TServiceCountersPerBlendingStageRef TService::GetCountersPerBlendingStage(NTravelProto::NOfferCache::NApi::ERespMode respMode, bool full, const TString& stage) {
    return CountersPerBlendingStage_.GetOrCreate({ToString(respMode), ToString(full), stage});
}

bool TService::IsSearchSubKeyAllowed(NOrdinalDate::TOrdinalDate today, const THotelId& hotelId, const TSearchSubKey& key, TString* restrictReason) const {
    TVector<EPartnerId> emergencyDisabledPartners;
    {
        TReadGuard guard(EmergencyDisabledPartnersLock_);
        emergencyDisabledPartners = EmergencyDisabledPartners_;
    }
    if (IsIn(emergencyDisabledPartners, hotelId.PartnerId)) {
        if (restrictReason) {
            *restrictReason = "Emergency disable";
        }
        return false;
    }

    NTravelProto::NConfig::TSearchKeyRestrictions r;
    if (auto partR = YtConfigSearchKeyRestrictions_.GetById({})) {
        r.MergeFrom(*partR);
    }
    if (auto partR = YtConfigSearchKeyRestrictions_.GetById(THotelId{hotelId.PartnerId, {}})) {
        r.MergeFrom(*partR);
    }
    if (auto partR = YtConfigSearchKeyRestrictions_.GetById(hotelId)) {
        r.MergeFrom(*partR);
    }

    if (!r.GetEnabled()) {
        if (restrictReason) {
            *restrictReason = "Search key disabled";
        }
        return false;
    }
#define CHECK_RESTRICTION(_VAR_, _CMP_, _PROTOFIELD_)                    \
    if (r.Has##_PROTOFIELD_() && !(_VAR_ _CMP_ r.Get##_PROTOFIELD_())) { \
        if (restrictReason) {                                            \
            *restrictReason = ToString(key) +                            \
            " restricted, because " #_VAR_ " = " + ToString(_VAR_) +     \
            " should be " #_CMP_ " " + ToString(r.Get##_PROTOFIELD_());  \
        }                                                                \
        return false;                                                    \
    }

    int checkInRel = key.Date - today;
    int checkOutRel = checkInRel + key.Nights;
    CHECK_RESTRICTION(checkInRel, >=, CheckInRelMin);
    CHECK_RESTRICTION(checkInRel, <=, CheckInRelMax);
    CHECK_RESTRICTION(checkOutRel, >=, CheckOutRelMin);
    CHECK_RESTRICTION(checkOutRel, <=, CheckOutRelMax);
    CHECK_RESTRICTION(key.Nights, >=, NightsMin);
    CHECK_RESTRICTION(key.Nights, <=, NightsMax);
    CHECK_RESTRICTION(key.Ages.GetAdultCount(), >=, AdultsMin);
    CHECK_RESTRICTION(key.Ages.GetAdultCount(), <=, AdultsMax);
    if (key.Ages.HasChildren()) {
        CHECK_RESTRICTION(key.Ages.GetChildAgeMin(), >=, ChildAgeMin);
        CHECK_RESTRICTION(key.Ages.GetChildAgeMax(), <=, ChildAgeMax);
    }
    return true;
#undef CHECK_RESTRICTION
}


TInstant TService::GetFakeableNow() const {
    if (Config_.GetOther().HasFakeNowSec()) {
        return TInstant::Seconds(Config_.GetOther().GetFakeNowSec());
    } else {
        return Now();
    }
}

bool TService::IsWizardBanned(TPermalink permalink) const {
    auto entry = YtConfigHotelWizardBan_.GetById(permalink);
    return entry && entry->GetBan();
}

size_t TService::GetUserConfirmedHotelOrderCount(TPassportUid passportUid) const {
    auto entry = YtConfigUserOrderCounters_.GetById(passportUid);
    if (!entry) {
        return 0;
    }
    return entry->GetConfirmedHotelOrders();
}

}// namespace NOfferCache
}// namespace NTravel

