#pragma once

#include "cache.h"
#include "cache_heatmap.h"
#include "counters.h"
#include "req_cache.h"
#include "rooms/room_service.h"
#include "http_read_job.h"
#include "grpc_job.h"
#include "searcher_client.h"
#include "hotels_whitelist.h"
#include "partner_data_offer_filter.h"
#include "grpc_read_job.h"
#include "user_events_storage.h"
#include "promo_service.h"
#include "outdated_record_builder.h"
#include "outdated_offers_transmitter.h"

#include <travel/hotels/proto/offercache_api/main.pb.h>
#include <travel/hotels/proto/offercache_grpc/offercache_service.pb.h>
#include <travel/hotels/proto/promo_service/promo_service.pb.h>
#include <travel/hotels/offercache/proto/config.pb.h>
#include <travel/hotels/offercache/proto/reqans_logrecord.pb.h>
#include <travel/hotels/offercache/proto/grpc_reqans_logrecord.pb.h>

#include <travel/hotels/proto/data_config/hotel_wizard_ban.pb.h>
#include <travel/hotels/proto/data_config/offercache_client.pb.h>
#include <travel/hotels/proto/data_config/operator.pb.h>
#include <travel/hotels/proto/data_config/partner.pb.h>
#include <travel/hotels/proto/data_config/searchkey_restrictions.pb.h>
#include <travel/hotels/proto2/bus_messages.pb.h>
#include <travel/hotels/proto2/hotels.pb.h>

#include <travel/hotels/lib/cpp/grpc/grpc_async_client.h>
#include <travel/hotels/lib/cpp/grpc/grpc_async_server.h>
#include <travel/hotels/lib/cpp/encryption/codec.h>
#include <travel/hotels/lib/cpp/permalink_mappers/mappers/permalink_to_original_ids_mapper.h>
#include <travel/hotels/lib/cpp/permalink_mappers/mappers/permalink_to_cluster_mapper.h>
#include <travel/hotels/lib/cpp/http/http_service.h>
#include <travel/hotels/lib/cpp/label/label.h>
#include <travel/hotels/lib/cpp/logging/common_logger.h>
#include <travel/hotels/lib/cpp/util/flag.h>
#include <travel/hotels/lib/cpp/util/string_deduplicator.h>
#include <travel/hotels/lib/cpp/util/profiletimer.h>
#include <travel/hotels/lib/cpp/util/restart_detector.h>
#include <travel/hotels/lib/cpp/util/object_deduplicator.h>
#include <travel/hotels/lib/cpp/yt/persistent_config.h>
#include <travel/hotels/lib/cpp/yt/queue_reader.h>
#include <travel/hotels/lib/cpp/yt/queue_writer.h>
#include <travel/hotels/lib/cpp/memusage/memusage.h>
#include <travel/hotels/lib/cpp/mon/page.h>
#include <travel/hotels/lib/cpp/offer_cache_invalidation/cache_invalidation_service.h>

#include <travel/proto/user_order_counters/user_order_counters.pb.h>

#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/system/event.h>
#include <util/system/mutex.h>
#include <util/thread/pool.h>
#include <util/generic/ptr.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>

#include <tuple>
#include <functional>

namespace NTravel {
namespace NOfferCache {

class TService
{
public:
    using TSearcherGrpcClient = NGrpc::TAsyncClient<ru::yandex::travel::hotels::OfferSearchServiceV1>;

    using TReqAnsLogger = TCommonLogger<NTravelProto::NOfferCache::NReqAnsLog::TReqAnsLogRecord>;
    using TGrpcReqAnsLogger = TCommonLogger<NTravelProto::NOfferCache::NGrpcReqAnsLog::TGrpcReqAnsLogRecord>;

    TService(const NTravelProto::NOfferCache::TConfig& pbCfg, const TString& environment);
    ~TService();
    void Start();
    void Wait();
    void Stop();

    TSourceCountersRef GetSourceCounters(const NTravelProto::TSearchOffersReq& subReq);
    TSourceCountersRef GetSourceCounters(const TString& name);
    TServiceCountersPerOperatorRef GetCountersPerOperator(EOperatorId operatorId);
    TServiceCountersPerPartnerRef GetCountersPerPartner(EPartnerId pId);
    TCacheHitCountersRef GetCacheHitCounters(ERequestType rt, EPermalinkType pt, const TString& ocClientId);
    TCatRoomCountersPerDataSourceRef GetCatRoomCounters(TCatRoomDataSourceIdStr dsId, bool partnerOffersMatching);
    TOfferShowCountersRef GetOfferShowCounters(ERequestType rt, EPermalinkType pt, EOperatorId operatorId);
    TServiceCountersPerStageRef GetCountersPerStage(ERequestSize size, ERequestStage stage, bool useSearcher);
    TServiceCountersPerBlendingStageRef GetCountersPerBlendingStage(NTravelProto::NOfferCache::NApi::ERespMode respMode, bool full, const TString& stage);

    bool GetPartnerIdByCode(const TString& code, EPartnerId* pId) const;
    TString GetPartnerCode(EPartnerId pId) const;
    std::shared_ptr<const NTravelProto::NConfig::TPartner> GetPartner(EPartnerId pId) const;

    THashSet<EPartnerId> GetPartnersForOperators(const THashSet<EOperatorId>& opIds) const;
    THashSet<EOperatorId> GetOperatorsForPartners(const THashSet<EPartnerId>& partnerIds) const;

    const TVector<EOperatorId>& GetBumpedOperators() const;

    size_t GetOperatorOrder(EOperatorId opId) const;
    std::shared_ptr<const NTravelProto::NConfig::TOperator> GetOperator(EOperatorId opId) const;
    THashSet<EOperatorId> GetEnabledOperatorsDefault() const;

    bool IsBoYPartner(EPartnerId pId) const;
    bool IsBoYDirectPartner(EPartnerId pId) const;
    bool IsBoYOperator(EOperatorId opId) const;
    bool IsBoYDirectOperator(EOperatorId opId) const;// "напрямую у отеля"
    THashSet<EOperatorId> GetBoYOperators() const;

    void DoSearcherRequest(const TString& logPrefix, const NTravelProto::TSearchOffersRpcReq& rpcReq, const NGrpc::TClientMetadata& meta, TSearcherClient::TOnResponse respCb);

    TServiceCounters& GetCounters() {
        return ServiceCounters_;
    }
    NLabel::TLabelCodec& LabelCodec() {
        return LabelCodec_;
    }
    NEncryption::TTokenCodec& RedirAddInfoCodec() {
        return RedirAddInfoCodec_;
    }
    const NTravelProto::NOfferCache::TConfig& Config() const {
        return Config_;
    }
    TCache& Cache() {
        return Cache_;
    }
    TReqCache& ReqCache() {
        return ReqCache_;
    }
    TReqAnsLogger& ReqAnsLogger() {
        return ReqAnsLogger_;
    }
    TGrpcReqAnsLogger& GrpcReqAnsLogger() {
        return GrpcReqAnsLogger_;
    }
    TYtQueueWriter& OfferReqBus() {
        return OfferReqBus_;
    }
    const NPermalinkMappers::TPermalinkToOriginalIdsMapper& PermalinkToOriginalIdsMapper() const {
        return PermalinkToOriginalIdsMapper_;
    }
    const THotelsWhitelist& HotelsWhitelist() const {
        return HotelsWhitelist_;
    }
    const NPermalinkMappers::TPermalinkToOriginalIdsMapper& HotelsGreylist() const {
        return HotelsGreylist_;
    }
    const NPermalinkMappers::TPermalinkToOriginalIdsMapper& HotelsBlacklist() const {
        return HotelsBlacklist_;
    }
    const NPermalinkMappers::TPermalinkToClusterMapper& PermalinkToClusterMapper() const {
        return PermalinkToClusterMapper_;
    }
    const TPartnerDataOfferFilter& PartnerDataOfferFilter() const {
        return PartnerDataOfferFilter_;
    }
    TRoomService& RoomService() {
        return RoomService_;
    }
    TUserEventsStorage& UserEventsStorage() {
        return UserEventsStorage_;
    }
    TYtQueueWriter& ICBusWriter() {
        return ICBusWriter_;
    }
    TObjectDeduplicator& ObjectDeduplicator() {
        return ObjectDeduplicator_;
    }
    const TCacheInvalidationService& CacheInvalidationService() const {
        return CacheInvalidationService_;
    }
    TString GetEnvironment() const {
        return Environment_;
    }
    const TPromoService& GetPromoService() const {
        return PromoService_;
    }
    bool IsPingEnabled() const {
        return IsPingEnabled_;
    }
    TString GetPansionDisplayName(EOCPansion pt) const {
        if (auto pansion = Pansions_.FindPtr(pt)) {
            return *pansion;
        }
        return {};
    }

    bool IsGrpcClientIdKnown(const TString& grpcClientId) const;

    NTravelProto::NConfig::TOfferCacheClient FindHttpOfferCacheClient(const TString& geoOrigin, const TString& geoClientId) const;

    bool IsReady(TString* reason = nullptr) const;

    bool IsPinged() const;

    bool IsSearchSubKeyAllowed(NOrdinalDate::TOrdinalDate today, const THotelId& hotelId, const TSearchSubKey& key, TString* restrictReason = nullptr) const;

    bool IsWizardBanned(TPermalink permalink) const;

    size_t GetUserConfirmedHotelOrderCount(TPassportUid passportUid) const;

private:
    struct TProcessingStat {
        TDuration CacheTime;
        TDuration ReqCacheTime;
        TDuration EncodingTime;
    };

    // Configuration
    const NTravelProto::NOfferCache::TConfig Config_;
    const TString                            Environment_;
    const TString                            DummyPayload_;

    TYtPersistentConfig<EOperatorId, NTravelProto::NConfig::TOperator> YtConfigOperators_;
    TYtPersistentConfig<EPartnerId, NTravelProto::NConfig::TPartner> YtConfigPartners_;
    TYtPersistentConfig<TOfferCacheClientKey, NTravelProto::NConfig::TOfferCacheClient> YtConfigOfferCacheClients_;// HTTP-only clients
    TYtPersistentConfig<TSearchKeyRestrictionsKey, NTravelProto::NConfig::TSearchKeyRestrictions> YtConfigSearchKeyRestrictions_;
    TYtPersistentConfig<TPermalink, NTravelProto::NConfig::THotelWizardBan> YtConfigHotelWizardBan_;
    TYtPersistentConfig<TPassportUid, ru::yandex::travel::user_order_counters::TUserOrderCounter> YtConfigUserOrderCounters_;

    TRWMutex OperatorPartnerLock_;
    THashSet<EOperatorId> EnabledOperatorsDefault_;
    THashMap<TString, EPartnerId> PartnerIdByCode_;
    THashSet<EOperatorId> BoYOperators_;
    THashSet<EOperatorId> BoYDirectOperators_;

    TVector<EOperatorId> BumpedOperators_;
    THashMap<EOCPansion, TString> Pansions_;

    THashSet<TString>              GrpcExactClientIds_;
    TVector<TString>               GrpcPrefixClientIds_;

    // Variables
    TMutex                         LastPingLock_;
    TProfileTimer                  LastPing_;
    TAtomicFlag                    SeenBalancerPing_;

    TAtomicFlag                    IsShuttingDown_;
    TAutoEvent                     ShuttingDownEvent_;
    TAtomicFlag                    IsPingEnabled_;
    TAutoEvent                     StopEvent_;
    TAtomic                        LastJobId_;


    // Composite members

    NMonitor::TBasicPage            CountersPage_;
    mutable TServiceCounters        ServiceCounters_;
    TMemUsageCounters               MemUsageCounters_;
    NMonitor::TCounterHypercube<TSourceCounters>             CountersPerSource_;
    NMonitor::TCounterHypercube<TServiceCountersPerOperator> CountersPerOperator_;
    NMonitor::TCounterHypercube<TServiceCountersPerPartner>  CountersPerPartner_;
    NMonitor::TCounterHypercube<TCacheHitCounters>           CountersCacheHit_;
    NMonitor::TCounterHypercube<TCatRoomCountersPerDataSource> CountersCatRoom_;
    NMonitor::TCounterHypercube<TOfferShowCounters> CountersOfferShow_;
    NMonitor::TCounterHypercube<TServiceCountersPerStage>    CountersPerStage_;
    NMonitor::TCounterHypercube<TServiceCountersPerBlendingStage> CountersPerBlendingStage_;

    NMonitor::THttpService          MonWebService_;
    NLabel::TLabelCodec             LabelCodec_;
    NEncryption::TTokenCodec        RedirAddInfoCodec_;
    TOutdatedRecordBuilder          OutdatedRecordBuilder_;
    TCacheInvalidationService       CacheInvalidationService_;
    TCache                          Cache_;
    TCacheHeatmap                   CacheHeatmap_;
    TReqCache                       ReqCache_;
    TYtQueueReader                  CacheFiller_;
    TYtQueueReader                  OutdatedOfferBusReader_;
    TYtQueueWriter                  OutdatedOfferBusWriter_;
    TAtomicFlag                     InitialOutdatedOfferBusReadDone_;
    TSearcherClient                 SearcherClient_;
    TYtQueueWriter                  OfferReqBus_;
    TReqAnsLogger                   ReqAnsLogger_;
    TGrpcReqAnsLogger               GrpcReqAnsLogger_;
    NPermalinkMappers::TPermalinkToOriginalIdsMapper PermalinkToOriginalIdsMapper_;
    THotelsWhitelist                HotelsWhitelist_;
    NPermalinkMappers::TPermalinkToOriginalIdsMapper HotelsGreylist_;
    NPermalinkMappers::TPermalinkToOriginalIdsMapper HotelsBlacklist_;
    NPermalinkMappers::TPermalinkToClusterMapper PermalinkToClusterMapper_;
    NHttp::TServer                  Http_;
    TOfferCacheGrpcServer           OfferCacheGrpcServer_;
    TRestartDetector                RestartDetector_;

    TStringDeduplicator             OfferCacheClientDeduplicator_;

    TPartnerDataOfferFilter         PartnerDataOfferFilter_;

    TYtQueueReader                  ICBusReader_;
    TYtQueueWriter                  ICBusWriter_;

    TRoomService                    RoomService_;

    TUserEventsStorage              UserEventsStorage_;

    TObjectDeduplicator                ObjectDeduplicator_;
    TAutoPtr<IThreadFactory::IThread>  ObjectDeduplicatorGcThread_;

    TPromoService                   PromoService_;

    TPromoServiceGrpcServer         PromoServiceGrpcServer_;

    TOutdatedOffersTransmitter      OutdatedOffersTransmitter_;

    TRWMutex            EmergencyDisabledPartnersLock_;
    TVector<EPartnerId> EmergencyDisabledPartners_;

private:
    void OnPing(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnPingNanny(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnPingDisable(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnShutdown(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnSetLogLevel(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnReopenReqAnsLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnReopenGrpcReqAnsLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnReopenCacheUsageLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnReopenMainLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnRead(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnHeatmap(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnGetPermalink(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnWaitFlush(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);
    void OnEmergencyPartnerDisable(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb);

    void OnGrpcSearchOffers(const NTravelProto::TSearchOffersRpcReq& req, const NGrpc::TServerReqMetadata& srvMeta, const TOfferCacheGrpcServer::TResponseCb<NTravelProto::TSearchOffersRpcRsp>& responseCb);
    void OnGrpcPing(const NTravelProto::TPingRpcReq& req, const NGrpc::TServerReqMetadata& srvMeta, const TOfferCacheGrpcServer::TResponseCb<NTravelProto::TPingRpcRsp>& responseCb);

    void OnGrpcRead(const NTravelProto::NOfferCache::NApi::TReadReq& req, const NGrpc::TServerReqMetadata& srvMeta, const TOfferCacheGrpcServer::TResponseCb<NTravelProto::NOfferCache::NApi::TReadResp>& responseCb);

    void OnPromoServiceGrpcDeterminePromosForOffer(const NTravelProto::NPromoService::TDeterminePromosForOfferReq& req,
                                                   const NGrpc::TServerReqMetadata& srvMeta,
                                                   const TPromoServiceGrpcServer::TResponseCb<NTravelProto::NPromoService::TDeterminePromosForOfferRsp>& responseCb);
    void OnPromoServiceGrpcGetActivePromos(const NTravelProto::NPromoService::TGetActivePromosReq& req,
                                           const NGrpc::TServerReqMetadata& srvMeta,
                                           const TPromoServiceGrpcServer::TResponseCb<NTravelProto::NPromoService::TGetActivePromosRsp>& responseCb);

    void OnPromoServiceGrpcCalculateDiscountForOffer(const NTravelProto::NPromoService::TCalculateDiscountForOfferReq& req,
                                                     const NGrpc::TServerReqMetadata& srvMeta,
                                                     const TPromoServiceGrpcServer::TResponseCb<NTravelProto::NPromoService::TCalculateDiscountForOfferRsp>& responseCb);
    void OnPromoServiceGrpcGetWhiteLabelPointsProps(const NTravelProto::NPromoService::TGetWhiteLabelPointsPropsReq& req,
                                                    const NGrpc::TServerReqMetadata& srvMeta,
                                                    const TPromoServiceGrpcServer::TResponseCb<NTravelProto::NPromoService::TGetWhiteLabelPointsPropsRsp>& responseCb);

    void AddReadyFlag(bool subReady, const TString& name, bool* ready, TString* reason) const;

    bool ProcessOfferBusSearcherMessage(const TYtQueueMessage& busMessage);
    void ProcessOfferBusSearcherMessageParsed(TInstant timestamp, TInstant messageExpireTimestamp, const TString& messageId, const ru::yandex::travel::hotels::TSearcherMessage& message, TProcessingStat* stat);
    bool ProcessOutdatedOfferBusSearcherMessage(const TYtQueueMessage& busMessage);
    TVector<TCacheRecordRef> BuildAndCacheRecordsFromSearcherMessage(TInstant timestamp,
                                                                     TInstant messageExpireTimestamp,
                                                                     const ru::yandex::travel::hotels::TSearcherMessage& message,
                                                                     TInstant now,
                                                                     TProcessingStat* stat,
                                                                     bool updateReqCacheInfo,
                                                                     std::function<TMaybe<TCacheRecordRef>(const TCacheRecordRef&)> preprocessRecord,
                                                                     bool fromOutdatedOffersBus);
    void AddOfferToRecord(const TCacheRecordRef& rec, const NTravelProto::TOffer& pbOffer, TProcessingStat* stat, bool fromOutdatedOffersBus);
    void ProcessPromo(const NTravelProto::TOffer& pbOffer, TOffer* offer, NTravelProto::ECurrency offerCurrency) const;
    void ProcessRefundRules(const NTravelProto::TOffer& pbOffer, TOffer* offer, NTravelProto::ECurrency offerCurrency);
    bool ProcessPartnerSpecificData(const NTravelProto::TOffer& pbOffer, EPartnerId partnerId, TOffer* offer, bool isOutdated);

    void InitSources();

    void UpdateBoYOperatorsUnlocked();

    TInstant GetFakeableNow() const;
};

}// namespace NOfferCache
}// namespace NTravel
