#include "user_events_storage.h"

#include <travel/hotels/lib/cpp/scheduler/scheduler.h>

namespace NTravel::NOfferCache {

void TUserEventsStorage::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NInteractiveSearchEvents));
    ct->insert(MAKE_COUNTER_PAIR(NBytes));
}

size_t TUserEventsStorage::TInteractiveSearchEvent::GetAllocSize() const {
    return SubKey.GetAllocSize();
}

TUserEventsStorage::TUserEventsStorage(const NTravelProto::NOfferCache::TConfig::TUserEventsStorage& cfg)
    : Enabled_(cfg.GetEnabled())
    , CleanupPeriod_(TDuration::MilliSeconds(cfg.GetMemoryCleanupPeriodMSec()))
    , InteractiveSearchEventTTL_(TDuration::Days(cfg.GetInteractiveSearchEventTTLDays()))
{
}

TUserEventsStorage::~TUserEventsStorage() {
}

bool TUserEventsStorage::IsEnabled() const {
    return Enabled_;
}

void TUserEventsStorage::RegisterCounters(NMonitor::TCounterSource& counters) {
    counters.RegisterSource(&Counters_, "UserEventsStorage");
}

bool TUserEventsStorage::OnInteractiveSearchEvent(const NTravelProto::NOfferCache::TInteractiveSearchEvent& pbEvent, TInstant eventTs) {
    if (!Enabled_) {
        return false;
    }
    TInstant expTime = eventTs + InteractiveSearchEventTTL_;
    if (expTime < ::Now()) {
        return false;
    }
    TMaybe<TYandexUid> yandexUid;
    TMaybe<TPassportUid> passportUid;
    if (pbEvent.HasYandexUid()) {
        yandexUid = pbEvent.GetYandexUid();
    }
    if (pbEvent.HasPassportUid()) {
        passportUid = pbEvent.GetPassportUid();
    }
    TMaybe<TUserIdentifier> userId = TUserIdentifier::Create(yandexUid, passportUid);
    if (!userId) {
        return false; // strange...
    }
    TInteractiveSearchEvent ise;
    ise.SubKey.Date = pbEvent.GetDate();
    ise.SubKey.Nights = pbEvent.GetNights();
    ise.SubKey.Ages = TAges::FromAgesString(pbEvent.GetAges());
    ise.ExpTime = expTime;

    TInstant cleanupTime = expTime + TDuration::MicroSeconds(CleanupPeriod_.MicroSeconds() - (expTime.MicroSeconds() % CleanupPeriod_.MicroSeconds()));
    bool isCleanupScheduleRequired;
    {
        auto& bucket = InteractiveSearchEvents_.GetBucketForKey(userId.GetRef());
        {
            TWriteGuard g(bucket.GetMutex());
            auto itR = bucket.GetMap().find(userId.GetRef());
            if (itR == bucket.GetMap().end()) {
                itR = bucket.GetMap().insert({userId.GetRef(), {}}).first;
                Counters_.NBytes += sizeof *itR + itR->first.GetAllocSize();
                itR->second = ise;
                OnRecordAdded(ise);
            } else {
                TInteractiveSearchEvent& iseOld = itR->second;
                if (ise.ExpTime <= iseOld.ExpTime) {
                    return false;
                }
                OnRecordRemoved(iseOld);
                iseOld = ise;
                OnRecordAdded(ise);
            }
        }
        {
            auto& cleanupBucket = CleanupSchedule_.GetBucketForKey(cleanupTime);
            TWriteGuard g(cleanupBucket.GetMutex());
            auto& bucketPtrsToCleanup = cleanupBucket.GetMap()[cleanupTime];
            isCleanupScheduleRequired = bucketPtrsToCleanup.empty();
            if (bucketPtrsToCleanup.insert(&bucket).second) {
                Counters_.NBytes += sizeof &bucket;
            }
        }
    }
    if (isCleanupScheduleRequired) {
        TScheduler::Instance().Enqueue(cleanupTime, [this, cleanupTime]() {ExecuteCleanup(cleanupTime);});
    }
    return true;
}

TMaybe<TSearchSubKey> TUserEventsStorage::FindSearchSubKey(const TUserIdentifier& userId) const {
    auto& bucket = InteractiveSearchEvents_.GetBucketForKey(userId);
    TReadGuard g(bucket.GetMutex());
    auto itR = bucket.GetMap().find(userId);
    if (itR == bucket.GetMap().end()) {
        return Nothing();
    }
    const TInteractiveSearchEvent& ise = itR->second;
    if (ise.ExpTime < ::Now()) {
        return Nothing();
    }
    return ise.SubKey;
}

void TUserEventsStorage::ExecuteCleanup(TInstant cleanupTime) {
    TProfileTimer started;
    TDuration maxLock = TDuration::Zero();
    THashSet<TInteractiveSearchEventMap::TBucket*> bucketPtrsToCleanup;
    {
        auto& cleanupBucket = CleanupSchedule_.GetBucketForKey(cleanupTime);
        TWriteGuard g(cleanupBucket.GetMutex());
        TProfileTimer lockStarted;
        auto itB = cleanupBucket.GetMap().find(cleanupTime);
        if (itB == cleanupBucket.GetMap().end()) {
            return;
        }
        itB->second.swap(bucketPtrsToCleanup);
        Counters_.NBytes -= sizeof *itB;
        cleanupBucket.GetMap().erase(itB);
        maxLock = Max(maxLock, lockStarted.Get());
    }
    for (auto bucketPtr : bucketPtrsToCleanup) {
        TDuration dur = CleanupBucket(bucketPtr, cleanupTime);
        maxLock = Max(maxLock, dur);
    }
    INFO_LOG << "Cleanup done in " << started.Get() << ", cleaned " << bucketPtrsToCleanup.size() << " buckets, maxLock: " << maxLock << Endl;
}

TDuration TUserEventsStorage::CleanupBucket(TInteractiveSearchEventMap::TBucket* bucketPtr, TInstant expTime) {
    TProfileTimer started;
    TWriteGuard g(bucketPtr->GetMutex());
    for (auto itR = bucketPtr->GetMap().begin(); itR != bucketPtr->GetMap().end();) {
        if (itR->second.ExpTime < expTime) {
            OnRecordRemoved(itR->second);
            Counters_.NBytes -= sizeof *itR + itR->first.GetAllocSize();
            bucketPtr->GetMap().erase(itR++);
        } else {
            ++itR;
        }
    }
    return started.Get();
}

void TUserEventsStorage::OnRecordRemoved(const TInteractiveSearchEvent& ise) const {
    Counters_.NInteractiveSearchEvents.Dec();
    Counters_.NBytes -= sizeof ise + ise.GetAllocSize();
}

void TUserEventsStorage::OnRecordAdded(const TInteractiveSearchEvent& ise) const {
    Counters_.NInteractiveSearchEvents.Inc();
    Counters_.NBytes += sizeof ise + ise.GetAllocSize();
}

} // namespace NTravel::NOfferCache
