#include "white_label_promo_service.h"

namespace NTravel::NOfferCache {
    TWhiteLabelPromoService::TWhiteLabelPromoService(const TWiteLabelProtoConfig& config)
        : PointsTypeLinguistic_(MakeConfig(config.GetPointsTypeLinguistic()))
        , S7Promo_(MakeConfig(config.GetS7Promo()))
    {
    }

    TWhiteLabelPromoService::~TWhiteLabelPromoService() {
    }

    THashMap<EWhiteLabelPointsType, TWhiteLabelPromoService::TLinguistic> TWhiteLabelPromoService::MakeConfig(const TWiteLabelProtoConfig::TPointsTypeLinguistic& pbCfg) const {
        THashMap<EWhiteLabelPointsType, TWhiteLabelPromoService::TLinguistic> res;
        for (const auto& linguistics : pbCfg.GetLinguistics()) {
            res[linguistics.GetPointsType()] = {
                .SingleNominative = linguistics.GetSingleNominative(),
                .SingleGenitive = linguistics.GetSingleGenitive(),
                .PluralGenitive = linguistics.GetPluralGenitive()};
        }
        return res;
    }

    TWhiteLabelPromoService::TWlPromoConfig TWhiteLabelPromoService::MakeConfig(const TWiteLabelProtoConfig::TWLPromoConfig& pbCfg) const {
        TWhiteLabelPromoService::TWlPromoConfig res = {
            .Enabled = pbCfg.GetEnabled(),
            .DefaultPointsConfig = ParsePointsConfig(pbCfg.GetDefaultPointsConfig()),
            .DefaultPointsType = pbCfg.GetDefaultPointsType()};
        for (const auto& event : pbCfg.GetEvents()) {
            TWLEventConfig wlEvent = {
                .EventId = event.GetEventId(),
                .PointsConfig = ParsePointsConfig(event.GetPointsConfig()),
                .PointsType = event.GetPointsType()};
            if (event.HasOrderTimeStartUtc()) {
                wlEvent.OrderTimeStartUtc = TInstant::ParseIso8601(event.GetOrderTimeStartUtc());
            }
            if (event.HasOrderTimeEndUtc()) {
                wlEvent.OrderTimeEndUtc = TInstant::ParseIso8601(event.GetOrderTimeEndUtc());
            }

            res.Events.push_back(wlEvent);
        }
        return res;
    }

    TWLBasePointsConfigRef TWhiteLabelPromoService::ParsePointsConfig(const TWiteLabelProtoConfig::TWLPointsConfig& ppCfg) const {
        if (ppCfg.HasPointsPerAmountConfig()) {
            auto config = ppCfg.GetPointsPerAmountConfig();
            return MakeAtomicShared<TWLPointsPerAmountConfig>(TWLPointsPerAmountConfig(config.GetPoints(), config.GetAmount()));
        } else {
            throw yexception() << "PartnerPointsConfig for White Label is not defined";
        }
    }

    TString TWhiteLabelPromoService::GetNameForNumeralNominative(ui32 pointsAmount, EWhiteLabelPointsType pointsType) const {
        const TLinguistic* linguistic = PointsTypeLinguistic_.FindPtr(pointsType);
        if (!linguistic) {
            throw yexception() << "Linguistic for " << NTravelProto::NWhiteLabel::EWhiteLabelPointsType_Name(pointsType) << " is not defined";
        }

        ui32 firstDecimal = pointsAmount % 10;
        ui32 secondDecimal = pointsAmount % 100 / 10;
        if (secondDecimal == 1) {
            return linguistic->PluralGenitive;
        }
        if (firstDecimal == 1) {
            return linguistic->SingleNominative;
        }
        if (2 <= firstDecimal && firstDecimal <= 4) {
            return linguistic->SingleGenitive;
        }
        return linguistic->PluralGenitive;
    }

    void TWhiteLabelPromoService::FillPointsLinguistics(ui32 pointsAmount, EWhiteLabelPointsType pointsType, NTravelProto::NPromoService::TWhiteLabelPointsLinguistics* linguistics) const {
        linguistics->SetNameForNumeralNominative(GetNameForNumeralNominative(pointsAmount, pointsType));
    }

    void TWhiteLabelPromoService::FillWLStatus(const TForOfferInternalReq& req, NTravelProto::NPromoService::TWhiteLabelStatus* res, const TWhiteLabelPromoService::TWlPromoConfig& promoConfig) const {
        if (!promoConfig.Enabled) {
            res->SetEligibility(NTravelProto::NPromoService::WLE_PROMO_DISABLED);
            return;
        }

        res->SetEligibility(NTravelProto::NPromoService::WLE_ELIGIBLE);
        res->SetPartnerId(req.WhiteLabelInfo.GetPartnerId());

        TPointsOffer bestPointsOffer = {
            .EventId = "default",
            .PointsAmount = promoConfig.DefaultPointsConfig->CalculatePointsOffer(req.LatestPrice.Value),
            .PointsType = promoConfig.DefaultPointsType};

        for (const auto& event : promoConfig.Events) {
            if (event.OrderTimeStartUtc.Defined() && req.Now < event.OrderTimeStartUtc.GetRef()) {
                continue;
            }
            if (event.OrderTimeEndUtc.Defined() && req.Now >= event.OrderTimeEndUtc.GetRef()) {
                continue;
            }

            ui32 pointsAmount = event.PointsConfig->CalculatePointsOffer(req.LatestPrice.Value);
            if (pointsAmount <= bestPointsOffer.PointsAmount) {
                continue;
            }
            bestPointsOffer.EventId = event.EventId;
            bestPointsOffer.PointsAmount = pointsAmount;
            bestPointsOffer.PointsType = event.PointsType;
        }

        res->MutablePoints()->SetAmount(bestPointsOffer.PointsAmount);
        res->MutablePoints()->SetPointsType(bestPointsOffer.PointsType);
        FillPointsLinguistics(bestPointsOffer.PointsAmount, bestPointsOffer.PointsType, res->MutablePointsLinguistics());
        res->MutablePromoInfo()->SetEventId(bestPointsOffer.EventId);
    }

    void TWhiteLabelPromoService::FillStatus(const TForOfferInternalReq& req, NTravelProto::NPromoService::TWhiteLabelStatus* res) const {
        if (!req.WhiteLabelInfo.IsInitialized() || req.WhiteLabelInfo.GetPartnerId() == NTravelProto::NWhiteLabel::WL_UNKNOWN) {
            res->SetEligibility(NTravelProto::NPromoService::WLE_UNUSED);
            return;
        }
        switch (req.WhiteLabelInfo.GetPartnerId()) {
            case NTravelProto::NWhiteLabel::WL_S7:
                FillWLStatus(req, res, S7Promo_);
                break;
            default:
                res->SetEligibility(NTravelProto::NPromoService::WLE_UNKNOWN_PARTNER);
                break;
        }
    }

    TWLPointsPerAmountConfig::TWLPointsPerAmountConfig(ui32 points, ui32 amount)
        : Points_(points)
        , Amount_(amount)
    {
    }

    ui32 TWLPointsPerAmountConfig::CalculatePointsOffer(ui32 totalPrice) const {
        return (totalPrice / Amount_) * Points_;
    }
} // namespace NTravel::NOfferCache
