import abc
import base64
import datetime
from abc import ABCMeta

from dateutil import parser as dateparser
import json
import random
import string
import time
from timeit import default_timer as timer
from flask import jsonify
from flask import render_template

from travel.hotels.partner_mocks.objects import Offer, Result, Response, Request, Occupancy, HotelInfo

MAX_OFFERS_PER_HOTEL = 9
MINIMUM_DAY_PRICE = 500
MAXIMUM_DAY_PRICE = 15000
MIN_OFFER_COEFFICIENT = 0.7
MAX_OFFER_COEFFICIENT = 1.4
MIN_DELAY = 0.05
MAX_DELAY = 3
MAX_CITY_COUNT = 2000


class PartnerMock(object):
    filters = {}

    def __init__(self):
        self.delay_histogram = {}
        self.avg_number_of_retries = 0
        self.static_initialized = False

    def init_static(self):
        self.static_initialized = True

    def handle(self, method, path, args, body):
        if not self.static_initialized:
            self.init_static()
        if method == "GET":
            return self.handle_get(path, args)
        elif method == "POST":
            return self.handle_post(path, args, body)
        else:
            return None

    def handle_get(self, path, args):
        return self.default_search(args)

    def handle_post(self, path, args, body):
        return None

    def default_search(self, args):
        started = timer()
        seed = datetime.datetime.now().hour
        req = self.generate_request(args)
        resp = self.generate_response(seed, req)

        text = render_template(self.template, response=resp, request=req)
        res = jsonify(json.loads(text))
        self.delay(started)
        return res

    @abc.abstractmethod
    def generate_request(self, args):
        pass

    @property
    @abc.abstractmethod
    def name(self):
        pass

    @property
    def template(self):
        return self.name + ".jinja2"

    def generate_response(self, seed, request):
        rnd = random.Random()
        rnd.seed(seed)
        response_id = self.generate_response_id(rnd, seed, request)
        results = []
        for hotel_id in request.hotel_ids:
            rnd.seed((seed, hotel_id, request.checkin, request.checkout, request.occupancy.num_adults,
                      tuple(request.occupancy.child_ages)))
            num_offers = rnd.randint(0, MAX_OFFERS_PER_HOTEL)
            base_price = self.generate_base_price(rnd, hotel_id, request)
            offers = [Offer(id=self.generate_offer_id(rnd, i, hotel_id, request),
                            num=i,
                            room_id=self.generate_room_id(rnd, i, hotel_id, request),
                            room_type_id=self.generate_room_type(rnd, i, hotel_id, request),
                            price=self.generate_offer_price(rnd, base_price, i, hotel_id, request),
                            num_adults=request.occupancy.num_adults,
                            num_children=len(request.occupancy.child_ages),
                            pansion=self.generate_offer_pansion(rnd),
                            mobile=request.mobile) for i in range(num_offers)]
            results.append(Result(hotel_id, offers))

        return Response(response_id, results)

    def generate_response_id(self, rnd, seed, request):
        s = "RANDOM_SEED_{}_TIMESTAMP_{}".format(seed, int(time.time() * 1000))
        b = s.encode('ascii')
        return base64.b64encode(b).decode('ascii').strip()

    def generate_delay(self):
        if not self.delay_histogram:
            return 0
        rnd = random.Random()
        i = rnd.randint(0, len(self.delay_histogram) - 2)
        return ((self.delay_histogram[i] + self.delay_histogram[i + 1]) / 2) / 1000.0

    def check_success(self, attempt):
        if self.avg_number_of_retries == 0:
            return True
        else:
            rnd = random.Random()
            success_probability = 1 / self.avg_number_of_retries
            return rnd.random() < success_probability

    def set_delay_histogram(self, value):
        self.delay_histogram = value

    def set_avg_number_of_retries(self, value):
        self.avg_number_of_retries = value

    def delay(self, started=None):
        if started is None:
            started = timer()
        delay_seconds = self.generate_delay()
        ended = timer()
        elapsed_seconds = (ended - started)
        if delay_seconds > elapsed_seconds:
            v = delay_seconds - elapsed_seconds
            time.sleep(v)

    @abc.abstractmethod
    def generate_offer_id(self, rnd, index, hotel_id, request):
        pass

    @abc.abstractmethod
    def generate_room_id(self, rnd, index, hotel_id, request):
        pass

    @abc.abstractmethod
    def generate_room_type(self, rnd, index, hotel_id, request):
        pass

    def generate_base_price(self, rnd, hotel_id, request):
        return rnd.uniform(MINIMUM_DAY_PRICE, MAXIMUM_DAY_PRICE)

    def generate_offer_price(self, rnd, base_price, index, hotel_id, request):
        return round(base_price * rnd.uniform(MIN_OFFER_COEFFICIENT, MAX_OFFER_COEFFICIENT), 2) * (
            request.get_check_out_date() - request.get_checkin_date()).days

    def generate_offer_pansion(self, rnd):
        pass


class BookingMock(PartnerMock):
    name = "booking"

    def __init__(self):
        super(BookingMock, self).__init__()
        self.filters = {"occupancy": self.occupancy}

    def handle_get(self, path, args):
        if "hotels" in path:
            return self.handle_hotels(args)
        return self.default_search(args)

    def generate_request(self, args):
        num_adults = 0
        child_ages = []
        if 'room1' in args:
            for part in args['room1'].split(','):
                if part.upper() == 'A':
                    num_adults += 1
                else:
                    child_ages.append(int(part))
        mobile = False
        if 'user_platform' in args:
            mobile = args['user_platform'].lower() == "mobile"

        return Request(args['checkin'], args['checkout'], Occupancy(num_adults, child_ages),
                       args['hotel_ids'].split(','), mobile)

    def generate_room_id(self, rnd, index, hotel_id, request):
        return int("{}{}".format(hotel_id, index))

    def generate_room_type(self, rnd, index, hotel_id, request):
        return index

    def generate_offer_id(self, rnd, index, hotel_id, request):
        return "{}{}_{}".format(hotel_id, index, request.occupancy.num_adults)

    def occupancy(self, value):
        return ",".join(['A'] * value.num_adults + [str(age) for age in value.child_ages])

    def handle_hotels(self, args):
        started = timer()
        seed = datetime.datetime.now().hour
        req = self.generate_hotels_request(args)
        resp = self.generate_hotels_response(seed, req)

        template = "booking_hotels.jinja2"
        text = render_template(template, response=resp, request=req)
        res = jsonify(json.loads(text))
        self.delay(started)
        return res

    def generate_hotels_request(self, args):
        return Request(None, None, None, args['hotel_ids'].split(','))

    def generate_hotels_response(self, seed, request):
        rnd = random.Random()
        rnd.seed(seed)
        response_id = self.generate_response_id(rnd, seed, request)
        results = []
        for hotel_id in request.hotel_ids:
            rnd.seed((seed, hotel_id))
            city_id = rnd.randint(0, MAX_CITY_COUNT)
            hotel_info = HotelInfo(hotel_id, city_id)
            results.append(Result(hotel_id, [], hotel_info))

        return Response(response_id, results)


class OstrovokMock(PartnerMock):
    name = "ostrovok"
    OFFER_ID_LENGTH = 15

    def __init__(self):
        super(OstrovokMock, self).__init__()
        self.filters = {"occupancy": self.occupancy,
                        "date": self.date}

    def generate_request(self, args):
        data = json.loads(args['data'])
        return Request(data['checkin'], data['checkout'], Occupancy(data["adults"], data.get("children", [])),
                       data["ids"])

    def generate_offer_id(self, rnd, index, hotel_id, request):
        return "".join([rnd.choice(string.ascii_uppercase + string.digits) for i in range(self.OFFER_ID_LENGTH)])

    def generate_room_id(self, rnd, index, hotel_id, request):
        return "{}{}".format(hotel_id, index)

    def generate_room_type(self, rnd, index, hotel_id, request):
        return hotel_id

    def occupancy(self, occupancy):
        if len(occupancy.child_ages) == 0:
            return str(occupancy.num_adults)
        else:
            return "{}and{}".format(occupancy.num_adults, ".".join([str(age) for age in occupancy.child_ages]))

    def date(self, value):
        return dateparser.parse(value).strftime("%d.%m.%Y")


class ExpediaMock(PartnerMock):
    name = "expedia"
    ROOM_TYPE_LENGTH = 5

    def __init__(self):
        super(ExpediaMock, self).__init__()
        self.filters = {"occupancy": self.occupancy}

    def generate_room_id(self, rnd, index, hotel_id, request):
        return str(rnd.randint(1, 1000000000))

    def generate_request(self, args):
        occ_string = args["occupancy"]
        occ_parts = occ_string.split('-')
        num_adults = int(occ_parts[0])
        if len(occ_parts) > 1:
            child_ages = [int(age) for age in occ_parts[1].split(",")]
        else:
            child_ages = []
        return Request(args['checkin'], args['checkout'], Occupancy(num_adults, child_ages),
                       args.getlist('property_id'))

    def generate_room_type(self, rnd, index, hotel_id, request):
        return "".join([rnd.choice(string.ascii_uppercase) for i in range(self.ROOM_TYPE_LENGTH)])

    def generate_offer_id(self, rnd, index, hotel_id, request):
        return str(rnd.randint(1, 1000000000))

    def occupancy(self, value):
        if value.child_ages:
            return "{}-{}".format(value.num_adults, ",".join([str(age) for age in value.child_ages]))
        else:
            return str(value.num_adults)


class YandexMock(PartnerMock):
    __metaclass__ = ABCMeta
    PANSION = [
        "unknown",
        "alc",
        "bb",
        "dinner",
        "fbr",
        "hd",
        "da",
        "sc",
        "uai",
    ]

    def __init__(self):
        super(YandexMock, self).__init__()
        self.filters["flatten"] = self.flatten

    def generate_room_id(self, rnd, index, hotel_id, request):
        return ""

    def generate_request(self, args):
        checkout = dateparser.parse(args['date_from']) + datetime.timedelta(days=int(args['nights']))
        num_adults = int(args['adults'])
        child_ages = []
        if "children" in args:
            child_ages = args["children"].split(",")
        return Request(args['date_from'], checkout.strftime("%Y-%m-%d"), Occupancy(num_adults, child_ages),
                       args['hotel_ids'].split(','))

    def generate_room_type(self, rnd, index, hotel_id, request):
        return ""

    def generate_offer_id(self, rnd, index, hotel_id, request):
        return "{}_{}_{}_{}_{}".format(
            rnd.randint(100000000, 900000000),
            rnd.randint(100000000, 900000000),
            rnd.randint(0, 3),
            rnd.randint(0, 3),
            rnd.randint(0, 3))

    def flatten(self, iterator, attr_name):
        for item in iterator:
            if hasattr(item, attr_name):
                for subitem in getattr(item, attr_name, []):
                    yield item, subitem
            else:
                pass

    def generate_offer_pansion(self, rnd):
        return rnd.choice(self.PANSION)


class Hotels101Mock(YandexMock):
    name = "hotels101"


class TravellineMock(YandexMock):
    name = "travelline"


class HtcMock(PartnerMock):
    name = "hotelscombined"

    ALL_PROVIDERS = [
        {
            "code": "AGD",
            "name": "Agoda.com",
            "logo": "https://cdn.datahc.com/images/providers/logos/AGD.png?cdn=1.0.2018.260001-C500d91dbb4c8240fda7bbc0d513aa966c1f73ab4",
            "isDirect": False,
            "isLanguageSupported": True
        },
        {
            "code": "IAN",
            "name": "Hotels.com",
            "logo": "https://cdn.datahc.com/images/providers/logos/IAN.png?cdn=1.0.2018.260001-C500d91dbb4c8240fda7bbc0d513aa966c1f73ab4",
            "isDirect": False,
            "isLanguageSupported": True
        },
        {
            "code": "HDE",
            "name": "Hotel.info",
            "logo": "https://cdn.datahc.com/images/providers/logos/HDE.png?cdn=1.0.2018.260001-C500d91dbb4c8240fda7bbc0d513aa966c1f73ab4",
            "isDirect": False,
            "isLanguageSupported": True
        },
        {
            "code": "OLO",
            "name": "Amoma.com",
            "logo": "https://cdn.datahc.com/images/providers/logos/OLO.png?cdn=1.0.2019.036001-Caf23477cbeb36ebc2ab586fb7b09d678632f4b29",
            "isDirect": False,
            "isLanguageSupported": True
        }
    ]

    def generate_response(self, seed, request):
        response = super(HtcMock, self).generate_response(seed, request)
        response.extra["providers"] = self.generate_providers(seed, request, response)
        return response

    def check_success(self, attempt):
        if attempt == 0:
            return False
        else:
            return super(HtcMock, self).check_success(attempt)

    def generate_request(self, args):
        occ = args["rooms"].split(":")
        num_adults = int(occ[0])
        if len(occ) > 1:
            child_ages = [int(age) for age in occ[1].split(",")]
        else:
            child_ages = []
        return Request(args["checkin"], args["checkout"], Occupancy(num_adults, child_ages), [args["hotel"]])

    def generate_room_type(self, rnd, index, hotel_id, request):
        return ""

    def generate_room_id(self, rnd, index, hotel_id, request):
        return ""

    def generate_offer_id(self, rnd, index, hotel_id, request):
        return ""

    def generate_providers(self, seed, request, response):
        rnd = random.Random()
        if not response.results[0].offers:
            return []
        rnd.seed((seed, tuple(request.hotel_ids), request.checkin, request.checkout, request.occupancy.num_adults,
                  tuple(request.occupancy.child_ages)))
        num_providers = rnd.randint(1, min(len(response.results[0].offers), len(self.ALL_PROVIDERS)))
        providers = list(self.ALL_PROVIDERS)
        while len(providers) > num_providers:
            providers.pop(rnd.randint(0, len(providers) - 1))
        return providers

    def generate_response_id(self, rnd, seed, request):
        return rnd.randint(1000000, 9999999)


class HtcMhsMock(PartnerMock):
    name = "hotelscombinedmhs"

    ALL_PROVIDERS = [
        {
            "code": "AGD",
            "name": "Agoda.com",
            "logo": "https://cdn.datahc.com/images/providers/logos/AGD.png?cdn=1.0.2018.260001-C500d91dbb4c8240fda7bbc0d513aa966c1f73ab4",
            "isDirect": False,
            "isLanguageSupported": True
        },
        {
            "code": "IAN",
            "name": "Hotels.com",
            "logo": "https://cdn.datahc.com/images/providers/logos/IAN.png?cdn=1.0.2018.260001-C500d91dbb4c8240fda7bbc0d513aa966c1f73ab4",
            "isDirect": False,
            "isLanguageSupported": True
        },
        {
            "code": "HDE",
            "name": "Hotel.info",
            "logo": "https://cdn.datahc.com/images/providers/logos/HDE.png?cdn=1.0.2018.260001-C500d91dbb4c8240fda7bbc0d513aa966c1f73ab4",
            "isDirect": False,
            "isLanguageSupported": True
        },
        {
            "code": "OLO",
            "name": "Amoma.com",
            "logo": "https://cdn.datahc.com/images/providers/logos/OLO.png?cdn=1.0.2019.036001-Caf23477cbeb36ebc2ab586fb7b09d678632f4b29",
            "isDirect": False,
            "isLanguageSupported": True
        }
    ]

    def generate_response(self, seed, request):
        response = super(HtcMhsMock, self).generate_response(seed, request)
        response.extra["providers"] = self.generate_providers(seed, request, response)
        return response

    def check_success(self, attempt):
        if attempt == 0:
            return False
        else:
            return super(HtcMhsMock, self).check_success(attempt)

    def generate_request(self, args):
        occ = args["rooms"].split(":")
        num_adults = int(occ[0])
        if len(occ) > 1:
            child_ages = [int(age) for age in occ[1].split(",")]
        else:
            child_ages = []
        return Request(args["checkin"], args["checkout"], Occupancy(num_adults, child_ages),
                       args["destination"].strip("hotels:").split(","))

    def generate_room_type(self, rnd, index, hotel_id, request):
        return ""

    def generate_room_id(self, rnd, index, hotel_id, request):
        return ""

    def generate_offer_id(self, rnd, index, hotel_id, request):
        return ""

    def generate_providers(self, seed, request, response):
        rnd = random.Random()
        if not response.results[0].offers:
            return []
        rnd.seed((seed, tuple(request.hotel_ids), request.checkin, request.checkout, request.occupancy.num_adults,
                  tuple(request.occupancy.child_ages)))
        num_providers = rnd.randint(1, min(len(response.results[0].offers), len(self.ALL_PROVIDERS)))
        providers = list(self.ALL_PROVIDERS)
        while len(providers) > num_providers:
            providers.pop(rnd.randint(0, len(providers) - 1))
        return providers

    def generate_response_id(self, rnd, seed, request):
        return rnd.randint(1000000, 9999999)


class DolphinMock(PartnerMock):
    name = "dolphin"
    supported_statics = {"Areas", "Pansions", "RoomCategories", "Rooms"}
    statics = {}

    def __init__(self):
        super(DolphinMock, self).__init__()
        self.filters = {"date": self.dolphin_date}

    def init_static(self):
        super(DolphinMock, self).init_static()
        for static in self.supported_statics:
            self.statics[static] = json.loads(render_template("dolphin_static_{}.jinja2".format(static)))

    def generate_offer_pansion(self, rnd):
        return rnd.choice(self.statics["Pansions"])["Id"]

    def handle_get(self, path, args):
        if path in self.supported_statics:
            return json.dumps(self.statics[path])

    def handle_post(self, path, args, body):
        if path == "SearchPartnerCheckins":
            body_args = json.loads(body)
            return super(DolphinMock, self).default_search(body_args)

    def generate_request(self, args):
        checkin = datetime.datetime.strptime(args["CheckInDates"][0], "%d.%m.%Y").date().isoformat()
        checkout = (datetime.datetime.strptime(args["CheckInDates"][0], "%d.%m.%Y").date() + datetime.timedelta(
            days=args["Nights"][0])).isoformat()
        occupancy = Occupancy(args["Adults"], args["ChildAges"])
        hotels = args["Params"]["Location"]["Fields"][0]
        return Request(checkin, checkout, occupancy, hotels)

    def generate_response(self, seed, request):
        response = super(DolphinMock, self).generate_response(seed, request)
        response.results = [res for res in response.results if res.offers]
        return response

    def generate_offer_id(self, rnd, index, hotel_id, request):
        pass

    def generate_room_id(self, rnd, index, hotel_id, request):
        return rnd.choice(self.statics["Rooms"])["Id"]

    def generate_room_type(self, rnd, index, hotel_id, request):
        return rnd.choice(self.statics["RoomCategories"])["Id"]

    def dolphin_date(self, value):
        return datetime.datetime.strptime(value, "%Y-%m-%d").strftime("%d.%m.%Y")


class TvilMock(PartnerMock):
    name = "tvil"

    def __init__(self):
        super(TvilMock, self).__init__()

    def generate_request(self, args):
        # /yandexTravel/availability?hotels=424275,424280&check_in=2020-07-01&check_out=2020-07-07&guests=1,2,3&language=ru&currency=RUB&user_country=RU
        print("generate tvil req", args)
        guests = args['guests'].split(',')
        num_adults = int(guests[0])
        child_ages = [int(age) for age in guests[1:]]
        return Request(args['check_in'], args['check_out'], Occupancy(num_adults, child_ages),
                       args['hotels'].split(','))

    def handle_post(self, path, args, body):
        return super(TvilMock, self).default_search(args)
