from gevent import monkey
monkey.patch_all()

import argparse
import os
from os import path

from flask import abort
from flask import Flask
from flask import Response
from flask import request

from travel.hotels.partner_mocks.generators import BookingMock
from travel.hotels.partner_mocks.generators import ExpediaMock
from travel.hotels.partner_mocks.generators import HtcMock
from travel.hotels.partner_mocks.generators import HtcMhsMock
from travel.hotels.partner_mocks.generators import OstrovokMock
from travel.hotels.partner_mocks.generators import Hotels101Mock
from travel.hotels.partner_mocks.generators import TravellineMock
from travel.hotels.partner_mocks.generators import DolphinMock
from travel.hotels.partner_mocks.generators import TvilMock

PARTNERS = [
    BookingMock(),
    OstrovokMock(),
    ExpediaMock(),
    Hotels101Mock(),
    TravellineMock(),
    HtcMock(),
    HtcMhsMock(),
    DolphinMock(),
    TvilMock()
]

PARTNERS_MAP = {p.name: p for p in PARTNERS}


app = Flask(__name__)


@app.route("/ping", methods=["GET"])
def ping():
    return "OK"


@app.route("/<partner>", methods=["GET", "POST"])
@app.route("/<partner>/<path:path>", methods=["GET", "POST"])
def do(partner, path=None):
    p = PARTNERS_MAP.get(partner)
    if not p:
        abort(404)
    if "x-ya-hotels-searcher-attempt" in request.headers:
        attempt = int(request.headers["x-ya-hotels-searcher-attempt"])
    else:
        attempt = 0
    if p.check_success(attempt):
        result = p.handle(request.method, path, request.args, request.data)
        if result is None:
            abort(404)
        else:
            return result
    else:
        p.delay()
        return Response("", status=202)


def load_histogram(filename):
    partners_map = {p.name: p for p in PARTNERS}
    with open(filename) as f:
        for line in f:
            values = line.split('\t')
            partner = values[0]
            if partner in partners_map:
                partners_map[partner].set_delay_histogram([int(float(v)) for v in values[1:-1]])
                partners_map[partner].set_avg_number_of_retries(float(values[-1]))
                print("Delay histogram loaded for {}".format(partner))


def run():
    parser = argparse.ArgumentParser()
    parser.add_argument('-b', '--bind', default='::')
    parser.add_argument('-p', '--port', type=int, default=4242)
    parser.add_argument('-t', '--template-dir', default=path.join(os.curdir, "templates"))
    parser.add_argument('-d', '--delay-file', default=path.join(os.curdir, "histogram.tsv"))
    args = parser.parse_args()
    app.config['JSON_AS_ASCII'] = False
    if os.path.isabs(args.template_dir):
        app.template_folder = args.template_dir
    else:
        app.template_folder = os.path.abspath(os.path.join(os.curdir, args.template_dir))
    load_histogram(args.delay_file)

    # Register partner-specific jinja-filters
    for partner in PARTNERS:
        for name, flt in partner.filters.items():
            app.jinja_env.filters['{}_{}'.format(partner.name, name)] = flt

    app.run(host=args.bind, port=args.port)
