#include "offer_filter.h"

#include <travel/hotels/pricechecker/service/service.h>

#include <travel/hotels/lib/cpp/ordinal_date/ordinal_date.h>

#include <library/cpp/logger/global/global.h>

namespace NTravel::NPriceChecker {
    TOfferFilter::TOfferFilter(TService& service, double baseRate)
        : BaseRate_(baseRate)
        , Service_(service)
        , TotalBucket_({0, 0})
        , LastBucketRotationTimestamp_(TInstant::Days(Now().Days()) + TDuration::Hours(6)) // 9:00 msk, 11:00 ekb
        , Counters_({"PartnerId", "DaysBeforeUpperBound", "HourUpperBound"})
    {
    }

    void TOfferFilter::RegisterCounters(NMonitor::TCounterSource& source, const TString& name) {
        source.RegisterSource(&Counters_, name);
    }

    bool TOfferFilter::ShouldFollowOffer(TInstant timestamp, const NTravelProto::TSearchOffersReq& request, const NTravelProto::TOffer& offer) {
        auto bucketKey = GetBucketKey(timestamp, request, offer);

        auto inverseSampleRate = 0;

        with_lock (Lock_) {
            if (Now() - LastBucketRotationTimestamp_ > TDuration::Days(1)) {
                TotalBucket_.CurrentCount = TotalBucket_.NextCount;
                TotalBucket_.NextCount = 0;
                for (auto& [key, bucket] : Buckets_) {
                    bucket.CurrentCount = bucket.NextCount;
                    bucket.NextCount = 0;
                }
                LastBucketRotationTimestamp_ += TDuration::Days(1);
            }

            if (!Buckets_.contains(bucketKey)) {
                auto current = !Buckets_.empty()
                                   ? TotalBucket_.CurrentCount / Buckets_.size()
                                   : 1000; // could be 1, but 1000 to avoid problems with +-1
                Buckets_[bucketKey] = TBucket{current, 0};
                TotalBucket_.CurrentCount += current;
            }

            auto& bucket = Buckets_[bucketKey];
            bucket.NextCount++;
            TotalBucket_.NextCount++;

            /* We want rate to be inverse proportional to bucket size:
             * rate_i = k / size_i
             *
             * Also we need to have total rate equal to BaseRate:
             * Sum rate_i * size_i = BaseRate * Sum size_i
             *
             * So:
             * Sum k / size_i * size_i = BaseRate * Sum size_i
             * n * k = BaseRate * Sum size_i
             * k = BaseRate * Sum size_i / n
             *
             * sampleRate = k / newSize = BaseRate * Sum size_i / (n * newSize)
             * inverseSampleRate = k / newSize = n * newSize / (BaseRate * Sum size_i)
             *
             * Using max of new and old size, to make rate more inert and avoid peaks when bucket is almost empty
             * and when current bucket size is suddenly filled faster than earlier:
             * inverseSampleRate = k / newSize = n * max(oldSize, newSize) / (BaseRate * Sum size_i)
             *
             * */

            inverseSampleRate = Buckets_.size() * Max(bucket.CurrentCount, bucket.NextCount) / (BaseRate_ * TotalBucket_.CurrentCount);
        }

        size_t keyHash = THash<TString>()(offer.GetId());
        bool shouldSample = keyHash % Max(1, static_cast<int>(inverseSampleRate)) == 0;

        auto& [partnerId, daysBeforeUpperBound, hourUpperBound] = bucketKey;
        const auto& counters = Counters_.GetOrCreate({partnerId, ToString(daysBeforeUpperBound), ToString(hourUpperBound)});
        counters->NSeenOffers.Inc();
        counters->NSeenOffersRate.Inc();
        if (shouldSample) {
            counters->NSampledOffers.Inc();
            counters->NSampledOffersRate.Inc();
        }

        return shouldSample;
    }

    NTravelProto::NPriceChecker::TFilteringState TOfferFilter::ToProto() const {
        auto result = NTravelProto::NPriceChecker::TFilteringState();
        with_lock (Lock_) {
            for (auto& [key, bucket] : Buckets_) {
                auto kvPair = result.AddBuckets();
                auto mutableKey = kvPair->MutableKey();
                mutableKey->SetPartnerId(std::get<0>(key));
                mutableKey->SetDaysBeforeUpperBound(std::get<1>(key));
                mutableKey->SetHourUpperBound(std::get<2>(key));
                auto mutableValue = kvPair->MutableValue();
                mutableValue->SetCurrentCount(bucket.CurrentCount);
                mutableValue->SetNextCount(bucket.NextCount);
            }

            auto totalBucket = result.MutableTotalBucket();
            totalBucket->SetCurrentCount(TotalBucket_.CurrentCount);
            totalBucket->SetNextCount(TotalBucket_.NextCount);
            result.SetLastBucketRotationTimestamp(LastBucketRotationTimestamp_.Seconds());
        }
        return result;
    }

    void TOfferFilter::FromProto(const NTravelProto::NPriceChecker::TFilteringState& filteringState) {
        with_lock (Lock_) {
            Buckets_.clear();
            for (const auto& kvPair : filteringState.GetBuckets()) {
                const auto& pbKey = kvPair.GetKey();
                const auto& pbValue = kvPair.GetValue();
                auto key = std::make_tuple(pbKey.GetPartnerId(), pbKey.GetDaysBeforeUpperBound(), pbKey.GetHourUpperBound());
                auto value = TBucket{pbValue.GetCurrentCount(), pbValue.GetNextCount()};
                Buckets_[key] = value;
            }

            const auto& pbTotalBucket = filteringState.GetTotalBucket();
            TotalBucket_ = TBucket{pbTotalBucket.GetCurrentCount(), pbTotalBucket.GetNextCount()};
            LastBucketRotationTimestamp_ = TInstant::Seconds(filteringState.GetLastBucketRotationTimestamp());
        }
    }

    TOfferFilter::BucketKey TOfferFilter::GetBucketKey(TInstant timestamp, const NTravelProto::TSearchOffersReq& request, const NTravelProto::TOffer& offer) const {
        auto pId = Service_.GetPartnerIdByOperatorId(offer.GetOperatorId());
        auto partnerName = pId == NTravelProto::PI_UNUSED ? TString("UNKNOWN") : ToString(pId);

        auto checkInDate = NOrdinalDate::ToInstant(NOrdinalDate::FromString(request.GetCheckInDate()));
        auto actualizationTime = timestamp;

        auto daysBefore = (checkInDate - actualizationTime).Days();
        auto daysBeforeUpperBound = 1000000;
        for (size_t x : {1, 3, 5, 10}) {
            if (daysBefore <= x) {
                daysBeforeUpperBound = x;
                break;
            }
        }

        auto hourUpperBound = static_cast<int>(((actualizationTime.Hours() % 24) / 3 + 1) * 3 - 1);

        return {partnerName, daysBeforeUpperBound, hourUpperBound};
    }

    void TOfferFilter::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        ct->insert(MAKE_COUNTER_PAIR(NSampledOffers));
        ct->insert(MAKE_COUNTER_PAIR(NSeenOffers));

        ct->insert(MAKE_COUNTER_PAIR(NSampledOffersRate));
        ct->insert(MAKE_COUNTER_PAIR(NSeenOffersRate));
    }
}
