#include "offer_tracker.h"

#include <travel/hotels/lib/cpp/scheduler/scheduler.h>
#include <travel/hotels/proto2/bus_messages.pb.h>

#include <utility>
#include <sstream>

namespace NTravel::NPriceChecker {
    const TString TOfferTracker::PriceCheckerOfferCacheClientId = "price-checker-v2";

    TOfferTracker::TOfferTracker(TService& service,
                                 const TString& name,
                                 const NTravelProto::NAppConfig::TConfigGrpcClient& offerCacheClientConfig,
                                 const NTravelProto::NPriceChecker::TConfig::TTrackingStrategy& trackingStrategyConfig,
                                 const NTravelProto::NAppConfig::TConfigYtQueueReader& stateBusReaderConfig,
                                 const NTravelProto::NAppConfig::TConfigYtQueueWriter& stateBusWriterConfig,
                                 const NTravelProto::NPriceChecker::TConfig::TOfferTracker& offerTrackerConfig)
        : TombstoneDuration_(TDuration::Minutes(10))
        , SearcherResponseTimeout_(TDuration::Minutes(5))
        , Name_(name)
        , OfferFilter_(service, offerTrackerConfig.GetBaseSampleRate())
        , OfferCacheClient_(offerCacheClientConfig)
        , TrackingStrategy_(trackingStrategyConfig, &Counters_)
        , StateBusReader_(stateBusReaderConfig, Name_ + "StateReader")
        , StateBusWriter_(stateBusWriterConfig, Name_ + "StateWriter")
        , OfferTrackerConfig_(offerTrackerConfig)
        , MaxLoadedFilteringStateTimestamp_(TInstant::Zero())
        , MaxLoadedTimestamp_(TInstant::Zero())
    {
    }

    void TOfferTracker::RegisterCounters(NMonitor::TCounterSource& source) {
        source.RegisterSource(&Counters_, Name_);
        OfferFilter_.RegisterCounters(source, Name_ + "Filter");
        StateBusReader_.RegisterCounters(source);
        StateBusWriter_.RegisterCounters(source);
        TrackingStrategy_.RegisterCounters(source, Name_ + "TrackingStrategy");
    }

    void TOfferTracker::Subscribe(std::function<void(const TOfferTrackingState&)> handler) {
        FinishedCheckHandler_ = std::move(handler);
    }

    bool TOfferTracker::ProcessSearcherMessage(const TYtQueueMessage& busMessage, const ru::yandex::travel::hotels::TSearcherMessage& parsedMessage) {
        auto timestamp = busMessage.Timestamp;
        if (busMessage.Timestamp <= MaxLoadedTimestamp_) {
            return false;
        }

        const auto& request = parsedMessage.GetRequest();
        const auto& response = parsedMessage.GetResponse();

        auto clientId = request.GetAttribution().GetOfferCacheClientId();
        if (clientId == "price-checker" || clientId == "price-checker-filters") {
            return false;
        }

        if (clientId == PriceCheckerOfferCacheClientId) {
            DEBUG_LOG << "Got message with PC utm source. RequestId: " << request.GetId() << Endl;
            HandleCheckResult(timestamp, request, response);
            return true; // sic! we don't want to sample our own requests, so returning here
        }

        bool sampled = false;
        if (response.HasOffers()) {
            for (const auto& offer : response.GetOffers().GetOffer()) {
                if (OfferFilter_.ShouldFollowOffer(timestamp, request, offer)) {
                    DEBUG_LOG << "Will follow offer. RequestId: " << request.GetId() << ", OfferId: " << offer.GetId() << Endl;
                    StartNewTracking(timestamp, request, offer);
                    sampled = true;
                    if (!OfferTrackerConfig_.GetAllowSampleMoreThanOneOfferFromResponse()) {
                        break;
                    }
                }
            }
        }

        return sampled;
    }

    void TOfferTracker::SetInitializationFinishHandler(std::function<void(void)> handler) {
        InitializationFinishHandler_ = std::move(handler);
    }

    void TOfferTracker::Start() {
        NTravel::TScheduler::Instance().EnqueuePeriodical(TDuration::Minutes(1), [this]() {
            ReportStateSize();
        });

        StateBusReader_.Subscribe(NTravelProto::NPriceChecker::TTrackingState(), this, &TOfferTracker::ProcessTrackingStateMessage);
        StateBusReader_.Subscribe(NTravelProto::NPriceChecker::TFilteringState(), this, &TOfferTracker::ProcessFilteringStateMessage);
        StateBusReader_.Ignore(ru::yandex::travel::hotels::TSearcherMessage());
        StateBusReader_.SetReadinessNotifier([this]() {
            NTravel::TScheduler::Instance().Enqueue(Now(), [this]() {
                INFO_LOG << "State bus read finished" << Endl;
                StateBusReader_.Stop();
                INFO_LOG << "Starting state bus writer" << Endl;
                StateBusWriter_.Start();
                NTravel::TScheduler::Instance().EnqueuePeriodical(TDuration::Minutes(1), [this]() {
                    StateBusWriter_.Write(OfferFilter_.ToProto(), TDuration::Hours(1));
                });
                if (MaxLoadedTimestamp_ == TInstant::Zero()) {
                    MaxLoadedTimestamp_ = Now();
                }
                with_lock (TrackingStatesLock_) {
                    for (auto& [key, trackingState] : TrackingStates_) {
                        if (trackingState->IsActive()) {
                            Counters_.NActiveTrackings.Inc();
                        }
                        if (trackingState->IsWaiting()) {
                            Counters_.NWaitingChecks.Inc();
                        }
                    }
                    for (auto& [key, trackingState] : TrackingStates_) {
                        ScheduleNextCheckOrStop(trackingState, Now());
                    }
                }
                if (InitializationFinishHandler_) {
                    InitializationFinishHandler_();
                }
            });
        });
        StateBusReader_.Start();
    }

    void TOfferTracker::Stop() {
        StateBusReader_.Stop();
        OfferCacheClient_.Shutdown();
        StateBusWriter_.Stop();
    }

    bool TOfferTracker::ProcessTrackingStateMessage(const TYtQueueMessage& busMessage) {
        NTravelProto::NPriceChecker::TTrackingState trackingState;
        if (!trackingState.ParseFromString(busMessage.Bytes)) {
            throw yexception() << "Failed to parse TTrackingState message";
        }

        with_lock (TrackingStatesLock_) {
            auto key = trackingState.GetKey();
            auto it = TrackingStates_.find(key);
            if (it == TrackingStates_.end() || it->second->LastLoadedTimestamp < busMessage.Timestamp) {
                TrackingStates_[key] = TOfferTrackingState::FromProto(trackingState, &Counters_);
                TrackingStates_[key]->LastLoadedTimestamp = busMessage.Timestamp;
            }
            MaxLoadedTimestamp_ = Max(MaxLoadedTimestamp_, busMessage.Timestamp);
        }
        return true;
    }

    bool TOfferTracker::ProcessFilteringStateMessage(const TYtQueueMessage& busMessage) {
        NTravelProto::NPriceChecker::TFilteringState filteringState;
        if (!filteringState.ParseFromString(busMessage.Bytes)) {
            throw yexception() << "Failed to parse TFilteringState message";
        }
        with_lock (MaxLoadedFilteringStateTimestampLock_) {
            if (MaxLoadedFilteringStateTimestamp_ < busMessage.Timestamp) {
                OfferFilter_.FromProto(filteringState);
                MaxLoadedFilteringStateTimestamp_ = busMessage.Timestamp;
            }
        }
        return true;
    }

    void TOfferTracker::StartNewTracking(TInstant timestamp,
                                         const NTravelProto::TSearchOffersReq& requestPb,
                                         const NTravelProto::TOffer& offerPb) {
        auto key = offerPb.GetId();
        std::shared_ptr<TOfferTrackingState> trackingState;
        with_lock (TrackingStatesLock_) {
            auto insertResult = TrackingStates_.emplace(key, std::make_shared<TOfferTrackingState>(key, requestPb, offerPb, timestamp, &Counters_));

            if (!insertResult.second) {
                WARNING_LOG << "Duplicate request id: " << key << Endl;
                Counters_.NDuplicateRequestId.Inc();
                return;
            }

            trackingState = insertResult.first->second;
        }
        Counters_.NCreatedTrackings.Inc();
        with_lock (trackingState->Lock) {
            trackingState->ToWaitingForScheduling();
            ScheduleNextCheckOrStop(trackingState, timestamp);
        }
    }

    void TOfferTracker::HandleCheckResult(TInstant timestamp, const NTravelProto::TSearchOffersReq& request, const NTravelProto::TSearchOffersRsp& response) {
        if (response.HasPlaceholder()) {
            return;
        }
        auto keyAndIndex = GetKeyByRequestId(request.GetId());
        if (keyAndIndex.Empty()) {
            return;
        }
        auto key = keyAndIndex.Get()->first;
        auto checkIndex = keyAndIndex.Get()->second;

        std::shared_ptr<TOfferTrackingState> trackingState;
        with_lock (TrackingStatesLock_) {
            auto trackingStateIter = TrackingStates_.find(key);
            if (trackingStateIter == TrackingStates_.end()) {
                Counters_.NNotFoundTracking.Inc();
                return;
            }
            trackingState = trackingStateIter->second;
        }
        with_lock (trackingState->Lock) {
            if (!trackingState->IsActive()) {
                WARNING_LOG << "Response to not active tracking. Key: " << trackingState->Key << Endl;
                Counters_.NResponseToNotActiveTracking.Inc();
                return;
            }

            if (trackingState->OfferTrackingResult.HasCheckResult(checkIndex)) {
                Counters_.NResponseToFinishedChecking.Inc();
                return;
            }

            trackingState->ToWaitingForScheduling();

            if (response.HasError()) {
                const auto& error = response.GetError();
                ERROR_LOG << "Bus response error: " << error.GetMessage() << " RequestId: " << request.GetId() << Endl;
                Counters_.NBusResponseErrors.Inc();
                TrackingStrategy_.HandleCheckError(trackingState.get(), timestamp, error.GetMessage());
            }

            if (response.HasOffers()) {
                const auto& offersPb = response.GetOffers().GetOffer();
                TVector<NTravelProto::TOffer> offers(offersPb.begin(), offersPb.end());
                TrackingStrategy_.HandleCheckResult(trackingState.get(), timestamp, checkIndex, offers);
            }

            ScheduleNextCheckOrStop(trackingState, timestamp);
        }
    }

    void TOfferTracker::HandleRpcError(const std::shared_ptr<TOfferTrackingState>& trackingState, size_t checkIndex, const TString& error) {
        ERROR_LOG << "RPC error (key: " << trackingState->Key << "): " << error << Endl;
        with_lock (trackingState->Lock) {
            if (!trackingState->IsActive()) {
                WARNING_LOG << "Response to not active tracking. Key: " << trackingState->Key << Endl;
                Counters_.NResponseToNotActiveTracking.Inc();
                return;
            }
            if (trackingState->OfferTrackingResult.HasCheckResult(checkIndex)) {
                Counters_.NResponseToFinishedChecking.Inc();
                return;
            }
            trackingState->ToWaitingForScheduling();
            Counters_.NRpcErrors.Inc();
            TrackingStrategy_.HandleCheckError(trackingState.get(), Now(), error);
            ScheduleNextCheckOrStop(trackingState, Now());
        }
    }

    void TOfferTracker::ScheduleNextCheckOrStop(const std::shared_ptr<TOfferTrackingState>& trackingState, TInstant now) {
        if (trackingState->IsTerminating()) {
            StopTracking(trackingState.get());
            return;
        }

        TInstant nextCheckTime;
        size_t nextCheckId;
        if (!TrackingStrategy_.GetNextCheckInfoOrError(trackingState.get(), now, &nextCheckTime, &nextCheckId)) {
            WARNING_LOG << "No next check time got from strategy. Key: " << trackingState->Key << Endl;
            StopTracking(trackingState.get());
            return;
        }

        if (!trackingState->IsReadyToSchedule()) {
            ERROR_LOG << "Not scheduling tracking because tracking is not in appropriate state. Key: " << trackingState->Key << Endl;
            return;
        }

        SaveState(trackingState.get());

        trackingState->ToScheduled();

        DEBUG_LOG << "Scheduling tracking (nextCheckTime: " << nextCheckTime << ", key: " << trackingState->Key << ")" << Endl;

        NTravel::TScheduler::Instance().Enqueue(nextCheckTime, [this, trackingState, nextCheckId]() {
            trackingState->ToInFly();
            auto requestId = GetRequestIdByKey(trackingState->Key, nextCheckId);

            NTravelProto::TSearchOffersRpcReq rpcReq;
            rpcReq.SetSync(false);
            auto subReq = rpcReq.AddSubrequest();
            *subReq = trackingState->OfferTrackingResult.InitialRequest;
            subReq->SetOfferCacheIgnoreBlacklist(true);
            subReq->SetId(requestId);
            subReq->SetRequestClass(NTravelProto::ERequestClass::RC_BACKGROUND);
            subReq->MutableAttribution()->SetOfferCacheClientId(PriceCheckerOfferCacheClientId);
            subReq->SetOfferCacheUseCache(false);
            subReq->SetOfferCacheUseSearcher(true);

            Counters_.NSentRpcRequests.Inc();
            OfferCacheClient_.Request<NTravelProto::TSearchOffersRpcReq,
                                      NTravelProto::TSearchOffersRpcRsp,
                                      &NTravelProto::NOfferCacheGrpc::OfferCacheServiceV1::Stub::AsyncSearchOffers>(
                rpcReq,
                NGrpc::TClientMetadata(),
                [this, trackingState, nextCheckId](const TString& grpcError, const TString& /*remoteFQDN*/, const NTravelProto::TSearchOffersRpcRsp& resp) {
                    if (!grpcError.empty()) {
                        HandleRpcError(trackingState, nextCheckId, grpcError);
                        return;
                    }

                    if (resp.SubresponseSize() != 1) {
                        auto sz = ToString(resp.SubresponseSize());
                        HandleRpcError(trackingState, nextCheckId, "The number of subresponses equals to " + sz + ", expected to be 1");
                        return;
                    }

                    if (resp.GetSubresponse(0).HasError()) {
                        HandleRpcError(trackingState, nextCheckId, resp.GetSubresponse(0).GetError().GetMessage());
                        return;
                    }
                });

            NTravel::TScheduler::Instance().Enqueue(Now() + SearcherResponseTimeout_, [this, trackingState, nextCheckId]() {
                with_lock (trackingState->Lock) {
                    if (!trackingState->IsInFly() ||
                        trackingState->OfferTrackingResult.HasCheckResult(nextCheckId) ||
                        (!trackingState->LastSuccessiveErrors.empty() &&
                         trackingState->LastSuccessiveErrors.rbegin()->Timestamp > Now() - SearcherResponseTimeout_))
                    {
                        return;
                    }

                    ERROR_LOG << "No response from searcher after " << SearcherResponseTimeout_ << Endl;
                    Counters_.NSearcherTimeouts.Inc();
                    TrackingStrategy_.HandleSearcherTimeout(trackingState.get(), Now());
                    ScheduleNextCheckOrStop(trackingState, Now());
                }
            });
        });
    }

    void TOfferTracker::StopTracking(TOfferTrackingState* trackingState) {
        if (!trackingState->IsTerminating()) {
            ERROR_LOG << "Tried to stop tracking not in terminating state. Key: " << trackingState->Key << Endl;
            return;
        }

        if (!trackingState->IsReported()) {
            DEBUG_LOG << "Stopping tracking. Key: " << trackingState->Key << Endl;
            trackingState->ToReported();
            SaveState(trackingState);
            Counters_.NCompletedTrackings.Inc();
            if (FinishedCheckHandler_) {
                FinishedCheckHandler_(*trackingState);
            }
        }

        auto key = trackingState->Key;
        NTravel::TScheduler::Instance().Enqueue(Now() + TombstoneDuration_, [this, key]() {
            with_lock (TrackingStatesLock_) {
                TrackingStates_.erase(key);
            }
        });
    }

    void TOfferTracker::SaveState(TOfferTrackingState* trackingState) {
        StateBusWriter_.Write(trackingState->ToProto(), TrackingStrategy_.GetLifetime(trackingState, Now()));
    }

    void TOfferTracker::ReportStateSize() {
        size_t byteSize = 0;
        size_t spaceUsed = 0;
        auto start = Now();
        with_lock (TrackingStatesLock_) {
            byteSize += sizeof(TrackingStates_);
            spaceUsed += sizeof(TrackingStates_);
            for (const auto& [key, trackingState] : TrackingStates_) {
                byteSize += sizeof(key) + key.length();
                spaceUsed += sizeof(key) + key.length();
                byteSize += trackingState->GetSize(false);
                spaceUsed += trackingState->GetSize(true);
            }
        }
        Counters_.TrackerByteSize = byteSize;
        Counters_.TrackerSpaceUsed = spaceUsed;
        Counters_.NMemoryMeasuringTimeSec = (Now() - start).Seconds();
    }

    TString TOfferTracker::GetRequestIdByKey(const TString& key, int index) {
        return Sprintf("PC-%03d-", index) + key;
    }

    TMaybe<std::pair<TString, size_t>> TOfferTracker::GetKeyByRequestId(const TString& requestId) {
        if (requestId.substr(0, 3) != "PC-") {
            ERROR_LOG << "Request id doesn't start with PC- (RequestId: " << requestId << ")" << Endl;
            return {};
        }
        auto index = FromString<size_t>(requestId.substr(3, 3));
        return {{requestId.substr(7), index}};
    }

}
