#pragma once

#include "offer_tracker_counters.h"

#include <travel/hotels/pricechecker/proto/tracking_state_message.pb.h>
#include <travel/hotels/proto2/bus_messages.pb.h>

#include <util/datetime/base.h>
#include <util/generic/maybe.h>
#include <util/generic/variant.h>
#include <util/system/mutex.h>

#include <utility>

namespace NTravel::NPriceChecker {
    struct TSingleCheckResult {
        TInstant Timestamp;
        TMaybe<NTravelProto::TOffer> MatchedOffer;
    };

    struct TUnknownTrackingResult {
    };

    struct TPriceDifferTrackingResult {
        TPriceDifferTrackingResult(TDuration lifetimeMin, TDuration lifetimeMax, int oldPrice, int newPrice)
            : LifetimeMin(lifetimeMin)
            , LifetimeMax(lifetimeMax)
            , OldPrice(oldPrice)
            , NewPrice(newPrice)
        {
        }
        TDuration LifetimeMin;
        TDuration LifetimeMax;
        int OldPrice;
        int NewPrice;
    };

    struct TPriceNotFoundTrackingResult {
        TPriceNotFoundTrackingResult(TDuration lifetimeMin, TDuration lifetimeMax, int oldPrice)
            : LifetimeMin(lifetimeMin)
            , LifetimeMax(lifetimeMax)
            , OldPrice(oldPrice)
        {
        }
        TDuration LifetimeMin;
        TDuration LifetimeMax;
        int OldPrice;
    };

    struct TNoChangesTrackingResult {
    };

    struct TErrorTrackingResult {
        explicit TErrorTrackingResult(TString error)
            : Error(std::move(error))
        {
        }
        TString Error;
    };

    struct TOfferTrackingResult {
        TInstant InitialTimestamp;
        NTravelProto::TSearchOffersReq InitialRequest;
        NTravelProto::TOffer InitialOffer;
        TVector<TSingleCheckResult> CheckResults;
        std::variant<TUnknownTrackingResult,
                 TPriceDifferTrackingResult,
                 TPriceNotFoundTrackingResult,
                 TNoChangesTrackingResult,
                 TErrorTrackingResult>
            Result;

        bool HasCheckResult(size_t checkId) {
            return checkId < CheckResults.size();
        }
    };

    struct TError {
        TError(TString error, TInstant timestamp)
            : Error(error)
            , Timestamp(timestamp)
        {
        }
        TString Error;
        TInstant Timestamp;
    };

    class TOfferTrackingState {
    public:
        TOfferTrackingState(const TString& key,
                            const NTravelProto::TSearchOffersReq& requestPb,
                            const NTravelProto::TOffer& offerPb,
                            TInstant timestamp,
                            TOfferTrackerCounters* counters);

        enum class EOfferTrackingStateType {
            New,
            WaitingForScheduling,
            Scheduled,
            InFly,
            Done,
            Reported,
        };

        using PbTrackingStateType = NTravelProto::NPriceChecker::TTrackingState::ETrackingStateType;

        TMutex Lock;

        TString Key;
        TInstant LastLoadedTimestamp;
        TOfferTrackingResult OfferTrackingResult;
        TVector<TError> LastSuccessiveErrors;

        bool IsActive();
        bool IsTerminating();
        bool IsReadyToSchedule();
        bool IsWaiting();
        bool IsInFly();
        bool IsReported();

        void ToWaitingForScheduling();
        void ToScheduled();
        void ToInFly();
        void ToDone();
        void ToReported();

        size_t GetSize(bool useSpaceUsedForPb);

        NTravelProto::NPriceChecker::TTrackingState ToProto() const;
        static std::shared_ptr<TOfferTrackingState> FromProto(const NTravelProto::NPriceChecker::TTrackingState& trackingState, TOfferTrackerCounters* counters);

    private:
        struct TResultVisitorToProto {
            explicit TResultVisitorToProto(NTravelProto::NPriceChecker::TTrackingState::TOfferTrackingResult* trackingResult);

            void operator()(const TUnknownTrackingResult& result) const;

            void operator()(const TPriceDifferTrackingResult& result) const;

            void operator()(const TPriceNotFoundTrackingResult& result) const;

            void operator()(const TNoChangesTrackingResult& result) const;

            void operator()(const TErrorTrackingResult& result) const;

            NTravelProto::NPriceChecker::TTrackingState::TOfferTrackingResult* TrackingResult;
        };

        EOfferTrackingStateType StateType_;
        TOfferTrackerCounters* Counters_;

        void TrySwitch(EOfferTrackingStateType targetState, const std::function<bool()>& condition);
        static EOfferTrackingStateType TrackingStateFromPbToInner(PbTrackingStateType stateType);
        static PbTrackingStateType TrackingStateFromInnerToPb(EOfferTrackingStateType stateType);
    };
}