#include <travel/hotels/proto/app_config/yt_table_cache.pb.h>

#include <travel/hotels/lib/cpp/scheduler/scheduler.h>
#include <travel/hotels/lib/cpp/yt/data.h>
#include <travel/hotels/proto2/bus_messages.pb.h>
#include <travel/hotels/lib/cpp/ordinal_date/ordinal_date.h>
#include "price_filter_checker.h"

#include <util/random/fast.h>
#include <util/random/entropy.h>

namespace NTravel::NPriceChecker {
    const TString TPriceFilterChecker::PriceFilterCheckerOfferCacheClientId = "price-checker-filters";

    TPriceFilterChecker::TPriceFilterChecker(
        const NTravelProto::NAppConfig::TConfigGrpcClient& offerCacheClientConfig,
        const NTravelProto::NAppConfig::TYtTableCacheConfig& priceFilterTableConfig,
        const NTravelProto::NAppConfig::TYtTableCacheConfig& permalinkToClusterMapper,
        const NTravelProto::NAppConfig::TYtTableCacheConfig& permalinkToOriginalIdsMapper,
        const NTravelProto::NAppConfig::TYtTableCacheConfig& ytConfigPartners,
        double rps)
        : OfferCacheClient_(offerCacheClientConfig)
        , PriceFilterTable_("PriceFilterTable", priceFilterTableConfig)
        , PermalinkToClusterMapper_("PermalinkToClusterMapper", permalinkToClusterMapper)
        , PermalinkToOriginalIdsMapper_("PermalinkToOriginalIdsMapper", permalinkToOriginalIdsMapper)
        , YtConfigPartners_("YtConfigPartners", ytConfigPartners)
        , Start_(Now())
        , Epoch_(0)
        , SpecificCounters_({"filter_type"})
        , Rps_(rps)
        , PriceFilters_(MakeAtomicShared<TVector<NTravelProto::NPriceChecker::TPriceFilterRecord>>())
        , NewPriceFilters_(MakeAtomicShared<TVector<NTravelProto::NPriceChecker::TPriceFilterRecord>>())
    {
        InitPriceFilterTable();
        InitMappers();

        YtConfigPartners_.SetOnUpdateHandler([this](bool first) {
            auto partners = YtConfigPartners_.GetAll();
            THashMap<TString, EPartnerId> partnerIdByCode;
            for (const auto& it : *partners) {
                partnerIdByCode[it.second.GetCode()] = it.first;
            }
            PermalinkToOriginalIdsMapper_.SetPartnerIdByCode(partnerIdByCode);
            if (first) {
                PermalinkToOriginalIdsMapper_.Start();
            }
        });
    }

    void TPriceFilterChecker::RegisterCounters(NMonitor::TCounterSource& source, const TString& name) {
        YtConfigPartners_.RegisterCounters(source);
        PriceFilterTable_.RegisterCounters(source, name + "PriceFilterTable");
        PermalinkToOriginalIdsMapper_.RegisterCounters(source);
        PermalinkToClusterMapper_.RegisterCounters(source);
        source.RegisterSource(&CommonCounters_, name + "Common");
        source.RegisterSource(&SpecificCounters_, name + "Specific");
    }

    void TPriceFilterChecker::Start() {
        YtConfigPartners_.Start();
        PriceFilterTable_.Start();
        PermalinkToClusterMapper_.Start();
        SystemThreadFactory()->Run([this]() {
            CheckPriceFilters();
        });
    }

    void TPriceFilterChecker::Stop() {
        PermalinkToClusterMapper_.Stop();
        PermalinkToOriginalIdsMapper_.Stop();
        PriceFilterTable_.Stop();
        YtConfigPartners_.Stop();
        IsStopped_.Set();
    }

    void TPriceFilterChecker::CheckPriceFilters() {
        while (!IsStopped_) {
            Sleep(TDuration::Seconds(2));

            if (Epoch_.load() == 0) {
                INFO_LOG << "Skipping price filters check because no price filters ready" << Endl;
                continue;
            }
            if (!PermalinkToClusterMapper_.IsReady() || !PermalinkToOriginalIdsMapper_.IsReady()) {
                INFO_LOG << "Skipping price filters check because permalink mappers are not ready" << Endl;
                continue;
            }

            INFO_LOG << "Starting price filters check" << Endl;
            TAtomicSharedPtr<TVector<NTravelProto::NPriceChecker::TPriceFilterRecord>> priceFilters;
            int currEpoch;
            with_lock (PriceFiltersMutex_) {
                priceFilters = PriceFilters_;
                currEpoch = Epoch_.load();
            }

            auto interval = TDuration::MilliSeconds(1000 / Rps_);
            TInstant nextAction = Now();
            for (size_t i = 0; i < priceFilters->size() && currEpoch == Epoch_.load() && !IsStopped_; i++) {
                auto now = Now();
                if (nextAction > now) {
                    Sleep(nextAction - now);
                }
                CheckPriceFilter(currEpoch, i, priceFilters->at(i));
                nextAction = Max(Now(), nextAction + interval);
            }
        }
    }

    void TPriceFilterChecker::CheckPriceFilter(int epoch, int priceFilterInd, NTravelProto::NPriceChecker::TPriceFilterRecord priceFilter) {
        NTravel::TScheduler::Instance().Enqueue(Now(), [this, epoch, priceFilterInd, priceFilter]() {
            auto requestId = "PCF-" + ToString(epoch) + "-" + CreateGuidAsString();

            DEBUG_LOG << "Preparing request (requestId: " << requestId
                      << " Permalink: " << priceFilter.GetPermalink()
                      << " CheckIn: " << priceFilter.GetCheckIn()
                      << " CheckOut: " << priceFilter.GetCheckOut()
                      << Endl;

            auto clusterPermalink = PermalinkToClusterMapper_.GetClusterPermalink(priceFilter.GetPermalink());
            auto originalIds = PermalinkToOriginalIdsMapper_.GetMapping(clusterPermalink);
            if (!originalIds) {
                WARNING_LOG << "No mapping for permalink: " << clusterPermalink << Endl;
                return;
            }

            NTravelProto::TSearchOffersRpcReq rpcReq;
            rpcReq.SetSync(false);
            bool withDates = false;
            int subReqIndex = 0;
            for (const auto& hotelId : originalIds->PartnerIds) {
                auto subReq = rpcReq.AddSubrequest();
                subReq->SetPermalink(priceFilter.GetPermalink());
                subReq->MutableHotelId()->SetPartnerId(hotelId.PartnerId);
                subReq->MutableHotelId()->SetOriginalId(hotelId.OriginalId);
                if (priceFilter.HasCheckIn() && priceFilter.HasCheckOut()) {
                    if (NOrdinalDate::FromString(priceFilter.GetCheckOut()) - NOrdinalDate::FromString(priceFilter.GetCheckIn()) != 1) {
                        WARNING_LOG << "Found price filter for several nights" << Endl;
                        return;
                    }
                    subReq->SetCheckInDate(NOrdinalDate::ToString(NOrdinalDate::FromString(priceFilter.GetCheckIn())));
                    subReq->SetCheckOutDate(NOrdinalDate::ToString(NOrdinalDate::FromString(priceFilter.GetCheckOut())));
                    withDates = true;
                } else {
                    auto today = NOrdinalDate::FromInstant(Now());
                    subReq->SetCheckInDate(NOrdinalDate::ToString(today + 1));
                    subReq->SetCheckOutDate(NOrdinalDate::ToString(today + 2));
                }
                subReq->SetOccupancy("2");
                subReq->SetCurrency(NTravelProto::ECurrency::C_RUB);

                subReq->SetOfferCacheIgnoreBlacklist(false);
                subReq->SetId(requestId + "~" + ToString(subReqIndex));
                subReq->SetRequestClass(NTravelProto::ERequestClass::RC_BACKGROUND);
                subReq->MutableAttribution()->SetOfferCacheClientId(PriceFilterCheckerOfferCacheClientId);
                subReq->SetOfferCacheUseCache(false);
                subReq->SetOfferCacheUseSearcher(true);
                subReqIndex++;
            }

            DEBUG_LOG << "Sending request (requestId: " << requestId
                      << " Permalink: " << priceFilter.GetPermalink()
                      << " CheckIn: " << priceFilter.GetCheckIn()
                      << " CheckOut: " << priceFilter.GetCheckOut()
                      << Endl;
            with_lock (RunningChecksMutex_) {
                RunningChecks_[requestId] = RunningCheck(withDates, priceFilterInd, originalIds->PartnerIds.size());
            }
            NTravel::TScheduler::Instance().Enqueue(Now() + MaxCheckWaitTime_, [this, requestId]() {
                with_lock (RunningChecksMutex_) {
                    RunningChecks_.erase(requestId);
                }
            });
            CommonCounters_.NSentRpcRequests.Inc();
            OfferCacheClient_.Request<NTravelProto::TSearchOffersRpcReq,
                                      NTravelProto::TSearchOffersRpcRsp,
                                      &NTravelProto::NOfferCacheGrpc::OfferCacheServiceV1::Stub::AsyncSearchOffers>(
                rpcReq,
                NGrpc::TClientMetadata(),
                [this, requestId](const TString& grpcError, const TString& /*remoteFQDN*/, const NTravelProto::TSearchOffersRpcRsp& resp) {
                    if (!grpcError.empty()) {
                        ERROR_LOG << "RPC error (requestId: " << requestId << "): " << grpcError << Endl;
                        CommonCounters_.NRpcErrors.Inc();
                        return;
                    }

                    for (const auto& subResp : resp.GetSubresponse()) {
                        if (subResp.HasError()) {
                            ERROR_LOG << "RPC error (requestId: " << requestId << "): " << subResp.GetError().GetMessage() << Endl;
                            CommonCounters_.NRpcErrors.Inc();
                            return;
                        }
                    }
                });
        });
    }

    bool TPriceFilterChecker::ProcessSearcherMessage(const TYtQueueMessage& busMessage, const ru::yandex::travel::hotels::TSearcherMessage& parsedMessage) {
        if (busMessage.Timestamp <= Start_) {
            return false;
        }

        const auto& request = parsedMessage.GetRequest();
        const auto& response = parsedMessage.GetResponse();

        if (request.GetAttribution().GetOfferCacheClientId() != PriceFilterCheckerOfferCacheClientId) {
            return false;
        }

        DEBUG_LOG << "Got message with " << PriceFilterCheckerOfferCacheClientId << " client id. RequestId: " << request.GetId() << Endl;
        HandleCheckResult(request, response);
        return true;
    }

    void TPriceFilterChecker::HandleCheckResult(const NTravelProto::TSearchOffersReq& request, const NTravelProto::TSearchOffersRsp& response) {
        if (response.HasPlaceholder()) {
            return;
        }

        const auto& requestId = request.GetId();
        if (requestId.substr(0, 4) != "PCF-") {
            ERROR_LOG << "Request id doesn't start with PCF- (RequestId: " << requestId << ")" << Endl;
            return;
        }
        auto requestIdWithNoSuffix = SplitString(requestId, "~")[0];
        auto parts = SplitString(requestIdWithNoSuffix, "-");
        auto epoch = FromString<int>(parts[1]);

        if (response.HasError()) {
            const auto& error = response.GetError();
            ERROR_LOG << "Bus response error: " << error.GetMessage() << " RequestId: " << request.GetId() << Endl;
            CommonCounters_.NBusResponseErrors.Inc();
        }

        with_lock (PriceFiltersMutex_) {
            if (epoch != Epoch_.load()) {
                ERROR_LOG << "Skipping response for old epoch" << Endl;
                return;
            }
        }

        RunningCheck runningCheck;
        if (response.HasOffers()) {
            with_lock (RunningChecksMutex_) {
                if (!RunningChecks_.contains(requestIdWithNoSuffix)) {
                    ERROR_LOG << "Running check not found (RequestId: " << requestId << ")" << Endl;
                    return;
                }
                RunningChecks_[requestIdWithNoSuffix].Responses.push_back(response);
                runningCheck = RunningChecks_[requestIdWithNoSuffix];
            }
        }

        NTravelProto::NPriceChecker::TPriceFilterRecord priceFilter;
        with_lock (PriceFiltersMutex_) {
            priceFilter = PriceFilters_->at(runningCheck.PriceFilterInd);
        }

        DEBUG_LOG << "RequestId: " << requestId
                  << " got responses: " << runningCheck.Responses.size() << "/" << runningCheck.TotalRequests
                  << Endl;

        if (runningCheck.Responses.size() == runningCheck.TotalRequests) {
            uint minPrice = std::numeric_limits<uint>::max();
            uint maxPrice = std::numeric_limits<uint>::min();
            bool hasOffers = false;

            for (const auto& rps : runningCheck.Responses) {
                const auto& offersPb = rps.GetOffers().GetOffer();
                TVector<NTravelProto::TOffer> offers(offersPb.begin(), offersPb.end());
                for (const auto& offer : offers) {
                    auto price = static_cast<uint>(offer.GetPrice().GetAmount());
                    minPrice = Min(minPrice, price);
                    maxPrice = Max(minPrice, price);
                    hasOffers = true;
                }
            }

            auto counters = SpecificCounters_.GetOrCreate({runningCheck.WithDates ? "WithDates" : "WithoutDates"});
            TString borderResult;
            TString overlapResult;
            if (!hasOffers) {
                counters->NNoOffers.Inc();
                borderResult = "NoOffers";
                overlapResult = "NoOffers";
            } else {
                if (priceFilter.GetMinPricePerNight() <= minPrice && maxPrice <= priceFilter.GetMaxPricePerNight()) {
                    counters->NBordersBothOk.Inc();
                    borderResult = "Ok";
                } else if (priceFilter.GetMinPricePerNight() <= minPrice) {
                    counters->NBordersWrongMax.Inc();
                    borderResult = "WrongMax";
                } else if (maxPrice <= priceFilter.GetMaxPricePerNight()) {
                    counters->NBordersWrongMin.Inc();
                    borderResult = "WrongMin";
                } else {
                    counters->NBordersBothWrong.Inc();
                    borderResult = "BothWrong";
                }

                if (priceFilter.GetMinPricePerNight() <= minPrice && maxPrice <= priceFilter.GetMaxPricePerNight()) {
                    counters->NOverlapRealRangeInside.Inc();
                    overlapResult = "Inside";
                } else if (Max(priceFilter.GetMinPricePerNight(), minPrice) <= Min(priceFilter.GetMaxPricePerNight(), maxPrice)) {
                    counters->NOverlapHasIntersection.Inc();
                    overlapResult = "HasIntersection";
                } else {
                    counters->NOverlapNoIntersection.Inc();
                    overlapResult = "NoIntersection";
                }
            }

            INFO_LOG << "Permalink: " << priceFilter.GetPermalink()
                     << " CheckIn: " << priceFilter.GetCheckIn()
                     << " CheckOut: " << priceFilter.GetCheckOut()
                     << " Border result: " << borderResult
                     << " Overlap result: " << overlapResult
                     << Endl;
        }
    }

    void TPriceFilterChecker::InitPriceFilterTable() {
        auto conv = [](const NYT::TNode& node, NTravelProto::NPriceChecker::TPriceFilterRecord* proto) {
            proto->SetPermalink(node["permalink"].IntCast<ui64>());
            proto->SetMinPricePerNight(node["min_price_per_night"].IntCast<ui32>());
            proto->SetMaxPricePerNight(node["max_price_per_night"].IntCast<ui32>());
            if (!node["checkin"].IsNull()) {
                proto->SetCheckIn(node["checkin"].AsString());
            }
            if (!node["checkout"].IsNull()) {
                proto->SetCheckOut(node["checkout"].AsString());
            }
        };
        auto data = [this](const NTravelProto::NPriceChecker::TPriceFilterRecord& proto) {
            NewPriceFilters_->push_back(proto);
        };
        auto finish = [this](bool ok, bool /*initial*/) {
            if (ok) {
                std::shuffle(NewPriceFilters_->begin(), NewPriceFilters_->end(), TFastRng<ui32>(Seed()));
                with_lock (PriceFiltersMutex_) {
                    swap(NewPriceFilters_, PriceFilters_);
                    Epoch_.store(Now().Seconds());
                }
            }
            NewPriceFilters_ = MakeAtomicShared<TVector<NTravelProto::NPriceChecker::TPriceFilterRecord>>();
        };
        PriceFilterTable_.SetCallbacks(conv, data, finish);
    }

    void TPriceFilterChecker::InitMappers() {
        auto permalinkMapper = [this](NPermalinkMappers::TPermalinkToOriginalIdsMapper::TKey* key) {
            *key = PermalinkToClusterMapper_.GetClusterPermalink(*key);
        };
        PermalinkToOriginalIdsMapper_.SetMappingFilters(permalinkMapper, NPermalinkMappers::TPermalinkToOriginalIdsMappingRecJoiner::Join);

        PermalinkToClusterMapper_.SetOnFinishHandler([this]() {
            INFO_LOG << "Started reloading of mappings" << Endl;
            PermalinkToOriginalIdsMapper_.Reload();
        });
    }

    void TPriceFilterChecker::TSpecificCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        ct->insert(MAKE_COUNTER_PAIR(NNoOffers));

        ct->insert(MAKE_COUNTER_PAIR(NBordersBothOk));
        ct->insert(MAKE_COUNTER_PAIR(NBordersWrongMin));
        ct->insert(MAKE_COUNTER_PAIR(NBordersWrongMax));
        ct->insert(MAKE_COUNTER_PAIR(NBordersBothWrong));

        ct->insert(MAKE_COUNTER_PAIR(NOverlapRealRangeInside));
        ct->insert(MAKE_COUNTER_PAIR(NOverlapHasIntersection));
        ct->insert(MAKE_COUNTER_PAIR(NOverlapNoIntersection));
    }

    void TPriceFilterChecker::TCommonCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        ct->insert(MAKE_COUNTER_PAIR(NSentRpcRequests));
        ct->insert(MAKE_COUNTER_PAIR(NRpcErrors));
        ct->insert(MAKE_COUNTER_PAIR(NBusResponseErrors));
    }
}
