#include "tracking_results_consumer.h"

namespace NTravel::NPriceChecker {
    TTrackingResultsConsumer::TTrackingResultsConsumer(NTravelProto::NAppConfig::TConfigLogger config)
        : Counters_({"operator"})
        , PriceCheckerLogger_("TrackingResultsLogger", std::move(config))
    {
        auto descriptor = NTravelProto::EOperatorId_descriptor();
        for (int index = 0; index < descriptor->value_count(); ++index) {
            Counters_.GetOrCreate({descriptor->value(index)->name()});
        }
    }

    void TTrackingResultsConsumer::Start() {
        PriceCheckerLogger_.Start();
    }

    void TTrackingResultsConsumer::Stop() {
        PriceCheckerLogger_.Stop();
    }

    void TTrackingResultsConsumer::HandleTrackingResult(const TOfferTrackingState& trackingState) {
        auto& trackingResult = trackingState.OfferTrackingResult;

        auto logRecord = BuildLogRecord(trackingResult);
        auto operatorId = trackingResult.InitialOffer.GetOperatorId();

        std::visit(TResultVisitor(&logRecord, GetCountersForOperator(operatorId)), trackingResult.Result);
        PriceCheckerLogger_.AddRecord(logRecord);
    }

    void TTrackingResultsConsumer::RegisterCounters(NMonitor::TCounterSource& source, const TString& name) {
        source.RegisterSource(&Counters_, name);
        PriceCheckerLogger_.RegisterCounters(source);
    }

    void TTrackingResultsConsumer::ReopenLog() {
        PriceCheckerLogger_.Reopen();
    }

    TTrackingResultsConsumer::TPriceCheckerLogRecord TTrackingResultsConsumer::BuildLogRecord(const TOfferTrackingResult& trackingResult) const {
        const auto now = Now();
        TPriceCheckerLogRecord logRecord;
        logRecord.SetTimestamp(now.Seconds());
        *logRecord.MutableRequest() = trackingResult.InitialRequest;
        *logRecord.MutableOriginalOffer() = trackingResult.InitialOffer;

        auto mutableCheckResults = logRecord.MutableCheckResults();
        for (auto& checkResult : trackingResult.CheckResults) {
            auto singleCheckResultLogRecord = mutableCheckResults->Add();
            singleCheckResultLogRecord->SetTimestamp(checkResult.Timestamp.Seconds());
            if (checkResult.MatchedOffer.Defined()) {
                *singleCheckResultLogRecord->MutableMatchedOffer() = *checkResult.MatchedOffer.Get();
            }
        }

        return logRecord;
    }

    TAtomicSharedPtr<TTrackingResultsConsumer::TCounters> TTrackingResultsConsumer::GetCountersForOperator(NTravelProto::EOperatorId operatorId) {
        const auto& name = NTravelProto::EOperatorId_Name(operatorId);
        return Counters_.GetOrCreate({name.empty() ? "_OTHER_" : name});
    }

    TTrackingResultsConsumer::TCounters::TCounters()
        : NPricePercentSmaller(HistogramBuckets, "Inf")
        , NPricePercentBigger(HistogramBuckets, "Inf")
    {
    }

    void TTrackingResultsConsumer::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
        ct->insert(MAKE_COUNTER_PAIR(NError));
        ct->insert(MAKE_COUNTER_PAIR(NPriceDiffer));
        ct->insert(MAKE_COUNTER_PAIR(NPriceNotFound));
        ct->insert(MAKE_COUNTER_PAIR(NNoChanges));
        ct->insert(MAKE_COUNTER_PAIR(NError));

        ct->insert(MAKE_COUNTER_PAIR(NPriceIsSmaller));
        ct->insert(MAKE_COUNTER_PAIR(NPriceIsBigger));
        ct->insert(MAKE_COUNTER_PAIR(NPriceIsApproxSame));
        ct->insert(MAKE_COUNTER_PAIR(NPriceIsApproxSmaller));
        ct->insert(MAKE_COUNTER_PAIR(NPriceIsApproxBigger));

        NPricePercentSmaller.QueryCounters("NPriceIsUpTo", "PercentSmaller", ct);
        NPricePercentBigger.QueryCounters("NPriceIsUpTo", "PercentBigger", ct);
    }

    TTrackingResultsConsumer::TResultVisitor::TResultVisitor(TTrackingResultsConsumer::TPriceCheckerLogRecord* logRecord, TAtomicSharedPtr<TCounters> counters)
        : LogRecord(logRecord)
        , Counters(counters)
    {
    }

    void TTrackingResultsConsumer::TResultVisitor::operator()(const TUnknownTrackingResult&) const {
        LogRecord->SetTrackingResult(TPriceCheckerLogRecord::TTrackingResult::TPriceCheckerLogRecord_TTrackingResult_CR_UNKNOWN);
        Counters->NUnknown.Inc();
    }

    void TTrackingResultsConsumer::TResultVisitor::operator()(const TPriceDifferTrackingResult& result) const {
        auto priceDifference = result.NewPrice - result.OldPrice;
        auto relativePriceDifference = static_cast<double>(priceDifference) / result.OldPrice;

        if (priceDifference > 0) {
            Counters->NPriceIsBigger.Inc();
            Counters->NPricePercentBigger.Update(static_cast<int>(ceil(100 * relativePriceDifference)));
        } else {
            Counters->NPriceIsSmaller.Inc();
            Counters->NPricePercentSmaller.Update(static_cast<int>(floor(100 * relativePriceDifference)));
        }

        if (abs(relativePriceDifference) <= PriceRelTolerance) {
            Counters->NPriceIsApproxSame.Inc();
        } else if (relativePriceDifference > PriceRelTolerance) {
            Counters->NPriceIsApproxBigger.Inc();
        } else {
            Counters->NPriceIsApproxSmaller.Inc();
        }

        Counters->NPriceDiffer.Inc();
        LogRecord->SetTrackingResult(TPriceCheckerLogRecord::TTrackingResult::TPriceCheckerLogRecord_TTrackingResult_CR_PRICE_DIFFER);
        LogRecord->SetLifetimeLowerBound(result.LifetimeMin.MilliSeconds());
        LogRecord->SetLifetimeUpperBound(result.LifetimeMax.MilliSeconds());
        LogRecord->SetPriceDifference(priceDifference);
        LogRecord->SetRelativePriceDifference(relativePriceDifference);
    }

    void TTrackingResultsConsumer::TResultVisitor::operator()(const TPriceNotFoundTrackingResult& result) const {
        Counters->NPriceNotFound.Inc();
        LogRecord->SetTrackingResult(TPriceCheckerLogRecord::TTrackingResult::TPriceCheckerLogRecord_TTrackingResult_CR_NOT_FOUND);
        LogRecord->SetLifetimeLowerBound(result.LifetimeMin.MilliSeconds());
        LogRecord->SetLifetimeUpperBound(result.LifetimeMax.MilliSeconds());
    }

    void TTrackingResultsConsumer::TResultVisitor::operator()(const TNoChangesTrackingResult&) const {
        Counters->NNoChanges.Inc();
        LogRecord->SetTrackingResult(TPriceCheckerLogRecord::TTrackingResult::TPriceCheckerLogRecord_TTrackingResult_CR_NO_CHANGES);
    }

    void TTrackingResultsConsumer::TResultVisitor::operator()(const TErrorTrackingResult& result) const {
        Counters->NError.Inc();
        LogRecord->SetTrackingResult(TPriceCheckerLogRecord::TTrackingResult::TPriceCheckerLogRecord_TTrackingResult_CR_ERROR);
        LogRecord->SetErrorMessage(result.Error);
    }

}
