#include <library/cpp/logger/global/global.h>
#include <util/generic/ymath.h>

#include "tracking_strategy.h"

namespace NTravel::NPriceChecker {
    TTrackingStrategy::TTrackingStrategy(const NTravelProto::NPriceChecker::TConfig_TTrackingStrategy& trackingStrategyConfig, TOfferTrackerCounters* counters)
        : CheckingTimes_(trackingStrategyConfig.GetCheckingTimeMin().begin(), trackingStrategyConfig.GetCheckingTimeMin().end())
        , Counters_(counters)
    {
    }

    void TTrackingStrategy::RegisterCounters(NMonitor::TCounterSource&, const TString&) {
    }

    bool TTrackingStrategy::GetNextCheckInfoOrError(TOfferTrackingState* state, TInstant now, TInstant* nextCheckTime, size_t* nextCheckId) const {
        if (!state->IsActive()) {
            ERROR_LOG << "Tried to get next check time for not active state. Key: " << state->Key << Endl;
            return false;
        }

        auto& offerTrackingResult = state->OfferTrackingResult;

        if (!TryGetNextCheckInfo(state, nextCheckTime, nextCheckId)) {
            offerTrackingResult.Result = TErrorTrackingResult("Finished check");
            state->ToDone();
            WARNING_LOG << "Tried to get next check time for finished check. Key: " << state->Key << Endl;
            return false;
        }

        if (*nextCheckTime < now - MaxCheckDelay_) {
            offerTrackingResult.Result = TErrorTrackingResult("Outdated check");
            state->ToDone();
            Counters_->NOutdatedChecks.Inc();
            WARNING_LOG << "Tried to get next check time for outdated check. Key: " << state->Key << Endl;
            return false;
        }

        return true;
    }

    void TTrackingStrategy::HandleCheckResult(TOfferTrackingState* state,
                                              TInstant timestamp,
                                              size_t checkId,
                                              const TVector<NTravelProto::TOffer>& offers) {
        if (checkId < state->OfferTrackingResult.CheckResults.size()) {
            Counters_->NCorruptedTrackings.Inc();
            ERROR_LOG << "Found response to finished checking. Key: " << state->Key << Endl;
            state->OfferTrackingResult.Result = TErrorTrackingResult("Tracking is corrupted");
            state->ToDone();
            return;
        }

        if (checkId > state->OfferTrackingResult.CheckResults.size()) {
            Counters_->NCorruptedTrackings.Inc();
            ERROR_LOG << "Found response to not started checking. Key: " << state->Key << Endl;
            state->OfferTrackingResult.Result = TErrorTrackingResult("Tracking is corrupted");
            state->ToDone();
            return;
        }

        state->LastSuccessiveErrors.clear();
        auto& checkResult = state->OfferTrackingResult.CheckResults.emplace_back();
        checkResult.Timestamp = timestamp;

        const NTravelProto::TOffer* bestMatchedOffer = nullptr;

        const auto& oldOffer = state->OfferTrackingResult.InitialOffer;
        const int oldOfferPrice = GetOfferPrice(oldOffer);

        for (const auto& newOffer : offers) {
            if (AreOffersEquivalent(oldOffer, newOffer)) {
                if (!bestMatchedOffer || abs(GetOfferPrice(*bestMatchedOffer) - oldOfferPrice) > abs(GetOfferPrice(newOffer) - oldOfferPrice)) {
                    bestMatchedOffer = &newOffer;
                }
            }
        }

        if (bestMatchedOffer) {
            checkResult.MatchedOffer = *bestMatchedOffer;
        }

        if (!bestMatchedOffer || !ArePricesEquivalent(oldOfferPrice, GetOfferPrice(*bestMatchedOffer))) {
            auto lifetimeMin = checkId == 0
                                   ? TDuration::Seconds(0)
                                   : state->OfferTrackingResult.CheckResults[checkId - 1].Timestamp - state->OfferTrackingResult.InitialTimestamp;
            auto lifetimeMax = timestamp - state->OfferTrackingResult.InitialTimestamp;

            if (bestMatchedOffer) {
                state->OfferTrackingResult.Result = TPriceDifferTrackingResult(lifetimeMin, lifetimeMax, oldOfferPrice, GetOfferPrice(*bestMatchedOffer));
            } else {
                state->OfferTrackingResult.Result = TPriceNotFoundTrackingResult(lifetimeMin, lifetimeMax, oldOfferPrice);
            }
            state->ToDone();
        } else if (state->OfferTrackingResult.CheckResults.size() >= CheckingTimes_.size()) {
            state->OfferTrackingResult.Result = TNoChangesTrackingResult();
            state->ToDone();
        }
    }

    void TTrackingStrategy::HandleCheckError(TOfferTrackingState* state, TInstant timestamp, const TString& error) {
        HandleError(state, timestamp, error);
    }

    void TTrackingStrategy::HandleSearcherTimeout(TOfferTrackingState* state, TInstant timestamp) {
        HandleError(state, timestamp, "Searcher timeout on pricechecker side");
    }

    TDuration TTrackingStrategy::GetLifetime(TOfferTrackingState* state, TInstant now) {
        auto delay = TDuration::Minutes(30);
        TInstant nextCheckTime;
        size_t nextCheckId;
        if (!TryGetNextCheckInfo(state, &nextCheckTime, &nextCheckId)) {
            return delay;
        }
        return (nextCheckTime - now) + delay;
    }

    bool TTrackingStrategy::TryGetNextCheckInfo(TOfferTrackingState* state, TInstant* nextCheckTime, size_t* nextCheckId) const {
        auto& offerTrackingResult = state->OfferTrackingResult;

        if (offerTrackingResult.CheckResults.size() >= CheckingTimes_.size()) {
            return false;
        }

        *nextCheckTime = offerTrackingResult.InitialTimestamp + TDuration::Minutes(CheckingTimes_[offerTrackingResult.CheckResults.size()]);
        *nextCheckId = offerTrackingResult.CheckResults.size();

        if (!state->LastSuccessiveErrors.empty()) {
            *nextCheckTime = state->LastSuccessiveErrors.rbegin()->Timestamp + RetryDelay_;
        }

        return true;
    }

    void TTrackingStrategy::HandleError(TOfferTrackingState* state, TInstant timestamp, const TString& error) {
        state->LastSuccessiveErrors.emplace_back(error, timestamp);
        if (state->LastSuccessiveErrors.size() >= 2) {
            state->OfferTrackingResult.Result = TErrorTrackingResult(error);
            state->ToDone();
        }
    }

    bool TTrackingStrategy::AreOffersEquivalent(const NTravelProto::TOffer& firstOffer, const NTravelProto::TOffer& secondOffer) {
        auto result = firstOffer.GetOperatorId() == secondOffer.GetOperatorId() &&
                      firstOffer.GetPrice().GetCurrency() == secondOffer.GetPrice().GetCurrency() &&
                      firstOffer.GetCapacity() == secondOffer.GetCapacity() &&
                      firstOffer.GetDisplayedTitle().value() == secondOffer.GetDisplayedTitle().value() &&
                      firstOffer.GetPansion() == secondOffer.GetPansion() &&
                      firstOffer.HasFreeCancellation() == secondOffer.HasFreeCancellation() &&
                      firstOffer.GetFreeCancellation().value() == secondOffer.GetFreeCancellation().value();
        return result;
    }

    int TTrackingStrategy::GetOfferPrice(const NTravelProto::TOffer& offer) {
        return offer.GetPrice().GetAmount();
    }

    bool TTrackingStrategy::ArePricesEquivalent(double oldPrice, double newPrice) {
        double relativePriceDifference = (newPrice - oldPrice) / oldPrice;
        return Abs(relativePriceDifference) < 0.01;
    }
}
