#include "cache.h"

#include <travel/hotels/lib/cpp/scheduler/scheduler.h>

#include <util/generic/vector.h>
#include <util/random/random.h>

namespace NTravel {
namespace NPriceChecker {

void TCache::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NCachedOffers));
    ct->insert(MAKE_COUNTER_PAIR(NCachedRequests));
}

TCache::TCache(const NTravelProto::NPriceChecker::TConfig::TCache& config)
    : AdditionalCacheTime_(TDuration::Seconds(config.GetAdditionalCacheTimeSec()))
{
}

void TCache::RegisterCounters(NMonitor::TCounterSource& source) {
    source.RegisterSource(&Counters_, "Cache");
}

void TCache::AddOffers(TInstant expireTimestamp, const TSearcherResponse& searcherResponse) {
    if (searcherResponse.Type != ESearcherResponseType::OFFER) {
        return;
    }

    auto deadline = expireTimestamp + AdditionalCacheTime_;
    if (deadline < Now()) {
        return;
    }

    TVector<TOfferId> offerIds;
    for (const auto& offer : searcherResponse.OffersWithRequests) {
        auto& bucket = Offers_.GetBucketForKey(offer->OfferId);
        TWriteGuard g(bucket.GetMutex());
        if (bucket.GetMap().insert(std::make_pair(offer->OfferId, offer)).second) {
            offerIds.push_back(offer->OfferId);
        }
    }
    if (!offerIds) {
        return;
    }
    Counters_.NCachedOffers += offerIds.size();
    Counters_.NCachedRequests.Inc();

    NTravel::TScheduler::Instance().Enqueue(deadline, [this, offerIds] () {
        for (const auto& offerId: offerIds) {
            auto& bucket = Offers_.GetBucketForKey(offerId);
            TWriteGuard g(bucket.GetMutex());
            bucket.GetMap().erase(offerId);
        }
        Counters_.NCachedOffers -= offerIds.size();
        Counters_.NCachedRequests.Dec();
    });
}

TOfferWithRequestRef TCache::GetOfferWithRequest(const TOfferId& offerId) const {
    const auto& bucket = Offers_.GetBucketForKey(offerId);
    TReadGuard g(bucket.GetMutex());
    auto it = bucket.GetMap().find(offerId);
    if (it != bucket.GetMap().end()) {
        return it->second;
    }
    return {};
}

} // namespace NPriceChecker
} // namespace NTravel
