#include "job_processor.h"

#include <travel/hotels/lib/cpp/scheduler/scheduler.h>

#include <util/generic/guid.h>

namespace NTravel {
namespace NPriceChecker {

const TString EXTERNAL_TYPE_NAME = "redir";
const TString MISSING_OPERATOR_NAME = "_OTHER_";

const std::initializer_list<int> HISTOGRAM_BUCKETS = {1, 5, 20, 50};
const double PRICE_REL_TOLERANCE = 0.01;

const TString TJobProcessor::OfferCacheClientId = "price-checker";

void TJobProcessor::TGenericCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NAncientCacheMisses));
    ct->insert(MAKE_COUNTER_PAIR(NOldCacheMisses));
    ct->insert(MAKE_COUNTER_PAIR(NFreshCacheMisses));
}

TJobProcessor::TSpecificCounters::TSpecificCounters()
    : NPricePercentSmaller(HISTOGRAM_BUCKETS, "Inf")
    , NPricePercentBigger(HISTOGRAM_BUCKETS, "Inf")
{
}

void TJobProcessor::TSpecificCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(NRunningChecks));

    ct->insert(MAKE_COUNTER_PAIR(NCompletedChecks));
    ct->insert(MAKE_COUNTER_PAIR(NExpiredChecks));
    ct->insert(MAKE_COUNTER_PAIR(NBusResponseErrors));
    ct->insert(MAKE_COUNTER_PAIR(NRpcErrors));

    ct->insert(MAKE_COUNTER_PAIR(NSentRpcRequests));

    ct->insert(MAKE_COUNTER_PAIR(NPriceNotFound));
    ct->insert(MAKE_COUNTER_PAIR(NPriceIsSame));
    ct->insert(MAKE_COUNTER_PAIR(NPriceIsSmaller));
    ct->insert(MAKE_COUNTER_PAIR(NPriceIsBigger));
    ct->insert(MAKE_COUNTER_PAIR(NPriceIsApproxSame));
    ct->insert(MAKE_COUNTER_PAIR(NPriceIsApproxSmaller));
    ct->insert(MAKE_COUNTER_PAIR(NPriceIsApproxBigger));

    NPricePercentSmaller.QueryCounters("NPriceIsUpTo", "PercentSmaller", ct);
    NPricePercentBigger.QueryCounters("NPriceIsUpTo", "PercentBigger", ct);
}

TJobProcessor::TJob::TJob(const TOfferId& offerId, const TString& typeName, TInstant receiptTime)
    : OfferId(offerId)
    , TypeName(typeName)
    , ReceiptTime(receiptTime)
    , LogPrefix("| offerId = " + OfferIdToString(OfferId) + ", type = " + TypeName + " | ")
{
}

TJobProcessor::TLogRecord TJobProcessor::TJob::GenerateLogRecord(TLogRecord::TCheckResult checkResult,
                                                                 const TString& errorMessage)
{
    const auto now = Now();
    TLogRecord record;
    record.SetTimestamp(now.Seconds());
    if (StartTime) {
        record.SetAwaitTime((StartTime - ReceiptTime).MilliSeconds());
        record.SetLifetime((now - StartTime).MilliSeconds());
    } else {
        record.SetAwaitTime((now - ReceiptTime).MilliSeconds());
    }
    record.SetJobType(TypeName);
    record.SetCheckResult(checkResult);
    if (!errorMessage.empty()) {
        record.SetErrorMessage(errorMessage);
    }
    if (OfferWithRequest) {
        *record.MutableRequest() = *OfferWithRequest->RequestPb;
        record.MutableOriginalOffer()->SetId(OfferIdToString(OfferWithRequest->OfferId));
        record.MutableOriginalOffer()->SetOperatorId(OfferWithRequest->OperatorId);
        record.MutableOriginalOffer()->MutablePrice()->SetAmount(OfferWithRequest->OfferPrice);

    }
    return record;
}

TJobProcessor::TJobProcessor(const NTravelProto::NPriceChecker::TConfig::TJobProcessor& config, const TCache& cache,
                             TCommonLogger<TLogRecord>& reqAnsLogger)
    : GenerationSwapPeriod_(TDuration::Seconds(config.GetGenerationSwapPeriodSec()))
    , AncientCacheMissLag_(TDuration::Seconds(config.GetAncientCacheMissLagSec()))
    , OldCacheMissLag_(TDuration::Seconds(config.GetOldCacheMissLagSec()))
    , FreshCacheMissRetryDelay_(TDuration::Seconds(config.GetFreshCacheMissRetryDelaySec()))
    , FreshCacheMissRetryCount_(config.GetFreshCacheMissRetryCount())
    , InternalJobRequestClass_(config.GetInternalJobRequestClass())
    , ExternalJobRequestClass_(config.GetExternalJobRequestClass())
    , Cache_(cache)
    , ReqAnsLogger_(reqAnsLogger)
    , OfferCacheClient_(config.GetOfferCache())
    , SpecificCounters_({"type", "operator"})
{
    if (AncientCacheMissLag_ < OldCacheMissLag_) {
        ythrow yexception() << "Make sure that AncientCacheMissLagSec >= OldCacheMissLagSec";
    }

    auto descriptor = NTravelProto::EOperatorId_descriptor();
    for (int index = 0; index < descriptor->value_count(); ++index) {
        auto valueDescriptor = descriptor->value(index);
        TString operatorName = valueDescriptor->name();
        if (operatorName.StartsWith("OI_")) {
            operatorName = operatorName.substr(3);
            operatorName.to_lower(0, operatorName.size());
        } else {
            ythrow yexception() << "Make sure that EOperatorId elements are named as OI_*";
        }
        OperatorIdToName_[valueDescriptor->number()] = operatorName;
    }

    InitializeCountersForJobType(EXTERNAL_TYPE_NAME);
}

void TJobProcessor::RegisterCounters(NMonitor::TCounterSource& source) {
    source.RegisterSource(&GenericCounters_, "JobProcessorGeneric");
    source.RegisterSource(&SpecificCounters_, "JobProcessorSpecific");
}

void TJobProcessor::Start() {
    NTravel::TScheduler::Instance().EnqueuePeriodical(GenerationSwapPeriod_, [this]() { SwapGenerations(); });
}

void TJobProcessor::Stop() {
    OfferCacheClient_.Shutdown();
}

TString TJobProcessor::ConvertOperatorIdToName(int operatorId) {
    auto it = OperatorIdToName_.find(operatorId);
    if (it != OperatorIdToName_.end()) {
        return it->second;
    }
    return MISSING_OPERATOR_NAME;
}

void TJobProcessor::SwapGenerations() {
    with_lock (Lock_) {
        CheckingRequestIdToJobFirstGen_.swap(CheckingRequestIdToJobSecondGen_);
        for (const auto& requestIdAndJob : CheckingRequestIdToJobSecondGen_) {
            const auto& job = requestIdAndJob.second;
            job->Counters->NRunningChecks.Dec();
            job->Counters->NExpiredChecks.Inc();
            ERROR_LOG << job->LogPrefix << "expired while waiting for the bus response" << Endl;
            FinishJob(job, job->GenerateLogRecord(TLogRecord::TCheckResult::TReqAnsLogRecord_TCheckResult_CR_RESPONSE_EXPIRED));
        }
        CheckingRequestIdToJobSecondGen_.clear();
    }
}

void TJobProcessor::InitializeCountersForJobType(const TString& typeName) {
    SpecificCounters_.GetOrCreate({typeName, MISSING_OPERATOR_NAME});
    for (const auto& operatorIdAndName : OperatorIdToName_) {
        SpecificCounters_.GetOrCreate({typeName, operatorIdAndName.second});
    }
}

void TJobProcessor::AddInternalJob(const TOfferWithRequestRef& offerWithRequest, const TString& typeName, TInstant receiptTime) {
    auto job = MakeIntrusive<TJob>(offerWithRequest->OfferId, typeName, receiptTime);
    INFO_LOG << job->LogPrefix << "adding" << Endl;

    job->OfferWithRequest = offerWithRequest;

    StartJob(job, InternalJobRequestClass_);
}

void TJobProcessor::AddExternalJob(const TOfferId& offerId, TMaybe<TInstant> cacheTimestamp,
                                   const TOnJobCompletion& onCompletionCallback, TInstant receiptTime)
{
    auto job = MakeIntrusive<TJob>(offerId, EXTERNAL_TYPE_NAME, receiptTime);
    job->OnCompletionCallback = onCompletionCallback;
    INFO_LOG << job->LogPrefix << "adding" << Endl;

    TDuration requestLag;
    if (cacheTimestamp) {
        requestLag = Now() - *cacheTimestamp;
    }

    PrepareExternalJob(job, requestLag, 1);
}

void TJobProcessor::PrepareExternalJob(TJobRef job, TDuration requestLag, size_t attempt) {
    job->OfferWithRequest = Cache_.GetOfferWithRequest(job->OfferId);
    if (job->OfferWithRequest) {
        StartJob(job, ExternalJobRequestClass_);
        return;
    }
    TString cacheMissType;
    TLogRecord::TCheckResult checkResult;
    if (requestLag >= AncientCacheMissLag_) {
        cacheMissType = "ancient";
        checkResult = TLogRecord::TCheckResult::TReqAnsLogRecord_TCheckResult_CR_ANCIENT_CACHE_MISS;
        GenericCounters_.NAncientCacheMisses.Inc();
    } else if (requestLag >= OldCacheMissLag_) {
        cacheMissType = "old";
        checkResult = TLogRecord::TCheckResult::TReqAnsLogRecord_TCheckResult_CR_OLD_CACHE_MISS;
        GenericCounters_.NOldCacheMisses.Inc();
    } else {
        if (attempt < FreshCacheMissRetryCount_) {
            WARNING_LOG << job->LogPrefix << "Fresh cache miss at attempt " << attempt
                        << ", requestLag " << requestLag
                        << ", will retry in " << FreshCacheMissRetryDelay_ << Endl;
            NTravel::TScheduler::Instance().Enqueue(FreshCacheMissRetryDelay_, [this, job, requestLag, attempt]{
                PrepareExternalJob(job, requestLag, attempt + 1);
            });
            return;
        }
        cacheMissType = "fresh";
        checkResult = TLogRecord::TCheckResult::TReqAnsLogRecord_TCheckResult_CR_FRESH_CACHE_MISS;
        GenericCounters_.NFreshCacheMisses.Inc();
    }
    ERROR_LOG << job->LogPrefix << cacheMissType << " cache miss, requestLag " << requestLag << Endl;
    FinishJob(job, job->GenerateLogRecord(checkResult, cacheMissType + " cache miss"));
}

void TJobProcessor::StartJob(const TJobRef& job, NTravelProto::ERequestClass requestClass) {
    INFO_LOG << job->LogPrefix << "starting" << Endl;
    job->StartTime = Now();
    job->Counters = SpecificCounters_.GetOrCreate(
            {job->TypeName, ConvertOperatorIdToName(job->OfferWithRequest->OperatorId)});
    job->Counters->NRunningChecks.Inc();

    auto requestId = CreateGuidAsString();
    NTravelProto::TSearchOffersRpcReq rpcReq;
    rpcReq.SetSync(false);
    auto subReq = rpcReq.AddSubrequest();
    *subReq = *job->OfferWithRequest->RequestPb;
    subReq->SetOfferCacheIgnoreBlacklist(true);
    subReq->SetId(requestId);
    subReq->SetRequestClass(requestClass);
    subReq->MutableAttribution()->SetOfferCacheClientId(OfferCacheClientId);
    subReq->SetOfferCacheUseCache(false);
    subReq->SetOfferCacheUseSearcher(true);

    OfferCacheClient_.Request<NTravelProto::TSearchOffersRpcReq, NTravelProto::TSearchOffersRpcRsp,
                              &NTravelProto::NOfferCacheGrpc::OfferCacheServiceV1::Stub::AsyncSearchOffers>
        (rpcReq, NGrpc::TClientMetadata(), [this, requestId, job](const TString& grpcError, const TString& remoteFQDN,
                                                                  const NTravelProto::TSearchOffersRpcRsp& resp)
        {
            TMaybe<TString> errorMessage;
            if (!grpcError.empty()) {
                errorMessage = grpcError;
            } else {
                INFO_LOG << job->LogPrefix << "completed a RPC request, remote FQDN is " << remoteFQDN << Endl;
                if (resp.SubresponseSize() != 1) {
                    errorMessage = "The number of subresponses equals " + ToString(resp.SubresponseSize()) +
                                   ", expected to be 1";
                } else {
                    const auto& subResp = resp.GetSubresponse(0);
                    if (subResp.HasError()) {
                        errorMessage = subResp.GetError().GetMessage();
                    }
                }
            }
            if (errorMessage) {
                job->Counters->NRunningChecks.Dec();
                job->Counters->NRpcErrors.Inc();
                ERROR_LOG << job->LogPrefix << "RPC error: " << *errorMessage << Endl;
                FinishJob(job, job->GenerateLogRecord(TLogRecord::TCheckResult::TReqAnsLogRecord_TCheckResult_CR_RPC_ERROR,
                                                      *errorMessage));
            } else {
                with_lock (Lock_) {
                    CheckingRequestIdToJobSecondGen_[requestId] = job;
                }
            }
        });

    job->Counters->NSentRpcRequests.Inc();
}

void TJobProcessor::FinishJob(const TJobRef& job, const TLogRecord& logRecord) {
    INFO_LOG << job->LogPrefix << "finishing" << Endl;
    ReqAnsLogger_.AddRecord(logRecord);
    if (job->OnCompletionCallback) {
        job->OnCompletionCallback();
    }
}

void TJobProcessor::MaybeProcessResponse(const TSearcherResponse& searcherResponse) {
    if (searcherResponse.Type == ESearcherResponseType::PLACEHOLDER) {
        return;
    }

    TJobRef job;
    with_lock (Lock_) {
        auto iter = CheckingRequestIdToJobSecondGen_.find(searcherResponse.RequestId);
        if (iter != CheckingRequestIdToJobSecondGen_.end()) {
            job = iter->second;
            CheckingRequestIdToJobSecondGen_.erase(iter);
        } else {
            iter = CheckingRequestIdToJobFirstGen_.find(searcherResponse.RequestId);
            if (iter != CheckingRequestIdToJobFirstGen_.end()) {
                job = iter->second;
                CheckingRequestIdToJobFirstGen_.erase(iter);
            } else {
                return;
            }
        }
    }

    if (searcherResponse.Type == ESearcherResponseType::ERROR) {
        job->Counters->NRunningChecks.Dec();
        job->Counters->NBusResponseErrors.Inc();
        ERROR_LOG << job->LogPrefix << "bus response error: " << searcherResponse.Error << Endl;
        FinishJob(job, job->GenerateLogRecord(TLogRecord::TCheckResult::TReqAnsLogRecord_TCheckResult_CR_RESPONSE_ERROR,
                                              searcherResponse.Error));
    } else {
        CheckDifference(job, searcherResponse.OffersWithRequests);
        job->Counters->NRunningChecks.Dec();
        job->Counters->NCompletedChecks.Inc();
    }
}

void TJobProcessor::CheckDifference(const TJobRef& job, const TVector<TOfferWithRequestRef>& newOffersWithRequests) {
    auto logRecord = job->GenerateLogRecord(TLogRecord::TCheckResult::TReqAnsLogRecord_TCheckResult_CR_COMPLETED_NOT_FOUND);

    const auto& oldOfferHash = job->OfferWithRequest->OfferHash;
    const auto& oldOfferPrice = job->OfferWithRequest->OfferPrice;
    const int Undefined = Max<int>();
    int minPriceDifference = Undefined;
    for (const auto& newOfferWithRequest : newOffersWithRequests) {
        auto logResponseOffer = logRecord.AddResponseOffers();
        logResponseOffer->SetId(OfferIdToString(newOfferWithRequest->OfferId));
        logResponseOffer->SetOperatorId(newOfferWithRequest->OperatorId);
        logResponseOffer->MutablePrice()->SetAmount(newOfferWithRequest->OfferPrice);
        if (newOfferWithRequest->OfferHash == oldOfferHash) {
            int candidatePriceDifference = newOfferWithRequest->OfferPrice - oldOfferPrice;
            if (abs(minPriceDifference) > abs(candidatePriceDifference)) {
                minPriceDifference = candidatePriceDifference;
                logRecord.MutableMatchedOffer()->SetId(OfferIdToString(newOfferWithRequest->OfferId));
                logRecord.MutableMatchedOffer()->SetOperatorId(newOfferWithRequest->OperatorId);
                logRecord.MutableMatchedOffer()->MutablePrice()->SetAmount(newOfferWithRequest->OfferPrice);
            }
        }
    }
    if (minPriceDifference == Undefined) {
        job->Counters->NPriceNotFound.Inc();
        INFO_LOG << job->LogPrefix << "completed, not found" << Endl;
        FinishJob(job, logRecord);
        return;
    }

    double relativePriceDifference = static_cast<double>(minPriceDifference) / oldOfferPrice;
    // Strict comparison
    if (minPriceDifference == 0) {
        job->Counters->NPriceIsSame.Inc();
    } else if (minPriceDifference > 0) {
        job->Counters->NPriceIsBigger.Inc();
        // Round towards +inf
        job->Counters->NPricePercentBigger.Update(static_cast<int>(ceil(100 * relativePriceDifference)));
    } else {
        job->Counters->NPriceIsSmaller.Inc();
        // Round towards -inf
        job->Counters->NPricePercentSmaller.Update(static_cast<int>(floor(100 * relativePriceDifference)));
    }
    // Tolerant comparison
    if (abs(relativePriceDifference) <= PRICE_REL_TOLERANCE) {
        job->Counters->NPriceIsApproxSame.Inc();
    } else if (relativePriceDifference > PRICE_REL_TOLERANCE) {
        job->Counters->NPriceIsApproxBigger.Inc();
    } else {
        job->Counters->NPriceIsApproxSmaller.Inc();
    }

    logRecord.SetCheckResult(TLogRecord::TCheckResult::TReqAnsLogRecord_TCheckResult_CR_COMPLETED_FOUND);
    logRecord.SetPriceDifference(minPriceDifference);
    logRecord.SetRelativePriceDifference(relativePriceDifference);

    INFO_LOG << job->LogPrefix << "completed, relative price difference is " <<
                FloatToString(relativePriceDifference, PREC_POINT_DIGITS, 2) << Endl;
    FinishJob(job, logRecord);
}

} // namespace NPriceChecker
} // namespace NTravel
